/*
 *  Notes - panel plugin for Xfce Desktop Environment
 *  Copyright (c) 2009  Mike Massonnet <mmassonnet@xfce.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <stdlib.h>
#include <string.h>
#include <xfconf/xfconf.h>
#include <color.h>
#include <glib/gi18n-lib.h>
#include <glib/gstdio.h>
#include <float.h>
#include <math.h>
#include <gdk/gdk.h>


#define XNP_TYPE_APPLICATION (xnp_application_get_type ())
#define XNP_APPLICATION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), XNP_TYPE_APPLICATION, XnpApplication))
#define XNP_APPLICATION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), XNP_TYPE_APPLICATION, XnpApplicationClass))
#define XNP_IS_APPLICATION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), XNP_TYPE_APPLICATION))
#define XNP_IS_APPLICATION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), XNP_TYPE_APPLICATION))
#define XNP_APPLICATION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), XNP_TYPE_APPLICATION, XnpApplicationClass))

typedef struct _XnpApplication XnpApplication;
typedef struct _XnpApplicationClass XnpApplicationClass;
typedef struct _XnpApplicationPrivate XnpApplicationPrivate;

#define XNP_TYPE_WINDOW (xnp_window_get_type ())
#define XNP_WINDOW(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), XNP_TYPE_WINDOW, XnpWindow))
#define XNP_WINDOW_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), XNP_TYPE_WINDOW, XnpWindowClass))
#define XNP_IS_WINDOW(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), XNP_TYPE_WINDOW))
#define XNP_IS_WINDOW_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), XNP_TYPE_WINDOW))
#define XNP_WINDOW_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), XNP_TYPE_WINDOW, XnpWindowClass))

typedef struct _XnpWindow XnpWindow;
typedef struct _XnpWindowClass XnpWindowClass;

#define XNP_TYPE_NOTE (xnp_note_get_type ())
#define XNP_NOTE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), XNP_TYPE_NOTE, XnpNote))
#define XNP_NOTE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), XNP_TYPE_NOTE, XnpNoteClass))
#define XNP_IS_NOTE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), XNP_TYPE_NOTE))
#define XNP_IS_NOTE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), XNP_TYPE_NOTE))
#define XNP_NOTE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), XNP_TYPE_NOTE, XnpNoteClass))

typedef struct _XnpNote XnpNote;
typedef struct _XnpNoteClass XnpNoteClass;
typedef struct _XnpNotePrivate XnpNotePrivate;

#define XNP_TYPE_HYPERTEXT_VIEW (xnp_hypertext_view_get_type ())
#define XNP_HYPERTEXT_VIEW(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), XNP_TYPE_HYPERTEXT_VIEW, XnpHypertextView))
#define XNP_HYPERTEXT_VIEW_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), XNP_TYPE_HYPERTEXT_VIEW, XnpHypertextViewClass))
#define XNP_IS_HYPERTEXT_VIEW(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), XNP_TYPE_HYPERTEXT_VIEW))
#define XNP_IS_HYPERTEXT_VIEW_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), XNP_TYPE_HYPERTEXT_VIEW))
#define XNP_HYPERTEXT_VIEW_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), XNP_TYPE_HYPERTEXT_VIEW, XnpHypertextViewClass))

typedef struct _XnpHypertextView XnpHypertextView;
typedef struct _XnpHypertextViewClass XnpHypertextViewClass;

struct _XnpApplication {
	GObject parent_instance;
	XnpApplicationPrivate * priv;
};

struct _XnpApplicationClass {
	GObjectClass parent_class;
};

struct _XnpApplicationPrivate {
	GSList* window_list;
	char* notes_path;
	char* _config_file;
	XfconfChannel* xfconf_channel;
};

struct _XnpNote {
	GtkBin parent_instance;
	XnpNotePrivate * priv;
	GtkScrolledWindow* scrolled_window;
	XnpHypertextView* text_view;
};

struct _XnpNoteClass {
	GtkBinClass parent_class;
};


static gpointer xnp_application_parent_class = NULL;

GType xnp_application_get_type (void);
GType xnp_window_get_type (void);
#define XNP_APPLICATION_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), XNP_TYPE_APPLICATION, XnpApplicationPrivate))
enum  {
	XNP_APPLICATION_DUMMY_PROPERTY,
	XNP_APPLICATION_CONFIG_FILE
};
static void _g_slist_free_g_object_unref (GSList* self);
static void xnp_application_set_config_file (XnpApplication* self, const char* value);
static void _lambda17_ (XfconfChannel* channel, const char* prop, const GValue* val, XnpApplication* self);
static void __lambda17__xfconf_channel_property_changed (XfconfChannel* _sender, const char* p0, const GValue* p1, gpointer self);
XnpWindow* xnp_application_create_window (XnpApplication* self, const char* name);
XnpApplication* xnp_application_new (const char* config_file);
XnpApplication* xnp_application_construct (GType object_type, const char* config_file);
XnpWindow* xnp_window_new (void);
XnpWindow* xnp_window_construct (GType object_type);
void xnp_window_set_above (XnpWindow* self, gboolean value);
void xnp_window_set_sticky (XnpWindow* self, gboolean value);
static gboolean xnp_application_window_name_exists (XnpApplication* self, const char* name);
void xnp_window_set_name (XnpWindow* self, const char* value);
gint xnp_window_compare_func (XnpWindow* self, XnpWindow* win2);
void xnp_window_set_window_list (XnpWindow* self, GSList* list);
GType xnp_note_get_type (void);
XnpNote* xnp_window_insert_note (XnpWindow* self);
GType xnp_hypertext_view_get_type (void);
const char* xnp_window_get_name (XnpWindow* self);
const char* xnp_note_get_name (XnpNote* self);
static void xnp_application_load_window_data (XnpApplication* self, XnpWindow* window);
static void xnp_application_rename_window (XnpApplication* self, XnpWindow* window);
static void xnp_application_delete_window (XnpApplication* self, XnpWindow* window);
void xnp_application_open_settings_dialog (XnpApplication* self);
static void _lambda18_ (XnpWindow* win, const char* action, XnpApplication* self);
static void __lambda18__xnp_window_action (XnpWindow* _sender, const char* action, gpointer self);
static void xnp_application_save_note (XnpApplication* self, XnpWindow* window, XnpNote* note);
static void _lambda19_ (XnpWindow* win, XnpNote* note, XnpApplication* self);
static void __lambda19__xnp_window_save_data (XnpWindow* _sender, XnpNote* note, gpointer self);
static void _lambda20_ (XnpWindow* win, XnpNote* note, XnpApplication* self);
static void __lambda20__xnp_window_note_inserted (XnpWindow* _sender, XnpNote* note, gpointer self);
static void _lambda21_ (XnpWindow* win, XnpNote* note, XnpApplication* self);
static void __lambda21__xnp_window_note_deleted (XnpWindow* _sender, XnpNote* note, gpointer self);
static gboolean xnp_application_name_is_valid (XnpApplication* self, const char* name);
void xnp_note_set_name (XnpNote* self, const char* value);
static void _lambda22_ (XnpWindow* win, XnpNote* note, const char* old_name, XnpApplication* self);
static void __lambda22__xnp_window_note_renamed (XnpWindow* _sender, XnpNote* note, const char* old_name, gpointer self);
const char* xnp_application_get_config_file (XnpApplication* self);
void xnp_window_move_note (XnpWindow* self, const char* note_name, gint page);
void xnp_window_set_current_page (XnpWindow* self, gint page);
void xnp_window_get_geometry (XnpWindow* self, gint* winx, gint* winy, gint* width, gint* height);
char** xnp_window_get_note_names (XnpWindow* self, int* result_length1);
gint xnp_window_get_current_page (XnpWindow* self);
gboolean xnp_window_get_above (XnpWindow* self);
gboolean xnp_window_get_sticky (XnpWindow* self);
void xnp_application_save_windows_configuration (XnpApplication* self);
void xnp_window_save_notes (XnpWindow* self);
void xnp_application_save_notes (XnpApplication* self);
gint xnp_window_get_n_pages (XnpWindow* self);
void xnp_window_hide (XnpWindow* self);
void xnp_application_show_hide_notes (XnpApplication* self);
static GObject * xnp_application_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static void xnp_application_finalize (GObject* obj);
static void xnp_application_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void xnp_application_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



static void _g_slist_free_g_object_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) g_object_unref, NULL);
	g_slist_free (self);
}


static void _lambda17_ (XfconfChannel* channel, const char* prop, const GValue* val, XnpApplication* self) {
	g_return_if_fail (channel != NULL);
	g_return_if_fail (prop != NULL);
	if (_vala_strcmp0 (prop, "/global/background-color") == 0) {
		color_set_background (g_value_get_string (&(*val)));
	}
}


static void __lambda17__xfconf_channel_property_changed (XfconfChannel* _sender, const char* p0, const GValue* p1, gpointer self) {
	_lambda17_ (_sender, p0, p1, self);
}


XnpApplication* xnp_application_construct (GType object_type, const char* config_file) {
	GError * _inner_error_;
	GParameter * __params;
	GParameter * __params_it;
	XnpApplication * self;
	XfconfChannel* _tmp0_;
	char* color;
	char* name;
	gboolean found;
	g_return_val_if_fail (config_file != NULL, NULL);
	_inner_error_ = NULL;
	__params = g_new0 (GParameter, 1);
	__params_it = __params;
	__params_it->name = "config-file";
	g_value_init (&__params_it->value, G_TYPE_STRING);
	g_value_set_string (&__params_it->value, config_file);
	__params_it++;
	self = g_object_newv (object_type, __params_it - __params, __params);
	_tmp0_ = NULL;
	self->priv->xfconf_channel = (_tmp0_ = xfconf_channel_new_with_property_base ("xfce4-panel", "/plugins/notes"), (self->priv->xfconf_channel == NULL) ? NULL : (self->priv->xfconf_channel = (g_object_unref (self->priv->xfconf_channel), NULL)), _tmp0_);
	color = xfconf_channel_get_string (self->priv->xfconf_channel, "/global/background-color", "#F2F1EF");
	color_set_background (color);
	g_signal_connect (self->priv->xfconf_channel, "property-changed", (GCallback) __lambda17__xfconf_channel_property_changed, self);
	name = NULL;
	found = FALSE;
	{
		GDir* dir;
		dir = g_dir_open (self->priv->notes_path, (guint) 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch4_g_error;
			goto __finally4;
		}
		while (TRUE) {
			char* _tmp2_;
			const char* _tmp1_;
			XnpWindow* _tmp3_;
			_tmp2_ = NULL;
			_tmp1_ = NULL;
			if (!((name = (_tmp2_ = (_tmp1_ = g_dir_read_name (dir), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), name = (g_free (name), NULL), _tmp2_)) != NULL)) {
				break;
			}
			_tmp3_ = NULL;
			_tmp3_ = xnp_application_create_window (self, name);
			(_tmp3_ == NULL) ? NULL : (_tmp3_ = (g_object_unref (_tmp3_), NULL));
			found = TRUE;
		}
		(dir == NULL) ? NULL : (dir = (g_dir_close (dir), NULL));
	}
	goto __finally4;
	__catch4_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_mkdir_with_parents (self->priv->notes_path, 0700);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally4:
	if (_inner_error_ != NULL) {
		color = (g_free (color), NULL);
		name = (g_free (name), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return NULL;
	}
	if (found == FALSE) {
		XnpWindow* _tmp4_;
		_tmp4_ = NULL;
		_tmp4_ = xnp_application_create_window (self, NULL);
		(_tmp4_ == NULL) ? NULL : (_tmp4_ = (g_object_unref (_tmp4_), NULL));
	}
	color = (g_free (color), NULL);
	name = (g_free (name), NULL);
	while (__params_it > __params) {
		--__params_it;
		g_value_unset (&__params_it->value);
	}
	g_free (__params);
	return self;
}


XnpApplication* xnp_application_new (const char* config_file) {
	return xnp_application_construct (XNP_TYPE_APPLICATION, config_file);
}


static void _lambda18_ (XnpWindow* win, const char* action, XnpApplication* self) {
	g_return_if_fail (win != NULL);
	g_return_if_fail (action != NULL);
	if (_vala_strcmp0 (action, "rename") == 0) {
		xnp_application_rename_window (self, win);
	} else {
		if (_vala_strcmp0 (action, "delete") == 0) {
			xnp_application_delete_window (self, win);
		} else {
			if (_vala_strcmp0 (action, "create-new-window") == 0) {
				XnpWindow* new_win;
				new_win = xnp_application_create_window (self, NULL);
				gtk_widget_show ((GtkWidget*) new_win);
				(new_win == NULL) ? NULL : (new_win = (g_object_unref (new_win), NULL));
			} else {
				if (_vala_strcmp0 (action, "properties") == 0) {
					xnp_application_open_settings_dialog (self);
				}
			}
		}
	}
}


static void __lambda18__xnp_window_action (XnpWindow* _sender, const char* action, gpointer self) {
	_lambda18_ (_sender, action, self);
}


static void _lambda19_ (XnpWindow* win, XnpNote* note, XnpApplication* self) {
	g_return_if_fail (win != NULL);
	g_return_if_fail (note != NULL);
	xnp_application_save_note (self, win, note);
}


static void __lambda19__xnp_window_save_data (XnpWindow* _sender, XnpNote* note, gpointer self) {
	_lambda19_ (_sender, note, self);
}


static void _lambda20_ (XnpWindow* win, XnpNote* note, XnpApplication* self) {
	GError * _inner_error_;
	char* path;
	g_return_if_fail (win != NULL);
	g_return_if_fail (note != NULL);
	_inner_error_ = NULL;
	xfconf_g_property_bind (self->priv->xfconf_channel, "/global/font-description", G_TYPE_STRING, note->text_view, "font");
	path = g_strdup_printf ("%s/%s/%s", self->priv->notes_path, xnp_window_get_name (win), xnp_note_get_name (note));
	{
		g_file_set_contents (path, "", (glong) (-1), &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch6_g_file_error;
			}
			goto __finally6;
		}
	}
	goto __finally6;
	__catch6_g_file_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally6:
	if (_inner_error_ != NULL) {
		path = (g_free (path), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	path = (g_free (path), NULL);
}


static void __lambda20__xnp_window_note_inserted (XnpWindow* _sender, XnpNote* note, gpointer self) {
	_lambda20_ (_sender, note, self);
}


static void _lambda21_ (XnpWindow* win, XnpNote* note, XnpApplication* self) {
	char* path;
	g_return_if_fail (win != NULL);
	g_return_if_fail (note != NULL);
	path = g_strdup_printf ("%s/%s/%s", self->priv->notes_path, xnp_window_get_name (win), xnp_note_get_name (note));
	g_unlink (path);
	path = (g_free (path), NULL);
}


static void __lambda21__xnp_window_note_deleted (XnpWindow* _sender, XnpNote* note, gpointer self) {
	_lambda21_ (_sender, note, self);
}


static void _lambda22_ (XnpWindow* win, XnpNote* note, const char* old_name, XnpApplication* self) {
	char* old_path;
	char* new_path;
	g_return_if_fail (win != NULL);
	g_return_if_fail (note != NULL);
	g_return_if_fail (old_name != NULL);
	if (!xnp_application_name_is_valid (self, xnp_note_get_name (note))) {
		xnp_note_set_name (note, old_name);
		return;
	}
	old_path = g_strdup_printf ("%s/%s/%s", self->priv->notes_path, xnp_window_get_name (win), old_name);
	new_path = g_strdup_printf ("%s/%s/%s", self->priv->notes_path, xnp_window_get_name (win), xnp_note_get_name (note));
	g_rename (old_path, new_path);
	old_path = (g_free (old_path), NULL);
	new_path = (g_free (new_path), NULL);
}


static void __lambda22__xnp_window_note_renamed (XnpWindow* _sender, XnpNote* note, const char* old_name, gpointer self) {
	_lambda22_ (_sender, note, old_name, self);
}


/*
 * Window management
 
*
 * create_window:
 *
 * Creates a new Xnp.Window and stores it inside window_list.
 * If a name is given, it assumes it can load existing notes.
 */
XnpWindow* xnp_application_create_window (XnpApplication* self, const char* name) {
	XnpWindow* result;
	GError * _inner_error_;
	XnpWindow* window;
	XnpWindow* _tmp4_;
	g_return_val_if_fail (self != NULL, NULL);
	_inner_error_ = NULL;
	window = g_object_ref_sink (xnp_window_new ());
	/* Global settings */
	xfconf_g_property_bind (self->priv->xfconf_channel, "/global/skip-taskbar-hint", G_TYPE_BOOLEAN, window, "skip-taskbar-hint");
	xfconf_g_property_bind (self->priv->xfconf_channel, "/global/show-tabs", G_TYPE_BOOLEAN, window, "show-tabs");
	/* Default settings */
	if (name == NULL) {
		gint width;
		gint height;
		gboolean _tmp0_;
		xnp_window_set_above (window, xfconf_channel_get_bool (self->priv->xfconf_channel, "/new-window/always-on-top", FALSE));
		xnp_window_set_sticky (window, xfconf_channel_get_bool (self->priv->xfconf_channel, "/new-window/sticky", TRUE));
		width = (gint) xfconf_channel_get_int (self->priv->xfconf_channel, "/new-window/width", (gint32) 0);
		height = (gint) xfconf_channel_get_int (self->priv->xfconf_channel, "/new-window/height", (gint32) 0);
		_tmp0_ = FALSE;
		if (width > 0) {
			_tmp0_ = height > 0;
		} else {
			_tmp0_ = FALSE;
		}
		if (_tmp0_) {
			gtk_window_resize ((GtkWindow*) window, width, height);
		}
	}
	/* Set window name */
	if (name == NULL) {
		const char* _tmp1_;
		char* window_name;
		gint len;
		_tmp1_ = NULL;
		window_name = (_tmp1_ = _ ("Notes"), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
		len = (gint) g_slist_length (self->priv->window_list);
		{
			gint id;
			id = 1;
			{
				gboolean _tmp2_;
				_tmp2_ = TRUE;
				while (TRUE) {
					if (!_tmp2_) {
						id++;
					}
					_tmp2_ = FALSE;
					if (!(id <= (len + 1))) {
						break;
					}
					if (id > 1) {
						char* _tmp3_;
						_tmp3_ = NULL;
						window_name = (_tmp3_ = g_strdup_printf (_ ("Notes %d"), id), window_name = (g_free (window_name), NULL), _tmp3_);
					}
					if (!xnp_application_window_name_exists (self, window_name)) {
						break;
					}
				}
			}
		}
		xnp_window_set_name (window, window_name);
		window_name = (g_free (window_name), NULL);
	} else {
		xnp_window_set_name (window, name);
	}
	/* Add to window_list */
	_tmp4_ = NULL;
	self->priv->window_list = g_slist_insert_sorted (self->priv->window_list, (_tmp4_ = window, (_tmp4_ == NULL) ? NULL : g_object_ref (_tmp4_)), (GCompareFunc) xnp_window_compare_func);
	{
		GSList* win_collection;
		GSList* win_it;
		win_collection = self->priv->window_list;
		for (win_it = win_collection; win_it != NULL; win_it = win_it->next) {
			XnpWindow* _tmp5_;
			XnpWindow* win;
			_tmp5_ = NULL;
			win = (_tmp5_ = (XnpWindow*) win_it->data, (_tmp5_ == NULL) ? NULL : g_object_ref (_tmp5_));
			{
				xnp_window_set_window_list (win, self->priv->window_list);
				(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
			}
		}
	}
	/* Insert initial notes */
	if (name == NULL) {
		XnpNote* note;
		char* window_path;
		note = xnp_window_insert_note (window);
		xfconf_g_property_bind (self->priv->xfconf_channel, "/global/font-description", G_TYPE_STRING, note->text_view, "font");
		window_path = g_strdup_printf ("%s/%s", self->priv->notes_path, xnp_window_get_name (window));
		g_mkdir_with_parents (window_path, 0700);
		{
			char* note_path;
			note_path = g_strdup_printf ("%s/%s", window_path, xnp_note_get_name (note));
			g_file_set_contents (note_path, "", (glong) (-1), &_inner_error_);
			if (_inner_error_ != NULL) {
				note_path = (g_free (note_path), NULL);
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch5_g_file_error;
				}
				goto __finally5;
			}
			note_path = (g_free (note_path), NULL);
		}
		goto __finally5;
		__catch5_g_file_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally5:
		if (_inner_error_ != NULL) {
			(note == NULL) ? NULL : (note = (g_object_unref (note), NULL));
			window_path = (g_free (window_path), NULL);
			(window == NULL) ? NULL : (window = (g_object_unref (window), NULL));
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return NULL;
		}
		(note == NULL) ? NULL : (note = (g_object_unref (note), NULL));
		window_path = (g_free (window_path), NULL);
	} else {
		xnp_application_load_window_data (self, window);
	}
	g_signal_connect (window, "action", (GCallback) __lambda18__xnp_window_action, self);
	g_signal_connect (window, "save-data", (GCallback) __lambda19__xnp_window_save_data, self);
	g_signal_connect (window, "note-inserted", (GCallback) __lambda20__xnp_window_note_inserted, self);
	g_signal_connect (window, "note-deleted", (GCallback) __lambda21__xnp_window_note_deleted, self);
	g_signal_connect (window, "note-renamed", (GCallback) __lambda22__xnp_window_note_renamed, self);
	result = window;
	return result;
}


/**
 * load_window_data:
 *
 * Load existing notes and configuration inside the window.
 */
static void xnp_application_load_window_data (XnpApplication* self, XnpWindow* window) {
	GError * _inner_error_;
	char* name;
	char* path;
	GKeyFile* keyfile;
	g_return_if_fail (self != NULL);
	g_return_if_fail (window != NULL);
	_inner_error_ = NULL;
	/* Load notes */
	name = NULL;
	path = g_strdup_printf ("%s/%s", self->priv->notes_path, xnp_window_get_name (window));
	{
		GDir* dir;
		dir = g_dir_open (path, (guint) 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch7_g_file_error;
			}
			goto __finally7;
		}
		while (TRUE) {
			char* _tmp1_;
			const char* _tmp0_;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			if (!((name = (_tmp1_ = (_tmp0_ = g_dir_read_name (dir), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), name = (g_free (name), NULL), _tmp1_)) != NULL)) {
				break;
			}
			{
				char* contents;
				char* filename;
				char* _tmp4_;
				gboolean _tmp3_;
				char* _tmp2_;
				XnpNote* note;
				GtkTextBuffer* _tmp5_;
				GtkTextBuffer* buffer;
				contents = NULL;
				filename = g_strdup_printf ("%s/%s", path, name);
				_tmp4_ = NULL;
				_tmp2_ = NULL;
				_tmp3_ = g_file_get_contents (filename, &_tmp2_, NULL, &_inner_error_);
				contents = (_tmp4_ = _tmp2_, contents = (g_free (contents), NULL), _tmp4_);
				_tmp3_;
				if (_inner_error_ != NULL) {
					contents = (g_free (contents), NULL);
					filename = (g_free (filename), NULL);
					if (_inner_error_->domain == G_FILE_ERROR) {
						goto __catch8_g_file_error;
					}
					goto __finally8;
				}
				note = xnp_window_insert_note (window);
				xnp_note_set_name (note, name);
				_tmp5_ = NULL;
				buffer = (_tmp5_ = gtk_text_view_get_buffer ((GtkTextView*) note->text_view), (_tmp5_ == NULL) ? NULL : g_object_ref (_tmp5_));
				gtk_text_buffer_set_text (buffer, contents, -1);
				xfconf_g_property_bind (self->priv->xfconf_channel, "/global/font-description", G_TYPE_STRING, note->text_view, "font");
				contents = (g_free (contents), NULL);
				filename = (g_free (filename), NULL);
				(note == NULL) ? NULL : (note = (g_object_unref (note), NULL));
				(buffer == NULL) ? NULL : (buffer = (g_object_unref (buffer), NULL));
			}
			goto __finally8;
			__catch8_g_file_error:
			{
				GError * e;
				e = _inner_error_;
				_inner_error_ = NULL;
				{
					g_warning ("application.vala:215: %s", e->message);
					(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
				}
			}
			__finally8:
			if (_inner_error_ != NULL) {
				(dir == NULL) ? NULL : (dir = (g_dir_close (dir), NULL));
				if (_inner_error_->domain == G_FILE_ERROR) {
					goto __catch7_g_file_error;
				}
				goto __finally7;
			}
		}
		(dir == NULL) ? NULL : (dir = (g_dir_close (dir), NULL));
	}
	goto __finally7;
	__catch7_g_file_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally7:
	if (_inner_error_ != NULL) {
		name = (g_free (name), NULL);
		path = (g_free (path), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	/* Load window configuration */
	keyfile = g_key_file_new ();
	{
		gint winx;
		gint winy;
		gint width;
		gint height;
		char** _tmp7_;
		gint tabs_order_size;
		gint tabs_order_length1;
		gint _tmp6_;
		char** tabs_order;
		gint last_page;
		gboolean above;
		gboolean sticky;
		gint _tmp8_;
		double opacity;
		gboolean visible;
		g_key_file_load_from_file (keyfile, self->priv->_config_file, G_KEY_FILE_NONE, &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch9_g_error;
			goto __finally9;
		}
		winx = g_key_file_get_integer (keyfile, xnp_window_get_name (window), "PosX", &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch9_g_error;
			goto __finally9;
		}
		winy = g_key_file_get_integer (keyfile, xnp_window_get_name (window), "PosY", &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch9_g_error;
			goto __finally9;
		}
		width = g_key_file_get_integer (keyfile, xnp_window_get_name (window), "Width", &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch9_g_error;
			goto __finally9;
		}
		height = g_key_file_get_integer (keyfile, xnp_window_get_name (window), "Height", &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch9_g_error;
			goto __finally9;
		}
		_tmp7_ = NULL;
		tabs_order = (_tmp7_ = g_key_file_get_string_list (keyfile, xnp_window_get_name (window), "TabsOrder", &_tmp6_, &_inner_error_), tabs_order_length1 = _tmp6_, tabs_order_size = tabs_order_length1, _tmp7_);
		if (_inner_error_ != NULL) {
			goto __catch9_g_error;
			goto __finally9;
		}
		last_page = g_key_file_get_integer (keyfile, xnp_window_get_name (window), "LastTab", &_inner_error_);
		if (_inner_error_ != NULL) {
			tabs_order = (_vala_array_free (tabs_order, tabs_order_length1, (GDestroyNotify) g_free), NULL);
			goto __catch9_g_error;
			goto __finally9;
		}
		above = g_key_file_get_boolean (keyfile, xnp_window_get_name (window), "Above", &_inner_error_);
		if (_inner_error_ != NULL) {
			tabs_order = (_vala_array_free (tabs_order, tabs_order_length1, (GDestroyNotify) g_free), NULL);
			goto __catch9_g_error;
			goto __finally9;
		}
		sticky = g_key_file_get_boolean (keyfile, xnp_window_get_name (window), "Sticky", &_inner_error_);
		if (_inner_error_ != NULL) {
			tabs_order = (_vala_array_free (tabs_order, tabs_order_length1, (GDestroyNotify) g_free), NULL);
			goto __catch9_g_error;
			goto __finally9;
		}
		_tmp8_ = g_key_file_get_integer (keyfile, xnp_window_get_name (window), "Transparency", &_inner_error_);
		if (_inner_error_ != NULL) {
			tabs_order = (_vala_array_free (tabs_order, tabs_order_length1, (GDestroyNotify) g_free), NULL);
			goto __catch9_g_error;
			goto __finally9;
		}
		opacity = 1 - (((double) _tmp8_) / 100);
		visible = g_key_file_get_boolean (keyfile, xnp_window_get_name (window), "Visible", &_inner_error_);
		if (_inner_error_ != NULL) {
			tabs_order = (_vala_array_free (tabs_order, tabs_order_length1, (GDestroyNotify) g_free), NULL);
			goto __catch9_g_error;
			goto __finally9;
		}
		gtk_window_move ((GtkWindow*) window, winx, winy);
		gtk_window_resize ((GtkWindow*) window, width, height);
		{
			gint i;
			i = 0;
			{
				gboolean _tmp9_;
				_tmp9_ = TRUE;
				while (TRUE) {
					if (!_tmp9_) {
						i++;
					}
					_tmp9_ = FALSE;
					if (!(i < tabs_order_length1)) {
						break;
					}
					xnp_window_move_note (window, tabs_order[i], i);
				}
			}
		}
		xnp_window_set_current_page (window, last_page);
		xnp_window_set_above (window, above);
		xnp_window_set_sticky (window, sticky);
		gtk_window_set_opacity ((GtkWindow*) window, opacity);
		if (visible) {
			gtk_widget_show ((GtkWidget*) window);
		}
		tabs_order = (_vala_array_free (tabs_order, tabs_order_length1, (GDestroyNotify) g_free), NULL);
	}
	goto __finally9;
	__catch9_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_warning ("application.vala:249: Unable to load window configuration from %s: %s", self->priv->_config_file, e->message);
			gtk_widget_show ((GtkWidget*) window);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally9:
	if (_inner_error_ != NULL) {
		name = (g_free (name), NULL);
		path = (g_free (path), NULL);
		(keyfile == NULL) ? NULL : (keyfile = (g_key_file_free (keyfile), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	name = (g_free (name), NULL);
	path = (g_free (path), NULL);
	(keyfile == NULL) ? NULL : (keyfile = (g_key_file_free (keyfile), NULL));
}


/**
 * save_windows_configuration:
 *
 * Save window configuration inside rc file.
 */
void xnp_application_save_windows_configuration (XnpApplication* self) {
	GError * _inner_error_;
	GKeyFile* keyfile;
	g_return_if_fail (self != NULL);
	_inner_error_ = NULL;
	keyfile = g_key_file_new ();
	{
		char* contents;
		{
			GSList* win_collection;
			GSList* win_it;
			win_collection = self->priv->window_list;
			for (win_it = win_collection; win_it != NULL; win_it = win_it->next) {
				XnpWindow* _tmp2_;
				XnpWindow* win;
				_tmp2_ = NULL;
				win = (_tmp2_ = (XnpWindow*) win_it->data, (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_));
				{
					gint winx;
					gint winy;
					gint width;
					gint height;
					char** _tmp1_;
					gint tabs_order_size;
					gint tabs_order_length1;
					gint _tmp0_;
					char** tabs_order;
					gint last_page;
					gint transparency;
					gboolean visible;
					winx = 0;
					winy = 0;
					width = 0;
					height = 0;
					xnp_window_get_geometry (win, &winx, &winy, &width, &height);
					_tmp1_ = NULL;
					tabs_order = (_tmp1_ = xnp_window_get_note_names (win, &_tmp0_), tabs_order_length1 = _tmp0_, tabs_order_size = tabs_order_length1, _tmp1_);
					last_page = xnp_window_get_current_page (win);
					transparency = (gint) ((1 - gtk_window_get_opacity ((GtkWindow*) win)) * 100);
					visible = (gboolean) (GTK_WIDGET_FLAGS ((GtkWidget*) win) & GTK_VISIBLE);
					g_key_file_set_integer (keyfile, xnp_window_get_name (win), "PosX", winx);
					g_key_file_set_integer (keyfile, xnp_window_get_name (win), "PosY", winy);
					g_key_file_set_integer (keyfile, xnp_window_get_name (win), "Width", width);
					g_key_file_set_integer (keyfile, xnp_window_get_name (win), "Height", height);
					g_key_file_set_string_list (keyfile, xnp_window_get_name (win), "TabsOrder", tabs_order, tabs_order_length1);
					g_key_file_set_integer (keyfile, xnp_window_get_name (win), "LastTab", last_page);
					g_key_file_set_boolean (keyfile, xnp_window_get_name (win), "Above", xnp_window_get_above (win));
					g_key_file_set_boolean (keyfile, xnp_window_get_name (win), "Sticky", xnp_window_get_sticky (win));
					g_key_file_set_double (keyfile, xnp_window_get_name (win), "Transparency", (double) transparency);
					g_key_file_set_boolean (keyfile, xnp_window_get_name (win), "Visible", visible);
					(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
					tabs_order = (_vala_array_free (tabs_order, tabs_order_length1, (GDestroyNotify) g_free), NULL);
				}
			}
		}
		contents = g_key_file_to_data (keyfile, NULL, NULL);
		g_file_set_contents (self->priv->_config_file, contents, -1, &_inner_error_);
		if (_inner_error_ != NULL) {
			contents = (g_free (contents), NULL);
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch10_g_file_error;
			}
			goto __finally10;
		}
		contents = (g_free (contents), NULL);
	}
	goto __finally10;
	__catch10_g_file_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally10:
	if (_inner_error_ != NULL) {
		(keyfile == NULL) ? NULL : (keyfile = (g_key_file_free (keyfile), NULL));
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	(keyfile == NULL) ? NULL : (keyfile = (g_key_file_free (keyfile), NULL));
}


/**
 * save_notes:
 *
 * Save the contents of every existing notes.
 */
void xnp_application_save_notes (XnpApplication* self) {
	g_return_if_fail (self != NULL);
	{
		GSList* win_collection;
		GSList* win_it;
		win_collection = self->priv->window_list;
		for (win_it = win_collection; win_it != NULL; win_it = win_it->next) {
			XnpWindow* _tmp0_;
			XnpWindow* win;
			_tmp0_ = NULL;
			win = (_tmp0_ = (XnpWindow*) win_it->data, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			{
				xnp_window_save_notes (win);
				(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
			}
		}
	}
}


/**
 * save_note:
 *
 * Save the contents of the given note.
 */
static void xnp_application_save_note (XnpApplication* self, XnpWindow* window, XnpNote* note) {
	GError * _inner_error_;
	char* path;
	g_return_if_fail (self != NULL);
	g_return_if_fail (window != NULL);
	g_return_if_fail (note != NULL);
	_inner_error_ = NULL;
	path = g_strdup_printf ("%s/%s/%s", self->priv->notes_path, xnp_window_get_name (window), xnp_note_get_name (note));
	{
		GtkTextIter start = {0};
		GtkTextIter end = {0};
		GtkTextBuffer* _tmp0_;
		GtkTextBuffer* buffer;
		const char* _tmp1_;
		char* contents;
		_tmp0_ = NULL;
		buffer = (_tmp0_ = gtk_text_view_get_buffer ((GtkTextView*) note->text_view), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
		gtk_text_buffer_get_bounds (buffer, &start, &end);
		_tmp1_ = NULL;
		contents = (_tmp1_ = gtk_text_buffer_get_text (buffer, &start, &end, TRUE), (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_));
		g_file_set_contents (path, contents, (glong) (-1), &_inner_error_);
		if (_inner_error_ != NULL) {
			(buffer == NULL) ? NULL : (buffer = (g_object_unref (buffer), NULL));
			contents = (g_free (contents), NULL);
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch11_g_file_error;
			}
			goto __finally11;
		}
		(buffer == NULL) ? NULL : (buffer = (g_object_unref (buffer), NULL));
		contents = (g_free (contents), NULL);
	}
	goto __finally11;
	__catch11_g_file_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			g_warning ("application.vala:314: %s", e->message);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally11:
	if (_inner_error_ != NULL) {
		path = (g_free (path), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	path = (g_free (path), NULL);
}


/**
 * rename_window:
 *
 * Renames the window name.
 */
static void xnp_application_rename_window (XnpApplication* self, XnpWindow* window) {
	GtkDialog* dialog;
	GtkEntry* entry;
	gint res;
	g_return_if_fail (self != NULL);
	g_return_if_fail (window != NULL);
	dialog = g_object_ref_sink ((GtkDialog*) gtk_dialog_new_with_buttons (_ ("Rename group"), (GtkWindow*) window, GTK_DIALOG_DESTROY_WITH_PARENT | GTK_DIALOG_NO_SEPARATOR, GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL, GTK_STOCK_OK, GTK_RESPONSE_OK, NULL));
	gtk_dialog_set_default_response (dialog, (gint) GTK_RESPONSE_OK);
	gtk_window_set_resizable ((GtkWindow*) dialog, FALSE);
	gtk_window_set_icon_name ((GtkWindow*) dialog, GTK_STOCK_EDIT);
	gtk_container_set_border_width ((GtkContainer*) dialog, (guint) 4);
	gtk_container_set_border_width ((GtkContainer*) dialog->vbox, (guint) 6);
	entry = g_object_ref_sink ((GtkEntry*) gtk_entry_new ());
	gtk_entry_set_text (entry, xnp_window_get_name (window));
	gtk_entry_set_activates_default (entry, TRUE);
	gtk_container_add ((GtkContainer*) dialog->vbox, (GtkWidget*) entry);
	gtk_widget_show_all ((GtkWidget*) dialog->vbox);
	res = gtk_dialog_run (dialog);
	gtk_widget_hide ((GtkWidget*) dialog);
	if (res == GTK_RESPONSE_OK) {
		const char* name;
		name = gtk_entry_get_text (entry);
		if (xnp_application_window_name_exists (self, name)) {
			GtkMessageDialog* error_dialog;
			error_dialog = g_object_ref_sink ((GtkMessageDialog*) gtk_message_dialog_new ((GtkWindow*) window, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, _ ("The name %s is already in use"), name, NULL));
			gtk_dialog_run ((GtkDialog*) error_dialog);
			gtk_object_destroy ((GtkObject*) error_dialog);
			(error_dialog == NULL) ? NULL : (error_dialog = (g_object_unref (error_dialog), NULL));
		} else {
			char* old_path;
			char* new_path;
			if (!xnp_application_name_is_valid (self, name)) {
				(dialog == NULL) ? NULL : (dialog = (g_object_unref (dialog), NULL));
				(entry == NULL) ? NULL : (entry = (g_object_unref (entry), NULL));
				return;
			}
			old_path = g_strdup_printf ("%s/%s", self->priv->notes_path, xnp_window_get_name (window));
			new_path = g_strdup_printf ("%s/%s", self->priv->notes_path, name);
			xnp_window_set_name (window, name);
			g_rename (old_path, new_path);
			self->priv->window_list = g_slist_sort (self->priv->window_list, (GCompareFunc) xnp_window_compare_func);
			old_path = (g_free (old_path), NULL);
			new_path = (g_free (new_path), NULL);
		}
	}
	gtk_object_destroy ((GtkObject*) dialog);
	(dialog == NULL) ? NULL : (dialog = (g_object_unref (dialog), NULL));
	(entry == NULL) ? NULL : (entry = (g_object_unref (entry), NULL));
}


/**
 * delete_window:
 *
 * Delete the window.
 */
static void xnp_application_delete_window (XnpApplication* self, XnpWindow* window) {
	GError * _inner_error_;
	char* name;
	char* path;
	g_return_if_fail (self != NULL);
	g_return_if_fail (window != NULL);
	_inner_error_ = NULL;
	if (xnp_window_get_n_pages (window) >= 1) {
		GtkMessageDialog* dialog;
		gint res;
		dialog = g_object_ref_sink ((GtkMessageDialog*) gtk_message_dialog_new ((GtkWindow*) window, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_QUESTION, GTK_BUTTONS_YES_NO, _ ("Are you sure you want to delete this group?"), NULL));
		res = gtk_dialog_run ((GtkDialog*) dialog);
		gtk_object_destroy ((GtkObject*) dialog);
		if (res != GTK_RESPONSE_YES) {
			(dialog == NULL) ? NULL : (dialog = (g_object_unref (dialog), NULL));
			return;
		}
		(dialog == NULL) ? NULL : (dialog = (g_object_unref (dialog), NULL));
	}
	name = NULL;
	path = g_strdup_printf ("%s/%s", self->priv->notes_path, xnp_window_get_name (window));
	{
		GDir* dir;
		dir = g_dir_open (path, (guint) 0, &_inner_error_);
		if (_inner_error_ != NULL) {
			if (_inner_error_->domain == G_FILE_ERROR) {
				goto __catch12_g_file_error;
			}
			goto __finally12;
		}
		while (TRUE) {
			char* _tmp1_;
			const char* _tmp0_;
			char* filename;
			_tmp1_ = NULL;
			_tmp0_ = NULL;
			if (!((name = (_tmp1_ = (_tmp0_ = g_dir_read_name (dir), (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), name = (g_free (name), NULL), _tmp1_)) != NULL)) {
				break;
			}
			filename = g_strdup_printf ("%s/%s", path, name);
			g_unlink (filename);
			filename = (g_free (filename), NULL);
		}
		g_rmdir (path);
		(dir == NULL) ? NULL : (dir = (g_dir_close (dir), NULL));
	}
	goto __finally12;
	__catch12_g_file_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
		}
	}
	__finally12:
	if (_inner_error_ != NULL) {
		name = (g_free (name), NULL);
		path = (g_free (path), NULL);
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
	self->priv->window_list = g_slist_remove (self->priv->window_list, window);
	gtk_object_destroy ((GtkObject*) window);
	if (g_slist_length (self->priv->window_list) >= 1) {
		{
			GSList* win_collection;
			GSList* win_it;
			win_collection = self->priv->window_list;
			for (win_it = win_collection; win_it != NULL; win_it = win_it->next) {
				XnpWindow* _tmp2_;
				XnpWindow* win;
				_tmp2_ = NULL;
				win = (_tmp2_ = (XnpWindow*) win_it->data, (_tmp2_ == NULL) ? NULL : g_object_ref (_tmp2_));
				{
					xnp_window_set_window_list (win, self->priv->window_list);
					(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
				}
			}
		}
	} else {
		XnpWindow* new_win;
		new_win = xnp_application_create_window (self, NULL);
		gtk_widget_show ((GtkWidget*) new_win);
		(new_win == NULL) ? NULL : (new_win = (g_object_unref (new_win), NULL));
	}
	name = (g_free (name), NULL);
	path = (g_free (path), NULL);
}


/**
 * window_name_exists:
 *
 * Verify if the given name already exists in the window list.
 */
static gboolean xnp_application_window_name_exists (XnpApplication* self, const char* name) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	{
		GSList* win_collection;
		GSList* win_it;
		win_collection = self->priv->window_list;
		for (win_it = win_collection; win_it != NULL; win_it = win_it->next) {
			XnpWindow* _tmp0_;
			XnpWindow* win;
			_tmp0_ = NULL;
			win = (_tmp0_ = (XnpWindow*) win_it->data, (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_));
			{
				if (_vala_strcmp0 (xnp_window_get_name (win), name) == 0) {
					result = TRUE;
					(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
					return result;
				}
				(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
			}
		}
	}
	result = FALSE;
	return result;
}


/**
 * name_is_valid:
 *
 * Verify if the given name is valid for window and notes.
 */
static gboolean xnp_application_name_is_valid (XnpApplication* self, const char* name) {
	gboolean result;
	gboolean res;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (name != NULL, FALSE);
	res = g_regex_match_simple ("^[^*|/\\:\"<>?]+$", name, 0, 0);
	if (!res) {
		GtkMessageDialog* error_dialog;
		char* _tmp0_;
		error_dialog = g_object_ref_sink ((GtkMessageDialog*) gtk_message_dialog_new (NULL, 0, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, _ ("The name \"%s\" is invalid."), name, NULL));
		_tmp0_ = NULL;
		gtk_message_dialog_format_secondary_markup (error_dialog, _tmp0_ = g_strdup_printf (_ ("The invalid characters are: %s"), "<tt>*|/\\:\"&lt;&gt;?</tt>"), NULL);
		_tmp0_ = (g_free (_tmp0_), NULL);
		gtk_dialog_run ((GtkDialog*) error_dialog);
		gtk_object_destroy ((GtkObject*) error_dialog);
		(error_dialog == NULL) ? NULL : (error_dialog = (g_object_unref (error_dialog), NULL));
	}
	result = res;
	return result;
}


/**
 * show_hide_notes:
 *
 * Show all the notes or hide them if they are all shown.
 */
void xnp_application_show_hide_notes (XnpApplication* self) {
	gboolean invisible_found;
	gboolean visible_found;
	gboolean active_found;
	g_return_if_fail (self != NULL);
	invisible_found = FALSE;
	visible_found = FALSE;
	active_found = FALSE;
	{
		GSList* win_collection;
		GSList* win_it;
		win_collection = self->priv->window_list;
		for (win_it = win_collection; win_it != NULL; win_it = win_it->next) {
			XnpWindow* _tmp1_;
			XnpWindow* win;
			_tmp1_ = NULL;
			win = (_tmp1_ = (XnpWindow*) win_it->data, (_tmp1_ == NULL) ? NULL : g_object_ref (_tmp1_));
			{
				gboolean _tmp0_;
				if ((g_object_get ((GtkWindow*) win, "is-active", &_tmp0_, NULL), _tmp0_)) {
					active_found = TRUE;
				}
				if (!((gboolean) (GTK_WIDGET_FLAGS ((GtkWidget*) win) & GTK_VISIBLE))) {
					invisible_found = TRUE;
				} else {
					visible_found = TRUE;
				}
				(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
			}
		}
	}
	{
		GSList* win_collection;
		GSList* win_it;
		win_collection = self->priv->window_list;
		for (win_it = win_collection; win_it != NULL; win_it = win_it->next) {
			XnpWindow* _tmp3_;
			XnpWindow* win;
			_tmp3_ = NULL;
			win = (_tmp3_ = (XnpWindow*) win_it->data, (_tmp3_ == NULL) ? NULL : g_object_ref (_tmp3_));
			{
				gboolean _tmp2_;
				_tmp2_ = FALSE;
				if (!active_found) {
					_tmp2_ = visible_found;
				} else {
					_tmp2_ = FALSE;
				}
				/* Present visible windows*/
				if (_tmp2_) {
					if ((gboolean) (GTK_WIDGET_FLAGS ((GtkWidget*) win) & GTK_VISIBLE)) {
						gtk_window_present ((GtkWindow*) win);
					}
				} else {
					if (invisible_found) {
						gtk_widget_show ((GtkWidget*) win);
					} else {
						/* Hide all windows*/
						xnp_window_hide (win);
					}
				}
				(win == NULL) ? NULL : (win = (g_object_unref (win), NULL));
			}
		}
	}
}


/**
 * open_settings_dialog:
 *
 * Open the settings dialog.
 */
void xnp_application_open_settings_dialog (XnpApplication* self) {
	GError * _inner_error_;
	g_return_if_fail (self != NULL);
	_inner_error_ = NULL;
	{
		gdk_spawn_command_line_on_screen (gdk_screen_get_default (), "xfce4-notes-settings", &_inner_error_);
		if (_inner_error_ != NULL) {
			goto __catch13_g_error;
			goto __finally13;
		}
	}
	goto __finally13;
	__catch13_g_error:
	{
		GError * e;
		e = _inner_error_;
		_inner_error_ = NULL;
		{
			GtkMessageDialog* error_dialog;
			error_dialog = g_object_ref_sink ((GtkMessageDialog*) gtk_message_dialog_new (NULL, GTK_DIALOG_DESTROY_WITH_PARENT, GTK_MESSAGE_ERROR, GTK_BUTTONS_CLOSE, _ ("Unable to open the settings dialog"), NULL));
			gtk_message_dialog_format_secondary_text (error_dialog, "%s", e->message, NULL);
			gtk_dialog_run ((GtkDialog*) error_dialog);
			gtk_object_destroy ((GtkObject*) error_dialog);
			(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			(error_dialog == NULL) ? NULL : (error_dialog = (g_object_unref (error_dialog), NULL));
		}
	}
	__finally13:
	if (_inner_error_ != NULL) {
		g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
		g_clear_error (&_inner_error_);
		return;
	}
}


const char* xnp_application_get_config_file (XnpApplication* self) {
	const char* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_config_file;
	return result;
}


static void xnp_application_set_config_file (XnpApplication* self, const char* value) {
	char* _tmp1_;
	const char* _tmp0_;
	g_return_if_fail (self != NULL);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->_config_file = (_tmp1_ = (_tmp0_ = value, (_tmp0_ == NULL) ? NULL : g_strdup (_tmp0_)), self->priv->_config_file = (g_free (self->priv->_config_file), NULL), _tmp1_);
	g_object_notify ((GObject *) self, "config-file");
}


static GObject * xnp_application_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	XnpApplicationClass * klass;
	GObjectClass * parent_class;
	XnpApplication * self;
	klass = XNP_APPLICATION_CLASS (g_type_class_peek (XNP_TYPE_APPLICATION));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = XNP_APPLICATION (obj);
	{
		char* _tmp0_;
		_tmp0_ = NULL;
		self->priv->notes_path = (_tmp0_ = g_strdup_printf ("%s/notes", g_get_user_data_dir ()), self->priv->notes_path = (g_free (self->priv->notes_path), NULL), _tmp0_);
	}
	return obj;
}


static void xnp_application_class_init (XnpApplicationClass * klass) {
	xnp_application_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (XnpApplicationPrivate));
	G_OBJECT_CLASS (klass)->get_property = xnp_application_get_property;
	G_OBJECT_CLASS (klass)->set_property = xnp_application_set_property;
	G_OBJECT_CLASS (klass)->constructor = xnp_application_constructor;
	G_OBJECT_CLASS (klass)->finalize = xnp_application_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), XNP_APPLICATION_CONFIG_FILE, g_param_spec_string ("config-file", "config-file", "config-file", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT_ONLY));
}


static void xnp_application_instance_init (XnpApplication * self) {
	self->priv = XNP_APPLICATION_GET_PRIVATE (self);
}


static void xnp_application_finalize (GObject* obj) {
	XnpApplication * self;
	self = XNP_APPLICATION (obj);
	{
		xnp_application_save_windows_configuration (self);
		xnp_application_save_notes (self);
		g_object_unref ((GObject*) self->priv->xfconf_channel);
	}
	(self->priv->window_list == NULL) ? NULL : (self->priv->window_list = (_g_slist_free_g_object_unref (self->priv->window_list), NULL));
	self->priv->notes_path = (g_free (self->priv->notes_path), NULL);
	self->priv->_config_file = (g_free (self->priv->_config_file), NULL);
	(self->priv->xfconf_channel == NULL) ? NULL : (self->priv->xfconf_channel = (g_object_unref (self->priv->xfconf_channel), NULL));
	G_OBJECT_CLASS (xnp_application_parent_class)->finalize (obj);
}


GType xnp_application_get_type (void) {
	static GType xnp_application_type_id = 0;
	if (xnp_application_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (XnpApplicationClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) xnp_application_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (XnpApplication), 0, (GInstanceInitFunc) xnp_application_instance_init, NULL };
		xnp_application_type_id = g_type_register_static (G_TYPE_OBJECT, "XnpApplication", &g_define_type_info, 0);
	}
	return xnp_application_type_id;
}


static void xnp_application_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	XnpApplication * self;
	gpointer boxed;
	self = XNP_APPLICATION (object);
	switch (property_id) {
		case XNP_APPLICATION_CONFIG_FILE:
		g_value_set_string (value, xnp_application_get_config_file (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void xnp_application_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	XnpApplication * self;
	self = XNP_APPLICATION (object);
	switch (property_id) {
		case XNP_APPLICATION_CONFIG_FILE:
		xnp_application_set_config_file (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void _vala_array_destroy (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if ((array != NULL) && (destroy_func != NULL)) {
		int i;
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) array)[i] != NULL) {
				destroy_func (((gpointer*) array)[i]);
			}
		}
	}
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	_vala_array_destroy (array, array_length, destroy_func);
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




