/*
 *  Notes - panel plugin for Xfce Desktop Environment
 *  Copyright (c) 2009  Mike Massonnet <mmassonnet@xfce.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include <glib.h>
#include <glib-object.h>
#include <gtk/gtk.h>
#include <gdk/gdk.h>
#include <stdlib.h>
#include <string.h>
#include <pango/pango.h>
#include <gio/gio.h>


#define XNP_TYPE_HYPERTEXT_VIEW (xnp_hypertext_view_get_type ())
#define XNP_HYPERTEXT_VIEW(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), XNP_TYPE_HYPERTEXT_VIEW, XnpHypertextView))
#define XNP_HYPERTEXT_VIEW_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), XNP_TYPE_HYPERTEXT_VIEW, XnpHypertextViewClass))
#define XNP_IS_HYPERTEXT_VIEW(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), XNP_TYPE_HYPERTEXT_VIEW))
#define XNP_IS_HYPERTEXT_VIEW_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), XNP_TYPE_HYPERTEXT_VIEW))
#define XNP_HYPERTEXT_VIEW_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), XNP_TYPE_HYPERTEXT_VIEW, XnpHypertextViewClass))

typedef struct _XnpHypertextView XnpHypertextView;
typedef struct _XnpHypertextViewClass XnpHypertextViewClass;
typedef struct _XnpHypertextViewPrivate XnpHypertextViewPrivate;

struct _XnpHypertextView {
	GtkTextView parent_instance;
	XnpHypertextViewPrivate * priv;
};

struct _XnpHypertextViewClass {
	GtkTextViewClass parent_class;
};

struct _XnpHypertextViewPrivate {
	GdkCursor* hand_cursor;
	GdkCursor* regular_cursor;
	gboolean cursor_over_link;
	guint undo_timeout;
	gint undo_cursor_pos;
	char* undo_text;
	char* redo_text;
	guint tag_timeout;
	GtkTextTag* tag_link;
	char* _font;
};



GType xnp_hypertext_view_get_type (void);
#define XNP_HYPERTEXT_VIEW_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), XNP_TYPE_HYPERTEXT_VIEW, XnpHypertextViewPrivate))
enum  {
	XNP_HYPERTEXT_VIEW_DUMMY_PROPERTY,
	XNP_HYPERTEXT_VIEW_FONT
};
static gboolean xnp_hypertext_view_button_release_event_cb (XnpHypertextView* self, XnpHypertextView* hypertextview, const GdkEventButton* event);
static gboolean _xnp_hypertext_view_button_release_event_cb_gtk_widget_button_release_event (XnpHypertextView* _sender, const GdkEventButton* event, gpointer self);
static gboolean xnp_hypertext_view_motion_notify_event_cb (XnpHypertextView* self, XnpHypertextView* hypertextview, const GdkEventMotion* event);
static gboolean _xnp_hypertext_view_motion_notify_event_cb_gtk_widget_motion_notify_event (XnpHypertextView* _sender, const GdkEventMotion* event, gpointer self);
static void xnp_hypertext_view_move_cursor_cb (XnpHypertextView* self, XnpHypertextView* hypertextview, GtkMovementStep step, gint count, gboolean extend_selection);
static void _xnp_hypertext_view_move_cursor_cb_gtk_text_view_move_cursor (XnpHypertextView* _sender, GtkMovementStep step, gint count, gboolean extend_selection, gpointer self);
static void xnp_hypertext_view_buffer_changed_cb (XnpHypertextView* self);
static void _xnp_hypertext_view_buffer_changed_cb_gtk_text_buffer_changed (GtkTextBuffer* _sender, gpointer self);
static void xnp_hypertext_view_insert_text_cb (XnpHypertextView* self, GtkTextBuffer* buffer, const GtkTextIter* location, const char* text, gint len);
static void _xnp_hypertext_view_insert_text_cb_gtk_text_buffer_insert_text (GtkTextBuffer* _sender, const GtkTextIter* pos, const char* text, gint length, gpointer self);
static void xnp_hypertext_view_delete_range_cb (XnpHypertextView* self, GtkTextBuffer* buffer, const GtkTextIter* start, const GtkTextIter* end);
static void _xnp_hypertext_view_delete_range_cb_gtk_text_buffer_delete_range (GtkTextBuffer* _sender, const GtkTextIter* start, const GtkTextIter* end, gpointer self);
XnpHypertextView* xnp_hypertext_view_new (void);
XnpHypertextView* xnp_hypertext_view_construct (GType object_type);
XnpHypertextView* xnp_hypertext_view_new (void);
static gboolean xnp_hypertext_view_undo_snapshot (XnpHypertextView* self);
void xnp_hypertext_view_update_tags (XnpHypertextView* self);
static gboolean _xnp_hypertext_view_undo_snapshot_gsource_func (gpointer self);
static gboolean xnp_hypertext_view_tag_timeout_cb (XnpHypertextView* self);
static gboolean _xnp_hypertext_view_tag_timeout_cb_gsource_func (gpointer self);
static void xnp_hypertext_view_tag_timeout_init (XnpHypertextView* self);
void xnp_hypertext_view_undo (XnpHypertextView* self);
const char* xnp_hypertext_view_get_font (XnpHypertextView* self);
void xnp_hypertext_view_set_font (XnpHypertextView* self, const char* value);
static GObject * xnp_hypertext_view_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer xnp_hypertext_view_parent_class = NULL;
static void xnp_hypertext_view_finalize (GObject* obj);
static void xnp_hypertext_view_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void xnp_hypertext_view_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);
static int _vala_strcmp0 (const char * str1, const char * str2);



static gboolean _xnp_hypertext_view_button_release_event_cb_gtk_widget_button_release_event (XnpHypertextView* _sender, const GdkEventButton* event, gpointer self) {
	return xnp_hypertext_view_button_release_event_cb (self, _sender, event);
}


static gboolean _xnp_hypertext_view_motion_notify_event_cb_gtk_widget_motion_notify_event (XnpHypertextView* _sender, const GdkEventMotion* event, gpointer self) {
	return xnp_hypertext_view_motion_notify_event_cb (self, _sender, event);
}


static void _xnp_hypertext_view_move_cursor_cb_gtk_text_view_move_cursor (XnpHypertextView* _sender, GtkMovementStep step, gint count, gboolean extend_selection, gpointer self) {
	xnp_hypertext_view_move_cursor_cb (self, _sender, step, count, extend_selection);
}


static void _xnp_hypertext_view_buffer_changed_cb_gtk_text_buffer_changed (GtkTextBuffer* _sender, gpointer self) {
	xnp_hypertext_view_buffer_changed_cb (self);
}


static void _xnp_hypertext_view_insert_text_cb_gtk_text_buffer_insert_text (GtkTextBuffer* _sender, const GtkTextIter* pos, const char* text, gint length, gpointer self) {
	xnp_hypertext_view_insert_text_cb (self, _sender, pos, text, length);
}


static void _xnp_hypertext_view_delete_range_cb_gtk_text_buffer_delete_range (GtkTextBuffer* _sender, const GtkTextIter* start, const GtkTextIter* end, gpointer self) {
	xnp_hypertext_view_delete_range_cb (self, _sender, start, end);
}


XnpHypertextView* xnp_hypertext_view_construct (GType object_type) {
	XnpHypertextView * self;
	GtkTextIter iter = {0};
	GtkTextTag* _tmp1_;
	GtkTextTag* _tmp0_;
	self = g_object_newv (object_type, 0, NULL);
	g_signal_connect_object ((GtkWidget*) self, "button-release-event", (GCallback) _xnp_hypertext_view_button_release_event_cb_gtk_widget_button_release_event, self, 0);
	g_signal_connect_object ((GtkWidget*) self, "motion-notify-event", (GCallback) _xnp_hypertext_view_motion_notify_event_cb_gtk_widget_motion_notify_event, self, 0);
	g_signal_connect_object ((GtkTextView*) self, "move-cursor", (GCallback) _xnp_hypertext_view_move_cursor_cb_gtk_text_view_move_cursor, self, 0);
	g_signal_connect_object (gtk_text_view_get_buffer ((GtkTextView*) self), "changed", (GCallback) _xnp_hypertext_view_buffer_changed_cb_gtk_text_buffer_changed, self, 0);
	g_signal_connect_object (gtk_text_view_get_buffer ((GtkTextView*) self), "insert-text", (GCallback) _xnp_hypertext_view_insert_text_cb_gtk_text_buffer_insert_text, self, 0);
	g_signal_connect_object (gtk_text_view_get_buffer ((GtkTextView*) self), "delete-range", (GCallback) _xnp_hypertext_view_delete_range_cb_gtk_text_buffer_delete_range, self, 0);
	gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer ((GtkTextView*) self), &iter, 0);
	gtk_text_buffer_create_mark (gtk_text_view_get_buffer ((GtkTextView*) self), "undo-pos", &iter, FALSE);
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	self->priv->tag_link = (_tmp1_ = (_tmp0_ = gtk_text_buffer_create_tag (gtk_text_view_get_buffer ((GtkTextView*) self), "link", "foreground", "blue", "underline", PANGO_UNDERLINE_SINGLE, NULL, NULL), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (self->priv->tag_link == NULL) ? NULL : (self->priv->tag_link = (g_object_unref (self->priv->tag_link), NULL)), _tmp1_);
	return self;
}


XnpHypertextView* xnp_hypertext_view_new (void) {
	return xnp_hypertext_view_construct (XNP_TYPE_HYPERTEXT_VIEW);
}


/*
 * Signal callbacks
 
*
 * button_release_event_cb:
 *
 * Event to open links.
 */
static gboolean xnp_hypertext_view_button_release_event_cb (XnpHypertextView* self, XnpHypertextView* hypertextview, const GdkEventButton* event) {
	GError * _inner_error_;
	GtkTextIter start = {0};
	GtkTextIter end = {0};
	GtkTextIter iter = {0};
	char* link;
	gint x;
	gint y;
	gboolean _tmp9_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (hypertextview != NULL, FALSE);
	_inner_error_ = NULL;
	link = NULL;
	x = 0;
	y = 0;
	if ((*event).button != 1) {
		gboolean _tmp0_;
		return (_tmp0_ = FALSE, link = (g_free (link), NULL), _tmp0_);
	}
	gtk_text_buffer_get_selection_bounds (gtk_text_view_get_buffer ((GtkTextView*) self), &start, &end);
	if (gtk_text_iter_get_offset (&start) != gtk_text_iter_get_offset (&end)) {
		gboolean _tmp1_;
		return (_tmp1_ = FALSE, link = (g_free (link), NULL), _tmp1_);
	}
	gtk_text_view_window_to_buffer_coords ((GtkTextView*) self, GTK_TEXT_WINDOW_WIDGET, (gint) (*event).x, (gint) (*event).y, &x, &y);
	gtk_text_view_get_iter_at_location ((GtkTextView*) self, &iter, x, y);
	if (gtk_text_iter_has_tag (&iter, self->priv->tag_link)) {
		char* _tmp3_;
		const char* _tmp2_;
		start = end = iter;
		if (!gtk_text_iter_begins_tag (&start, self->priv->tag_link)) {
			gtk_text_iter_backward_to_tag_toggle (&start, self->priv->tag_link);
		}
		gtk_text_iter_forward_to_tag_toggle (&end, self->priv->tag_link);
		_tmp3_ = NULL;
		_tmp2_ = NULL;
		link = (_tmp3_ = (_tmp2_ = gtk_text_iter_get_text (&start, &end), (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), link = (g_free (link), NULL), _tmp3_);
		{
			gboolean _tmp4_;
			g_app_info_launch_default_for_uri (link, NULL, &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch0_g_error;
				goto __finally0;
			}
			return (_tmp4_ = FALSE, link = (g_free (link), NULL), _tmp4_);
		}
		goto __finally0;
		__catch0_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				g_critical ("hypertextview.vala:121: Unable to open link with default handler: %s", e->message);
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally0:
		if (_inner_error_ != NULL) {
			link = (g_free (link), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return FALSE;
		}
		{
			char* _tmp5_;
			gboolean _tmp6_;
			_tmp5_ = NULL;
			gdk_spawn_command_line_on_screen (gdk_screen_get_default (), _tmp5_ = g_strconcat ("xdg-open ", link, NULL), &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch1_g_error;
				goto __finally1;
			}
			_tmp5_ = (g_free (_tmp5_), NULL);
			return (_tmp6_ = FALSE, link = (g_free (link), NULL), _tmp6_);
		}
		goto __finally1;
		__catch1_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally1:
		if (_inner_error_ != NULL) {
			link = (g_free (link), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return FALSE;
		}
		{
			char* _tmp7_;
			gboolean _tmp8_;
			_tmp7_ = NULL;
			gdk_spawn_command_line_on_screen (gdk_screen_get_default (), _tmp7_ = g_strconcat ("firefox ", link, NULL), &_inner_error_);
			if (_inner_error_ != NULL) {
				goto __catch2_g_error;
				goto __finally2;
			}
			_tmp7_ = (g_free (_tmp7_), NULL);
			return (_tmp8_ = FALSE, link = (g_free (link), NULL), _tmp8_);
		}
		goto __finally2;
		__catch2_g_error:
		{
			GError * e;
			e = _inner_error_;
			_inner_error_ = NULL;
			{
				(e == NULL) ? NULL : (e = (g_error_free (e), NULL));
			}
		}
		__finally2:
		if (_inner_error_ != NULL) {
			link = (g_free (link), NULL);
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, _inner_error_->message);
			g_clear_error (&_inner_error_);
			return FALSE;
		}
		g_critical ("hypertextview.vala:136: Unable to find an appropriate fallback to open the link");
	}
	return (_tmp9_ = FALSE, link = (g_free (link), NULL), _tmp9_);
}


/**
 * motion_notify_event_cb:
 *
 * Event to update the cursor of the pointer.
 */
static gboolean xnp_hypertext_view_motion_notify_event_cb (XnpHypertextView* self, XnpHypertextView* hypertextview, const GdkEventMotion* event) {
	GtkTextIter iter = {0};
	GdkWindow* win;
	gboolean over_link;
	gint x;
	gint y;
	gboolean _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (hypertextview != NULL, FALSE);
	win = NULL;
	over_link = FALSE;
	x = 0;
	y = 0;
	gtk_text_view_window_to_buffer_coords ((GtkTextView*) self, GTK_TEXT_WINDOW_WIDGET, (gint) (*event).x, (gint) (*event).y, &x, &y);
	gtk_text_view_get_iter_at_location ((GtkTextView*) self, &iter, x, y);
	over_link = gtk_text_iter_has_tag (&iter, self->priv->tag_link);
	if (over_link != self->priv->cursor_over_link) {
		GdkWindow* _tmp1_;
		GdkWindow* _tmp0_;
		GdkCursor* _tmp2_;
		self->priv->cursor_over_link = over_link;
		_tmp1_ = NULL;
		_tmp0_ = NULL;
		win = (_tmp1_ = (_tmp0_ = gtk_text_view_get_window ((GtkTextView*) self, GTK_TEXT_WINDOW_TEXT), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (win == NULL) ? NULL : (win = (g_object_unref (win), NULL)), _tmp1_);
		_tmp2_ = NULL;
		if (over_link) {
			_tmp2_ = self->priv->hand_cursor;
		} else {
			_tmp2_ = self->priv->regular_cursor;
		}
		gdk_window_set_cursor (win, _tmp2_);
	}
	return (_tmp3_ = FALSE, (win == NULL) ? NULL : (win = (g_object_unref (win), NULL)), _tmp3_);
}


/**
 * move_cursor_cb:
 *
 * Destroys existing timeouts and executes the actions immediately.
 */
static void xnp_hypertext_view_move_cursor_cb (XnpHypertextView* self, XnpHypertextView* hypertextview, GtkMovementStep step, gint count, gboolean extend_selection) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (hypertextview != NULL);
	if (self->priv->undo_timeout > 0) {
		gint _tmp0_;
		/* Make an undo snapshot and save cursor_position before it really moves */
		g_source_remove (self->priv->undo_timeout);
		self->priv->undo_timeout = (guint) 0;
		xnp_hypertext_view_undo_snapshot (self);
		self->priv->undo_cursor_pos = (g_object_get (gtk_text_view_get_buffer ((GtkTextView*) self), "cursor-position", &_tmp0_, NULL), _tmp0_);
	}
	if (self->priv->tag_timeout > 0) {
		g_source_remove (self->priv->tag_timeout);
		self->priv->tag_timeout = (guint) 0;
		xnp_hypertext_view_update_tags (self);
	}
}


static gboolean _xnp_hypertext_view_undo_snapshot_gsource_func (gpointer self) {
	return xnp_hypertext_view_undo_snapshot (self);
}


static gboolean _xnp_hypertext_view_tag_timeout_cb_gsource_func (gpointer self) {
	return xnp_hypertext_view_tag_timeout_cb (self);
}


/**
 * buffer_changed_cb:
 *
 * Initializes timeouts to postpone actions.
 */
static void xnp_hypertext_view_buffer_changed_cb (XnpHypertextView* self) {
	g_return_if_fail (self != NULL);
	/* Initialize undo_timeout */
	if (self->priv->undo_timeout > 0) {
		g_source_remove (self->priv->undo_timeout);
		self->priv->undo_timeout = (guint) 0;
	}
	self->priv->undo_timeout = g_timeout_add_seconds ((guint) 2, _xnp_hypertext_view_undo_snapshot_gsource_func, self);
	/* Reinit tag_timeout as long as the buffer is under constant changes */
	if (self->priv->tag_timeout > 0) {
		g_source_remove (self->priv->tag_timeout);
		self->priv->tag_timeout = (guint) 0;
		self->priv->tag_timeout = g_timeout_add_seconds ((guint) 2, _xnp_hypertext_view_tag_timeout_cb_gsource_func, self);
	}
}


static gboolean string_contains (const char* self, const char* needle) {
	g_return_val_if_fail (self != NULL, FALSE);
	g_return_val_if_fail (needle != NULL, FALSE);
	return strstr (self, needle) != NULL;
}


/**
 * insert_text_cb:
 *
 * Event to create and update existing tags within the buffer.
 */
static void xnp_hypertext_view_insert_text_cb (XnpHypertextView* self, GtkTextBuffer* buffer, const GtkTextIter* location, const char* text, gint len) {
	GtkTextIter start = {0};
	GtkTextIter end = {0};
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (buffer != NULL);
	g_return_if_fail (text != NULL);
	_tmp0_ = FALSE;
	if (gtk_text_iter_has_tag (&(*location), self->priv->tag_link)) {
		_tmp0_ = !gtk_text_iter_begins_tag (&(*location), self->priv->tag_link);
	} else {
		_tmp0_ = FALSE;
	}
	/* Text is inserted inside a tag */
	if (_tmp0_) {
		start = (*location);
		gtk_text_iter_backward_to_tag_toggle (&start, self->priv->tag_link);
		if ((gtk_text_iter_get_offset (&(*location)) - gtk_text_iter_get_offset (&start)) < 7) {
			gboolean _tmp1_;
			end = start;
			gtk_text_iter_forward_to_tag_toggle (&end, self->priv->tag_link);
			gtk_text_buffer_remove_tag (gtk_text_view_get_buffer ((GtkTextView*) self), self->priv->tag_link, &start, &end);
			_tmp1_ = FALSE;
			if (len > 1) {
				gboolean _tmp2_;
				_tmp2_ = FALSE;
				if (string_contains (text, " ")) {
					_tmp2_ = TRUE;
				} else {
					_tmp2_ = string_contains (text, "\n");
				}
				_tmp1_ = _tmp2_;
			} else {
				_tmp1_ = FALSE;
			}
			if (_tmp1_) {
				/* We are here because there is a chance in a million that the
				 * user pasted a text that ends with " ht" in front of "tp://"
				 */
				xnp_hypertext_view_tag_timeout_init (self);
			}
		} else {
			gboolean _tmp3_;
			_tmp3_ = FALSE;
			if (string_contains (text, " ")) {
				_tmp3_ = TRUE;
			} else {
				_tmp3_ = string_contains (text, "\n");
			}
			if (_tmp3_) {
				end = (*location);
				gtk_text_iter_forward_to_tag_toggle (&end, self->priv->tag_link);
				gtk_text_buffer_remove_tag (gtk_text_view_get_buffer ((GtkTextView*) self), self->priv->tag_link, &start, &end);
				xnp_hypertext_view_tag_timeout_init (self);
			}
		}
	} else {
		if (gtk_text_iter_ends_tag (&(*location), self->priv->tag_link)) {
			gboolean _tmp4_;
			_tmp4_ = FALSE;
			if (len >= 1) {
				gboolean _tmp5_;
				_tmp5_ = FALSE;
				if (g_utf8_get_char (g_utf8_offset_to_pointer (text, 0)) == ' ') {
					_tmp5_ = TRUE;
				} else {
					_tmp5_ = g_utf8_get_char (g_utf8_offset_to_pointer (text, 0)) == '\n';
				}
				_tmp4_ = !_tmp5_;
			} else {
				_tmp4_ = FALSE;
			}
			if (_tmp4_) {
				start = (*location);
				gtk_text_iter_backward_to_tag_toggle (&start, self->priv->tag_link);
				gtk_text_buffer_remove_tag (gtk_text_view_get_buffer ((GtkTextView*) self), self->priv->tag_link, &start, &(*location));
				xnp_hypertext_view_tag_timeout_init (self);
			}
		} else {
			gboolean _tmp6_;
			_tmp6_ = FALSE;
			if (len == 1) {
				_tmp6_ = g_utf8_get_char (g_utf8_offset_to_pointer (text, 0)) == '/';
			} else {
				_tmp6_ = FALSE;
			}
			if (_tmp6_) {
				gboolean _tmp7_;
				start = (*location);
				_tmp7_ = FALSE;
				if (!gtk_text_iter_backward_chars (&start, 6)) {
					_tmp7_ = TRUE;
				} else {
					char* _tmp8_;
					_tmp8_ = NULL;
					_tmp7_ = _vala_strcmp0 (_tmp8_ = g_utf8_strdown (gtk_text_iter_get_text (&start, &(*location)), -1), "http:/") != 0;
					_tmp8_ = (g_free (_tmp8_), NULL);
				}
				if (_tmp7_) {
					return;
				}
				xnp_hypertext_view_tag_timeout_init (self);
			} else {
				gboolean _tmp9_;
				_tmp9_ = FALSE;
				if (len == 1) {
					gboolean _tmp10_;
					_tmp10_ = FALSE;
					if (g_utf8_get_char (g_utf8_offset_to_pointer (text, 0)) == ' ') {
						_tmp10_ = TRUE;
					} else {
						_tmp10_ = g_utf8_get_char (g_utf8_offset_to_pointer (text, 0)) == '\n';
					}
					_tmp9_ = _tmp10_;
				} else {
					_tmp9_ = FALSE;
				}
				if (_tmp9_) {
					xnp_hypertext_view_update_tags (self);
				} else {
					gboolean _tmp11_;
					_tmp11_ = FALSE;
					if (len > 1) {
						_tmp11_ = string_contains (text, "http://");
					} else {
						_tmp11_ = FALSE;
					}
					if (_tmp11_) {
						xnp_hypertext_view_tag_timeout_init (self);
					}
				}
			}
		}
	}
}


/**
 * delete_range_cb:
 *
 * Event to delete and update existing tags within the buffer.
 */
static void xnp_hypertext_view_delete_range_cb (XnpHypertextView* self, GtkTextBuffer* buffer, const GtkTextIter* start, const GtkTextIter* end) {
	GtkTextIter iter = {0};
	gboolean _tmp0_;
	g_return_if_fail (self != NULL);
	g_return_if_fail (buffer != NULL);
	_tmp0_ = FALSE;
	if (!gtk_text_iter_has_tag (&(*start), self->priv->tag_link)) {
		_tmp0_ = !gtk_text_iter_has_tag (&(*end), self->priv->tag_link);
	} else {
		_tmp0_ = FALSE;
	}
	if (_tmp0_) {
		return;
	}
	if (gtk_text_iter_has_tag (&(*start), self->priv->tag_link)) {
		iter = (*start);
		gtk_text_iter_backward_to_tag_toggle (&iter, self->priv->tag_link);
		gtk_text_buffer_remove_tag (gtk_text_view_get_buffer ((GtkTextView*) self), self->priv->tag_link, &iter, &(*start));
	}
	if (gtk_text_iter_has_tag (&(*end), self->priv->tag_link)) {
		iter = (*end);
		gtk_text_iter_forward_to_tag_toggle (&iter, self->priv->tag_link);
		gtk_text_buffer_remove_tag (gtk_text_view_get_buffer ((GtkTextView*) self), self->priv->tag_link, &(*end), &iter);
	}
	xnp_hypertext_view_tag_timeout_init (self);
}


/*
 * Undo
 
*
 * undo_snapshot:
 *
 * Makes a snapshot of the current buffer and swaps undo/redo texts.
 */
static gboolean xnp_hypertext_view_undo_snapshot (XnpHypertextView* self) {
	GtkTextIter start = {0};
	GtkTextIter end = {0};
	gint _tmp0_;
	char* _tmp2_;
	const char* _tmp1_;
	char* _tmp4_;
	const char* _tmp3_;
	g_return_val_if_fail (self != NULL, FALSE);
	self->priv->undo_cursor_pos = (g_object_get (gtk_text_view_get_buffer ((GtkTextView*) self), "cursor-position", &_tmp0_, NULL), _tmp0_);
	gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer ((GtkTextView*) self), &start, 0);
	gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer ((GtkTextView*) self), &end, -1);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->undo_text = (_tmp2_ = (_tmp1_ = self->priv->redo_text, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->undo_text = (g_free (self->priv->undo_text), NULL), _tmp2_);
	_tmp4_ = NULL;
	_tmp3_ = NULL;
	self->priv->redo_text = (_tmp4_ = (_tmp3_ = gtk_text_buffer_get_text (gtk_text_view_get_buffer ((GtkTextView*) self), &start, &end, FALSE), (_tmp3_ == NULL) ? NULL : g_strdup (_tmp3_)), self->priv->redo_text = (g_free (self->priv->redo_text), NULL), _tmp4_);
	return FALSE;
}


/**
 * undo:
 *
 * Revert the buffer to the undo text and swaps undo/redo texts.
 */
void xnp_hypertext_view_undo (XnpHypertextView* self) {
	GtkTextIter iter = {0};
	GtkTextMark* mark;
	char* tmp;
	GtkTextMark* _tmp1_;
	GtkTextMark* _tmp0_;
	char* _tmp3_;
	const char* _tmp2_;
	char* _tmp5_;
	const char* _tmp4_;
	char* _tmp7_;
	const char* _tmp6_;
	g_return_if_fail (self != NULL);
	mark = NULL;
	tmp = NULL;
	if (self->priv->undo_timeout > 0) {
		/* Make an undo snaphot */
		g_source_remove (self->priv->undo_timeout);
		self->priv->undo_timeout = (guint) 0;
		xnp_hypertext_view_undo_snapshot (self);
	}
	gtk_text_buffer_set_text (gtk_text_view_get_buffer ((GtkTextView*) self), self->priv->undo_text, -1);
	gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer ((GtkTextView*) self), &iter, self->priv->undo_cursor_pos);
	gtk_text_buffer_place_cursor (gtk_text_view_get_buffer ((GtkTextView*) self), &iter);
	/* Scroll to the cursor position */
	_tmp1_ = NULL;
	_tmp0_ = NULL;
	mark = (_tmp1_ = (_tmp0_ = gtk_text_buffer_get_mark (gtk_text_view_get_buffer ((GtkTextView*) self), "undo-pos"), (_tmp0_ == NULL) ? NULL : g_object_ref (_tmp0_)), (mark == NULL) ? NULL : (mark = (g_object_unref (mark), NULL)), _tmp1_);
	gtk_text_buffer_move_mark (gtk_text_view_get_buffer ((GtkTextView*) self), mark, &iter);
	gtk_text_view_scroll_to_mark ((GtkTextView*) self, mark, 0.0, FALSE, 0.5, 0.5);
	_tmp3_ = NULL;
	_tmp2_ = NULL;
	tmp = (_tmp3_ = (_tmp2_ = self->priv->undo_text, (_tmp2_ == NULL) ? NULL : g_strdup (_tmp2_)), tmp = (g_free (tmp), NULL), _tmp3_);
	_tmp5_ = NULL;
	_tmp4_ = NULL;
	self->priv->undo_text = (_tmp5_ = (_tmp4_ = self->priv->redo_text, (_tmp4_ == NULL) ? NULL : g_strdup (_tmp4_)), self->priv->undo_text = (g_free (self->priv->undo_text), NULL), _tmp5_);
	_tmp7_ = NULL;
	_tmp6_ = NULL;
	self->priv->redo_text = (_tmp7_ = (_tmp6_ = tmp, (_tmp6_ == NULL) ? NULL : g_strdup (_tmp6_)), self->priv->redo_text = (g_free (self->priv->redo_text), NULL), _tmp7_);
	if (self->priv->undo_timeout > 0) {
		g_source_remove (self->priv->undo_timeout);
		self->priv->undo_timeout = (guint) 0;
	}
	(mark == NULL) ? NULL : (mark = (g_object_unref (mark), NULL));
	tmp = (g_free (tmp), NULL);
}


/*
 * Tags
 */
static gboolean xnp_hypertext_view_tag_timeout_cb (XnpHypertextView* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	xnp_hypertext_view_update_tags (self);
	return FALSE;
}


static void xnp_hypertext_view_tag_timeout_init (XnpHypertextView* self) {
	g_return_if_fail (self != NULL);
	if (self->priv->tag_timeout > 0) {
		g_source_remove (self->priv->tag_timeout);
		self->priv->tag_timeout = (guint) 0;
	}
	self->priv->tag_timeout = g_timeout_add_seconds ((guint) 2, _xnp_hypertext_view_tag_timeout_cb_gsource_func, self);
}


/**
 * update_tags:
 *
 * Goes through the entire document to search for untagged HTTP links and tag them.
 */
void xnp_hypertext_view_update_tags (XnpHypertextView* self) {
	GtkTextIter iter = {0};
	GtkTextIter start = {0};
	GtkTextIter end = {0};
	GtkTextIter tmp = {0};
	g_return_if_fail (self != NULL);
	if (self->priv->tag_timeout > 0) {
		g_source_remove (self->priv->tag_timeout);
		self->priv->tag_timeout = (guint) 0;
	}
	gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer ((GtkTextView*) self), &iter, 0);
	while (gtk_text_iter_forward_search (&iter, "http://", GTK_TEXT_SEARCH_TEXT_ONLY, &start, &end, NULL)) {
		iter = end;
		if (gtk_text_iter_begins_tag (&start, self->priv->tag_link)) {
			continue;
		}
		if (!gtk_text_iter_forward_search (&iter, " ", GTK_TEXT_SEARCH_TEXT_ONLY, &end, NULL, NULL)) {
			if (!gtk_text_iter_forward_search (&iter, "\n", GTK_TEXT_SEARCH_TEXT_ONLY, &end, NULL, NULL)) {
				gtk_text_buffer_get_iter_at_offset (gtk_text_view_get_buffer ((GtkTextView*) self), &end, -1);
			}
		} else {
			if (gtk_text_iter_forward_search (&iter, "\n", GTK_TEXT_SEARCH_TEXT_ONLY, &tmp, NULL, NULL)) {
				if (gtk_text_iter_get_offset (&tmp) < gtk_text_iter_get_offset (&end)) {
					end = tmp;
				}
			}
		}
		if ((gtk_text_iter_get_offset (&end) - gtk_text_iter_get_offset (&start)) >= 7) {
			gtk_text_buffer_apply_tag (gtk_text_view_get_buffer ((GtkTextView*) self), self->priv->tag_link, &start, &end);
		}
	}
}


const char* xnp_hypertext_view_get_font (XnpHypertextView* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_font;
}


void xnp_hypertext_view_set_font (XnpHypertextView* self, const char* value) {
	char* _tmp2_;
	const char* _tmp1_;
	PangoFontDescription* font_descr;
	g_return_if_fail (self != NULL);
	_tmp2_ = NULL;
	_tmp1_ = NULL;
	self->priv->_font = (_tmp2_ = (_tmp1_ = value, (_tmp1_ == NULL) ? NULL : g_strdup (_tmp1_)), self->priv->_font = (g_free (self->priv->_font), NULL), _tmp2_);
	font_descr = pango_font_description_from_string (value);
	gtk_widget_modify_font ((GtkWidget*) self, font_descr);
	(font_descr == NULL) ? NULL : (font_descr = (pango_font_description_free (font_descr), NULL));
	g_object_notify ((GObject *) self, "font");
}


static GObject * xnp_hypertext_view_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	XnpHypertextViewClass * klass;
	GObjectClass * parent_class;
	XnpHypertextView * self;
	klass = XNP_HYPERTEXT_VIEW_CLASS (g_type_class_peek (XNP_TYPE_HYPERTEXT_VIEW));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = XNP_HYPERTEXT_VIEW (obj);
	{
		xnp_hypertext_view_set_font (self, "Sans 12");
	}
	return obj;
}


static void xnp_hypertext_view_class_init (XnpHypertextViewClass * klass) {
	xnp_hypertext_view_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (XnpHypertextViewPrivate));
	G_OBJECT_CLASS (klass)->get_property = xnp_hypertext_view_get_property;
	G_OBJECT_CLASS (klass)->set_property = xnp_hypertext_view_set_property;
	G_OBJECT_CLASS (klass)->constructor = xnp_hypertext_view_constructor;
	G_OBJECT_CLASS (klass)->finalize = xnp_hypertext_view_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), XNP_HYPERTEXT_VIEW_FONT, g_param_spec_string ("font", "font", "font", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void xnp_hypertext_view_instance_init (XnpHypertextView * self) {
	self->priv = XNP_HYPERTEXT_VIEW_GET_PRIVATE (self);
	self->priv->hand_cursor = gdk_cursor_new (GDK_HAND2);
	self->priv->regular_cursor = gdk_cursor_new (GDK_XTERM);
	self->priv->cursor_over_link = FALSE;
	self->priv->undo_timeout = (guint) 0;
	self->priv->undo_text = g_strdup ("");
	self->priv->redo_text = g_strdup ("");
	self->priv->tag_timeout = (guint) 0;
}


static void xnp_hypertext_view_finalize (GObject* obj) {
	XnpHypertextView * self;
	self = XNP_HYPERTEXT_VIEW (obj);
	{
		if (self->priv->undo_timeout != 0) {
			g_source_remove (self->priv->undo_timeout);
		}
		if (self->priv->tag_timeout != 0) {
			g_source_remove (self->priv->tag_timeout);
		}
	}
	(self->priv->hand_cursor == NULL) ? NULL : (self->priv->hand_cursor = (gdk_cursor_unref (self->priv->hand_cursor), NULL));
	(self->priv->regular_cursor == NULL) ? NULL : (self->priv->regular_cursor = (gdk_cursor_unref (self->priv->regular_cursor), NULL));
	self->priv->undo_text = (g_free (self->priv->undo_text), NULL);
	self->priv->redo_text = (g_free (self->priv->redo_text), NULL);
	(self->priv->tag_link == NULL) ? NULL : (self->priv->tag_link = (g_object_unref (self->priv->tag_link), NULL));
	self->priv->_font = (g_free (self->priv->_font), NULL);
	G_OBJECT_CLASS (xnp_hypertext_view_parent_class)->finalize (obj);
}


GType xnp_hypertext_view_get_type (void) {
	static GType xnp_hypertext_view_type_id = 0;
	if (xnp_hypertext_view_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (XnpHypertextViewClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) xnp_hypertext_view_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (XnpHypertextView), 0, (GInstanceInitFunc) xnp_hypertext_view_instance_init, NULL };
		xnp_hypertext_view_type_id = g_type_register_static (GTK_TYPE_TEXT_VIEW, "XnpHypertextView", &g_define_type_info, 0);
	}
	return xnp_hypertext_view_type_id;
}


static void xnp_hypertext_view_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	XnpHypertextView * self;
	gpointer boxed;
	self = XNP_HYPERTEXT_VIEW (object);
	switch (property_id) {
		case XNP_HYPERTEXT_VIEW_FONT:
		g_value_set_string (value, xnp_hypertext_view_get_font (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void xnp_hypertext_view_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	XnpHypertextView * self;
	self = XNP_HYPERTEXT_VIEW (object);
	switch (property_id) {
		case XNP_HYPERTEXT_VIEW_FONT:
		xnp_hypertext_view_set_font (self, g_value_get_string (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return str1 != str2;
	}
	return strcmp (str1, str2);
}




