// -*- c++ -*-

/*****************************************************************

Copyright (c) 1996-2002 the kicker authors. See file AUTHORS.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN
AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************/

#ifndef KASITEM_H
#define KASITEM_H

class QPainter;
class QMouseEvent;
class KasPopup;

#include <qobject.h>
#include "kasbar.h"

/**
 * Abstract base class for items that can be in a KasBar.
 *
 * @version $Id: kasitem.h,v 1.12 2002/06/10 02:29:04 rich Exp $
 * @author Richard Moore, rich@kde.org
 */
class KasItem : public QObject
{
    Q_OBJECT

public:
    friend class KasBar;

    KasItem( KasBar *parent );
    virtual ~KasItem();

    /**
     * Returns the parent KasBar of this item.
     */
    KasBar *kasbar() const { return kas; }

    /**
     * Returns the size of the item in pixels.
     */
    int extent() const { return kas->itemExtent(); }

    //
    // Popup
    //

    /**
     * Returns true iff this item is showing a popup.
     */
    bool isShowingPopup() const { return (pop ? true : false ); }

    /**
     * Returns the active popup or 0.
     */
    KasPopup *popup() const { return pop; }

    /**
     * Returns true iff this item uses a custom popup policy.  If this
     * flag is set, the default popup behaviour is disabled. This
     * means you must call show/hide/toggle yourself if you want the
     * popup to be shown.
     */
    bool hasCustomPopup() const { return customPopup; }

    /**
     * Enables or disables custom popup handling.
     */
    void setCustomPopup( bool enable = true ) { customPopup = enable; }

    void setGroupItem( bool enable = true  ) { groupItem = enable; }

    //
    // Drawing Methods
    //

    /**
     * Translates the QPainter then calls paintItem().
     */
    void paint( QPainter *p, int x, int y );

    /**
     * Subclasses should reimplement this method to paint
     * themselves. The painter is setup so that the item is always at
     * 0, 0.
     */
    virtual void paint( QPainter *p );

    /**
     * Draw a standard frame for the item.
     */
    void paintFrame( QPainter *p );

    /**
     * Paint the background.
     */
    void paintBackground( QPainter *p );

    /**
     * Draw the label for the item.
     */
    void paintLabel( QPainter *p );

    /**
     * Draw a label with an arrow, the parameters specify the position
     * and size of the arrow.
     */
    void paintArrowLabel( QPainter *p, int arrowSize, bool arrowOnLeft );

    /**
     * Paints a progress graph.
     */
    void paintProgress( QPainter *p, int percent );

public slots:
    /**
     * Asks the KasBar to update this item. If erase is true then the
     * item will be erased before the paint method is called.
     */
    void update( bool erase );

    void update();

    void setActive( bool yes );
    void setText( const QString &title );

    /**
     * Shows the items popup.
     */
    void showPopup();

    /**
     * Hides the items popup.
     */
    void hidePopup();

    /**
     * Hides or shows the popup.
     */
    void togglePopup();

    /**
     * Called when something being dragged is held over the item for a while.
     */
    virtual void dragOverAction() {}

protected:
    /**
     * Gets the font metrics from the parent.
     */
    QFontMetrics fontMetrics() const { return kas->fontMetrics(); }

    /**
     * Gets the color group from the parent.
     */
    const QColorGroup &colorGroup() const { return kas->colorGroup(); }

    /**
     * Factory method that creates a popup widget for the item.
     */
    virtual KasPopup *createPopup() { return 0; }

    //
    // Event Handlers
    //

    /**
     * Called when the item receives a mouse event.
     */
    virtual void mousePressEvent( QMouseEvent * ) {}

    /**
     * Called when the mouse enters the item.
     */
    virtual void mouseEnter();

    /**
     * Called when the mouse leaves the item.
     */
    virtual void mouseLeave();

    /**
     * Called when a drag enters the item.
     */
    virtual void dragEnter();

    /**
     * Called when a drag leaves the item.
     */
    virtual void dragLeave();

private:
    KasBar *kas;
    KasPopup *pop;
    QTimer *popupTimer;
    QTimer *dragTimer;
    QString title;
    bool mouseOver;
    bool activated;
    bool customPopup;
    bool groupItem;
};

#endif // KASITEM_H

