/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "filelist.h"

#include "cache.h"
#include "cache_maint.h"
#include "img-main.h"
#include "menu.h"
#include "slideshow.h"
#include "thumb.h"
#include "utilops.h"
#include "ui_clist_edit.h"
#include "ui_fileops.h"
#include "ui_tabcomp.h"
#include "ui_utildlg.h"


typedef struct _FileData FileData;
struct _FileData {
	gchar *name;
	gint size;
	time_t date;
};

static SortType sort_method = SORT_NAME;

GList *dir_list = NULL;
GList *file_list = NULL;

static gint filelist_click_row = -1;
static guint32 filelist_click_time = 0;
static gint filelist_rename_row = -1;


static void filelist_update_thumbs(void);


/*
 *-----------------------------------------------------------------------------
 * file status information (private)
 *-----------------------------------------------------------------------------
 */

static void update_progressbar(gfloat val)
{
	gtk_progress_bar_update (GTK_PROGRESS_BAR(info_progress_bar), val);
}

void update_status_label(const gchar *text)
{
	gchar *buf;
	gint count;
	gchar *ss = "";

	if (text)
		{
		gtk_label_set(GTK_LABEL(info_status), text);
		return;
		}

	if (main_image_slideshow_active())
		{
		if (!main_image_slideshow_paused())
			{
			ss = _(" Slideshow");
			}
		else
			{
			ss = _(" Paused");
			}
		}

	count = file_selection_count();
	if (count > 0)
		buf = g_strdup_printf(_("%d files (%d)%s"), file_count(), count, ss);
	else
		buf = g_strdup_printf(_("%d files%s"), file_count(), ss);

	gtk_label_set(GTK_LABEL(info_status), buf);
	g_free(buf);
}

/*
 *-----------------------------------------------------------------------------
 * file filtering
 *-----------------------------------------------------------------------------
 */

gint file_is_hidden(const gchar *name)
{
	if (name[0] != '.') return FALSE;
	if (name[1] == '\0' || (name[1] == '.' && name[2] == '\0')) return FALSE;
	return TRUE;
}

gint file_is_in_filter(const gchar *name)
{
	GList *work;
	if (!filename_filter || file_filter_disable) return TRUE;

	work = filename_filter;
	while (work)
		{
		gchar *filter = work->data;
		gint lf = strlen(filter);
		gint ln = strlen(name);
		if (ln >= lf)
			{
			if (strncasecmp(name + ln - lf, filter, lf) == 0) return TRUE;
			}
		work = work->next;
		}

	return FALSE;
}

GList *path_list_filter(GList *list, gint is_dir_list)
{
	GList *work;

	if (file_filter_disable && show_dot_files) return list;

	work = list;
	while (work)
		{
		gchar *name = work->data;
		const gchar *base;

		base = filename_from_path(name);

		if ((!show_dot_files && file_is_hidden(base)) ||
		    (!is_dir_list && !file_is_in_filter(base)) )
			{
			GList *link = work;
			work = work->next;
			list = g_list_remove_link(list, link);
			g_free(name);
			g_list_free(link);
			}
		else
			{
			work = work->next;
			}
		}

	return list;
}

static void add_to_filter(gchar *text, gint add)
{
	if (add) filename_filter = g_list_append(filename_filter, g_strdup(text));
}

void rebuild_file_filter(void)
{
	if (filename_filter)
		{
		g_list_foreach(filename_filter,(GFunc)g_free,NULL);
		g_list_free(filename_filter);
		filename_filter = NULL;
		}

	add_to_filter(".jpg", filter_include_jpg);
	add_to_filter(".jpeg", filter_include_jpg);
	add_to_filter(".xpm", filter_include_xpm);
	add_to_filter(".tif", filter_include_tif);
	add_to_filter(".tiff", filter_include_tif);
	add_to_filter(".gif", filter_include_gif);
	add_to_filter(".png", filter_include_png);
	add_to_filter(".ppm", filter_include_ppm);
	add_to_filter(".pgm", filter_include_pgm);
	add_to_filter(".pcx", filter_include_pcx);
	add_to_filter(".bmp", filter_include_bmp);

	if (custom_filter)
		{
		gchar *buf = g_strdup(custom_filter);
		gchar *pos_ptr_b;
		gchar *pos_ptr_e = buf;
		while(pos_ptr_e[0] != '\0')
			{
			pos_ptr_b = pos_ptr_e;
			while (pos_ptr_e[0] != ';' && pos_ptr_e[0] != '\0') pos_ptr_e++;
			if (pos_ptr_e[0] == ';')
				{
				pos_ptr_e[0] = '\0';
				pos_ptr_e++;
				}
			add_to_filter(pos_ptr_b, TRUE);
			}
		g_free(buf);
		}
}

/*
 *-----------------------------------------------------------------------------
 * path list recursive (should probably be someplace else)
 *-----------------------------------------------------------------------------
 */

static gint path_list_sort_cb(gconstpointer a, gconstpointer b)
{
	return strcmp((gchar *)a, (gchar *)b);
}

GList *path_list_sort(GList *list)
{
	return g_list_sort(list, path_list_sort_cb);
}

static void path_list_recursive_append(GList **list, GList *dirs)
{
	GList *work;

	work = dirs;
	while (work)
		{
		const gchar *path = work->data;
		GList *f = NULL;
		GList *d = NULL;

		if (path_list(path, &f, &d))
			{
			f = path_list_filter(f, FALSE);
			f = path_list_sort(f);
			*list = g_list_concat(*list, f);

			d = path_list_filter(d, TRUE);
			d = path_list_sort(d);
			path_list_recursive_append(list, d);
			g_list_free(d);
			}

		work = work->next;
		}
}

GList *path_list_recursive(const gchar *path)
{
	GList *list = NULL;
	GList *d = NULL;

	if (!path_list(path, &list, &d)) return NULL;
	list = path_list_filter(list, FALSE);
	list = path_list_sort(list);

	d = path_list_filter(d, TRUE);
	d = path_list_sort(d);
	path_list_recursive_append(&list, d);
	path_list_free(d);

	return list;
}

/*
 *-----------------------------------------------------------------------------
 * text conversion utils (public)
 *-----------------------------------------------------------------------------
 */

gchar *text_from_size(int size)
{
	gchar *a, *b;
	gchar *s, *d;
	gint l, n, i;

	/* what I would like to use is printf("%'d", size)
	 * BUT: not supported on every libc :(
	 */

	a = g_strdup_printf("%d", size);
	l = strlen(a);
	n = (l - 1)/ 3;
	if (n < 1) return a;

	b = g_new(gchar, l + n + 1);

	s = a;
	d = b;
	i = l - n * 3;
	while (*s != '\0')
		{
		if (i < 1)
			{
			i = 3;
			*d = ',';
			d++;
			}

		*d = *s;
		s++;
		d++;
		i--;
		}
	*d = '\0';

	g_free(a);
	return b;
}

const gchar *text_from_time(time_t t)
{
	static gchar ret[64];
	struct tm *btime;

	btime = localtime(&t);

	/* the %x warning about 2 digit years is not an error */
	if (strftime(ret, sizeof(ret), "%x %H:%M", btime) < 1) return "";

	return ret;
}

/*
 *-----------------------------------------------------------------------------
 * misc
 *-----------------------------------------------------------------------------
 */

static FileData *file_data_new(gchar *name, struct stat *st)
{
	FileData *fd;

	fd = g_new0(FileData, 1);
	fd->name = g_strdup(name);
	fd->size = st->st_size;
	fd->date = st->st_mtime;

	return fd;
}

static void file_data_free(FileData *fd)
{
	g_free(fd->name);
	g_free(fd);
}

/*
 *-----------------------------------------------------------------------------
 * load file list (private)
 *-----------------------------------------------------------------------------
 */

static gint sort_dir_cb(void *a, void *b)
{
	return strcmp((gchar *)a, (gchar *)b);
}

static gint sort_file_cb(void *a, void *b)
{
	FileData *fa = a;
	FileData *fb = b;

	if (!file_sort_ascending)
		{
		fa = b;
		fb = a;
		}

	switch (sort_method)
		{
		case SORT_SIZE:
			if (fa->size < fb->size) return -1;
			if (fa->size > fb->size) return 1;
			return 0;
			break;
		case SORT_TIME:
			if (fa->date < fb->date) return -1;
			if (fa->date > fb->date) return 1;
			return 0;
			break;
#ifdef HAVE_STRVERSCMP
		case SORT_NUMBER:
			return strverscmp(fa->name, fb->name);
			break;
#endif
		case SORT_NAME:
		default:
			return strcmp(fa->name, fb->name);
			break;
		}
}

static void file_data_free_fe_cb(gpointer data, gpointer ignore)
{
	file_data_free((FileData *)data);
}

static void filelist_read(gchar *path)
{
	DIR *dp;
	struct dirent *dir;
	struct stat ent_sbuf;

	g_list_foreach(dir_list,(GFunc)g_free,NULL);
	g_list_free(dir_list);
	dir_list = NULL;

	g_list_foreach(file_list, file_data_free_fe_cb, NULL);
	g_list_free(file_list);
	file_list = NULL;

	if((dp = opendir(path))==NULL)
		{
		/* dir not found, or no read access - allow backup */
		dir_list = g_list_prepend(dir_list, g_strdup(".."));
		dir_list = g_list_prepend(dir_list, g_strdup("."));
		return;
		}

	while ((dir = readdir(dp)) != NULL)
		{
		/* skips removed files */
		if (dir->d_ino > 0)
			{
			gchar *name = dir->d_name;
			if (show_dot_files || !file_is_hidden(name))
				{
				gchar *filepath = g_strconcat(path, "/", name, NULL);
				if (stat(filepath,&ent_sbuf) >= 0 && S_ISDIR(ent_sbuf.st_mode))
					{
					/* we ignore the .thumbnails dir for cleanliness */
					if (strcmp(name, GQVIEW_CACHE_DIR) != 0)
						{
						dir_list = g_list_prepend(dir_list, g_strdup(name));
						}
					}
				else
					{
					if (file_is_in_filter(name))
						file_list = g_list_prepend(file_list, file_data_new(name, &ent_sbuf));
					}
				g_free(filepath);
				}
			}
		}

	closedir(dp);

	dir_list = g_list_sort(dir_list, (GCompareFunc) sort_dir_cb);
	file_list = g_list_sort(file_list, (GCompareFunc) sort_file_cb);
}

/*
 *-----------------------------------------------------------------------------
 * file list utilities to retrieve information (public)
 *-----------------------------------------------------------------------------
 */

gint file_count(void)
{
	return g_list_length(file_list);
}

gint file_selection_count(void)
{
	gint count = 0;
	GList *work = GTK_CLIST(file_clist)->selection;
	while(work)
		{
		count++;
		if (debug) printf("s = %d\n", GPOINTER_TO_INT(work->data));
		work = work->next;
		}

	if (debug) printf("files selected = %d\n", count);

	return count;
}

gint find_file_in_list(const gchar *path)
{
	GList *work = file_list;
	gchar *buf;
	const gchar *name;
	gint count = -1;

	if (!path) return -1;

	buf = remove_level_from_path(path);
	if (strcmp(buf, current_path) != 0)
		{
		g_free(buf);
		return -1;
		}
	g_free(buf);

	name = filename_from_path(path);
	while(work)
		{
		FileData *fd = work->data;
		count++;
		if (strcmp(name, fd->name) == 0) return count;
		work = work->next;
		}

	return -1;
}

gchar *file_get_path(gint row)
{
	gchar *path = NULL;
	FileData *fd = gtk_clist_get_row_data(GTK_CLIST(file_clist), row);

	if (fd) path = g_strconcat(current_path, "/", fd->name, NULL);

	return path;
}

gint file_is_selected(gint row)
{
	GList *work = GTK_CLIST(file_clist)->selection;

	while(work)
		{
		if (GPOINTER_TO_INT(work->data) == row) return TRUE;
		work = work->next;
		}

	return FALSE;
}

/*
 *-----------------------------------------------------------------------------
 * utilities to retrieve list of selected files (public)
 *-----------------------------------------------------------------------------
 */

GList *file_get_complete_list(void)
{
	GList *list = NULL;
	GList *work = file_list;

	while(work)
		{
		FileData *fd = work->data;
		list = g_list_prepend(list, g_strconcat(current_path, "/", fd->name, NULL));
		work = work->next;
		}

	list = g_list_reverse(list);

	return list;
}

GList *file_get_selected_list(void)
{
	GList *list = NULL;
	GList *work = GTK_CLIST(file_clist)->selection;

	while(work)
		{
		FileData *fd = gtk_clist_get_row_data(GTK_CLIST(file_clist), GPOINTER_TO_INT(work->data));
		if (fd) list = g_list_prepend(list, g_strconcat(current_path, "/", fd->name, NULL));
		work = work->next;
		}

	list = g_list_reverse(list);

	return list;
}

gint file_clicked_is_selected(void)
{
	return file_is_selected(filelist_click_row);
}

gchar *file_clicked_get_path(void)
{
	return file_get_path(filelist_click_row);
}

/*
 *-----------------------------------------------------------------------------
 * image change routines
 *-----------------------------------------------------------------------------
 */

void file_image_change_to(gint row)
{
	gtk_clist_unselect_all(GTK_CLIST(file_clist));
	gtk_clist_select_row(GTK_CLIST(file_clist), row, -1);
	if (gtk_clist_row_is_visible(GTK_CLIST(file_clist), row) != GTK_VISIBILITY_FULL)
		{
		gtk_clist_moveto(GTK_CLIST(file_clist), row, -1, 0.5, 0.0);
		}
	main_image_slideshow_continue_check();
}

/*
 *-----------------------------------------------------------------------------
 * (un)select utils
 *-----------------------------------------------------------------------------
 */

void file_select_all(void)
{
	gtk_clist_select_all(GTK_CLIST(file_clist));
}

void file_unselect_all(void)
{
	gtk_clist_unselect_all(GTK_CLIST(file_clist));
}

/*
 *-----------------------------------------------------------------------------
 * file delete/rename update routines
 *-----------------------------------------------------------------------------
 */

static gint file_find_closest_unaccounted(gint row, gint count, GList *ignore_list)
{
	GList *list = NULL;
	GList *work;
	gint rev = row - 1;
	row ++;

	work = ignore_list;
	while(work)
		{
		gint f = find_file_in_list(work->data);
		if (f >= 0) list = g_list_append(list, GINT_TO_POINTER(f));
		work = work->next;
		}

	while(list)
		{
		gint c = TRUE;
		work = list;
		while(work && c)
			{
			gpointer p = work->data;
			work = work->next;
			if (row == GPOINTER_TO_INT(p))
				{
				row++;
				c = FALSE;
				}
			if (rev == GPOINTER_TO_INT(p))
				{
				rev--;
				c = FALSE;
				}
			if (!c) list = g_list_remove(list, p);
			}
		if (c && list)
			{
			g_list_free(list);
			list = NULL;
			}
		}
	if (row > count - 1)
		{
		if (rev < 0)
			return -1;
		else
			return rev;
		}
	else
		{
		return row;
		}
}

void file_is_gone(const gchar *path, GList *ignore_list)
{
	GList *list;
	FileData *fd;
	gint row;
	gint new_row = -1;
	row = find_file_in_list(path);
	if (row < 0) return;

	if (file_is_selected(row) /* && file_selection_count() == 1 */)
		{
		gint n = file_count();
		if (ignore_list)
			{
			new_row = file_find_closest_unaccounted(row, n, ignore_list);
			if (debug) printf("row = %d, closest is %d\n", row, new_row);
			}
		else
			{
			if (row + 1 < n)
				{
				new_row = row + 1;
				}
			else if (row > 0)
				{
				new_row = row - 1;
				}
			}
		gtk_clist_unselect_all(GTK_CLIST(file_clist));
		if (new_row >= 0)
			{
			gtk_clist_select_row(GTK_CLIST(file_clist), new_row, -1);
			file_image_change_to(new_row);
			}
		else
			{
			main_image_change_to(NULL, NULL);
			}
		}

	gtk_clist_remove(GTK_CLIST(file_clist), row);
	list = g_list_nth(file_list, row);
	fd = list->data;
	file_list = g_list_remove(file_list, fd);
	file_data_free(fd);
	update_status_label(NULL);
}

void file_is_renamed(const gchar *source, const gchar *dest)
{
	gint row;
	gchar *source_base;
	gchar *dest_base;

	if (main_image_get_path() && !strcmp(source, main_image_get_path()))
		{
		main_image_set_path(dest);
		}

	row = find_file_in_list(source);
	if (row < 0) return;

	source_base = remove_level_from_path(source);
	dest_base = remove_level_from_path(dest);

	if (strcmp(source_base, dest_base) == 0)
		{
		FileData *fd;
		gint n;
		GList *work = g_list_nth(file_list, row);
		fd = work->data;
		file_list = g_list_remove(file_list, fd);
		g_free(fd->name);
		fd->name = g_strdup(filename_from_path(dest));
		file_list = g_list_insert_sorted(file_list, fd, (GCompareFunc) sort_file_cb);
		n = g_list_index(file_list, fd);

		if (gtk_clist_get_cell_type(GTK_CLIST(file_clist), row, 0) != GTK_CELL_PIXTEXT)
			{
			gtk_clist_set_text (GTK_CLIST(file_clist), row, 0, fd->name);
			}
		else
			{
			guint8 spacing = 0;
			GdkPixmap *pixmap = NULL;
			GdkBitmap *mask = NULL;
			gtk_clist_get_pixtext(GTK_CLIST(file_clist), row, 0,
				NULL, &spacing, &pixmap, &mask);
			gtk_clist_set_pixtext(GTK_CLIST(file_clist), row, 0,
				fd->name, spacing, pixmap, mask);
			}

		gtk_clist_set_row_data(GTK_CLIST(file_clist), row, fd);
		gtk_clist_row_move(GTK_CLIST(file_clist), row, n);
		}
	else
		{
		GList *work = g_list_nth(file_list, row);
		FileData *fd = work->data;
		file_list = g_list_remove(file_list, fd);
		gtk_clist_remove(GTK_CLIST(file_clist), row);
		file_data_free(fd);
		update_status_label(NULL);
		}

	g_free(source_base);
	g_free(dest_base);
	
}

void file_is_moved(const gchar *source, const gchar *dest, GList *ignore_list)
{
	if (!source) return;

	if (main_image_get_collection(NULL) &&
	    main_image_get_path() && strcmp(source, main_image_get_path()) == 0)
		{
		main_image_set_path(dest);
		}

	file_is_gone(source, ignore_list);
}

static gint file_rename_edit_cb(ClistEditData *ced, const gchar *old, const gchar *new, gpointer data)
{
	gchar *old_path;
	gchar *new_path;

	old_path = g_strconcat(current_path, "/", old, NULL);
	new_path = g_strconcat(current_path, "/", new, NULL);

	if (strlen(new) == 0 || strchr(new, '/') != NULL)
		{
		gchar *text = g_strdup_printf(_("Invalid file name:\n%s"), new);
		file_util_warning_dialog(_("Error renaming file"), text);
		g_free(text);
		}
	else if (isfile(new_path))
		{
		gchar *text = g_strdup_printf(_("File named %s already exists."), new);
		file_util_warning_dialog(_("Error renaming file"), text);
		g_free(text);
		}
	else if (rename (old_path, new_path) < 0)
		{
		gchar *text = g_strdup_printf(_("Unable to rename file:\n%s\nto:\n%s"), old, new);
		file_util_warning_dialog(_("Error renaming file"), text);
		g_free(text);
		}
	else
		{
		file_maint_renamed(old_path, new_path);
		}

	g_free(old_path);
	g_free(new_path);

	return FALSE;
}

/*
 *-----------------------------------------------------------------------------
 * directory list callbacks
 *-----------------------------------------------------------------------------
 */

gchar *dir_get_path(gint row)
{
	gchar *name;
	gchar *new_path;

	name = gtk_clist_get_row_data (GTK_CLIST(dir_clist), row);

	if (!name) return NULL;

	if (strcmp(name, ".") == 0)
		{
		new_path = g_strdup(current_path);
		}
	else if (strcmp(name, "..") == 0)
		{
		new_path = remove_level_from_path(current_path);
		}
	else
		{
		if (strcmp(current_path, "/") == 0)
			new_path = g_strconcat(current_path, name, NULL);
		else
			new_path = g_strconcat(current_path, "/", name, NULL);
		}

	return new_path;
}

void dir_select_cb(GtkWidget *widget, gint row, gint col,
		   GdkEvent *event, gpointer data)
{
	gchar *path;

	path = dir_get_path(row);
	filelist_change_to(path);
	g_free(path);
}

static void dir_popup_destroy_cb(GtkWidget *widget, gpointer data)
{
	dir_clist_set_highlight(FALSE);
}

void dir_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	gint row = -1;
	gint col = -1;

	gtk_clist_get_selection_info(GTK_CLIST (widget), bevent->x, bevent->y, &row, &col);

	gtk_object_set_data(GTK_OBJECT(dir_clist), "click_row", GINT_TO_POINTER(row));

	if (bevent->button == 3)
		{
		GtkWidget *popup;

		dir_clist_set_highlight(TRUE);

		popup = menu_popup_dir_list((row >= 0));
		gtk_signal_connect(GTK_OBJECT(popup), "destroy",
				   GTK_SIGNAL_FUNC(dir_popup_destroy_cb), NULL);

		gtk_menu_popup(GTK_MENU(popup), NULL, NULL, NULL, NULL,
				bevent->button, bevent->time);
		}
}

/*
 *-----------------------------------------------------------------------------
 * file list callbacks
 *-----------------------------------------------------------------------------
 */

static void file_popup_destroy_cb(GtkWidget *widget, gpointer data)
{
	file_clist_highlight_unset();
}

void file_press_cb(GtkWidget *widget, GdkEventButton *bevent, gpointer data)
{
	gint row = -1;
	gint col = -1;

	gtk_clist_get_selection_info (GTK_CLIST (widget), bevent->x, bevent->y, &row, &col);

	filelist_click_row = row;

	if (bevent->button == 3)
		{
		GtkWidget *popup;

		file_clist_highlight_set();

		popup = menu_popup_file_list((row >= 0));

		gtk_signal_connect(GTK_OBJECT(popup), "destroy",
				   GTK_SIGNAL_FUNC(file_popup_destroy_cb), NULL);

		gtk_menu_popup (GTK_MENU(popup), NULL, NULL, NULL, NULL,
				bevent->button, bevent->time);
		return;
		}

	if (row == -1 || col == -1) return;

	if (bevent->button == 1 && bevent->type == GDK_BUTTON_PRESS)
		{
		if (bevent->state & GDK_SHIFT_MASK || bevent->state & GDK_CONTROL_MASK)
			{
			filelist_rename_row = -1;
			}
		else
			{
			if (bevent->time - filelist_click_time > 333)	/* 1/3 sec */
				{
				if (enable_in_place_rename && filelist_click_row == row && col == 0 && file_is_selected(row))
					{
					/* set up rename window */
					clist_edit_by_row(GTK_CLIST(file_clist), row, 0,
							  file_rename_edit_cb, NULL);
					return;
					}
				}
			else
				{
				filelist_rename_row = row;
				}
			}
		filelist_click_time = bevent->time;
		}
}

void file_select_cb(GtkWidget *widget, gint row, gint col,
		   GdkEvent *event, gpointer data)
{
	FileData *fd;
	gchar *path;
	gchar *read_ahead_path = NULL;

	if (file_selection_count() != 1)
		{
		update_status_label(NULL);
		return;
		}

	fd = gtk_clist_get_row_data(GTK_CLIST(file_clist), row);
	path = g_strconcat(current_path, "/", fd->name, NULL);
	if (enable_read_ahead)
		{
		if (row > find_file_in_list(main_image_get_path()) && row + 1 < file_count())
			{
			fd = gtk_clist_get_row_data(GTK_CLIST(file_clist), row + 1);
			}
		else if (row > 0)
			{
			fd = gtk_clist_get_row_data(GTK_CLIST(file_clist), row - 1);
			}
		else
			{
			fd = NULL;
			}
		if (fd) read_ahead_path = g_strconcat(current_path, "/", fd->name, NULL);
		}
	main_image_change_to(path, read_ahead_path);
	update_status_label(NULL);
	g_free(path);
	g_free(read_ahead_path);
}

void file_unselect_cb(GtkWidget *widget, gint row, gint col,
		   GdkEvent *event, gpointer data)
{
#if 0
	FileData *fd;
	gchar *path;

	fd = gtk_clist_get_row_data(GTK_CLIST(file_clist), row);
	path = g_strconcat(current_path, "/", fd->name, NULL);

	if (strcmp(path, image_get_path()) == 0)
		{
		if (file_selection_count() > 0 && !file_is_selected(find_file_in_list(image_get_path())) )
			{
			gint new_row = GPOINTER_TO_INT(GTK_CLIST(file_clist)->selection->data);
			gchar *new_name = gtk_clist_get_row_data(GTK_CLIST(file_clist), new_row);
			gchar *new_path = g_strconcat(current_path, "/", new_name, NULL);
			image_change_to(new_path);
			g_free(new_path);
			}
		}
	g_free(path);
#endif
	update_status_label(NULL);
}

/*
 *-----------------------------------------------------------------------------
 * file list highlight utils
 *-----------------------------------------------------------------------------
 */

void dir_clist_set_highlight(gint set)
{
	gint row = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(dir_clist), "click_row"));

	if (row < 0) return;

	if (set)
		{
		gtk_clist_set_background(GTK_CLIST(dir_clist), row,
			&GTK_WIDGET (dir_clist)->style->bg[GTK_STATE_ACTIVE]);
		gtk_clist_set_foreground(GTK_CLIST(dir_clist), row,
			&GTK_WIDGET (dir_clist)->style->fg[GTK_STATE_ACTIVE]);
		}
	else
		{
		gtk_clist_set_background(GTK_CLIST(dir_clist), row, NULL);
		gtk_clist_set_foreground(GTK_CLIST(dir_clist), row, NULL);
		}
}

void file_clist_highlight_set(void)
{
	if (file_clicked_is_selected()) return;

	gtk_clist_set_background(GTK_CLIST(file_clist), filelist_click_row,
		&GTK_WIDGET (file_clist)->style->bg[GTK_STATE_ACTIVE]);
	gtk_clist_set_foreground(GTK_CLIST(file_clist), filelist_click_row,
		&GTK_WIDGET (file_clist)->style->fg[GTK_STATE_ACTIVE]);
}

void file_clist_highlight_unset(void)
{
	if (file_clicked_is_selected()) return;

	gtk_clist_set_background(GTK_CLIST(file_clist), filelist_click_row, NULL);
	gtk_clist_set_foreground(GTK_CLIST(file_clist), filelist_click_row, NULL);
}

/*
 *-----------------------------------------------------------------------------
 * path entry and history menu
 *-----------------------------------------------------------------------------
 */

void path_entry_changed_cb(GtkWidget *widget, gpointer data)
{
	GList *work;
	gchar *buf;

	buf = g_strdup(gtk_entry_get_text(GTK_ENTRY(path_entry)));
	if (!buf || strcmp(buf, current_path) == 0)
		{
		g_free(buf);
		return;
		}

	work = history_list_get_by_key("path_list");
	while(work)
		{
		gchar *path = work->data;
		work = work->next;

		if (strcmp(path, buf) == 0)
			{
			filelist_change_to(buf);
			work = NULL;
			}
		}

	g_free(buf);
}

void path_entry_tab_cb(const gchar *newdir, gpointer data)
{
	gchar *new_path;
	gchar *buf;
	gint found = FALSE;

	new_path = g_strdup(newdir);
	parse_out_relatives(new_path);
	buf = remove_level_from_path(new_path);

	if (buf && current_path && strcmp(buf, current_path) == 0)
		{
		GList *work;
		const gchar *part;

		part = filename_from_path(new_path);
		work = file_list;

		while(part && work)
			{
			FileData *fd = work->data;
			work = work->next;

			if (strncmp(part, fd->name, strlen(part)) == 0)
				{
				gint row = g_list_index(file_list, fd);
				if (!gtk_clist_row_is_visible(GTK_CLIST(file_clist), row) != GTK_VISIBILITY_FULL)
					{
					gtk_clist_moveto(GTK_CLIST(file_clist), row, -1, 0.5, 0.0);
					}
				found = TRUE;
				break;
				}
			}
		}

	if (!found && new_path && current_path &&
	    strcmp(new_path, current_path) != 0 && isdir(new_path))
		{
		filelist_change_to(new_path);
		/* we are doing tab completion, add '/' back */
		gtk_entry_append_text(GTK_ENTRY(path_entry), "/");
		}

	g_free(buf);
	g_free(new_path);
}

void path_entry_cb(const gchar *newdir, gpointer data)
{
	gchar *new_path = g_strdup(newdir);
	parse_out_relatives(new_path);

	filelist_change_to_full_path(new_path);

	g_free(new_path);
}

static void history_menu_select_cb(GtkWidget *widget, gpointer data)
{
	gchar *new_path = data;
	filelist_change_to(new_path);
}

static gchar *truncate_hist_text(gchar *text, gint width, GdkFont *font)
{
	gchar *ptr;
	gchar *end;
	gint l;
	const gchar *ab = "...";
	gint ab_length;

	/* keep it useable (sane) */
	width = MAX(100, width);

	l = gdk_string_width(font, text);

	if (l < width)
		{
		return g_strdup(text);
		}

	ab_length = gdk_string_width(font, ab);	/* use the _wc version to support intl ? */
	ptr = text;
	end = text + strlen(text);

	while (ptr < end && l >= width - ab_length)
		{
		l -= gdk_text_width(font, ptr, 1);
		ptr++;
		}

	return g_strconcat(ab, ptr, NULL);
}

/*
 * peeked at gtkoptionmenu.c for these values
 * option_menu_char_space returns the space available for text.
 */

#define IND_WIDTH		12
#define IND_LEFT_PAD		3
#define IND_RIGHT_PAD		7
#define CHILD_LEFT_SPACING	5
#define CHILD_RIGHT_SPACING	1

static gint option_menu_char_space(GtkWidget *opmenu)
{
	gint width = 0;
	gint indicator_width = 0;
	gint button_padding;
	GtkWidget *menu;

	if (!opmenu) return 0;

	width = opmenu->allocation.width;
#if 0
	/* these calls are new in gtk 1.2.9, disabled for compatibility w/ older versions. */
	indicator_width += gtk_style_get_prop_experimental(opmenu->style,
							   "GtkOptionMenu::indicator_width", IND_WIDTH);
	indicator_width += gtk_style_get_prop_experimental(opmenu->style,
							   "GtkOptionMenu::indicator_left_spacing", IND_LEFT_PAD);
	indicator_width += gtk_style_get_prop_experimental(opmenu->style,
							   "GtkOptionMenu::indicator_right_spacing", IND_RIGHT_PAD);
#else
	indicator_width += IND_WIDTH + IND_LEFT_PAD + IND_RIGHT_PAD;
#endif

	button_padding = (GTK_CONTAINER(opmenu)->border_width + opmenu->style->klass->xthickness) * 2 +
			 CHILD_LEFT_SPACING + CHILD_RIGHT_SPACING + 2;

	menu = GTK_OPTION_MENU(opmenu)->menu;
	if (menu)
		{
		/* muliply by 4 to do this twice, accounting for menuitem borders */
		button_padding += 2 + (GTK_CONTAINER(menu)->border_width + menu->style->klass->xthickness) * 4;
		}

	return (width - indicator_width -  button_padding);
}

static void filelist_set_history(gchar *path)
{
	static GList *history_list = NULL;
	gchar *buf;
	gchar *buf_ptr;
	GtkWidget *menu;
	GtkWidget *item;
	gint width;
	GdkFont *font;

	if (!path) return;

	gtk_entry_set_text(GTK_ENTRY(path_entry), current_path);
	tab_completion_append_to_history(path_entry, current_path);

	/* get max size of string to display */

	width = option_menu_char_space(history_menu);
	font = history_menu->style->font;

	if (history_list)
                {
                g_list_foreach(history_list, (GFunc)g_free, NULL);
                g_list_free(history_list);
                history_list = NULL;
                }

	menu = gtk_menu_new();

	buf = g_strdup(path);
	buf_ptr = buf + strlen(buf) - 1 ;
	while (buf_ptr > buf)
		{
		gchar *full_path;
		gchar *truncated;
		truncated = truncate_hist_text(buf, width, font);

		full_path = g_strdup(buf);
		history_list = g_list_append(history_list, full_path);

		item = gtk_menu_item_new_with_label (truncated);
		gtk_signal_connect (GTK_OBJECT (item), "activate",
			(GtkSignalFunc) history_menu_select_cb, full_path);

		gtk_menu_append (GTK_MENU (menu), item);
		gtk_widget_show (item);

		g_free(truncated);

		while (buf_ptr[0] != '/' && buf_ptr > buf) buf_ptr--;
		buf_ptr[0] = '\0';
		}
	g_free(buf);

	item = gtk_menu_item_new_with_label ("/");

	gtk_signal_connect (GTK_OBJECT (item), "activate",
		(GtkSignalFunc) history_menu_select_cb, "/");

	gtk_menu_append (GTK_MENU (menu), item);
	gtk_widget_show (item);

	gtk_option_menu_set_menu(GTK_OPTION_MENU(history_menu), menu);
}

/*
 *-----------------------------------------------------------------------------
 * list update routines (public)
 *-----------------------------------------------------------------------------
 */

void filelist_populate_clist(void)
{
	GList *work;
	gint width;
	gint tmp_width;
	gint row;
	const gchar *image_name = NULL;
	gchar *buf;

	gint row_p = 0;
	gchar *text;
	guint8 spacing;
	GdkPixmap *nopixmap;
	GdkBitmap *nomask;

	interrupt_thumbs();

	filelist_set_history(current_path);

	gtk_clist_freeze (GTK_CLIST (dir_clist));
	gtk_clist_clear (GTK_CLIST (dir_clist));

	width = 0;
	work = dir_list;
	while(work)
		{
		gchar *buf[2];
		buf[0] = work->data;
		buf[1] = NULL;
		row = gtk_clist_append(GTK_CLIST(dir_clist), buf);
		gtk_clist_set_row_data (GTK_CLIST(dir_clist), row, work->data);
		tmp_width = gdk_string_width(dir_clist->style->font, buf[0]);
		if (tmp_width > width) width = tmp_width;
		work = work->next;
		}

	gtk_clist_set_column_width(GTK_CLIST(dir_clist), 0, width);
	gtk_clist_thaw(GTK_CLIST (dir_clist));

	buf = remove_level_from_path(main_image_get_path());
	if (buf && strcmp(buf, current_path) == 0)
		{
		image_name = main_image_get_name();
		}
	g_free(buf);

	gtk_clist_freeze (GTK_CLIST (file_clist));

	if (!thumbnails_enabled)
		{
		gtk_clist_set_row_height (GTK_CLIST(file_clist),
			GTK_WIDGET(file_clist)->style->font->ascent +
			GTK_WIDGET(file_clist)->style->font->descent + 1);
		}
	else
		{
		gtk_clist_set_row_height (GTK_CLIST(file_clist), thumb_max_height + 2);
		cache_maintain_home_dir(current_path, FALSE, FALSE);
		cache_maintain_dir(current_path, FALSE, FALSE);
		}

	width = 0;
	work = file_list;

	while(work)
		{
		gint has_pixmap = FALSE;
		gint match;
		FileData *fd = work->data;
		gint done = FALSE;

		while (!done)
			{
			if (GTK_CLIST(file_clist)->rows > row_p)
				{
				if (gtk_clist_get_cell_type(GTK_CLIST(file_clist),row_p, 0) == GTK_CELL_PIXTEXT)
					{
					gtk_clist_get_pixtext(GTK_CLIST(file_clist), row_p, 0, &text, &spacing, &nopixmap, &nomask);
					has_pixmap = TRUE;
					}
				else
					{
					gtk_clist_get_text(GTK_CLIST(file_clist), row_p, 0, &text);
					has_pixmap = FALSE;
					}
				match = strcmp(fd->name, text);
				}
			else
				{
				match = -1;
				}

			if (match < 0)
				{
				gchar *buf[4];

				buf[0] = fd->name;
				buf[1] = text_from_size(fd->size);
				buf[2] = (gchar *)text_from_time(fd->date);
				buf[3] = NULL;
				row = gtk_clist_insert(GTK_CLIST(file_clist), row_p, buf);
				g_free(buf[1]);

				gtk_clist_set_row_data (GTK_CLIST(file_clist), row, fd);
				if (thumbnails_enabled)
					{
					gtk_clist_set_shift(GTK_CLIST(file_clist), row, 0, 0, 5 + thumb_max_width);
					}
				done = TRUE;
				if (image_name && strcmp(fd->name, image_name) == 0)
					{
					gtk_clist_select_row(GTK_CLIST(file_clist), row, 0);
					}
				}
			else if (match > 0)
				{
				gtk_clist_remove(GTK_CLIST(file_clist), row_p);
				}
			else
				{
				if (thumbnails_enabled && !has_pixmap)
					{
					gtk_clist_set_shift(GTK_CLIST(file_clist), row_p, 0, 0, 5 + thumb_max_width);
					}
				if (!thumbnails_enabled)
					{
					gchar *buf;

					gtk_clist_set_text(GTK_CLIST(file_clist), row_p, 0, fd->name);

					buf = text_from_size(fd->size);
					gtk_clist_set_text(GTK_CLIST(file_clist), row_p, 1, buf);
					g_free(buf);

					gtk_clist_set_text(GTK_CLIST(file_clist), row_p, 2, text_from_time(fd->date));

					gtk_clist_set_shift(GTK_CLIST(file_clist), row_p, 0, 0, 0);
					}
				gtk_clist_set_row_data (GTK_CLIST(file_clist), row_p, fd);
				done = TRUE;
				}
			}
		row_p++;

		if (thumbnails_enabled)
			{
			tmp_width = gdk_string_width(file_clist->style->font, fd->name) + thumb_max_width + 5;
			}
		else
			{
			tmp_width = gdk_string_width(file_clist->style->font, fd->name);
			}
		if (tmp_width > width) width = tmp_width;
		work = work->next;
		}

	while (GTK_CLIST(file_clist)->rows > row_p)
		{
		gtk_clist_remove(GTK_CLIST(file_clist), row_p);
		}

	gtk_clist_set_column_width(GTK_CLIST(file_clist), 0, width);
	gtk_clist_thaw(GTK_CLIST (file_clist));

	update_status_label(NULL);

	filelist_update_thumbs();
}

void filelist_refresh(void)
{
	filelist_read(current_path);
	filelist_populate_clist();
	filelist_click_row = -1;
}

void filelist_change_to(const gchar *path)
{
	gchar *old_dir;
	gchar *buf;

	if (!isdir(path)) return;

	buf = remove_level_from_path(current_path);
	if (strcmp(buf, path) == 0)
		{
		old_dir = g_strdup(filename_from_path(current_path));
		}
	else
		{
		old_dir = NULL;
		}
	g_free(buf);

	g_free(current_path);
	current_path = g_strdup(path);

	gtk_clist_clear(GTK_CLIST(file_clist));
	filelist_refresh();

	if (old_dir)
		{
		/* make past dir visible in the list */
		gint row;
		gint found = FALSE;
		GList *work;

		row = -1;
		work = dir_list;
		while(work && !found)
			{
			if (strcmp(old_dir, work->data) == 0) found = TRUE;
			work = work->next;
			row++;
			}

		if (found) gtk_clist_moveto(GTK_CLIST(dir_clist), row, 0, 0.5, 0.0);
		g_free(old_dir);
		}
}

void filelist_change_to_full_path(const gchar *path)
{
	if (!isfile(path))
		{
		filelist_change_to(path);
		return;
		}
	else
		{
		gchar *sub_path;
		gint current;

		sub_path = remove_level_from_path(path);
		if (strcmp(current_path, sub_path) != 0)
			{
			filelist_change_to(sub_path);
			}
		else
			{
			filelist_refresh();
			}
		g_free(sub_path);

		current = find_file_in_list(path);
		if (current >= 0)
			{
			file_image_change_to(current);
			}
		else
			{
			main_image_change_to(path, NULL);
			}
		}
}

void filelist_set_sort_method(SortType method, gint force)
{
	GList *work;
	gint row;
	gint count;

	if (sort_method == method && !force) return;

	sort_method = method;

	if (!file_list) return;

	file_list = g_list_sort(file_list, (GCompareFunc) sort_file_cb);

	gtk_clist_freeze (GTK_CLIST (file_clist));

	count = 0;
	work = file_list;
	while(work)
		{
		row = gtk_clist_find_row_from_data (GTK_CLIST (file_clist), work->data);
		gtk_clist_row_move (GTK_CLIST (file_clist), row, count);
		work = work->next;
		count++;
		}

	gtk_clist_thaw (GTK_CLIST (file_clist));
}

/*
 *-----------------------------------------------------------------------------
 * thumb updates
 *-----------------------------------------------------------------------------
 */

static gint thumbs_running = 0;
static GList *thumbs_done_list = NULL;
static gint thumbs_count = 0;
static ThumbLoader *thumbs_loader;
static FileData *thumbs_filedata = NULL;

static gint filelist_update_thumb_next(void);

static void filelist_thumb_cleanup(void)
{
	update_status_label(NULL);

	update_progressbar(0.0);
	g_list_free(thumbs_done_list);
	thumbs_done_list = NULL;
	thumbs_count = 0;
	thumbs_running = FALSE;

	thumb_loader_free(thumbs_loader);
	thumbs_loader = NULL;

	thumbs_filedata = NULL;
}

void interrupt_thumbs(void)
{
        if (thumbs_running)
		{
		filelist_thumb_cleanup();
		}
}

static void filelist_update_thumb_do(ThumbLoader *tl, gint p, FileData *fd)
{
	GdkPixmap *pixmap = NULL;
	GdkBitmap *mask = NULL;
	gint spacing = -1;

	if (!fd) return;
	if (p < 0)
		{
		p = gtk_clist_find_row_from_data(GTK_CLIST(file_clist), fd);
		if (p < 0) return;
		}


	if (tl) spacing = thumb_loader_to_pixmap(thumbs_loader, &pixmap, &mask);

	if (spacing < 0)
		{
		/* unknown */
		spacing = thumb_max_width - thumb_from_xpm_d(img_unknown, thumb_max_width, thumb_max_height, &pixmap, &mask);
		}

	gtk_clist_set_pixtext(GTK_CLIST(file_clist), p, 0, fd->name, (guint8)(spacing + 5), pixmap, mask);
	gtk_clist_set_shift(GTK_CLIST(file_clist), p, 0, 0, 0);

	if (pixmap) gdk_pixmap_unref(pixmap);
	if (mask) gdk_bitmap_unref(mask);

	update_progressbar((gfloat)(thumbs_count) / GTK_CLIST(file_clist)->rows);
}

static void filelist_update_thumb_error_cb(ThumbLoader *tl, gpointer data)
{
	if (thumbs_filedata && thumbs_loader == tl)
		{
		filelist_update_thumb_do(NULL, -1, thumbs_filedata);
		}

	while (filelist_update_thumb_next());
}

static void filelist_update_thumb_done_cb(ThumbLoader *tl, gpointer data)
{
	if (thumbs_filedata && thumbs_loader == tl)
		{
		filelist_update_thumb_do(tl, -1, thumbs_filedata);
		}

	while(filelist_update_thumb_next());
}

static gint filelist_update_thumb_next(void)
{
	gint p = -1;
	gint r = -1;
	gint c = -1;

	gtk_clist_get_selection_info (GTK_CLIST(file_clist), 1, 1, &r, &c);

	/* find first visible undone */

	if (r != -1)
		{
		GList *work = g_list_nth(thumbs_done_list, r);
		while (work)
			{
			if (gtk_clist_row_is_visible(GTK_CLIST(file_clist), r))
				{
				if (!GPOINTER_TO_INT(work->data))
					{
					work->data = GINT_TO_POINTER(TRUE);
					p = r;
					work = NULL;
					}
				else
					{
					r++;
					work = work->next;
					}
				}
			else
				{
				work = NULL;
				}
			}
		}

	/* then find first undone */

	if (p == -1)
		{
		GList *work = thumbs_done_list;
		r = 0;
		while(work && p == -1)
			{
			if (!GPOINTER_TO_INT(work->data))
				{
				p = r;
				work->data = GINT_TO_POINTER(TRUE);
				}
			if (p == -1)
				{
				r++;
				work = work->next;
				if (!work)
					{
					filelist_thumb_cleanup();
					return FALSE;
					}
				}
			}
		}

	thumbs_count++;

	if (gtk_clist_get_cell_type(GTK_CLIST(file_clist), p, 0) != GTK_CELL_PIXTEXT)
		{
		FileData *fd;
		gchar *path;

		fd = gtk_clist_get_row_data(GTK_CLIST(file_clist), p);
		if (fd)
			{
			thumbs_filedata = fd;

			thumb_loader_free(thumbs_loader);

			path = g_strconcat (current_path, "/", fd->name, NULL);
			thumbs_loader = thumb_loader_new(path, thumb_max_width, thumb_max_height);
			g_free(path);

			thumb_loader_set_error_func(thumbs_loader, filelist_update_thumb_error_cb, NULL);

			if (!thumb_loader_start(thumbs_loader, filelist_update_thumb_done_cb, NULL))
				{
				/* set icon to unknown, continue */
				if (debug) printf("thumb loader start failed %s\n", thumbs_loader->path);
				filelist_update_thumb_do(NULL, p, fd);
				return TRUE;
				}
			return FALSE;
			}
		else
			{
			/* well, this should not happen, but handle it */
			if (debug) printf("thumb walked past end\n");
			filelist_thumb_cleanup();
			return FALSE;
			}
		}

	return TRUE;
}

static void filelist_update_thumbs(void)
{
	gint i;

	interrupt_thumbs();

	if (!thumbnails_enabled) return;

	update_status_label(_("Loading thumbs..."));

	for (i = 0; i < GTK_CLIST(file_clist)->rows; i++)
		{
		thumbs_done_list = g_list_prepend(thumbs_done_list, GINT_TO_POINTER(FALSE));
		}

	thumbs_running = TRUE;

	while (filelist_update_thumb_next());
}

