/*
 * GQview
 * (C) 2001 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License (GNU GPL).
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at your own risk!
 */


#include "gqview.h"
#include "utilops.h"


#include "cache_maint.h"
#include "collect.h"
#include "dupe.h"
#include "filelist.h"
#include "image.h"
#include "ui_fileops.h"
#include "ui_tabcomp.h"


/*
 *--------------------------------------------------------------------------
 * call these when names change, files move, deleted, etc.
 * so that any open windows are also updated
 *--------------------------------------------------------------------------
 */

void file_maint_renamed(const gchar *source, const gchar *dest)
{
	file_is_renamed(source, dest);
	collection_maint_renamed(source, dest);
	dupe_maint_renamed(source, dest);
	cache_maint_moved(source, dest);
}

/* under most cases ignore_list should be NULL */
void file_maint_removed(const gchar *path, GList *ignore_list)
{
	file_is_gone(path, ignore_list);
	collection_maint_removed(path);
	dupe_maint_removed(path);
	cache_maint_removed(path);
}

/* special case for correct main window behavior */
void file_maint_moved(const gchar *source, const gchar *dest, GList *ignore_list)
{
	file_is_moved(source, dest, ignore_list);
	collection_maint_renamed(source, dest);
	dupe_maint_renamed(source, dest);
	cache_maint_moved(source, dest);
}

/*
 *--------------------------------------------------------------------------
 * The file manipulation dialogs
 *--------------------------------------------------------------------------
 */


enum {
	DIALOG_NEW_DIR,
	DIALOG_COPY,
	DIALOG_MOVE,
	DIALOG_DELETE,
	DIALOG_RENAME
};

typedef struct _FileDataMult FileDataMult;
struct _FileDataMult
{
	gint confirm_all;
	gint confirmed;
	gint skip;
	GList *source_list;
	GList *source_next;
	gchar *dest_base;
	gchar *source;
	gchar *dest;
	gint copy;
};

typedef struct _FileDataSingle FileDataSingle;
struct _FileDataSingle
{
	gint confirmed;
	gchar *source;
	gchar *dest;
	gint copy;
};

/*
 *--------------------------------------------------------------------------
 * Adds 1 or 2 images (if 2, side by side) to a GenericDialog
 *--------------------------------------------------------------------------
 */

#define DIALOG_DEF_IMAGE_DIM_X 200
#define DIALOG_DEF_IMAGE_DIM_Y 150

static void generic_dialog_add_images(GenericDialog *gd, const gchar *path1, const gchar *path2)
{
	ImageWindow *imd;
	GtkWidget *hbox = NULL;
	GtkWidget *vbox;
	GtkWidget *sep;
	GtkWidget *label;

	if (!path1) return;

	sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(gd->vbox), sep, FALSE, FALSE, 5);
	gtk_widget_show(sep);

	if (path2)
		{
		hbox = gtk_hbox_new(TRUE, 5);
		gtk_box_pack_start(GTK_BOX(gd->vbox), hbox, TRUE, TRUE, 0);
		gtk_widget_show(hbox);
		}

	/* image 1 */

	vbox = gtk_vbox_new(FALSE, 0);
	if (hbox)
		{
		gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
		}
	else
		{
		gtk_box_pack_start(GTK_BOX(gd->vbox), vbox, TRUE, TRUE, 0);
		}
	gtk_widget_show(vbox);

	imd = image_new(TRUE);
	gtk_widget_set_usize(imd->widget, DIALOG_DEF_IMAGE_DIM_X, DIALOG_DEF_IMAGE_DIM_Y);
	gtk_box_pack_start(GTK_BOX(vbox), imd->widget, TRUE, TRUE, 0);
	image_change_path(imd, path1, 0.0);
	gtk_widget_show(imd->widget);

	label = gtk_label_new(filename_from_path(path1));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	/* image 2 */

	if (hbox && path2)
		{
		vbox = gtk_vbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(hbox), vbox, TRUE, TRUE, 0);
		gtk_widget_show(vbox);

		imd = image_new(TRUE);
		gtk_widget_set_usize(imd->widget, DIALOG_DEF_IMAGE_DIM_X, DIALOG_DEF_IMAGE_DIM_Y);
		gtk_box_pack_start(GTK_BOX(vbox), imd->widget, TRUE, TRUE, 0);
		image_change_path(imd, path2, 0.0);
		gtk_widget_show(imd->widget);

		label = gtk_label_new(filename_from_path(path2));
		gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
		gtk_widget_show(label);
		}
}

/*
 *--------------------------------------------------------------------------
 * Wrappers to aid in setting additional dialog properties (unde mouse, etc.)
 *--------------------------------------------------------------------------
 */

GenericDialog *file_util_gen_dlg(const gchar *title, const gchar *message,
				 const gchar *wmclass, const gchar *wmsubclass, gint auto_close,
				 void (*cancel_cb)(GenericDialog *, gpointer), gpointer data)
{
	GenericDialog *gd;

	gd = generic_dialog_new(title, message, wmclass, wmsubclass, auto_close, cancel_cb, data);
	if (place_dialogs_under_mouse)
		{
		gtk_window_position(GTK_WINDOW(gd->dialog), GTK_WIN_POS_MOUSE);
		}

	return gd;
}

FileDialog *file_util_file_dlg(const gchar *title, const gchar *message,
			       const gchar *wmclass, const gchar *wmsubclass,
			       void (*cancel_cb)(FileDialog *, gpointer), gpointer data)
{
	FileDialog *fd;

	fd = file_dialog_new(title, message, wmclass, wmsubclass, cancel_cb, data);
	if (place_dialogs_under_mouse)
		{
		gtk_window_position(GTK_WINDOW(GENERIC_DIALOG(fd)->dialog), GTK_WIN_POS_MOUSE);
		}

	return fd;
}

/* this warning dialog is copied from SLIK's ui_utildg.c,
 * because it does not have a mouse center option,
 * and we must center it before show, implement it here.
 */
static void file_util_warning_dialog_ok_cb(GenericDialog *gd, gpointer data)
{
	/* no op */
}

GenericDialog *file_util_warning_dialog(const gchar *title, const gchar *message)
{
	GenericDialog *gd;

	gd = file_util_gen_dlg(title, message, "GQview", "warning", TRUE, NULL, NULL);
	generic_dialog_add(gd, _("Ok"), file_util_warning_dialog_ok_cb, TRUE);
	if (place_dialogs_under_mouse)
		{
		gtk_window_position(GTK_WINDOW(gd->dialog), GTK_WIN_POS_MOUSE);
		}
	gtk_widget_show(gd->dialog);

	return gd;
}

/*
 *--------------------------------------------------------------------------
 * Move and Copy routines
 *--------------------------------------------------------------------------
 */

/*
 * Multi file move
 */

static FileDataMult *file_data_multiple_new(GList *source_list, gchar *dest, gint copy)
{
	FileDataMult *fdm = g_new0(FileDataMult, 1);
	fdm->confirm_all = FALSE;
	fdm->confirmed = FALSE;
	fdm->skip = FALSE;
	fdm->source_list = source_list;
	fdm->source_next = fdm->source_list;
	fdm->dest_base = g_strdup(dest);
	fdm->source = NULL;
	fdm->dest = NULL;
	fdm->copy = copy;
	return fdm;
}

static void file_data_multiple_free(FileDataMult *fdm)
{
	path_list_free(fdm->source_list);
	g_free(fdm->dest_base);
	g_free(fdm->dest);
	g_free(fdm);
}

static void file_util_move_multiple(FileDataMult *fdm);

static void file_util_move_multiple_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;

	fdm->confirmed = TRUE;
	file_util_move_multiple(fdm);
}

static void file_util_move_multiple_all_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;

	fdm->confirm_all = TRUE;
	file_util_move_multiple(fdm);
}

static void file_util_move_multiple_skip_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;

	fdm->skip = TRUE;
	file_util_move_multiple(fdm);
}

static void file_util_move_multiple_cancel_cb(GenericDialog *gd, gpointer data)
{
	FileDataMult *fdm = data;

	file_data_multiple_free(fdm);
}

static void file_util_move_multiple(FileDataMult *fdm)
{
	while (fdm->dest || fdm->source_next)
		{
		if (!fdm->dest)
			{
			GList *work = fdm->source_next;
			fdm->source = work->data;
			fdm->dest = g_strconcat(fdm->dest_base, "/", filename_from_path(fdm->source), NULL);
			fdm->source_next = work->next;
			}

		if (fdm->dest && fdm->source && strcmp(fdm->dest, fdm->source) == 0)
			{
			GenericDialog *gd;
			const gchar *title;
			gchar *text;

			if (fdm->copy)
				{
				title = _("Source to copy matches destination");
				text = g_strdup_printf(_("Unable to copy file:\n%s\nto itself."), fdm->dest);
				}
			else
				{
				title = _("Source to move matches destination");
				text = g_strdup_printf(_("Unable to move file:\n%s\nto itself."), fdm->dest);
				}

			gd = file_util_gen_dlg(title, text, "GQview", "dlg_confirm", TRUE,
						file_util_move_multiple_cancel_cb, fdm);
			g_free(text);
			generic_dialog_add(gd, _("Continue"), file_util_move_multiple_skip_cb, TRUE);

			gtk_widget_show(gd->dialog);
			return;
			}
		else if (isfile(fdm->dest) && !fdm->confirmed && !fdm->confirm_all && !fdm->skip)
			{
			GenericDialog *gd;
			gchar *text;

			text = g_strdup_printf(_("Overwrite file:\n %s\n with:\n %s"), fdm->dest, fdm->source);
			gd = file_util_gen_dlg(_("Overwrite file"), text, "GQview", "dlg_confirm", TRUE,
						file_util_move_multiple_cancel_cb, fdm);
			g_free(text);

			generic_dialog_add(gd, _("Yes"), file_util_move_multiple_ok_cb, TRUE);
			generic_dialog_add(gd, _("Yes to all"), file_util_move_multiple_all_cb, FALSE);
			generic_dialog_add(gd, _("Skip"), file_util_move_multiple_skip_cb, FALSE);
			generic_dialog_add_images(gd, fdm->dest, fdm->source);

			gtk_widget_show(gd->dialog);
			return;
			}
		else
			{
			gint success = FALSE;
			if (fdm->skip)
				{
				success = TRUE;
				fdm->skip = FALSE;
				}
			else
				{
				if (fdm->copy)
					{
					success = copy_file(fdm->source, fdm->dest);
					}
				else
					{
					if (move_file(fdm->source, fdm->dest))
						{
						success = TRUE;
						file_maint_moved(fdm->source, fdm->dest, fdm->source_list);
						}
					}
				}
			if (!success)
				{
				GenericDialog *gd;
				const gchar *title;
				gchar *text;

				if (fdm->copy)
					{
					title = _("Error copying file");
					text = g_strdup_printf(_("Unable to copy file:\n%sto:\n%s\n during multiple file copy."), fdm->source, fdm->dest);
					}
				else
					{
					title = _("Error moving file");
					text = g_strdup_printf(_("Unable to move file:\n%sto:\n%s\n during multiple file move."), fdm->source, fdm->dest);
					}
				gd = file_util_gen_dlg(title, text, "GQview", "dlg_confirm", TRUE,
							file_util_move_multiple_cancel_cb, fdm);
				g_free(text);

				generic_dialog_add(gd, _("Continue"), file_util_move_multiple_skip_cb, TRUE);

				gtk_widget_show(gd->dialog);
				return;
				}
			fdm->confirmed = FALSE;
			g_free(fdm->dest);
			fdm->dest = NULL;
			}
		}

	file_data_multiple_free(fdm);
}

/*
 * Single file move
 */

static FileDataSingle *file_data_single_new(gchar *source, gchar *dest, gint copy)
{
	FileDataSingle *fds = g_new0(FileDataSingle, 1);
	fds->confirmed = FALSE;
	fds->source = g_strdup(source);
	fds->dest = g_strdup(dest);
	fds->copy = copy;
	return fds;
}

static void file_data_single_free(FileDataSingle *fds)
{
	g_free(fds->source);
	g_free(fds->dest);
	g_free(fds);
}

static void file_util_move_single(FileDataSingle *fds);

static void file_util_move_single_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;

	fds->confirmed = TRUE;
	file_util_move_single(fds);
}

static void file_util_move_single_cancel_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;

	file_data_single_free(fds);
}

static void file_util_move_single(FileDataSingle *fds)
{
	if (fds->dest && fds->source && strcmp(fds->dest, fds->source) == 0)
		{
		file_util_warning_dialog(_("Source matches destination"),
					 _("Source and destination are the same, operation cancelled."));
		}
	else if (isfile(fds->dest) && !fds->confirmed)
		{
		GenericDialog *gd;
		gchar *text;

		text = g_strdup_printf(_("Overwrite file:\n%s\n with:\n%s"), fds->dest, fds->source);
		gd = file_util_gen_dlg(_("Overwrite file"), text, "GQview", "dlg_confirm", TRUE,
					file_util_move_single_cancel_cb, fds);
		g_free(text);

		generic_dialog_add(gd, _("Overwrite"), file_util_move_single_ok_cb, TRUE);
		generic_dialog_add_images(gd, fds->dest, fds->source);

		gtk_widget_show(gd->dialog);
		return;
		}
	else
		{
		gint success = FALSE;
		if (fds->copy)
			{
			success = copy_file(fds->source, fds->dest);
			}
		else
			{
			if (move_file(fds->source, fds->dest))
				{
				success = TRUE;
				file_maint_moved(fds->source, fds->dest, NULL);
				}
			}
		if (!success)
			{
			gchar *title;
			gchar *text;
			if (fds->copy)
				{
				title = _("Error copying file");
				text = g_strdup_printf(_("Unable to copy file:\n%s\nto:\n%s"), fds->source, fds->dest);
				}
			else
				{
				title = _("Error moving file");
				text = g_strdup_printf(_("Unable to move file:\n%s\nto:\n%s"), fds->source, fds->dest);
				}
			file_util_warning_dialog(title, text);
			g_free(text);
			}
		}

	file_data_single_free(fds);
}

/*
 * file move dialog
 */

static void file_util_move_do(FileDialog *fd)
{
	file_dialog_sync_history(fd, TRUE);

	if (fd->multiple_files)
		{
		file_util_move_multiple(file_data_multiple_new(fd->source_list, fd->dest_path, fd->type));
		fd->source_list = NULL;
		}
	else
		{
		if (isdir(fd->dest_path))
			{
			gchar *buf = g_strconcat(fd->dest_path, "/", filename_from_path(fd->source_path), NULL);
			gtk_entry_set_text(GTK_ENTRY(fd->entry), buf);
			g_free(buf);
			}
		file_util_move_single(file_data_single_new(fd->source_path, fd->dest_path, fd->type));
		}

	file_dialog_close(fd);
}

static void file_util_move_check(FileDialog *fd)
{
	if (fd->multiple_files && !isdir(fd->dest_path))
		{
		if (isfile(fd->dest_path))
			{
			file_util_warning_dialog(_("Invalid destination"),
						 _("When operating with multiple files, please select\n a directory, not file."));
			}
		else
			file_util_warning_dialog(_("Invalid directory"),
						 _("Please select an existing directory"));
		return;
		}

	file_util_move_do(fd);
}

static void file_util_move_cb(FileDialog *fd, gpointer data)
{
	file_util_move_check(fd);
}

static void file_util_move_cancel_cb(FileDialog *fd, gpointer data)
{
	file_dialog_close(fd);
}

static void real_file_util_move(const gchar *source_path, GList *source_list, const gchar *dest_path, gint copy)
{
	FileDialog *fd;
	gchar *path = NULL;
	gint multiple;
	gchar *text;
	const gchar *title;
	const gchar *op_text;

	if (!source_path && !source_list) return;

	if (source_path)
		{
		path = g_strdup(source_path);
		multiple = FALSE;
		}
	else if (source_list->next)
		{
		multiple = TRUE;
		}
	else
		{
		path = g_strdup(source_list->data);
		path_list_free(source_list);
		source_list = NULL;
		multiple = FALSE;
		}

	if (copy)
		{
		title = _("GQview - copy");
		op_text = _("Copy");
		if (path)
			text = g_strdup_printf(_("Copy file:\n%s\nto:"), path);
		else
			text = g_strdup(_("Copy multiple files to:"));
		}
	else
		{
		title = _("GQview - move");
		op_text = _("Move");
		if (path)
			text = g_strdup_printf(_("Move file:\n%s\nto:"), path);
		else
			text = g_strdup(_("Move multiple files to:"));
		}

	fd = file_util_file_dlg(title, text, "GQview", "dlg_copymove", file_util_move_cancel_cb, NULL);
	g_free(text);

	file_dialog_add_button(fd, op_text, file_util_move_cb, TRUE);

	file_dialog_add_path_widgets(fd, NULL, dest_path, "move_copy", NULL, NULL);

	fd->type = copy;
	fd->source_path = path;
	fd->source_list = source_list;
	fd->multiple_files = multiple;

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

void file_util_move(const gchar *source_path, GList *source_list, const gchar *dest_path)
{
	real_file_util_move(source_path, source_list, dest_path, FALSE);
}

void file_util_copy(const gchar *source_path, GList *source_list, const gchar *dest_path)
{
	real_file_util_move(source_path, source_list, dest_path, TRUE);
}

/*
 *--------------------------------------------------------------------------
 * Delete routines
 *--------------------------------------------------------------------------
 */

/*
 * delete multiple files
 */

static void file_util_delete_multiple_ok_cb(GenericDialog *gd, gpointer data);
static void file_util_delete_multiple_cancel_cb(GenericDialog *gd, gpointer data);

static void file_util_delete_multiple_ok_cb(GenericDialog *gd, gpointer data)
{
	GList *source_list = data;

	while(source_list)
		{
		gchar *path = source_list->data;

		source_list = g_list_remove(source_list, path);

		if (unlink(path) < 0)
			{
			if (source_list)
				{
				GenericDialog *d;
				gchar *text;

				text = g_strdup_printf(_("Unable to delete file:\n %s\n Continue multiple delete operation?"), path);
				d = file_util_gen_dlg(_("Delete failed"), text, "GQview", "dlg_confirm", TRUE,
						       file_util_delete_multiple_cancel_cb, source_list);
				g_free(text);

				generic_dialog_add(d, _("Continue"), file_util_delete_multiple_ok_cb, TRUE);
				gtk_widget_show(gd->dialog);
				}
			else
				{
				gchar *text;
				
				text = g_strdup_printf(_("Unable to delete file:\n%s"), path);
				file_util_warning_dialog(_("Delete failed"), text);
				g_free(text);
				}
			g_free(path);
			return;
			}
		else
			{
			file_maint_removed(path, source_list);
			}
		g_free(path);
		}
}

static void file_util_delete_multiple_cancel_cb(GenericDialog *gd, gpointer data)
{
	GList *source_list = data;

	path_list_free(source_list);
}

static void file_util_delete_multiple(GList *source_list)
{
	if (!confirm_delete)
		{
		file_util_delete_multiple_ok_cb(NULL, source_list);
		}
	else
		{
		GenericDialog *gd;

		gd = file_util_gen_dlg(_("Delete files"), _("About to delete multiple files..."),
					"GQview", "dlg_confirm", TRUE,
					file_util_delete_multiple_cancel_cb, source_list);

		generic_dialog_add(gd, _("Delete"), file_util_delete_multiple_ok_cb, TRUE);

		gtk_widget_show(gd->dialog);
		}
}

/*
 * delete single file
 */

static void file_util_delete_ok_cb(GenericDialog *gd, gpointer data)
{
	gchar *path = data;

	if (unlink (path) < 0)
		{
		gchar *text = g_strdup_printf(_("Unable to delete file:\n%s"), path);
		file_util_warning_dialog(_("File deletion failed"), text);
		g_free(text);
		}
	else
		{
		file_maint_removed(path, NULL);
		}

	g_free(path);
}

static void file_util_delete_cancel_cb(GenericDialog *gd, gpointer data)
{
	gchar *path = data;

	g_free(path);
}

static void file_util_delete_single(const gchar *path)
{
	gchar *buf = g_strdup(path);

	if (!confirm_delete)
		{
		file_util_delete_ok_cb(NULL, buf);
		}
	else
		{
		GenericDialog *gd;
		gchar *text;

		text = g_strdup_printf(_("About to delete the file:\n %s"), buf);
		gd = file_util_gen_dlg(_("Delete file"), text, "GQview", "dlg_confirm", TRUE,
					file_util_delete_cancel_cb, buf);
		g_free(text);

		generic_dialog_add(gd, _("Delete"), file_util_delete_ok_cb, TRUE);

		gtk_widget_show(gd->dialog);
		}
}

void file_util_delete(const gchar *source_path, GList *source_list)
{
	if (!source_path && !source_list) return;

	if (source_path)
		{
		file_util_delete_single(source_path);
		}
	else if (!source_list->next)
		{
		file_util_delete_single(source_list->data);
		path_list_free(source_list);
		}
	else
		{
		file_util_delete_multiple(source_list);
		}
}

/*
 *--------------------------------------------------------------------------
 * Rename routines
 *--------------------------------------------------------------------------
 */

/*
 * rename multiple files
 */

static void file_util_rename_multiple(FileDialog *fd);

static void file_util_rename_multiple_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDialog *fd = data;

	if (!GTK_WIDGET_VISIBLE(GENERIC_DIALOG(fd)->dialog)) gtk_widget_show(GENERIC_DIALOG(fd)->dialog);

	fd->type = TRUE;
	file_util_rename_multiple(fd);
}

static void file_util_rename_multiple_cancel_cb(GenericDialog *gd, gpointer data)
{
	FileDialog *fd = data;

	if (!GTK_WIDGET_VISIBLE(GENERIC_DIALOG(fd)->dialog)) gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

static void file_util_rename_multiple(FileDialog *fd)
{
	if (isfile(fd->dest_path) && !fd->type)
		{
		GenericDialog *gd;
		gchar *text;

		text = g_strdup_printf(_("Overwrite file:\n%s\nby renaming:\n%s"), fd->dest_path, fd->source_path);
		gd = file_util_gen_dlg(_("Overwrite file"), text, "GQview", "dlg_confirm", TRUE,
					file_util_rename_multiple_cancel_cb, fd);
		g_free(text);

		generic_dialog_add(gd, _("Overwrite"), file_util_rename_multiple_ok_cb, TRUE);
		generic_dialog_add_images(gd, fd->dest_path, fd->source_path);

		gtk_widget_hide(GENERIC_DIALOG(fd)->dialog);

		gtk_widget_show(gd->dialog);
		return;
		}
	else
		{
		if (rename (fd->source_path, fd->dest_path) < 0)
			{
			gchar *text = g_strdup_printf(_("Unable to rename file:\n%s\n to:\n%s"),
						      filename_from_path(fd->source_path),
						      filename_from_path(fd->dest_path));
			file_util_warning_dialog(_("Error renaming file"), text);
			g_free(text);
			}
		else
			{
			gint row;
			gint n;
			GtkWidget *clist;
			gchar *path;

			file_maint_renamed(fd->source_path, fd->dest_path);

			clist = gtk_object_get_data(GTK_OBJECT(fd->entry), "clist");
			path = gtk_object_get_data(GTK_OBJECT(clist), "source");
			row = gtk_clist_find_row_from_data(GTK_CLIST(clist), path);

			n = g_list_length(GTK_CLIST(clist)->row_list);
			if (debug) printf("r=%d n=%d\n", row, n);
			if (n - 1 > row)
				n = row;
			else if (n > 1)
				n = row - 1;
			else
				n = -1;

			if (n >= 0)
				{
				gtk_object_set_data(GTK_OBJECT(clist), "source", NULL);
				gtk_clist_remove(GTK_CLIST(clist), row);
				gtk_clist_select_row(GTK_CLIST(clist), n, -1);
				}
			else
				{
				if (debug) printf("closed by #%d\n", n);
				file_dialog_close(fd);
				}
			}
		}
}

static void file_util_rename_multiple_cb(FileDialog *fd, gpointer data)
{
	gchar *base;
	const gchar *name;

	name = gtk_entry_get_text(GTK_ENTRY(fd->entry));
	base = remove_level_from_path(fd->source_path);

	g_free(fd->dest_path);
	fd->dest_path = g_strconcat(base, "/", name, NULL);
	g_free(base);

	if (strlen(name) == 0 || strcmp(fd->source_path, fd->dest_path) == 0)
		{
		return;
		}

	fd->type = FALSE;
	file_util_rename_multiple(fd);
}

static void file_util_rename_multiple_close_cb(FileDialog *fd, gpointer data)
{
	file_dialog_close(fd);
}

static void file_util_rename_multiple_select_cb(GtkWidget *clist,
		gint row, gint column, GdkEventButton *bevent, gpointer data)
{
	FileDialog *fd = data;
	GtkWidget *label;
	const gchar *name;
	gchar *path;

	label = gtk_object_get_data(GTK_OBJECT(GENERIC_DIALOG(fd)->dialog), "label");
	path = gtk_clist_get_row_data(GTK_CLIST(clist), row);

	g_free(fd->source_path);
	fd->source_path = g_strdup(path);

	gtk_object_set_data(GTK_OBJECT(clist), "source", path);

	name = filename_from_path(fd->source_path);
	gtk_label_set(GTK_LABEL(label), name);
	gtk_entry_set_text(GTK_ENTRY(fd->entry), name);

	gtk_widget_grab_focus(fd->entry);
}

static void file_util_rename_multiple_do(GList *source_list)
{
	FileDialog *fd;
	GtkWidget *scrolled;
	GtkWidget *clist;
	GtkWidget *label;
	GList *work;

	fd = file_util_file_dlg(_("GQview - rename"), _("Rename multiple files:"), "GQview", "dlg_rename",
			     file_util_rename_multiple_close_cb, NULL);
	file_dialog_add_button(fd, _("Rename"), file_util_rename_multiple_cb, TRUE);

	fd->source_path = g_strdup(source_list->data);
	fd->dest_path = NULL;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolled),
				GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), scrolled, TRUE, TRUE, 0);
	gtk_widget_show(scrolled);

	clist = gtk_clist_new(1);
	gtk_clist_set_column_auto_resize(GTK_CLIST(clist), 0, TRUE);
	gtk_signal_connect(GTK_OBJECT(clist), "select_row",
			   GTK_SIGNAL_FUNC(file_util_rename_multiple_select_cb), fd);
	gtk_widget_set_usize(clist, 250, 150);
	gtk_container_add(GTK_CONTAINER(scrolled), clist);
	gtk_widget_show(clist);

	gtk_object_set_data(GTK_OBJECT(clist), "source", source_list->data);

	work = source_list;
	while(work)
		{
		gint row;
		gchar *buf[2];
		buf[0] = (gchar *)filename_from_path(work->data);
		buf[1] = NULL;
		row = gtk_clist_append(GTK_CLIST(clist), buf);
		gtk_clist_set_row_data_full(GTK_CLIST(clist), row,
					    work->data, (GtkDestroyNotify)g_free);
		work = work->next;
		}

	g_list_free(source_list);

	label = gtk_label_new(_("Rename:"));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	label = gtk_label_new(filename_from_path(fd->source_path));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);
	gtk_object_set_data(GTK_OBJECT(GENERIC_DIALOG(fd)->dialog), "label", label);

	label = gtk_label_new(_("to:"));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), label, FALSE, FALSE, 0);
	gtk_widget_show(label);

	fd->entry = gtk_entry_new();
	gtk_entry_set_text(GTK_ENTRY(fd->entry), filename_from_path(fd->source_path));
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), fd->entry, FALSE, FALSE, 0);
	generic_dialog_attach_default(GENERIC_DIALOG(fd), fd->entry);
	gtk_widget_grab_focus(fd->entry);
	gtk_widget_show(fd->entry);

	gtk_object_set_data(GTK_OBJECT(fd->entry), "clist", clist);

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

/*
 * rename single file
 */

static void file_util_rename_single(FileDataSingle *fds);

static void file_util_rename_single_ok_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;
	fds->confirmed = TRUE;
	file_util_rename_single(fds);
}

static void file_util_rename_single_cancel_cb(GenericDialog *gd, gpointer data)
{
	FileDataSingle *fds = data;
	file_data_single_free(fds);
}

static void file_util_rename_single(FileDataSingle *fds)
{
	if (isfile(fds->dest) && !fds->confirmed)
		{
		GenericDialog *gd;
		gchar *text;

		text = g_strdup_printf(_("Overwrite file:\n%s\nby renaming:\n%s"), fds->dest, fds->source);
		gd = file_util_gen_dlg(_("Overwrite file"), text, "GQview", "dlg_confirm", TRUE,
					file_util_rename_single_cancel_cb, fds);
		g_free(text);

		generic_dialog_add(gd, _("Overwrite"), file_util_rename_single_ok_cb, TRUE);
		generic_dialog_add_images(gd, fds->dest, fds->source);

		gtk_widget_show(gd->dialog);

		return;
		}
	else
		{
		if (rename(fds->source, fds->dest) < 0)
			{
			gchar *text = g_strdup_printf(_("Unable to rename file:\n%s\nto:\n%s"), filename_from_path(fds->source), filename_from_path(fds->dest));
			file_util_warning_dialog(_("Error renaming file"), text);
			g_free(text);
			}
		else
			{
			file_maint_renamed(fds->source, fds->dest);
			}
		}
	file_data_single_free(fds);
}

static void file_util_rename_single_cb(FileDialog *fd, gpointer data)
{
	const gchar *name;
	gchar *path;

	name = gtk_entry_get_text(GTK_ENTRY(fd->entry));
	path = g_strconcat(fd->dest_path, "/", name, NULL);

	if (strlen(name) == 0 || strcmp(fd->source_path, path) == 0)
		{
		g_free(path);
		return;
		}

	file_util_rename_single(file_data_single_new(fd->source_path, path, fd->type));

	g_free(path);
	file_dialog_close(fd);
}

static void file_util_rename_single_close_cb(FileDialog *fd, gpointer data)
{
	file_dialog_close(fd);
}

static void file_util_rename_single_do(const gchar *source_path)
{
	FileDialog *fd;
	gchar *text;
	const gchar *name = filename_from_path(source_path);

	text = g_strdup_printf(_("Rename file:\n%s\nto:"), name);
	fd = file_util_file_dlg(_("GQview - rename"), text, "GQview", "dlg_rename",
			     file_util_rename_single_close_cb, NULL);
	g_free(text);

	file_dialog_add_button(fd, _("Rename"), file_util_rename_single_cb, TRUE);

	fd->source_path = g_strdup(source_path);
	fd->dest_path = remove_level_from_path(source_path);

	fd->entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), fd->entry, FALSE, FALSE, 0);
	gtk_entry_set_text(GTK_ENTRY(fd->entry), filename_from_path(fd->source_path));
	gtk_entry_select_region(GTK_ENTRY(fd->entry), 0, strlen(gtk_entry_get_text(GTK_ENTRY(fd->entry))));
	generic_dialog_attach_default(GENERIC_DIALOG(fd), fd->entry);
	gtk_widget_grab_focus(fd->entry);
	gtk_widget_show(fd->entry);

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}

void file_util_rename(const gchar *source_path, GList *source_list)
{
	if (!source_path && !source_list) return;

	if (source_path)
		{
		file_util_rename_single_do(source_path);
		}
	else if (!source_list->next)
		{
		file_util_rename_single_do(source_list->data);
		path_list_free(source_list);
		}
	else
		{
		file_util_rename_multiple_do(source_list);
		}
}

/*
 *--------------------------------------------------------------------------
 * Create directory routines
 *--------------------------------------------------------------------------
 */

static void file_util_create_dir_do(const gchar *base, const gchar *name)
{
	gchar *path;

	path = g_strconcat(base, "/", name, NULL);

	if (isdir(path))
		{
		gchar *text = g_strdup_printf(_("The directory:\n%s\nalready exists."), name);
		file_util_warning_dialog(_("Directory exists"), text);
		g_free(text);
		}
	else if (isname(path))
		{
		gchar *text = g_strdup_printf(_("The path:\n%s\nalready exists as a file."), name);
		file_util_warning_dialog(_("Could not create directory"), text);
		g_free(text);
		}
	else
		{
		if (mkdir (path, 0755) < 0)
			{
			gchar *text = g_strdup_printf(_("Unable to create directory:\n%s"), name);
			file_util_warning_dialog(_("Error creating directory"), text);
			g_free(text);
			}
		else
			{
			if (strcmp(base, current_path) == 0)
				{
				gchar *buf = g_strdup(current_path);
				filelist_change_to(buf);
				g_free(buf);
				}
			}
		}

	g_free(path);
}

static void file_util_create_dir_cb(FileDialog *fd, gpointer data)
{
	const gchar *name;

	name = gtk_entry_get_text(GTK_ENTRY(fd->entry));

	if (strlen(name) == 0) return;

	if (name[0] == '/')
		{
		gchar *buf;
		buf  = remove_level_from_path(name);
		file_util_create_dir_do(buf, filename_from_path(name));
		g_free(buf);
		}
	else
		{
		file_util_create_dir_do(fd->dest_path, name);
		}

	file_dialog_close(fd);
}

static void file_util_create_dir_close_cb(FileDialog *fd, gpointer data)
{
	file_dialog_close(fd);
}

void file_util_create_dir(const gchar *path)
{
	FileDialog *fd;
	gchar *text;

	if (!isdir(path)) return;

	text = g_strdup_printf(_("Create directory in:\n%s\nnamed:"), path);
	fd = file_util_file_dlg(_("GQview - new directory"), text, "GQview", "dlg_newdir",
			     file_util_create_dir_close_cb, NULL);
	g_free(text);

	file_dialog_add_button(fd, _("Create"), file_util_create_dir_cb, TRUE);

	fd->dest_path = g_strdup(path);

	fd->entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(GENERIC_DIALOG(fd)->vbox), fd->entry, FALSE, FALSE, 0);
	generic_dialog_attach_default(GENERIC_DIALOG(fd), fd->entry);
	gtk_widget_grab_focus(fd->entry);
	gtk_widget_show(fd->entry);

	gtk_widget_show(GENERIC_DIALOG(fd)->dialog);
}





