/*****************************************************************************\
    hpijs.cpp : HP Inkjet Server

    Copyright (c) 2001 - 2002, Hewlett-Packard Co.
    All rights reserved.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions
    are met:
    1. Redistributions of source code must retain the above copyright
       notice, this list of conditions and the following disclaimer.
    2. Redistributions in binary form must reproduce the above copyright
       notice, this list of conditions and the following disclaimer in the
       documentation and/or other materials provided with the distribution.
    3. Neither the name of the Hewlett-Packard nor the names of its
       contributors may be used to endorse or promote products derived
       from this software without specific prior written permission.

    THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
    IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
    OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.
    IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT, INDIRECT,
    INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
    NOT LIMITED TO, PATENT INFRINGEMENT; PROCUREMENT OF SUBSTITUTE GOODS OR
    SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
    HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
    STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
    IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
    POSSIBILITY OF SUCH DAMAGE.
\*****************************************************************************/

#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <syslog.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include "header.h"
#include "ijs.h"
#include "ijs_server.h"
#include "services.h"
#include "hpijs.h"

extern int hpijs97();

int bug(const char *fmt, ...)
{
   char buf[256];
   va_list args;
   int n;

   va_start(args, fmt);

   if ((n = vsnprintf(buf, 256, fmt, args)) == -1)
      buf[255] = 0;     /* output was truncated */

   fprintf(stderr, buf);
   syslog(LOG_WARNING, buf);

   fflush(stderr);
   va_end(args);
   return n;
}

/* Set Print Context. */
int hpijs_set_context(UXServices *pSS)
{
   int r;

   if (pSS->PenSet != DUMMY_PEN)
   {
     if ((r = pSS->pPC->SetPenSet((PEN_TYPE)pSS->PenSet)) != NO_ERROR)
        bug("unable to SetPenSet set=%d, err=%d\n", pSS->PenSet, r);
   }

   if ((r = pSS->pPC->SelectPrintMode((QUALITY_MODE)pSS->Quality, (MEDIATYPE)pSS->MediaType, (COLORMODE)pSS->ColorMode)) !=  NO_ERROR)
   {
      bug("unable to set Quality=%d, ColorMode=%d, MediaType=%d, err=%d\n", pSS->Quality, pSS->ColorMode, pSS->MediaType, r);
      QUALITY_MODE   eQuality;
      MEDIATYPE      eMedia;
      COLORMODE      eColorMode;
      BOOL        bDevText;
      pSS->pPC->GetPrintModeSettings (eQuality, eMedia, eColorMode, bDevText);
      bug("following will be used Quality=%d, ColorMode=%d, MediaType=%d\n", eQuality, eColorMode, eMedia);
   }

   /* Map ghostscript paper size to APDK paper size. */
   pSS->MapPaperSize(pSS->PaperWidth, pSS->PaperHeight, pSS->Model); 

   /* Do duplex stuff now, since we have a valid print mode. */
   if (pSS->Duplex && !pSS->Tumble)
   {
      if (pSS->pPC->SelectDuplexPrinting(DUPLEXMODE_BOOK) != TRUE)
         bug("unable to set duplex mode=book\n");
   }
   else if (pSS->Duplex && pSS->Tumble)
   {
      if (pSS->pPC->SelectDuplexPrinting(DUPLEXMODE_TABLET) != TRUE)
         bug("unable to set duplex mode=tablet\n");
   }
   return 0;
}

int hpijs_status_cb(void *status_cb_data, IjsServerCtx *ctx, IjsJobId job_id)
{
  return 0;
}

int hpijs_list_cb(void *list_cb_data, IjsServerCtx *ctx, IjsJobId job_id,
        char *val_buf, int val_size)
{
  return 0;
}

int hpijs_enum_cb(void *enum_cb_data, IjsServerCtx *ctx, IjsJobId job_id,
       const char *key, char *val_buf, int val_size)
{
  if (!strcmp (key, "ColorSpace"))
    snprintf(val_buf, val_size, "%s", "sRGB");
  else if (!strcmp (key, "DeviceManufacturer"))
    snprintf(val_buf, val_size, "%s", "HEWLETT-PACKARD,APOLLO");
  else if (!strcmp (key, "PageImageFormat"))
    snprintf(val_buf, val_size, "%s", "Raster");
  else
    bug("unable to enum key=%s\n", key);    
  return 0;
}

/*
 * Set parameter (in the server) call back. Note, OutputFD is the only call that can be
 * preceded by set DeviceManufacturer and DeviceModel.
 */
int hpijs_set_cb (void *set_cb_data, IjsServerCtx *ctx, IjsJobId job_id,
        const char *key, const char *value, int value_size)
{
   UXServices *pSS = (UXServices*)set_cb_data;
   int fd, r;
   char *tail;
   int status = 0;
   char svalue[IJS_MAX_PARAM+1];   
   float w, h;

   /* Sanity check input value. */
   if (value_size > IJS_MAX_PARAM)
   {
     memcpy(svalue, value, IJS_MAX_PARAM);
     svalue[IJS_MAX_PARAM] = 0;
   }
   else
   {
     memcpy(svalue, value, value_size);
     svalue[value_size] = 0;
   }

   if (!strcmp (key, "OutputFD"))
   {
      fd = strtol(svalue, &tail, 10);
      pSS->OutputPath = fd;   /* set prn_stream as output of SS::ToDevice */
   }
   else if (!strcmp (key, "DeviceManufacturer"))
   {
      if ((strncmp(svalue, "HEWLETT-PACKARD", 15) != 0) && (strncmp(svalue, "APOLLO", 6) != 0))
      {
         bug("unable to set DeviceManufacturer=%s\n", svalue);
         status = -1;
      }
   }
   else if (!strcmp (key, "DeviceModel"))
   {
      /* Map device model to model number. */
      pSS->MapModel(svalue);

      if (pSS->Model == -1)
      {
         bug("unable to set DeviceModel=%s\n", svalue);
         status = -1;
      }

      if ((r = pSS->pPC->SelectDevice((PRINTER_TYPE)pSS->Model)) != NO_ERROR)
      {
         bug("unable to set device=%d, err=%d\n", pSS->Model, r);
         status = -1;
      }
   }
   else if ((strcmp (key, "PS:Duplex") == 0) || (strcmp (key, "Duplex") == 0))
   {
      if (strncmp(svalue, "true", 4) == 0)
         pSS->Duplex = 1;
      else
         pSS->Duplex = 0;
   }
   else if ((strcmp (key, "PS:Tumble") == 0) || (strcmp (key, "Tumble") == 0))
   {
      if (strncmp(svalue, "true", 4) == 0)
         pSS->Tumble = 1;
      else
         pSS->Tumble = 0;
   }
   else if (!strcmp (key, "PaperSize"))
   {
      w = (float)strtod(svalue, &tail);
      h = (float)strtod(tail+1, &tail);

      if (pSS->FirstRaster)
      {
         /* Normal start of print Job. */
         pSS->PaperWidth = w;
         pSS->PaperHeight = h;
		   hpijs_set_context(pSS);
      }
      else
      {
         /* Middle of print Job, ignore paper size if same. */
         if (!(w == pSS->PaperWidth && h == pSS->PaperHeight))
         {
            pSS->FirstRaster = 1;  /* force new Job */
            pSS->PaperWidth = w;   /* set new paper size */
            pSS->PaperHeight = h;
		      hpijs_set_context(pSS);
         }
      }
   }
   else if (!strcmp (key, "TopLeft"))
   {
      /* not currently used */
   }
   else if (!strcmp (key, "Quality:Quality"))
   {
      pSS->Quality = strtol(svalue, &tail, 10);
   }
   else if (!strcmp (key, "Quality:MediaType"))
   {
      pSS->MediaType = strtol(svalue, &tail, 10);
   }
   else if (!strcmp (key, "Quality:ColorMode"))
   {
      pSS->ColorMode = strtol(svalue, &tail, 10);
   }
   else if (!strcmp (key, "Quality:PenSet"))
   {
      pSS->PenSet = strtol(svalue, &tail, 10);
   }
   else if (!strcmp (key, "Quality:FullBleed"))
   {
      pSS->FullBleed = strtol(svalue, &tail, 10);
   }
   else
      bug("unable to set key=%s, value=%s\n", key, svalue);    

   return status;
}

/* Get parameter (from the server) call back. Note, all calls must be preceded by set DeviceName.
 * For duplex mode top and bottom margins must be 0.5".  */
int hpijs_get_cb(void *get_cb_data, IjsServerCtx *ctx, IjsJobId job_id, const char *key, char *value_buf, int value_size)
{
   UXServices *pSS = (UXServices*)get_cb_data;

   if (!strcmp (key, "PrintableArea"))
   {
      if (pSS->pPC->QueryDuplexMode() != DUPLEXMODE_NONE)
         return snprintf(value_buf, value_size, "%.4fx%.4f", pSS->pPC->PrintableWidth(), pSS->pPC->PhysicalPageSizeY()-1);
      else
         return snprintf(value_buf, value_size, "%.4fx%.4f", pSS->pPC->PrintableWidth(), pSS->pPC->PrintableHeight());
   }
   else if (!strcmp (key, "PrintableTopLeft"))
   {
      if (pSS->pPC->QueryDuplexMode() != DUPLEXMODE_NONE)
         return snprintf(value_buf, value_size, "%.4fx%.4f", pSS->pPC->PrintableStartX(), 0.5);
      else
         return snprintf(value_buf, value_size, "%.4fx%.4f", pSS->pPC->PrintableStartX(), pSS->pPC->PrintableStartY());
   }
   else if (!strcmp (key, "Duplex"))
   {
      return snprintf(value_buf, value_size, "%d", pSS->pPC->QueryDuplexMode());
   }
   else if (!strcmp (key, "PaperSize"))
   {
      return snprintf(value_buf, value_size, "%d", pSS->pPC->GetPaperSize());
   }
   else if (!strcmp (key, "Dpi"))
   {
      return snprintf(value_buf, value_size, "%dx%d", pSS->pPC->EffectiveResolutionX(), pSS->pPC->EffectiveResolutionY());
   }
   else if (!strcmp (key, "DeviceModel"))
   {
      return snprintf(value_buf, value_size, "%d", pSS->pPC->SelectedDevice());
   }
   else if (!strcmp (key, "PageImageFormat"))
      return snprintf(value_buf, value_size, "%s", "Raster");
   else
      bug("unable to get key=%s\n", key);    

   return 0;
}

int white(unsigned char *data, int size)
{
   int clean = 1;
   int i;
   for (i = 0; i < size; i++)
   {
     if (data[i] != 0xFF)
     {
        clean = 0;
        break;
     }
   }
   return clean;
}

/* Get raster from the client. */
int hpijs_get_client_raster(IjsServerCtx *ctx, void *buf, int size)
{
   int status;

   status = ijs_server_get_data(ctx, (char *)buf, size);

   if (status < 0)
      return status;  /* error */

   /* Check for blank raster. */
   if (white((unsigned char *)buf, size))
      return 0;

   return size;
}

int main(int argc, char *argv[])
{
   UXServices *pSS = NULL;
   IjsServerCtx *ctx;
   char *raster = NULL;
   int status = EXIT_FAILURE;
   int ret, n, i, width;

   if (argc > 1)
   {
      const char *arg = argv[1];
      if ((arg[0] == '-') && (arg[1] == 'h'))
      {
         fprintf(stdout, "\nHewlett-Packard Co. Inkjet Server %s\n", VERSION);
         fprintf(stdout, "Copyright (c) 2001-2002, Hewlett-Packard Co.\n");
         exit(0);
      }
   }

   /* Check for old hpijs interface. */
   if (getenv("SRV_TO_CLIENT") != NULL)
   {
      status = hpijs97();
      exit(status);
   }

   ctx = ijs_server_init();
   if (ctx == NULL)
   {
      bug("unable to init hpijs server\n");
      goto BUGOUT;
   }

   pSS = new UXServices();
   if (pSS->constructor_error != NO_ERROR)
   {
      bug("unable to open Services object err=%d\n", pSS->constructor_error);
      goto BUGOUT;
   }

#ifdef CAPTURE
   if ((pSS->InitScript("/tmp/capout", TRUE)) != NO_ERROR)
      bug("unable to init capture");
#endif

   pSS->pPC = new PrintContext(pSS, 0, 0);
   if (pSS->pPC->constructor_error != NO_ERROR)
   {
      bug("unable to open PrintContext object err=%d\n", pSS->pPC->constructor_error);
      goto BUGOUT;
   }

   ijs_server_install_status_cb (ctx, hpijs_status_cb, pSS);
   ijs_server_install_list_cb (ctx, hpijs_list_cb, pSS);
   ijs_server_install_enum_cb (ctx, hpijs_enum_cb, pSS);
   ijs_server_install_set_cb (ctx, hpijs_set_cb, pSS);
   ijs_server_install_get_cb (ctx, hpijs_get_cb, pSS);

   while (1)
   {
      if ((ret = ijs_server_get_page_header(ctx, &pSS->ph)) < 0)
      {
         bug("unable to read client data err=%d\n", ret);
         goto BUGOUT;
      }

      if (pSS->Model == -1)
         goto BUGOUT;      /* no device selected */

      if (ret)
      {
         status = 0; /* normal exit */
         break;
      }

      if (pSS->FirstRaster)
      {
         pSS->FirstRaster = 0;

         width = (int)(pSS->ph.xres * pSS->pPC->PrintableWidth() + 0.5);

         /* Desensitize input width, may be off by one due to paper size conversions. */
         if (pSS->ph.width < width)
            width = pSS->ph.width;

         if ((ret = pSS->pPC->SetPixelsPerRow(width, width)) != NO_ERROR)
         {
            bug("unable to SetPixelsPerRow width=%d, err=%d\n", pSS->ph.width, ret);
         }

         if (pSS->pJob != NULL)
            delete pSS->pJob;
         pSS->pJob = new Job(pSS->pPC);
         if (pSS->pJob->constructor_error != NO_ERROR)
         {
            bug("unable to create Job object err=%d\n", pSS->pJob->constructor_error);
            goto BUGOUT;
         }

         if (pSS->pPC->QueryDuplexMode() != DUPLEXMODE_NONE)
         {
            if (pSS->pPC->QueryDuplexMode() == DUPLEXMODE_BOOK)
               pSS->InitDuplexBuffer();
            pSS->BackPage = FALSE;
         }
#if 0
         bug("papersize=%d\n", pSS->pPC->GetPaperSize());
         bug("width=%d, height=%d\n", pSS->ph.width, pSS->ph.height);
         bug("EffResX=%d, EffResY=%d, InPixelsPerRow=%d, OutPixelsPerRow=%d\n", 
            pSS->pPC->EffectiveResolutionX(), pSS->pPC->EffectiveResolutionY(), 
            pSS->pPC->InputPixelsPerRow(), pSS->pPC->OutputPixelsPerRow());
#endif
      } // pSS->FirstRaster

      if ((raster = (char *)malloc(pSS->ph.width*3)) == NULL)
      {
         bug("unable to allocate raster buffer size=%d: %m\n", pSS->ph.width*3);
         goto BUGOUT;
      }

      for (i=0; i < pSS->ph.height; i++)      
      {
         if ((n = hpijs_get_client_raster(ctx, raster, pSS->ph.width*3)) < 0)
            break;    /* error */
         if (n == 0)
            pSS->ProcessRaster((char *)0);  /* blank raster */
         else
            pSS->ProcessRaster(raster);
      }

      free(raster);
      raster = NULL;

      if (pSS->pPC->QueryDuplexMode() != DUPLEXMODE_NONE)
      {
         if ((pSS->pPC->QueryDuplexMode() == DUPLEXMODE_BOOK) && pSS->BackPage)
         {
            pSS->SendBackPage();
         }
         pSS->BackPage = (BOOL)((int)pSS->BackPage + 1) % 2;
      }

      pSS->pJob->NewPage();

   } /* end while (1) */

   if (pSS->pPC->QueryDuplexMode() != DUPLEXMODE_NONE)
   {
      if (pSS->BackPage)
      {
         /* Send extra blank line & newpage to eject the page. (for VIP printers). */ 
         for (int i = 0; i < 201; i++)
            pSS->pJob->SendRasters((unsigned char *)0);
         pSS->pJob->NewPage();
      }
   }

 BUGOUT:
   if (pSS->pJob != NULL)
      delete pSS->pJob;
   if (pSS->pPC != NULL)
      delete pSS->pPC;
   if (pSS != NULL)
   {
#ifdef CAPTURE
      pSS->EndScript();
#endif
      delete pSS;
   }
   if (raster != NULL)
      free(raster);
   ijs_server_done(ctx);

   exit(status);
}

