/*
 * GQmpeg
 * (C)1998, 1999 John Ellis
 *
 * Author: John Ellis
 *
 * This software is released under the GNU General Public License.
 * Please read the included file COPYING for more information.
 * This software comes with no warranty of any kind, use at you own risk!
 */

#include "gqmpeg.h"

static GdkPixmap *create_background(gchar *text, gchar *dirpath);
static GdkBitmap *create_mask(gchar *text, gchar *dirpath);
static ButtonData *create_button(gint can_have_light, gchar *text, gchar *dirpath,
	void (*click_func)(void *), void (*press_func)(void *), void (*release_func)(void *));
static DigitData *create_digit(gchar *text, gchar *dirpath);
static NumberData *create_number(gchar *text, gchar *dirpath, SkinData *s, gint length, gint zeros);
static ItemData *create_item(gchar *text, gchar *dirpath, gint sections);
static FontData *create_font(gchar *text, gchar *dirpath);
static SliderData *create_slider(gchar *text, gchar *dirpath, ItemData *item,
	void (*press_func)(float), void (*release_func)(float), void (*move_func)(float));
static SkinData *load_skin(gchar *skin_path, gchar *datafile);

static GdkPixmap *create_background(gchar *text, gchar *dirpath)
{
	GdkPixmap *pixmap;
	gint p = 0;
	gchar *filename;
	gchar *pixmapfile;
	gint l = strlen(text);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	filename = text + p;

	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	if (p >= l) return NULL;
	text[p] = '\0';

	pixmapfile = g_strconcat(dirpath, "/", filename, NULL);

	pixmap = get_pixmap_from_file(pixmapfile);
	if (!pixmap) printf("unable to load background: %s\n", pixmapfile);

	g_free(pixmapfile);
	return pixmap;
}

static GdkBitmap *create_mask(gchar *text, gchar *dirpath)
{
	GdkBitmap *mask;
	gint p = 0;
	gchar *filename;
	gchar *pixmapfile;
	gint l = strlen(text);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	filename = text + p;

	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	if (p >= l) return NULL;
	text[p] = '\0';

	pixmapfile = g_strconcat(dirpath, "/", filename, NULL);

	mask = get_bitmap_from_file(pixmapfile);

	g_free(pixmapfile);
	return mask;
}

static ButtonData *create_button(gint can_have_light, gchar *text, gchar *dirpath,
	void (*click_func)(void *), void (*press_func)(void *), void (*release_func)(void *))
{
	ButtonData *button;
	gint p = 0;
	gint l = strlen(text);
	gchar *filename;
	gchar *pixmapfile;
	gchar *x_text;
	gchar *y_text;
	gint prelight;
	gint light;
	gint x;
	gint y;
	
	/* skip whitespace */
	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* filename */
	filename = text + p;

	/* find whitespace */
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	if (p >= l) return NULL;

	/* set whitespace to NULL so filename works */
	text[p] = '\0';
	p++;

	/* skip whitespace */
	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* load prelight True/False */
	while (text[p] != 'F' && text[p] != 'T' && p < l) p++;
	if (p >= l) return NULL;
	if (text[p] == 'T')
		prelight = TRUE;
	else
		prelight = FALSE;
		
	/* if we can be a lit button, parse */
	if (can_have_light)
		{
		gint pm = p;
		p++;
		while (text[p] != 'F' && text[p] != 'T' && p < l) p++;
		if (p >= l)
			{
			p = pm;
			light = FALSE;
			}
		else
			{			
			if (text[p] == 'T')
				light = TRUE;
			else
				light = FALSE;
			}
		}
	else
		light = FALSE;

	/* find whitespace */
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	/* skip whitespace */
	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* x coord */
	x_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	text[p] = '\0';
	p++;
	x = strtol(x_text,NULL,10);

	/* skip whitespace */
	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* y coord */
	y_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	text[p] = '\0';
	y = strtol(y_text,NULL,10);

	pixmapfile = g_strconcat(dirpath, "/", filename, NULL);

	button = new_button_from_file(pixmapfile, prelight, light, x, y,
				click_func, press_func, release_func);

	if (!button)
		printf("Unable to create button from: %s\n", pixmapfile);

	g_free(pixmapfile);

	return button;

}

static DigitData *create_digit(gchar *text, gchar *dirpath)
{
	DigitData *digit;
	gint p = 0;
	gchar *filename;
	gchar *pixmapfile;
	gint l = strlen(text);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	filename = text + p;

	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	if (p >= l) return NULL;
	text[p] = '\0';

	pixmapfile = g_strconcat(dirpath, "/", filename, NULL);

	if (!isfile(pixmapfile))
		{
		printf("unable to load digit: %s\n", pixmapfile);
		g_free(pixmapfile);
		return NULL;
		}

	digit =  new_digit_from_file(pixmapfile);
	g_free(pixmapfile);
	return digit;
}

static NumberData *create_number(gchar *text, gchar *dirpath, SkinData *s, gint length, gint zeros)
{
	DigitData *digit;
	gint p = 0;
	gint l = strlen(text);
	gchar *filename;
	gchar *pixmapfile;
	gchar *l_text;
	gchar *x_text;
	gchar *y_text;
	gint x;
	gint y;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	filename = text + p;

	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	if (p >= l) return NULL;
	text[p] = '\0';
	p++;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	l_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	text[p] = '\0';
	p++;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	x_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	text[p] = '\0';
	p++;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l || text[p] == '\0' || text[p] == '\r' || text[p] == '\n')
		{
		/* Must not have length specifier, use default */
		y_text = x_text;
		x_text = l_text;
		}
	else
		{
		y_text = text + p;
		while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
		text[p] = '\0';

		length = strtol(l_text,NULL,10);
		}

	x = strtol(x_text,NULL,10);
	y = strtol(y_text,NULL,10);

	if (!strcmp(filename,"Large"))
		{
		if (!s->dig_large)
			{
			printf("Large digit specified, but not defined\n");
			return NULL;
			}
		else
			digit = s->dig_large;
		}
	else if (!strcmp(filename,"Small"))
		{
		if (!s->dig_small)
			{
			printf("Small digit specified, but not defined\n");
			return NULL;
			}
		else
			digit = s->dig_small;
		
		}
	else
		{
		pixmapfile = g_strconcat(dirpath, "/", filename, NULL);
		digit = new_digit_from_file(pixmapfile);
		if (!digit)
			{
			printf("Unable to load number file: %s", pixmapfile);
			g_free(pixmapfile);
			return NULL;
			}
		g_free(pixmapfile);
		}

	return new_number(digit, length, zeros, x , y);
}

static ItemData *create_item(gchar *text, gchar *dirpath, gint sections)
{
	ItemData *item;
	gint p = 0;
	gint l = strlen(text);
	gchar *filename;
	gchar *pixmapfile;
	gchar *s_text;
	gchar *x_text;
	gchar *y_text;
	gint x;
	gint y;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	filename = text + p;

	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	if (p >= l) return NULL;
	text[p] = '\0';
	p++;

	if (sections == 0)
		{
		while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
		if (p >= l) return NULL;

		s_text = text + p;
		while (text[p] != ' ' && text[p] != 8 && p < l) p++;
		text[p] = '\0';
		p++;
		sections = strtol(s_text,NULL,10);
		}

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	x_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	text[p] = '\0';
	p++;
	x = strtol(x_text,NULL,10);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	y_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	text[p] = '\0';
	y = strtol(y_text,NULL,10);

	pixmapfile = g_strconcat(dirpath, "/", filename, NULL);

	item = new_item_from_file(pixmapfile, sections ,x , y);

	if (!item) printf("Unable to load item file: %s\n", pixmapfile);

	g_free(pixmapfile);

	return item;
}

static FontData *create_font(gchar *text, gchar *dirpath)
{
	FontData *font;
	gint p = 0;
	gint l = strlen(text);
	gchar *filename;
	gchar *pixmapfile;
	gchar *x_text;
	gchar *y_text;
	gchar *length_text;
	gint length;
	gint x;
	gint y;
	gint extended = FALSE;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	filename = text + p;

	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	if (p >= l) return NULL;
	text[p] = '\0';
	p++;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	length_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	text[p] = '\0';
	p++;
	length = strtol(length_text,NULL,10);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* ignore for backwards compatibilility (no f or t found) */
	if (text[p] == 'F' || text[p] == 'f' || text[p] == 'T' || text[p] == 't')
		{
		if (text[p] == 'T' || text[p] == 't')
			extended = TRUE;
		else
			extended = FALSE;

		while (text[p] != ' ' && text[p] != 8 && p < l) p++;
		while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
		if (p >= l) return NULL;
		}

	x_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	text[p] = '\0';
	p++;
	x = strtol(x_text,NULL,10);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	y_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	text[p] = '\0';
	y = strtol(y_text,NULL,10);

	pixmapfile = g_strconcat(dirpath, "/", filename, NULL);

	font = new_font_from_file(pixmapfile, length, extended, x, y);

	if (!font) printf("Unable to load font file: %s\n", pixmapfile);

	g_free(pixmapfile);

	return font;
}

static SliderData *create_slider(gchar *text, gchar *dirpath, ItemData *item,
	void (*press_func)(float), void (*release_func)(float), void (*move_func)(float))
{
	SliderData *slider;
	gint p = 0;
	gint l = strlen(text);
	gchar *filename;
	gchar *pixmapfile;
	gchar *l_text;
	gchar *x_text;
	gchar *y_text;
	gint prelight;
	gint verticle;
	gint reversed;
	gint length;
	gint x;
	gint y;
	gint pm;
	
	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	filename = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	if (p >= l) return NULL;
	text[p] = '\0';
	p++;

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* load prelight True/False */
	while (text[p] != 'F' && text[p] != 'T' && p < l) p++;
	if (p >= l) return NULL;
	if (text[p] == 'T')
		prelight = TRUE;
	else
		prelight = FALSE;

	/* horizontal / verticle */
	pm = p;
	p++;
	while (text[p] != 'F' && text[p] != 'T' && p < l) p++;
	if (p >= l)
		{
		/* ignore verticle TRUE/FALSE if not found, for backwards compatibility */
		p = pm;
		verticle = FALSE;
		}
	else
		{			
		if (text[p] == 'T')
			verticle = TRUE;
		else
			verticle = FALSE;
		}

	/* reversed */
	pm = p;
	p++;
	while (text[p] != 'F' && text[p] != 'T' && p < l) p++;
	if (p >= l)
		{
		/* ignore reverse TRUE/FALSE if not found, for backwards compatibility */
		p = pm;
		reversed = FALSE;
		}
	else
		{			
		if (text[p] == 'T')
			reversed = TRUE;
		else
			reversed = FALSE;
		}

	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* length */

	l_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	text[p] = '\0';
	p++;
	length = strtol(l_text,NULL,10);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* x coord */
	x_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && p < l) p++;
	text[p] = '\0';
	p++;
	x = strtol(x_text,NULL,10);

	while ((text[p] == ' ' || text[p] == 8 ) && p < l) p++;
	if (p >= l) return NULL;

	/* y coord */
	y_text = text + p;
	while (text[p] != ' ' && text[p] != 8 && text[p] !='\0' && text[p] != '\n' && p < l) p++;
	text[p] = '\0';
	y = strtol(y_text,NULL,10);

	pixmapfile = g_strconcat(dirpath, "/", filename, NULL);

	slider = new_slider_from_file(pixmapfile, prelight, verticle, reversed,
			length, x, y, item,
			press_func, release_func, move_func);

	if (!slider)
		printf("Unable to create slider from: %s\n", pixmapfile);

	g_free(pixmapfile);

	return slider;
}


static SkinData *load_skin(gchar *skin_path, gchar *datafile)
{
	SkinData *s;
	FILE *f;
	gchar buf[1024];

	f = fopen(datafile,"r");
	if (!f)
		{
		/* no file has been created yet */
		if (debug_mode) printf("Unable to open skin data file: %s\n",datafile);
		return NULL;
		}

	s = new_skin();

	while (fgets(buf,1024,f))
		{
		gint p;
		gint l;
		gchar *text;
		if (buf[0]=='#') continue;
		if (buf[0]=='\n') continue;

		/* get past label */
		p = 0;
		l = strlen(buf);
		while (buf[p] != ':' && p < l) p++;
		if (p >= l) continue;
		buf[p] = '\0';
		p++;
		if (p >= l) continue;

		text = buf + p;

		/* background */
		if (!strcmp(buf,"Background"))
			{
			s->background = create_background(text, skin_path);
			if (s->background)
				{
				gint width, height;
				gdk_window_get_size (s->background, &width, &height);
				s->width = width;
				s->height = height;
				}
			}
		/* background mask for shaped windows*/
		if (!strcmp(buf,"Mask"))
			{
			s->mask = create_mask(text, skin_path);
			}

		/* buttons */
		if (!strcmp(buf,"Play_Button"))
			s->play = create_button(TRUE, text, skin_path,
					btn_play_pressed , NULL, NULL);
		if (!strcmp(buf,"Stop_Button"))
			s->stop = create_button(TRUE, text, skin_path,
					btn_stop_pressed , NULL, NULL);
		if (!strcmp(buf,"Pause_Button"))
			s->pause = create_button(TRUE, text, skin_path,
					btn_pause_pressed , NULL, NULL);
		if (!strcmp(buf,"Shuffle_Button"))
			s->shuffle = create_button(TRUE, text, skin_path,
					btn_shuffle_pressed , NULL, NULL);
		if (!strcmp(buf,"Repeat_Button"))
			s->repeat = create_button(TRUE, text, skin_path,
					btn_repeat_pressed , NULL, NULL);

		if (!strcmp(buf,"Next_Button"))
			s->next = create_button(FALSE, text, skin_path,
					btn_next_pressed, btn_next_down, btn_next_up);
		if (!strcmp(buf,"Prev_Button"))
			s->prev = create_button(FALSE, text, skin_path,
					btn_prev_pressed, btn_prev_down, btn_prev_up);
		if (!strcmp(buf,"FF_Button"))
			s->ff = create_button(FALSE, text, skin_path,
					NULL, btn_ff_down, btn_ff_up);
		if (!strcmp(buf,"RW_Button"))
			s->rw = create_button(FALSE, text, skin_path,
					NULL, btn_rw_down, btn_rw_up);
		if (!strcmp(buf,"Config_Button"))
			s->config = create_button(FALSE, text, skin_path,
					show_configuration_window, NULL, NULL);
		if (!strcmp(buf,"Playlist_Button"))
			s->playlist = create_button(FALSE, text, skin_path,
					playlist_window_show , NULL, NULL);
		if (!strcmp(buf,"Exit_Button"))
			s->exit = create_button(FALSE, text, skin_path,
					gqmpeg_exit , NULL, NULL);
		if (!strcmp(buf,"Iconify_Button"))
			s->iconify = create_button(FALSE, text, skin_path,
					btn_iconify_pressed , NULL, NULL);
		if (!strcmp(buf,"Mixer_Button"))
			s->mixer = create_button(FALSE, text, skin_path,
					btn_mixer_pressed , NULL, NULL);
		if (!strcmp(buf,"Alt_Skin_Button"))
			s->skin_mode = create_button(FALSE, text, skin_path,
					mode_button_pressed , NULL, NULL);

		/* large / small digits */
		if (!strcmp(buf,"Digit_Large_Default") && !s->dig_large)
			s->dig_large = create_digit(text, skin_path);
		if (!strcmp(buf,"Digit_Small_Default") && !s->dig_small)
			s->dig_small = create_digit(text, skin_path);

		/* numbers */
		if (!strcmp(buf,"Hour_Number"))
			s->hour = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Minute_Number"))
			s->min = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Second_Number"))
			s->sec = create_number(text, skin_path, s, 2, TRUE);
		if (!strcmp(buf,"Hour_Total_Number"))
			s->hour_total = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Minute_Total_Number"))
			s->min_total = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Second_Total_Number"))
			s->sec_total = create_number(text, skin_path, s, 2, TRUE);
		if (!strcmp(buf,"Song_Number"))
			s->song_current = create_number(text, skin_path, s, 3, FALSE);
		if (!strcmp(buf,"Total_Number"))
			s->song_total = create_number(text, skin_path, s, 3, FALSE);
		if (!strcmp(buf,"In_Rate_Number"))
			s->input_bits = create_number(text, skin_path, s, 3, FALSE);
		if (!strcmp(buf,"In_Hz_Number"))
			s->input_hz = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Out_Bits_Number"))
			s->output_bits = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Out_Hz_Number"))
			s->output_hz = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Frame_Number"))
			s->frame = create_number(text, skin_path, s, 5, FALSE);
		if (!strcmp(buf,"Frame_Total_Number"))
			s->frame_total = create_number(text, skin_path, s, 5, FALSE);
		if (!strcmp(buf,"CPU_Number"))
			s->cpu = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Song_Minute_Number"))
			s->song_min = create_number(text, skin_path, s, 2, FALSE);
		if (!strcmp(buf,"Song_Second_Number"))
			s->song_sec = create_number(text, skin_path, s, 2, TRUE);

		/* items */
		if (!strcmp(buf,"Stereo_Item"))
			s->stereo = create_item(text, skin_path, 3);
		if (!strcmp(buf,"Shuffle_Item"))
			s->item_shuffle = create_item(text, skin_path, 2);
		if (!strcmp(buf,"Repeat_Item"))
			s->item_repeat = create_item(text, skin_path, 2);
		if (!strcmp(buf,"Mpegversion_Item"))
			s->mpegver = create_item(text, skin_path, 3);
		if (!strcmp(buf,"Mpeglayer_Item"))
			s->layer = create_item(text, skin_path, 4);
		if (!strcmp(buf,"Mpegmode_Item"))
			s->mode = create_item(text, skin_path, 5);
		if (!strcmp(buf,"Status_Item"))
			s->status = create_item(text, skin_path, 3);

		/* font / text */
		if (!strcmp(buf,"Title"))
			s->title = create_font(text, skin_path);
		if (!strcmp(buf,"Artist"))
			s->artist = create_font(text, skin_path);
		if (!strcmp(buf,"Album"))
			s->album = create_font(text, skin_path);
		if (!strcmp(buf,"Genre"))
			s->genre = create_font(text, skin_path);

		/* position slider & item */
		if (!strcmp(buf,"Position_Item"))
			s->position_item = create_item(text, skin_path, 0);
		if (!strcmp(buf,"Position_Slider"))
			s->position = create_slider(text, skin_path,
			s->position_item,
			position_slider_moved, position_slider_released,
			position_slider_moved);

		/* volume & balance */
		if (!strcmp(buf,"Volume_Up_Button"))
			s->volume_up = create_button(FALSE, text, skin_path,
					NULL, btn_volume_up_pressed , btn_volume_released);
		if (!strcmp(buf,"Volume_Down_Button"))
			s->volume_dn = create_button(FALSE, text, skin_path,
					NULL, btn_volume_down_pressed , btn_volume_released);
		if (!strcmp(buf,"Volume_Item"))
			s->volume_item = create_item(text, skin_path, 0);
		if (!strcmp(buf,"Volume_Slider"))
			s->volume = create_slider(text, skin_path,
			s->volume_item,
			volume_slider_moved, NULL, volume_slider_moved);

		if (!strcmp(buf,"Balance_Left_Button"))
			s->balance_left = create_button(FALSE, text, skin_path,
					NULL, btn_balance_left_pressed , btn_balance_released);
		if (!strcmp(buf,"Balance_Right_Button"))
			s->balance_right = create_button(FALSE, text, skin_path,
					NULL, btn_balance_right_pressed , btn_balance_released);
		if (!strcmp(buf,"Balance_Item"))
			s->balance_item = create_item(text, skin_path, 0);
		if (!strcmp(buf,"Balance_Slider"))
			s->balance = create_slider(text, skin_path,
			s->balance_item,
			balance_slider_moved, NULL, balance_slider_moved);

		if (!strcmp(buf,"Preset_1_Button"))
			s->preset[0] = create_button(FALSE, text, skin_path,
					preset_1_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_2_Button"))
			s->preset[1] = create_button(FALSE, text, skin_path,
					preset_2_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_3_Button"))
			s->preset[2] = create_button(FALSE, text, skin_path,
					preset_3_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_4_Button"))
			s->preset[3] = create_button(FALSE, text, skin_path,
					preset_4_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_5_Button"))
			s->preset[4] = create_button(FALSE, text, skin_path,
					preset_5_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_6_Button"))
			s->preset[5] = create_button(FALSE, text, skin_path,
					preset_6_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_7_Button"))
			s->preset[6] = create_button(FALSE, text, skin_path,
					preset_7_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_8_Button"))
			s->preset[7] = create_button(FALSE, text, skin_path,
					preset_8_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_9_Button"))
			s->preset[8] = create_button(FALSE, text, skin_path,
					preset_9_pressed, NULL, NULL);
		if (!strcmp(buf,"Preset_10_Button"))
			s->preset[9] = create_button(FALSE, text, skin_path,
					preset_10_pressed, NULL, NULL);
		}

	fclose(f);

	if (!s->background)
		{
		printf("No background image from: %s\n", skin_path);
		free_skin(s);
		return NULL;
		}

	/* if hours are defined, change minutes to have leading 0's */
	if (s->hour && s->min) s->min->zeros = TRUE;
	if (s->hour_total && s->min_total) s->min_total->zeros = TRUE;

	return s;

}

gint change_to_skin(gchar *file)
{
	SkinData *s = NULL;
	SkinData *ss = NULL;

	if (!file)
		{
		s = load_default_normal_skin();
		ss = load_default_small_skin();
		}
	else
		{
		gchar *datafile;
		datafile = g_strconcat(file, "/skindata", NULL);
		s = load_skin(file, datafile);
		g_free(datafile);
		if (!s)
			{
			s = load_wamp_normal_skin(file);
			if (!s)
				{
				printf("Unable to open skin data file: %s/skindata\n", file);
				return FALSE;
				}
			ss = load_wamp_small_skin(file);
			}
		else
			{
			datafile = g_strconcat(file, "/skindata_alt", NULL);
			ss = load_skin(file, datafile);
			g_free(datafile);
			}
		}

	skin = NULL;
	free_skin(skin_normal);
	free_skin(skin_small);
	skin_normal = s;
	skin_small = ss;
	skin = skin_normal;
	sync_window_to_skin();

	return TRUE;
}

