/* vi:set ts=8 sts=0 sw=8:
 * $Id: htmltags.c,v 1.49 2000/05/01 16:00:17 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "main.h"
#ifdef USE_HTMLTAGS
#include <string.h>
#include <gtk/gtk.h>
#include "htmltags.h"
#include "toolbar.h"
#include "doc.h"
#include "undo.h"
#include "dialog.h"
#include "prefs.h"
#include "misc.h"
#include "gtkefilesel.h"
#include "msgbar.h"
#include "menu.h"

#include "gnpintl.h"


/*** local types ***/
#define COL_VAL_HTML	255.0		/* HTML specifies 8-bit color in hex */

#define HTS_ABBR	"Abbreviation"
#define HTS_ACRO	"Acronym"
#define HTS_ADDRESS	"Address"
#define HTS_AREA	"Area"
#define HTS_APPLET	"Applet"
#define HTS_BASE	"Base"
#define HTS_BASEFONT	"Base Font"
#define HTS_BDO		"BDO"
#define HTS_BIG		"Big Font"
#define HTS_BLKQUOTE	"Blockquote"
#define HTS_BODY	"Body"
#define HTS_BOLD	"Bold Font"
#define HTS_BUTTON	"Button"
#define HTS_CAPTION	"Caption"
#define HTS_CENT	"Center Justify"
#define HTS_CITE	"Cite"
#define HTS_CMNT	"Comment"
#define HTS_CODE	"Code"
#define HTS_COL		"COL"
#define HTS_COLGRP	"COLGROUP"
#define HTS_DEL		"DEL"
#define HTS_DEF		"Definition"
#define HTS_DEFLIST	"Definition List"
#define HTS_DEFTERM	"Definition Term"
#define HTS_DFN		"DFN"
#define HTS_DIR		"DIR"
#define HTS_DIV		"DIV"
#define HTS_DOCTYPE	"!DOCTYPE"
#define HTS_EMPH	"Emphasis"
#define HTS_FIELDSET	"FIELDSET"
#define HTS_FONT	"Font"
#define HTS_FONTM	"Font Size -1"
#define HTS_FONTP	"Font Size +1"
#define HTS_FORMBUTTON	"Form: Button"
#define HTS_FORMCHECK	"Form: Check"
#define HTS_FORMFILE	"Form: File"
#define HTS_FORMHIDDEN	"Form: Hidden"
#define HTS_FORMIMG	"Form: Image"
#define HTS_FORMINPUT	"INPUT"
#define HTS_FORMNEW	"Form: New"
#define HTS_FORMOPTION	"Form: Option"
#define HTS_FORMPASSWD	"Form: Password"
#define HTS_FORMRADIO	"Form: Radio Button"
#define HTS_FORMRESET	"Form: Reset"
#define HTS_FORMSELECT	"Form: Select"
#define HTS_FORMSUBMIT	"Form: Submit"
#define HTS_FORMTAREA	"Form: Text Area"
#define HTS_FORMTFIELD	"Form: Text Field"
#define HTS_FRAMENEW	"New Frame"
#if 0
#define HTS_FRAMEWIZ	"Frame Wizard"
#endif
#define HTS_FRAMEBASE	"Frame Base"
#define HTS_FRAMEI	"FRAMEI"
#define HTS_FRAMENO	"No Frames"
#define HTS_FRAMESET	"Frame Set"
#define HTS_HDR1	"Header 1"
#define HTS_HDR2	"Header 2"
#define HTS_HDR3	"Header 3"
#define HTS_HDR4	"Header 4"
#define HTS_HDR5	"Header 5"
#define HTS_HDR6	"Header 6"
#define HTS_HEAD	"HEAD"
#define HTS_HTML	"HTML"
#define HTS_IMG		"Image"
#define HTS_INS		"INS"
#define HTS_INPUT	"Input"
#define HTS_ISINDEX	"ISINDEX"
#define HTS_ITAL	"Italic"
#define HTS_KBD		"KBD"
#define HTS_LABEL	"Label"
#define HTS_LBRK	"Line Break"
#define HTS_LEFT	"Left Justify"
#define HTS_LEGEND	"Legend"
#define HTS_LINK	"Link"
#define HTS_LINKREAL	"LINK"
#define HTS_LITM	"List Item"
#define HTS_MAP		"MAP"
#define HTS_MENU	"Menu"
#define HTS_META	"META"
#define HTS_NOSCRIPT	"NOSCRIPT"
#define HTS_OBJECT	"OBJECT"
#define HTS_OPTGRP	"OPTGROUP"
#define HTS_ORDR	"Ordered List"
#define HTS_PARA	"Paragraph"
#define HTS_PARAM	"PARAM"
#define HTS_PRE		"Preformatted"
#define HTS_Q		"<Q>"
#define HTS_RGHT	"Right Justify"
#define HTS_SAMP	"SAMP"
#define HTS_SCRIPT	"SCRIPT"
#define HTS_SEP		"Separator"
#define HTS_SMLL	"Small Font"
#define HTS_STRG	"Strong Font"
#define HTS_STRIKE	"STRIKE"
#define HTS_STRK	"Strikeout"
#define HTS_STYLE	"STYLE"
#define HTS_SUB		"Subscript"
#define HTS_SUP		"Superscript"
#define HTS_TARG	"Target"
#define HTS_TBLDATA	"Table Data"
#define HTS_TBLHDR	"Table Header"
#define HTS_TBLNEW	"New Table"
#define HTS_TBLROW	"Table Row"
#if 0
#define HTS_TBLWIZ	"Table Wizard"
#endif
#define HTS_TBODY	"TBODY"
#define HTS_TFOOT	"TFOOT"
#define HTS_THEAD	"THEAD"
#define HTS_TTL		"Title"
#define HTS_TTY		"Typewriter"
#define HTS_ULNE	"Underline"
#define HTS_UORD	"Unordered List"
#define HTS_VAR		"VAR"

/*
 * there are basically two types of html tags we'll be handling:
 *
 * 1. the simple kind, where the start of the tag and the end of the tag are
 *    more or less "symmetrical" (plus the addition of a forward slash).  for
 *    example, "<B></B>", "<I></I>".  simple tags use the callback function
 *    html_tag_simple_cb().
 *
 * 2. the somewhat more complex kind, where the start and end are asymmetrical.
 *    that is, the beginning of the tag may contain other arguments, which
 *    don't have a corresponding counterpart in the end of the tag.  for
 *    example, "<A HREF="http://www.foo.com"></A>".  complex tags use the
 *    callback function html_tag_complex_cb().  In general, complex tags
 *    require a popup dialog to get more information from the user.
 *
 * in both cases, the beginning of the tag will be referred to as the "prefix",
 * while the end of the tag is referred to as the "suffix".  (there is also a
 * case of an asymmetrical simple tag: those without a suffix.  e.g., "<HR>",
 * "<BR>", "<DT>")
 *
 * additionally, for readability purposes, we may prepend a carriage return
 * before the prefix, or append one after the suffix.
 *
 * the enums and struct below handle these various options.
 *
 * and yes, we'll be using lots and lots of tables instead of being a bonehead
 * and writing data specific code (which leads to serious code bloat) for
 * inserting html tags like in other GTK based text/html editors.
 */
/* HTML Tag group types */
typedef enum taggrp_id_e {
	TgDocStruct,		/* Global document structure */
	TgBodyStruct,		/* Body structure */
	TgTxtDir,		/* Text direction */
	TgTxtStruct,		/* Text structure */
	TgLinesPara,		/* Lines and Paragraphs */
	TgMarkup,		/* Text changes markup */
	TgLists,		/* Lists */
	TgTables,		/* Tables */
	TgLinks,		/* Links */
	TgImage,		/* Images and other objects */
	TgStyleSheets,		/* Style sheets */
	TgFonts,		/* Font style */
	TgFrames,		/* Frames */
	TgForms,		/* Forms */
	TgScripts,		/* Scripts */
	TgOther,		/* Other */
	TgCOUNT			/* count of how many group types */
} taggrp_id_t;


/* 'opt' flags for htmltags_t */
#define HT_BEFORE	(1 << 0)	/* set cursor before tag prefix */
#define HT_BETWEEN	(1 << 1)	/* set cursor between prefix & suffix */
#define HT_AFTER	(1 << 2)	/* set cursor after suffix */
#define HT_CRBEFORE	(1 << 3)	/* insert CR before prefix */
#define HT_CRAFTER	(1 << 4)	/* insert CR after suffix */
#define HT_NOSUFFIX	(1 << 5)	/* tag has no suffix */
#define HT_COMPLEX	(1 << 6)	/* use html_tag_complex_cb(). default */
					/* is html_tag_simple_cb(). */

#define HT_CRBOTH	(HT_CRBEFORE|HT_CRAFTER)

typedef struct htmltags_s {
	html_action_t	action;		/* see htmltags.h */
	taggrp_id_t	taggrp;		/* Tag group id */
	int		opt;		/* options, see above */
	char *		tagname;	/* see above #define's */
	char *		prefix;		/* see above for prefix */
	char *		data;		/* text that goes in prfx, not sffx */
} htmltags_t;

/*
 * see html_tag_chooser_cb() on how the 'html_id_list' and 'listcnt' fields
 * get used within tagdesc[].
 */
typedef struct taggrp_s {
	char *		grpname;	/* tag group description */
	html_action_t *	html_id_list;	/* array of html_id's */
	int		listcnt;	/* # of entries in html_id_list */
} taggrp_t;

static taggrp_t tagdesc[] = {
	/* TgDocStruct */	{ N_("Document structure") },
	/* TgBodyStruct */	{ N_("Body structure") },
	/* TgTxtDir */		{ N_("Text direction") },
	/* TgTxtStruct */	{ N_("Text structure") },
	/* TgLinesPara */	{ N_("Lines and Paragraphs") },
	/* TgMarkup */		{ N_("Text changes markup") },
	/* TgLists */		{ N_("Lists") },
	/* TgTables */		{ N_("Tables") },
	/* TgLinks */		{ N_("Links") },
	/* TgImage */		{ N_("Images and Objects") },
	/* TgStyleSheets */	{ N_("Style sheets") },
	/* TgFonts */		{ N_("Font style") },
	/* TgFrames */		{ N_("Frames") },
	/* TgForms */		{ N_("Forms") },
	/* TgScripts */		{ N_("Scripts") },
	/* TgOther */		{ N_("Other") }
}; /* tagdesc[] */


/*
 * this structure defines what each page item/entry represents.  e.g., what the
 * option name as part of the HTML tag itself.
 *
 * each page item/entry for a complex html dialog box can be of only two types:
 * an entry widget or a combo (pulldown) widget (note that the pulldown is
 * actually an extension of the entry widget).
 */
#define IS_ENTRY	(1 << 0)	/* Entry widget */
#define IS_PDOWN	(1 << 1)	/* Pulldown/combo widget */
#define IS_VSPACE	(1 << 2)	/* Vertical space */
#define IS_ASSIGN	(1 << 3)	/* HTML Tag is an assignment */
#define IS_NOQUOTE	(1 << 4)	/* Value is NOT put in double quotes */
#define IS_FILESEL	(1 << 5)	/* Use file selection dialog button */
#define IS_COLSEL	(1 << 6)	/* Use color selection dialog button */
#define IS_CHKBOX	(1 << 7)	/* Is a check button/box */
#define IS_SPINBUT	(1 << 8)	/* Is a spin button (entry) */

typedef struct html_wgt_info_s {
	int		hll_id;		/* label text id (HLL_xxx) */
	int		opt;		/* bit flag options.  see above */
	int		htl_id;		/* tagname text id (HTL_xxx) */
	gpointer	data;		/* data, which depends on type */
	GtkWidget *	wgt;		/* pointer to widget */
} html_wgt_info_t;

/*
 * this defines what a page is: a page name, and a list of widgets representing
 * the entries on that page.
 */
typedef struct html_page_s {
	int		hll_id;		/* HLL_xxx */
	html_wgt_info_t	*hwip;
} html_page_t;

/*
 * callback functions in this file really need more than one piece of informa-
 * tion for callback data.  this structure packages up the various things that
 * are usually needed.
 */
typedef struct {
	dialog_entry_t edp;	/* see dialog.h */
	doc_t *d;		/* the document where tag is to be inserted */
	htmltags_t *htp;	/* ptr to html tag info */
	html_page_t *dlg_pages;	/* the dialog pages for this html tag */
	int start, stop;	/* start/stop positions in text widget */
} html_entry_dlg_t;


/*** local function prototypes ***/
static void html_tag_simple_cb(GtkWidget *wgt, gpointer cbdata);
static void html_tag_complex_cb(GtkWidget *wgt, gpointer cbdata);
#if 0
static void html_wizard_cb(GtkWidget *wgt, gpointer cbdata);
#endif
static void html_tag_insert(doc_t *d, char **buf, htmltags_t *htp);
static void html_entry_dlg(doc_t *d, htmltags_t *htp, char *title, char *msg,
			   GtkSignalFunc);
static void html_comment_exec(GtkWidget *wgt, gpointer cbdata);
static void html_target_exec(GtkWidget *wgt, gpointer cbdata);
static void html_complex_exec(GtkWidget *wgt, gpointer cbdata);
static void html_complex_cancel(GtkWidget *wgt, gpointer cbdata);
static void html_complex_dlg(html_page_t *dlg_pages, doc_t *d,
			      htmltags_t *htp, char *title,
			      GtkSignalFunc execute_cb,
			      GtkSignalFunc cancel_cb);
static GtkWidget *html_tag_pulldown_create(char **itemlist);
static void html_dlg_destroy(GtkWidget *wgt, gpointer cbdata);
static void html_complex_dlg_pages_add(GtkWidget *toplev,
			html_page_t *dlg_pages, int maxent, GtkNotebook *nb);
static void html_complex_dlg_items_add(GtkWidget *toplev, GtkTable *table,
				       html_page_t *hpp);
static void html_complex_dlg_buttons_add(GtkWidget *toplev,
					  GtkSignalFunc execute_cb,
					  GtkSignalFunc cancel_cb,
					  html_entry_dlg_t *hedp);
static int html_complex_calc_len(html_entry_dlg_t *hedp);
static void html_complex_attr_add(html_page_t *dlg_pages, char *buf);
static char **html_tag_simple_buf_build(htmltags_t *htp, win_t *w);
static char *html_complex_buf_build(int len, html_entry_dlg_t *hedp);
static void html_menu_cb_common(html_action_t action, gpointer cbdata);
static htmltags_t *html_find_tag_by_name(char *tagname);
static void html_tag_simple_common(htmltags_t *htp, win_t *w);
static void html_tag_complex_common(htmltags_t *htp, doc_t *d, char *opttxt);
static htmltags_t *html_find_tag_by_action(html_action_t action);
static void html_complex_exec_pre(html_entry_dlg_t *hedp);
static void html_sel_dlg_cb(GtkWidget *wgt, gpointer cbdata);
static void html_sel_dlg_ok_cb(GtkWidget *wgt, gpointer cbdata);
static void html_sel_dlg_cancel_cb(GtkWidget *wgt, gpointer cbdata);
static void html_sel_dlg_destroy_cb(GtkWidget *wgt, gpointer cbdata);
static void html_tag_chooser_selrow(GtkWidget *wgt, int row, int col,
			GdkEventButton *bevent, gpointer cbdata);
static void html_tag_chooser_insert(GtkWidget *wgt, gpointer cbdata);
static void html_tag_chooser_close(GtkWidget *wgt, gpointer cbdata);


/*** local variables ***/
/*
 * Try to keep these alphabetical.  It's not important now, but eventually, if
 * this gets sufficiently large, we'll want to use a binary search on it.
 */
static htmltags_t htmltags[] = {
	{ HtmlAbbr,	TgTxtStruct,	HT_BETWEEN,
			HTS_ABBR,	"ABBR"
	},
	{ HtmlAcro,	TgTxtStruct,	HT_BETWEEN,
			HTS_ACRO,	"ACRONYM"
	},
	{ HtmlAddress,	TgDocStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_ADDRESS,	"ADDRESS"
	},
	{ HtmlArea,	TgImage,	HT_AFTER|HT_NOSUFFIX,
			HTS_AREA,	"AREA"
	},
	{ HtmlApplet,	TgImage,	HT_BETWEEN,
			HTS_APPLET,	"APPLET"
	},
	{ HtmlBase,	TgLinks,	HT_AFTER|HT_NOSUFFIX,
			HTS_BASE,	"BASE"
	},
	{ HtmlBaseFont,	TgFonts,	HT_AFTER|HT_NOSUFFIX,
			HTS_BASEFONT,	"BASEFONT"
	},
	{ HtmlBdo,	TgTxtDir,	HT_BETWEEN|HT_CRAFTER,
			HTS_BDO,	"BDO"
	},
	{ HtmlBig,	TgFonts,	HT_BETWEEN,
			HTS_BIG,	"BIG"
	},
	{ HtmlBlkQuote,	TgTxtStruct,	HT_BETWEEN,
			HTS_BLKQUOTE,	"BLOCKQUOTE"
	},
	{ HtmlBody,	TgDocStruct,	HT_BETWEEN|HT_CRBOTH,
			HTS_BODY,	"BODY"
	},
	{ HtmlBold,	TgFonts,	HT_BETWEEN,
			HTS_BOLD,	"B"
	},
	{ HtmlButton,	TgFrames,	HT_BETWEEN,
			HTS_BUTTON,	"BUTTON"
	},
	{ HtmlCaption,	TgTables,	HT_BETWEEN,
			HTS_CAPTION,	"CAPTION"
	},
	{ HtmlCent,	TgBodyStruct,	HT_BETWEEN,
			HTS_CENT,	"CENTER"
	},
	{ HtmlCite,	TgTxtStruct,	HT_BETWEEN,
			HTS_CITE,	"CITE"
	},
	{ HtmlCmnt,	TgDocStruct,	HT_COMPLEX|HT_AFTER,
			HTS_CMNT,	"!--"
	},
	{ HtmlCode,	TgTxtStruct,	HT_BETWEEN,
			HTS_CODE,	"CODE"
	},
	{ HtmlCol,	TgTables,	HT_BETWEEN|HT_NOSUFFIX,
			HTS_COL,	"COL"
	},
	{ HtmlColGrp,	TgTables,	HT_BETWEEN,
			HTS_COLGRP,	"COLGROUP"
	},
	{ HtmlDef,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_DEF,	"DD"
	},
	{ HtmlDefList,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_DEFLIST,	"DL"
	},
	{ HtmlDefTerm,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_DEFTERM,	"DT"
	},
	{ HtmlDel,	TgMarkup,	HT_BETWEEN,
			HTS_DEL,	"DEL"
	},
	{ HtmlDfn,	TgTxtStruct,	HT_BETWEEN,
			HTS_DFN,	"DFN"
	},
	{ HtmlDir,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_DIR,	"DIR"
	},
	{ HtmlDiv,	TgBodyStruct,	HT_BETWEEN,
			HTS_DIV,	"DIV"
	},
	{ HtmlDocType,	TgDocStruct,	HT_AFTER|HT_NOSUFFIX,
			HTS_DOCTYPE,	"!DOCTYPE"
	},
	{ HtmlEmph,	TgTxtStruct,	HT_BETWEEN,
			HTS_EMPH,	"EM"
	},
	{ HtmlFieldSet,	TgForms,	HT_BETWEEN,
			HTS_FIELDSET,	"FIELDSET"
	},
	{ HtmlFont,	TgFonts,	HT_BETWEEN,
			HTS_FONT,	"FONT"
	},
	{ HtmlFontM,	TgFonts,	HT_BETWEEN|HT_CRBOTH,
			HTS_FONTM,	"FONT",		"SIZE=\"-1\""
	},
	{ HtmlFontP,	TgFonts,	HT_BETWEEN|HT_CRBOTH,
			HTS_FONTP,	"FONT",		"SIZE=\"+1\""
	},
	{ HtmlFormBtn,	TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMBUTTON,	"INPUT",	"TYPE=\"BUTTON\""
	},
	{ HtmlFormChk,	TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMCHECK,	"INPUT",	"TYPE=\"CHECKBOX\""
	},
	{ HtmlFormFile,	TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMFILE,	"INPUT",	"TYPE=\"FILE\""
	},
	{ HtmlFormHid,	TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMHIDDEN,	"INPUT",	"TYPE=\"HIDDEN\""
	},
	{ HtmlFormImg,	TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMIMG,	"INPUT",	"TYPE=\"IMAGE\""
	},
	{ HtmlFormInput,TgForms,	HT_AFTER|HT_NOSUFFIX,
			HTS_FORMINPUT,	"INPUT"
	},
	{ HtmlFormNew,	TgForms,	HT_COMPLEX|HT_BETWEEN|HT_CRBOTH,
			HTS_FORMNEW,	"FORM"
	},
	{ HtmlFormOpt,	TgForms,	HT_COMPLEX|HT_BETWEEN,
			HTS_FORMOPTION,	"OPTION"
	},
	{ HtmlFormPass,	TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMPASSWD,	"INPUT",	"TYPE=\"PASSWORD\""
	},
	{ HtmlFormRadio,TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMRADIO,	"INPUT",	"TYPE=\"RADIO\""
	},
	{ HtmlFormReset,TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMRESET,	"INPUT",	"TYPE=\"RESET\""
	},
	{ HtmlFormSel,	TgForms,	HT_COMPLEX|HT_BETWEEN,
			HTS_FORMSELECT,	"SELECT"
	},
	{ HtmlFormSubmit,TgForms,	HT_AFTER,
			HTS_FORMSUBMIT,	"INPUT",	"TYPE=\"SUBMIT\""
	},
	{ HtmlFormTarea,TgForms,	HT_COMPLEX|HT_BETWEEN,
			HTS_FORMTAREA,	"TEXTAREA"
	},
	{ HtmlFormTfield,TgForms,	HT_COMPLEX|HT_AFTER,
			HTS_FORMTFIELD,	"INPUT",	"TYPE=\"TEXT\""
	},
	{ HtmlFrameBase,TgFrames,	HT_AFTER,
			HTS_FRAMEBASE,	"BASE",		"TARGET=\"\""
	},
	{ HtmlFrameI,	TgFrames,	HT_BETWEEN,
			HTS_FRAMEI,	"IFRAME"
	},
	{ HtmlFrameNew,	TgFrames,	HT_COMPLEX|HT_BETWEEN|HT_CRBOTH,
			HTS_FRAMENEW,	"FRAME"
	},
	{ HtmlFrameNo,	TgFrames,	HT_BETWEEN,
			HTS_FRAMENO,	"NOFRAMES"
	},
	{ HtmlFrameSet,	TgFrames,	HT_COMPLEX|HT_BETWEEN|HT_CRBOTH,
			HTS_FRAMESET,	"FRAMESET"
	},
	{ HtmlH1,	TgBodyStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_HDR1,	"H1"
	},
	{ HtmlH2,	TgBodyStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_HDR2,	"H2"
	},
	{ HtmlH3,	TgBodyStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_HDR3,	"H3"
	},
	{ HtmlH4,	TgBodyStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_HDR4,	"H4"
	},
	{ HtmlH5,	TgBodyStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_HDR5,	"H5"
	},
	{ HtmlH6,	TgBodyStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_HDR6,	"H6"
	},
	{ HtmlHead,	TgDocStruct,	HT_BETWEEN|HT_CRBEFORE|HT_CRAFTER,
			HTS_HEAD,	"HEAD"
	},
	{ HtmlHtml,	TgDocStruct,	HT_BETWEEN|HT_CRBEFORE|HT_CRAFTER,
			HTS_HTML,	"HTML"
	},
	{ HtmlImg,	TgImage,	HT_COMPLEX|HT_AFTER|HT_NOSUFFIX,
			HTS_IMG,	"IMG"
	},
	{ HtmlIns,	TgMarkup,	HT_BETWEEN,
			HTS_INS,	"INS"
	},
	{ HtmlIsIdx,	TgForms,	HT_AFTER|HT_NOSUFFIX,
			HTS_ISINDEX,	"ISINDEX"
	},
	{ HtmlItal,	TgFonts,	HT_BETWEEN,
			HTS_ITAL,	"I"
	},
	{ HtmlKbd,	TgTxtStruct,	HT_BETWEEN,
			HTS_KBD,	"KBD"
	},
	{ HtmlLabel,	TgForms,	HT_BETWEEN,
			HTS_LABEL,	"LABEL"
	},
	{ HtmlLbrk,	TgLinesPara,	HT_AFTER|HT_NOSUFFIX,
			HTS_LBRK,	"BR"
	},
	{ HtmlLeft,	TgBodyStruct,	HT_BETWEEN|HT_CRBOTH,
			HTS_LEFT,	"DIV",		"ALIGN=left"
	},
	{ HtmlLegend,	TgForms,	HT_BETWEEN,
			HTS_LEGEND,	"LEGEND"
	},
	{ HtmlLink,	TgLinks,	HT_COMPLEX|HT_BETWEEN,
			HTS_LINK,	"A"
	},
	{ HtmlLinkReal,	TgLinks,	HT_AFTER|HT_NOSUFFIX,
			HTS_LINKREAL,	"LINK"
	},
	{ HtmlLitm,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_LITM,	"LI"
	},
	{ HtmlMap,	TgImage,	HT_BETWEEN,
			HTS_MAP,	"MAP"
	},
	{ HtmlMeta,	TgDocStruct,	HT_AFTER|HT_NOSUFFIX,
			HTS_META,	"META"
	},
	{ HtmlMenu,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_MENU,	"MENU"
	},
	{ HtmlNoScript,	TgScripts,	HT_BETWEEN,
			HTS_NOSCRIPT,	"NOSCRIPT"
	},
	{ HtmlObject,	TgImage,	HT_BETWEEN,
			HTS_OBJECT,	"OBJECT"
	},
	{ HtmlOptGrp,	TgForms,	HT_BETWEEN,
			HTS_OPTGRP,	"OPTGROUP"
	},
	{ HtmlOrdr,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_ORDR,	"OL"
	},
	{ HtmlPara,	TgLinesPara,	HT_BETWEEN,
			HTS_PARA,	"P"
	},
	{ HtmlParam,	TgImage,	HT_AFTER|HT_NOSUFFIX,
			HTS_PARAM,	"PARAM"
	},
	{ HtmlPre,	TgLinesPara,	HT_BETWEEN,
			HTS_PRE,	"PRE"
	},
	{ HtmlQ,	TgTxtStruct,	HT_BETWEEN,
			HTS_Q,		"Q"
	},
	{ HtmlRght,	TgBodyStruct,	HT_BETWEEN|HT_CRBOTH,
			HTS_RGHT,	"DIV",		"ALIGN=right"
	},
	{ HtmlSamp,	TgTxtStruct,	HT_BETWEEN,
			HTS_SAMP,	"SAMP"
	},
	{ HtmlScript,	TgScripts,	HT_BETWEEN,
			HTS_SCRIPT,	"SCRIPT"
	},
	{ HtmlSep,	TgLinesPara,	HT_AFTER|HT_CRBOTH|HT_NOSUFFIX,
			HTS_SEP,	"HR"
	},
	{ HtmlSmll,	TgFonts,	HT_BETWEEN,
			HTS_SMLL,	"SMALL"
	},
	{ HtmlSpan,	TgBodyStruct,	HT_BETWEEN,
			HTS_DIV,	"SPAN"
	},
	{ HtmlStrg,	TgTxtStruct,	HT_BETWEEN,
			HTS_STRG,	"STRONG"
	},
	{ HtmlStrike,	TgFonts,	HT_BETWEEN,
			HTS_STRIKE,	"STRIKE"
	},
	{ HtmlStrk,	TgFonts,	HT_BETWEEN,
			HTS_STRK,	"S"
	},
	{ HtmlStyle,	TgStyleSheets,	HT_BETWEEN,
			HTS_STYLE,	"STYLE"
	},
	{ HtmlSub,	TgTxtStruct,	HT_BETWEEN,
			HTS_SUB,	"SUB"
	},
	{ HtmlSup,	TgTxtStruct,	HT_BETWEEN,
			HTS_SUP,	"SUP"
	},
	{ HtmlTarg,	TgLinks,	HT_COMPLEX|HT_BETWEEN,
			HTS_TARG,	"A"
	},
	{ HtmlTblData,	TgTables,	HT_COMPLEX|HT_BETWEEN,
			HTS_TBLDATA,	"TD"
	},
	{ HtmlTblHdr,	TgTables,	HT_COMPLEX|HT_BETWEEN,
			HTS_TBLHDR,	"TH"
	},
	{ HtmlTblNew,	TgTables,	HT_COMPLEX|HT_BETWEEN|HT_CRBOTH,
			HTS_TBLNEW,	"TABLE"
	},
	{ HtmlTblRow,	TgTables,	HT_COMPLEX|HT_BETWEEN,
			HTS_TBLROW,	"TR"
	},
	{ HtmlTbody,	TgTables,	HT_BETWEEN,
			HTS_TBODY,	"TBODY"
	},
	{ HtmlTfoot,	TgTables,	HT_BETWEEN,
			HTS_TFOOT,	"TFOOT"
	},
	{ HtmlThead,	TgTables,	HT_BETWEEN,
			HTS_THEAD,	"THEAD"
	},
	{ HtmlTtl,	TgDocStruct,	HT_BETWEEN|HT_CRAFTER,
			HTS_TTL,	"TITLE"
	},
	{ HtmlTty,	TgFonts,	HT_BETWEEN,
			HTS_TTY,	"TT"
	},
	{ HtmlUlne,	TgFonts,	HT_BETWEEN,
			HTS_ULNE,	"U"
	},
	{ HtmlUord,	TgLists,	HT_BETWEEN|HT_CRAFTER,
			HTS_UORD,	"UL"
	},
	{ HtmlVar,	TgTxtStruct,	HT_BETWEEN,
			HTS_VAR,	"VAR"
	},
	{ HtmlNONE }
}; /* htmltags[] */


/* various texts for pulldown items */
static char *shape_items[] =
	{ "none", "rect", "circle", "poly", "default", NULL };
static char *dir_items[] =
	{ "auto", "ltr", "rtl", NULL };
static char *align_items[] =
	{ "none", "top", "middle", "bottom", "left", "right", NULL };
static char *ismap_items[] =
	{ "AUTO", "ISMAP", NULL };
static char *tbl_frame_items[] =
	{ "auto", "void", "above", "below", "hsides", "vsides", "lhs",
	  "rhs", "box", "border", NULL };
static char *tbl_align_items[] =
	{ "auto", "left", "center", "right", NULL };
static char *tbl_rules_items[] =
	{ "auto", "none", "groups", "rows", "columns", "all", NULL };
static char *encoding_type_items[] =
	{ "application/x-www-form-urlencoded", "multipart/form-data", NULL };
static char *method_items[] =
	{ "auto", "GET", "POST", NULL };
static char *scrolling_items[] =
	{ "auto", "yes", "no", NULL };
#if 0
static char *type_items[] =
	{ "none", "text", "password", "checkbox", "radio button", "submit",
	  "reset", "file", "image", "hidden", "button" };
#endif

/*
 * spin button upper and lower limits.
 * 1st value: lower limit
 * 2nd value: upper limit
 * 3rd value: default value.  if SPIN_BUTTON_NO_VALUE, empty.
 */
#define SPIN_BUTTON_NO_VALUE	-99
static float width_limits[] = { 0, 9999, SPIN_BUTTON_NO_VALUE };
static float row_col_limits[] = { 0, 999, SPIN_BUTTON_NO_VALUE };

/*
 * Since a good number of the labels used in the dialogs are reused, it is
 * better to put them once in a table.  The enumerations index into this table,
 * so that tables used to specify the dialogs can just use the enumeration
 * instead of having to repeat/duplicate the same text over and over again.
 */
typedef enum html_label_list_names {
	HLL_NULL,		/* used to terminate table entries */
	HLL_ACCCHARSET,
	HLL_ACCEPTTYPES,
	HLL_ACCESSKEY,
	HLL_ACTION,
	HLL_ALIGN,
	HLL_ALT,
	HLL_BGCOLOR,
	HLL_BGIMG,
	HLL_BLUR,
	HLL_BORDER,
	HLL_CELLPADDING,	/* 10 */
	HLL_CELLSPACING,
	HLL_CHARENCODING,
	HLL_CHECKED,
	HLL_CLASS,
	HLL_CLIENTUSEMAP,
	HLL_COLUMNS,
	HLL_CONTENTTYPE,
	HLL_COORDINATES,
	HLL_CORE,
	HLL_DISABLED,		/* 20 */
	HLL_DOUBLECLICK,
	HLL_ENCTYPE,
	HLL_EVENTS,
	HLL_EXTRA,
	HLL_FOCUS,
	HLL_FRAME,
	HLL_FRAMEBORDER,
	HLL_FRAMESPACING,
	HLL_GRAPHICSURL,
	HLL_HEIGHT,		/* 30 */
	HLL_HORIZSPACING,
	HLL_ID,
	HLL_IMGMAP,
	HLL_LABEL,
	HLL_LANGCODE,
	HLL_LANGUAGE,
	HLL_LINKCOLOR,
	HLL_LINKDEST,
	HLL_LINKENDID,
	HLL_LINKRELATIONS,
	HLL_LONGDESCLINK,
	HLL_MAIN,
	HLL_MARGINHEIGHT,
	HLL_MARGINWIDTH,
	HLL_MAXLEN,
	HLL_METHOD,
	HLL_MISC,
	HLL_MULTSEL,
	HLL_NAME,
	HLL_NAMEDLINKEND,
	HLL_NORESIZE,
	HLL_ONBLUR,
	HLL_ONCHANGE,
	HLL_ONCLICK,
	HLL_ONFOCUS,
	HLL_ONKEYDOWN,
	HLL_ONKEYPRESS,
	HLL_ONKEYUP,
	HLL_ONMOUSEDOWN,
	HLL_ONMOUSEMOVE,
	HLL_ONMOUSEOUT,
	HLL_ONMOUSEOVER,
	HLL_ONMOUSEUP,
	HLL_ONRESET,
	HLL_ONSELECT,
	HLL_ONSUBMIT,
	HLL_READONLY,
	HLL_REVRELATIONS,
	HLL_ROWS,
	HLL_RULES,
	HLL_SCROLLING,
	HLL_SENSITIVEMAP,
	HLL_SHAPE,
	HLL_SIZE,
	HLL_SRC,
	HLL_STYLE,
	HLL_SUMMARY,
	HLL_TABINDEX,
	HLL_TARGET,
	HLL_TEXTCOLOR,
	HLL_TEXTDIR,
	HLL_TFNAME,
	HLL_TITLE,
	HLL_TYPE,
	HLL_VALUE,
	HLL_VERTSPACING,
	HLL_VLINKCOLOR,
	HLL_WIDTH,
	HLL_SPACE		/* must be last entry */
} html_label_list_t;



static char *html_label_list[] = {
	/* HLL_NULL */		" ",
	/* HLL_ACCCHARSET */	N_(" Accept-charset "),
	/* HLL_ACCEPTTYPES */	N_(" Accept Types "),
	/* HLL_ACCESSKEY */	N_(" Accessibility Key "),
	/* HLL_ACTION */	N_(" Action "),
	/* HLL_ALIGN */		N_(" Align "),
	/* HLL_ALT */		N_(" Alternative Text "),
	/* HLL_BGCOLOR */	N_(" Background Color "),
	/* HLL_BGIMG */		N_(" Background Image "),
	/* HLL_BLUR */		N_(" Blur "),
	/* HLL_BORDER */	N_(" Border "),
	/* HLL_CELLPADDING */	N_(" Cell Padding "),
	/* HLL_CELLSPACING */	N_(" Cell Spacing "),
	/* HLL_CHARENCODING */	N_(" Character Encoding "),
	/* HLL_CHECKED */	N_(" Checked "),
	/* HLL_CLASS */		N_(" Class "),
	/* HLL_CLIENTUSEMAP */	N_(" Client Usemap "),
	/* HLL_COLUMNS */	N_(" Columns "),
	/* HLL_CONTENTTYPE */	N_(" Content Type "),
	/* HLL_COORDINATES */	N_(" Coordinates "),
	/* HLL_CORE */		N_(" Core, i18n "),
	/* HLL_DISABLED */	N_(" Disabled "),
	/* HLL_DOUBLECLICK */	N_(" On Double Click "),
	/* HLL_ENCTYPE */	N_(" Encoding Type "),
	/* HLL_EVENTS */	N_(" Events "),
	/* HLL_EXTRA */		N_(" Extra "),
	/* HLL_FOCUS */		N_(" Focus "),
	/* HLL_FRAME */		N_(" Frame "),
	/* HLL_FRAMEBORDER */	N_(" Frame Border "),
	/* HLL_FRAMESPACING */	N_(" Frame Spacing "),
	/* HLL_GRAPHICSURL */	N_(" Graphics URL "),
	/* HLL_HEIGHT */	N_(" Height "),
	/* HLL_HORIZSPACING */	N_(" Horizontal Spacing "),
	/* HLL_ID */		N_(" Id "),
	/* HLL_IMGMAP */	N_(" Image Map "),
	/* HLL_LABEL */		N_(" Label "),
	/* HLL_LANGCODE */	N_(" Language Code "),
	/* HLL_LANGUAGE */	N_(" Language "),
	/* HLL_LINKCOLOR */	N_(" Link Color "),
	/* HLL_LINKDEST */	N_(" Link Destination "),
	/* HLL_LINKENDID */	N_(" Link End ID "),
	/* HLL_LINKRELATIONS */	N_(" Link Relations "),
	/* HLL_LONGDESCLINK */	N_(" Long Desc Link "),
	/* HLL_MAIN */		N_(" Main "),
	/* HLL_MARGINHEIGHT */	N_(" Margin Height "),
	/* HLL_MARGINWIDTH */	N_(" Margin Width "),
	/* HLL_MAXLEN */	N_(" Max Length "),
	/* HLL_METHOD */	N_(" Method "),
	/* HLL_MISC */		N_(" Miscellaneous "),
	/* HLL_MULTSEL */	N_(" Multiple Selection "),
	/* HLL_NAME */		N_(" Name "),
	/* HLL_NAMEDLINKEND */	N_(" Named Link End "),
	/* HLL_NORESIZE */	N_(" No Resize "),
	/* HLL_ONBLUR */	N_(" On Blur "),
	/* HLL_ONCHANGE */	N_(" On Change "),
	/* HLL_ONCLICK */	N_(" On Click "),
	/* HLL_ONFOCUS */	N_(" On Focus "),
	/* HLL_ONKEYDOWN */	N_(" On Key Down "),
	/* HLL_ONKEYPRESS */	N_(" On Key Press "),
	/* HLL_ONKEYUP */	N_(" On Key Up "),
	/* HLL_ONMOUSEDOWN */	N_(" On Mouse Down "),
	/* HLL_ONMOUSEMOVE */	N_(" On Mouse Move "),
	/* HLL_ONMOUSEOUT */	N_(" On Mouse Out "),
	/* HLL_ONMOUSEOVER */	N_(" On Mouse Over "),
	/* HLL_ONMOUSEUP */	N_(" On Mouse Up "),
	/* HLL_ONRESET */	N_(" On Reset "),
	/* HLL_ONSELECT */	N_(" On Select "),
	/* HLL_ONSUBMIT */	N_(" On Submit "),
	/* HLL_READONLY */	N_(" Read Only "),
	/* HLL_REVRELATIONS */	N_(" Reverse Relations "),
	/* HLL_ROWS */		N_(" Rows "),
	/* HLL_RULES */		N_(" Rules "),
	/* HLL_SCROLLING */	N_(" Scrolling "),
	/* HLL_SENSITIVEMAP */	N_(" Sensitive Map "),
	/* HLL_SHAPE */		N_(" Shape "),
	/* HLL_SIZE */		N_(" Size "),
	/* HLL_SRC */		N_(" Source "),
	/* HLL_STYLE */		N_(" Style "),
	/* HLL_SUMMARY */	N_(" Summary "),
	/* HLL_TABINDEX */	N_(" Tab Index "),
	/* HLL_TARGET */	N_(" Target "),
	/* HLL_TEXTCOLOR */	N_(" Text Color "),
	/* HLL_TEXTDIR */	N_(" Text Direction "),
	/* HLL_TFNAME */	N_(" Target Frame Name "),
	/* HLL_TITLE */		N_(" Title "),
	/* HLL_TYPE */		N_(" Type "),
	/* HLL_VALUE */		N_(" Value "),
	/* HLL_VERTSPACING */	N_(" Vertical Spacing "),
	/* HLL_VLINKCOLOR */	N_(" Visited Link Color "),
	/* HLL_WIDTH */		N_(" Width "),
	/* HLL_SPACE */		"SPACE",
	NULL
}; /* html_label_list */

typedef enum html_tag_list_names {
	HTL_NULL,
	HTL_ACCCHARSET,
	HTL_ACCEPTTYPES,
	HTL_ACCESS,
	HTL_ACCESSKEY,
	HTL_ACTION,
	HTL_ALIGN,
	HTL_ALT,
	HTL_BGCOLOR,
	HTL_BGIMG,
	HTL_BLUR,
	HTL_BORDER,
	HTL_CELLPADDING,
	HTL_CELLSPACING,
	HTL_CHARSET,
	HTL_CHECKED,
	HTL_CLASS,
	HTL_COLUMNS,
	HTL_CONTTYPE,
	HTL_COORDS,
	HTL_DIR,
	HTL_DISABLED,
	HTL_ENCTYPE,
	HTL_FOCUS,
	HTL_FRAME,
	HTL_FRAMEBORDER,
	HTL_FRAMESPACING,
	HTL_HEIGHT,
	HTL_HREF,
	HTL_HREFLANG,
	HTL_HSPACE,
	HTL_ID,
	HTL_IMGMAP,
	HTL_ISMAP,
	HTL_LABEL,
	HTL_LANG,
	HTL_LINKCOLOR,
	HTL_LONGDESC,
	HTL_MARGINHEIGHT,
	HTL_MARGINWIDTH,
	HTL_MAXLEN,
	HTL_METHOD,
	HTL_MISC,
	HTL_MULTSEL,
	HTL_NAME,
	HTL_NORESIZE,
	HTL_ONBLUR,
	HTL_ONCHANGE,
	HTL_ONCLICK,
	HTL_ONDBLCLICK,
	HTL_ONFOCUS,
	HTL_ONKEYDOWN,
	HTL_ONKEYPRESS,
	HTL_ONKEYUP,
	HTL_ONMOUSEDOWN,
	HTL_ONMOUSEMOVE,
	HTL_ONMOUSEOUT,
	HTL_ONMOUSEOVER,
	HTL_ONMOUSEUP,
	HTL_ONRESET,
	HTL_ONSELECT,
	HTL_ONSUBMIT,
	HTL_READONLY,
	HTL_REL,
	HTL_REV,
	HTL_ROWS,
	HTL_RULES,
	HTL_SCROLLING,
	HTL_SHAPE,
	HTL_SIZE,
	HTL_SRC,
	HTL_STYLE,
	HTL_SUMMARY,
	HTL_TABINDEX,
	HTL_TARGET,
	HTL_TEXTCOLOR,
	HTL_TITLE,
	HTL_TYPE,
	HTL_USEMAP,
	HTL_VALUE,
	HTL_VLINKCOLOR,
	HTL_VSPACE,
	HTL_WIDTH
} html_tag_list_t;

static char *html_tag_list[] = {
	/* HTL_NULL */		" ",
	/* HTL_ACCCHARSET */	"ACCEPT-CHARSET",
	/* HTL_ACCEPTTYPES */	"ACCEPT",
	/* HTL_ACCESS */	"ACCESS",
	/* HTL_ACCESSKEY */	"ACCESSKEY",
	/* HTL_ACTION */	"ACTION",
	/* HTL_ALIGN */		"ALIGN",
	/* HTL_ALT */		"ALT",
	/* HTL_BGCOLOR */	"BGCOLOR",
	/* HTL_BGIMG */		"IMG",
	/* HTL_BLUR */		"ONBLUR",
	/* HTL_BORDER */	"BORDER",
	/* HTL_CELLPADDING */	"CELLPADDING",
	/* HTL_CELLSPACING */	"CELLSPACING",
	/* HTL_CHARSET */	"CHARSET",
	/* HTL_CHECKED */	"CHECKED",
	/* HTL_CLASS */		"CLASS",
	/* HTL_COLUMNS */	"COLUMNS",
	/* HTL_CONTTYPE */	"CONTENT",
	/* HTL_COORDS */	"COORDS",
	/* HTL_DIR */		"DIR",
	/* HTL_DISABLED */	"DISABLED",
	/* HTL_ENCTYPE */	"ENCTYPE",
	/* HTL_FOCUS */		"FOCUS",
	/* HTL_FRAME */		"FRAME",
	/* HTL_FRAMEBORDER */	"FRAMEBORDER",
	/* HTL_FRAMESPACING */	"FRAMESPACING",
	/* HTL_HEIGHT */	"HEIGHT",
	/* HTL_HREF */		"HREF",
	/* HTL_HREFLANG */	"HREFLANG",
	/* HTL_HSPACE */	"HSPACE",
	/* HTL_ID */		"ID",
	/* HTL_IMGMAP, */	"USEMAP",
	/* HTL_ISMAP */		"ISMAP",
	/* HTL_LABEL */		"LABEL",
	/* HTL_LANG */		"LANG",
	/* HTL_LINKCOLOR */
	/* HTL_LONGDESC */	"LONGDESC",
	/* HTL_MARGINHEIGHT */	"MARGINHEIGHT",
	/* HTL_MARGINWIDTH */	"MARGINWIDTH",
	/* HTL_MAXLEN */	"MAXLENGTH",
	/* HTL_METHOD */	"METHOD",
	/* HTL_MISC */		"MISC",
	/* HTL_MULTSEL */	"MULTIPLE",
	/* HTL_NAME */		"NAME",
	/* HTL_NORESIZE */	"NORESIZE",
	/* HTL_ONBLUR */	"ONBLUR",
	/* HTL_ONCHANGE */	"ONCHANGE",
	/* HTL_ONCLICK */	"ONCLICK",
	/* HTL_ONDBLCLICK */	"ONDBLCLICK",
	/* HTL_ONFOCUS */	"ONFOCUS",
	/* HTL_ONKEYDOWN */	"ONKEYDOWN",
	/* HTL_ONKEYPRESS */	"ONKEYPRESS",
	/* HTL_ONKEYUP */	"ONKEYUP",
	/* HTL_ONMOUSEDOWN */	"ONMOUSEDOWN",
	/* HTL_ONMOUSEMOVE */	"ONMOUSEMOVE",
	/* HTL_ONMOUSEOUT */	"ONMOUSEOUT",
	/* HTL_ONMOUSEOVER */	"ONMOUSEOVER",
	/* HTL_ONMOUSEUP */	"ONMOUSEUP",
	/* HTL_ONRESET */	"ONRESET",
	/* HTL_ONSELECT */	"ONSELECT",
	/* HTL_ONSUBMIT */	"ONSUBMIT",
	/* HTL_READONLY */	"READONLY",
	/* HTL_REL */		"REL",
	/* HTL_REV */		"REV",
	/* HTL_ROWS */		"ROWS",
	/* HTL_RULES */		"RULES",
	/* HTL_SCROLLING */	"SCROLLING",
	/* HTL_SHAPE */		"SHAPE",
	/* HTL_SIZE */		"SIZE",
	/* HTL_SRC */		"SRC",
	/* HTL_STYLE */		"STYLE",
	/* HTL_SUMMARY */	"SUMMARY",
	/* HTL_TABINDEX */	"TABINDEX",
	/* HTL_TARGET */	"TARGET",
	/* HTL_TITLE */		"TITLE",
	/* HTL_TYPE */		"TYPE",
	/* HTL_USEMAP */	"USEMAP",
	/* HTL_VALUE */		"VALUE",
	/* HTL_VLINKCOLOR */	"VLINK",
	/* HTL_VSPACE */	"VSPACE",
	/* HTL_WIDTH */		"WIDTH",
	NULL
}; /* html_tag_list */


/* BEGIN: these are the pages that go into the LINK dialog notebook */
/*
 * SHAPE = { rect, circle, poly, default, NONE }  no quotes
 * DIR = { implied, ltr, rtl }  no quotes
 */
static html_wgt_info_t link_page_main[] = {
	{ HLL_LINKDEST,		IS_ENTRY|IS_ASSIGN|IS_FILESEL, HTL_HREF },
	{ HLL_NAMEDLINKEND,	IS_ENTRY|IS_ASSIGN,  HTL_NAME },
	{ HLL_LINKRELATIONS,	IS_ENTRY|IS_ASSIGN,  HTL_REL },
	{ HLL_REVRELATIONS,	IS_ENTRY|IS_ASSIGN,  HTL_REV },
	{ HLL_TFNAME,		IS_ENTRY|IS_ASSIGN,  HTL_TARGET },
	{ HLL_COORDINATES,	IS_ENTRY|IS_ASSIGN,  HTL_COORDS },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_SHAPE,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_SHAPE,
				(gpointer)shape_items },
	{ HLL_CONTENTTYPE,	IS_ENTRY|IS_ASSIGN,  HTL_CONTTYPE },
	{ HLL_NULL }
}; /* link_page_main */

static html_wgt_info_t link_page_core[] = {
	{ HLL_CHARENCODING,	IS_ENTRY|IS_ASSIGN, HTL_CHARSET },
	{ HLL_LANGCODE,	  	IS_ENTRY|IS_ASSIGN, HTL_HREFLANG },
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_LINKENDID,	IS_ENTRY|IS_ASSIGN, HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN, HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN, HTL_STYLE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,	 	IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				(gpointer)dir_items },
	{ HLL_NULL }
}; /* link_page_core */

static html_wgt_info_t link_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* link_page_events */

static html_wgt_info_t link_page_extra[] = {
	{ HLL_ACCESSKEY,	IS_ENTRY|IS_ASSIGN, HTL_ACCESS },
	{ HLL_FOCUS,		IS_ENTRY|IS_ASSIGN, HTL_FOCUS },
	{ HLL_BLUR,		IS_ENTRY|IS_ASSIGN, HTL_BLUR },
	{ HLL_TABINDEX,	 	IS_ENTRY|IS_ASSIGN, HTL_TABINDEX },
	{ HLL_NULL }
}; /* link_page_extra */
/* END: these are the pages that go into the LINK dialog notebook */

/* BEGIN: these are the pages that go into the IMAGE dialog notebook */
/*
 * DIR = { implied, ltr, rtl }  no quotes
 * ALIGN = { none, top, middle, bottom, left, right }   no quotes
 * ISMAP = { none }	if true, no assignment
 */
static html_wgt_info_t img_page_main[] = {
	{ HLL_GRAPHICSURL,	IS_ENTRY|IS_ASSIGN|IS_FILESEL, HTL_SRC },
	{ HLL_ALT,		IS_ENTRY|IS_ASSIGN, HTL_ALT },
	{ HLL_WIDTH,		IS_SPINBUT|IS_ASSIGN, HTL_WIDTH,
				(gpointer)width_limits },
	{ HLL_HEIGHT,		IS_SPINBUT|IS_ASSIGN, HTL_HEIGHT,
				(gpointer)width_limits },
	{ HLL_BORDER,		IS_SPINBUT|IS_ASSIGN, HTL_BORDER,
				(gpointer)width_limits },
	{ HLL_NULL }
}; /* img_page_main */

static html_wgt_info_t img_page_core[] = {
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN,  HTL_TITLE },
	{ HLL_LINKENDID,	IS_ENTRY|IS_ASSIGN,  HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN,  HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN,  HTL_STYLE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				(gpointer)dir_items },
	{ HLL_NULL }
}; /* img_page_core */

static html_wgt_info_t img_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* img_page_events */

static html_wgt_info_t img_page_align[] = {
	{ HLL_ALIGN,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_ALIGN,
				(gpointer)align_items },
	{ HLL_HORIZSPACING, 	IS_ENTRY|IS_ASSIGN,  HTL_HSPACE },
	{ HLL_VERTSPACING,	IS_ENTRY|IS_ASSIGN,  HTL_VSPACE },
	{ HLL_NULL }
}; /* img_page_align */

static html_wgt_info_t img_page_extra[] = {
	{ HLL_SENSITIVEMAP,	IS_PDOWN|IS_NOQUOTE, HTL_ISMAP,
				(gpointer)ismap_items },
	{ HLL_CLIENTUSEMAP,	IS_ENTRY|IS_ASSIGN,   HTL_USEMAP },
	{ HLL_LONGDESCLINK,	IS_ENTRY|IS_ASSIGN,   HTL_LONGDESC },
	{ HLL_NULL }
}; /* img_page_extra */
/* END: these are the pages that go into the IMAGE dialog notebook */

/* BEGIN: these are the pages that go into the TABLE diaglog notebook */
static html_wgt_info_t tbl_page_main[] = {
	{ HLL_WIDTH,		IS_SPINBUT|IS_ASSIGN, HTL_WIDTH,
				(gpointer)width_limits },
	{ HLL_BORDER,		IS_SPINBUT|IS_ASSIGN, HTL_BORDER,
				(gpointer)width_limits },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_FRAME,		IS_PDOWN|IS_ASSIGN, HTL_FRAME,
				(gpointer)tbl_frame_items },
	{ HLL_ALIGN,		IS_PDOWN|IS_ASSIGN, HTL_ALIGN,
				(gpointer)tbl_align_items },
	{ HLL_RULES,		IS_PDOWN|IS_ASSIGN, HTL_RULES,
				(gpointer)tbl_rules_items },
	{ HLL_BGCOLOR,		IS_ENTRY|IS_COLSEL, HTL_BGCOLOR },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_CELLSPACING,	IS_SPINBUT|IS_ASSIGN, HTL_CELLSPACING,
				(gpointer)width_limits },
	{ HLL_CELLPADDING,	IS_SPINBUT|IS_ASSIGN, HTL_CELLPADDING,
				(gpointer)width_limits },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_SUMMARY,		IS_ENTRY|IS_ASSIGN, HTL_SUMMARY },
	{ HLL_NULL }
}; /* tbl_page_main[] */

static html_wgt_info_t tbl_page_core[] = {
	{ HLL_ID,		IS_ENTRY|IS_ASSIGN, HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN, HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN, HTL_STYLE },
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				(gpointer)dir_items },
	{ HLL_NULL }
}; /* tbl_page_core[] */

static html_wgt_info_t tbl_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* tbl_page_events */
/* END: these are the pages that go into the TABLE diaglog notebook */


/*** BEGIN: Form INPUT dialog pages ***/
static html_wgt_info_t input_page_main[] = {
#if 0
	{ HLL_TYPE,		IS_PDOWN|IS_ASSIGN, HTL_TYPE, type_items },
#endif
	{ HLL_NAME,		IS_ENTRY|IS_ASSIGN, HTL_NAME },
	{ HLL_VALUE,		IS_ENTRY|IS_ASSIGN, HTL_VALUE },
	{ HLL_SRC,		IS_ENTRY|IS_ASSIGN|IS_FILESEL,
				HTL_SRC },
	{ HLL_ALT,		IS_ENTRY|IS_ASSIGN, HTL_ALT },
	{ HLL_IMGMAP,		IS_ENTRY|IS_ASSIGN|IS_FILESEL,
				HTL_IMGMAP },
	{ HLL_SIZE,		IS_ENTRY|IS_ASSIGN, HTL_SIZE },
	{ HLL_MAXLEN,		IS_ENTRY|IS_ASSIGN, HTL_MAXLEN },
	{ HLL_ACCESSKEY,	IS_ENTRY|IS_ASSIGN, HTL_ACCESSKEY },
	{ HLL_TABINDEX,		IS_ENTRY|IS_ASSIGN, HTL_TABINDEX },
	{ HLL_ALIGN,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_ALIGN,
				(gpointer)align_items },
	{ HLL_ACCEPTTYPES,	IS_ENTRY|IS_ASSIGN, HTL_ACCEPTTYPES },
	{ HLL_NULL }
}; /* input_page_main */


static html_wgt_info_t input_page_core[] = {
	{ HLL_ID,		IS_ENTRY|IS_ASSIGN, HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN, HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN, HTL_STYLE },
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				(gpointer)dir_items },
	{ HLL_NULL }
}; /* input_page_core */

static html_wgt_info_t input_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* input_page_events */

static html_wgt_info_t input_page_misc[] = {
	{ HLL_ONFOCUS,		IS_ENTRY|IS_ASSIGN, HTL_ONFOCUS },
	{ HLL_ONBLUR,		IS_ENTRY|IS_ASSIGN, HTL_ONBLUR },
	{ HLL_ONSELECT,		IS_ENTRY|IS_ASSIGN, HTL_ONSELECT },
	{ HLL_ONCHANGE,		IS_ENTRY|IS_ASSIGN, HTL_ONCHANGE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_READONLY,		IS_CHKBOX, HTL_READONLY },
	{ HLL_CHECKED,		IS_CHKBOX, HTL_CHECKED },
	{ HLL_DISABLED,		IS_CHKBOX, HTL_DISABLED },
	{ HLL_NULL }
}; /* input_page_misc */
/*** END: Form INPUT dialog pages ***/


/*** BEGIN: FORM dialog pages ***/
static html_wgt_info_t form_page_main[] = {
	{ HLL_ACTION,		IS_ENTRY|IS_ASSIGN|IS_FILESEL, HTL_ACTION },
	{ HLL_METHOD,		IS_PDOWN|IS_ASSIGN, HTL_METHOD, method_items },
	{ HLL_ENCTYPE,		IS_PDOWN|IS_ASSIGN, HTL_ENCTYPE,
				(gpointer)encoding_type_items },
	{ HLL_TARGET,		IS_ENTRY|IS_ASSIGN, HTL_TARGET },
	{ HLL_ACCCHARSET,	IS_ENTRY|IS_ASSIGN, HTL_ACCCHARSET },
	{ HLL_ONSUBMIT,		IS_ENTRY|IS_ASSIGN, HTL_ONSUBMIT },
	{ HLL_ONRESET,		IS_ENTRY|IS_ASSIGN, HTL_ONRESET },
	{ HLL_NULL }
}; /* form_page_main */


static html_wgt_info_t form_page_core[] = {
	{ HLL_ID,		IS_ENTRY|IS_ASSIGN, HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN, HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN, HTL_STYLE },
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				dir_items },
	{ HLL_NULL }
}; /* form_page_core */

static html_wgt_info_t form_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* form_page_events */
/*** END: FORM dialog pages ***/


/*** BEGIN: TEXTAREA dialog pages ***/
static html_wgt_info_t textarea_page_main[] = {
	{ HLL_NAME,		IS_ENTRY|IS_ASSIGN, HTL_NAME },
	{ HLL_ROWS,		IS_SPINBUT|IS_ASSIGN, HTL_ROWS,
				(gpointer)row_col_limits },
	{ HLL_COLUMNS,		IS_SPINBUT|IS_ASSIGN, HTL_COLUMNS,
				(gpointer)row_col_limits },
	{ HLL_ACCESSKEY,	IS_ENTRY|IS_ASSIGN, HTL_ACCESSKEY },
	{ HLL_TABINDEX,		IS_ENTRY|IS_ASSIGN, HTL_TABINDEX },
	{ HLL_ONFOCUS,		IS_ENTRY|IS_ASSIGN, HTL_ONFOCUS },
	{ HLL_ONBLUR,		IS_ENTRY|IS_ASSIGN, HTL_ONBLUR },
	{ HLL_ONSELECT,		IS_ENTRY|IS_ASSIGN, HTL_ONSELECT },
	{ HLL_ONCHANGE,		IS_ENTRY|IS_ASSIGN, HTL_ONCHANGE },
	{ HLL_NULL }
}; /* textarea_page_main */

static html_wgt_info_t textarea_page_core[] = {
	{ HLL_ID,		IS_ENTRY|IS_ASSIGN, HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN, HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN, HTL_STYLE },
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				(gpointer)dir_items },
	{ HLL_NULL }
}; /* textarea_page_core */

static html_wgt_info_t textarea_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* textarea_page_events */


/*** BEGIN: SELECT dialog pages ***/
static html_wgt_info_t select_page_main[] = {
	{ HLL_NAME,		IS_ENTRY|IS_ASSIGN, HTL_NAME },
	{ HLL_SIZE,		IS_ENTRY|IS_ASSIGN, HTL_SIZE },
	{ HLL_TABINDEX,		IS_ENTRY|IS_ASSIGN, HTL_TABINDEX },
	{ HLL_ONFOCUS,		IS_ENTRY|IS_ASSIGN, HTL_ONFOCUS },
	{ HLL_ONBLUR,		IS_ENTRY|IS_ASSIGN, HTL_ONBLUR },
	{ HLL_ONSELECT,		IS_ENTRY|IS_ASSIGN, HTL_ONSELECT },
	{ HLL_ONCHANGE,		IS_ENTRY|IS_ASSIGN, HTL_ONCHANGE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_MULTSEL,		IS_CHKBOX, HTL_MULTSEL },
	{ HLL_DISABLED,		IS_CHKBOX, HTL_DISABLED },
	{ HLL_NULL }
}; /* select_page_main */

static html_wgt_info_t select_page_core[] = {
	{ HLL_ID,		IS_ENTRY|IS_ASSIGN, HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN, HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN, HTL_STYLE },
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				(gpointer)dir_items },
	{ HLL_NULL }
}; /* select_page_core */

static html_wgt_info_t select_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* select_page_events */
/*** END: SELECT dialog pages ***/


/*** BEGIN: OPTION dialog pages ***/
static html_wgt_info_t option_page_main[] = {
	{ HLL_LABEL,		IS_ENTRY|IS_ASSIGN, HTL_LABEL },
	{ HLL_VALUE,		IS_ENTRY|IS_ASSIGN, HTL_VALUE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_MULTSEL,		IS_CHKBOX },
	{ HLL_DISABLED,		IS_CHKBOX },
	{ HLL_NULL }
}; /* option_page_main */

static html_wgt_info_t option_page_core[] = {
	{ HLL_ID,		IS_ENTRY|IS_ASSIGN, HTL_ID },
	{ HLL_CLASS,		IS_ENTRY|IS_ASSIGN, HTL_CLASS },
	{ HLL_STYLE,		IS_ENTRY|IS_ASSIGN, HTL_STYLE },
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_LANGUAGE,		IS_ENTRY|IS_ASSIGN,  HTL_LANG },
	{ HLL_TEXTDIR,		IS_PDOWN|IS_ASSIGN|IS_NOQUOTE, HTL_DIR,
				(gpointer)dir_items },
	{ HLL_NULL }
}; /* option_page_core */

static html_wgt_info_t option_page_events[] = {
	{ HLL_ONCLICK,		IS_ENTRY|IS_ASSIGN, HTL_ONCLICK },
	{ HLL_DOUBLECLICK,	IS_ENTRY|IS_ASSIGN, HTL_ONDBLCLICK },
	{ HLL_ONMOUSEOVER,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOVER },
	{ HLL_ONMOUSEDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEDOWN },
	{ HLL_ONMOUSEMOVE,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEMOVE },
	{ HLL_ONMOUSEOUT,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEOUT },
	{ HLL_ONMOUSEUP,	IS_ENTRY|IS_ASSIGN, HTL_ONMOUSEUP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_ONKEYDOWN,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYDOWN },
	{ HLL_ONKEYPRESS,	IS_ENTRY|IS_ASSIGN, HTL_ONKEYPRESS },
	{ HLL_ONKEYUP,		IS_ENTRY|IS_ASSIGN, HTL_ONKEYUP },
	{ HLL_NULL }
}; /* option_page_events */
/*** END: OPTION dialog pages ***/

/*** BEGIN: FRAMENEW dialog pages ***/
static html_wgt_info_t framenew_page_main[] = {
	{ HLL_SRC,		IS_ENTRY|IS_ASSIGN, HTL_SRC },
	{ HLL_NAME,		IS_ENTRY|IS_ASSIGN, HTL_NAME },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_FRAMEBORDER,	IS_SPINBUT|IS_ASSIGN, HTL_FRAMEBORDER,
				(gpointer)width_limits },
	{ HLL_FRAMESPACING,	IS_SPINBUT|IS_ASSIGN, HTL_FRAMESPACING,
				(gpointer)width_limits },
	{ HLL_SCROLLING,	IS_PDOWN|IS_ASSIGN, HTL_SCROLLING,
				(gpointer)scrolling_items },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_MARGINWIDTH,	IS_SPINBUT|IS_ASSIGN, HTL_MARGINWIDTH,
				(gpointer)width_limits },
	{ HLL_MARGINHEIGHT,	IS_SPINBUT|IS_ASSIGN, HTL_MARGINHEIGHT,
				(gpointer)width_limits },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_NORESIZE,		IS_CHKBOX, HTL_NORESIZE },
	{ HLL_NULL }
}; /* framenew_page_main */
/*** END: FRAMENEW dialog pages ***/

/*** BEGIN: FRAMESET dialog pages ***/
static html_wgt_info_t frameset_page_main[] = {
	{ HLL_ROWS,		IS_SPINBUT|IS_ASSIGN, HTL_ROWS,
				(gpointer)row_col_limits },
	{ HLL_COLUMNS,		IS_SPINBUT|IS_ASSIGN, HTL_COLUMNS,
				(gpointer)row_col_limits },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_FRAMEBORDER,	IS_SPINBUT|IS_ASSIGN, HTL_FRAMEBORDER,
				(gpointer)width_limits },
	{ HLL_FRAMESPACING,	IS_SPINBUT|IS_ASSIGN, HTL_FRAMESPACING,
				(gpointer)width_limits },
	{ HLL_NULL }
}; /* frameset_page_main */
/*** END: FRAMESET dialog pages ***/

#if 0
static html_wgt_info_t qstart_page_main[] = {
	{ HLL_TITLE,		IS_ENTRY|IS_ASSIGN, HTL_TITLE },
	{ HLL_BGIMG,		IS_ENTRY|IS_ASSIGN|IS_FILESEL, HTL_IMGMAP },
	{ HLL_SPACE,		IS_VSPACE },
	{ HLL_BGCOLOR,		IS_ENTRY|IS_COLSEL, HTL_BGCOLOR },
	{ HLL_TEXTCOLOR,	IS_ENTRY|IS_COLSEL, HTL_TEXTCOLOR },
	{ HLL_LINKCOLOR,	IS_ENTRY|IS_COLSEL, HTL_LINKCOLOR },
	{ HLL_VLINKCOLOR,	IS_ENTRY|IS_COLSEL, HTL_VLINKCOLOR },
}; /* qstart_page_main */

static html_page_t qstart_dlg[] = {
	{ HLL_MAIN,	qstart_page_main },
	{ HLL_NULL }
}; /* qstart_dlg */
#endif

/*** BEGIN: here are the actual dialogs */
static html_page_t link_tag_dlg[] = {
	{ HLL_MAIN,	link_page_main },
	{ HLL_CORE,	link_page_core },
	{ HLL_EVENTS,	link_page_events },
	{ HLL_EXTRA,	link_page_extra },
	{ HLL_NULL }
}; /* link_tag_dlg[] */

static html_page_t img_tag_dlg[] = {
	{ HLL_MAIN,	img_page_main },
	{ HLL_CORE,	img_page_core },
	{ HLL_EVENTS,	img_page_events },	/* same as before */
	{ HLL_ALIGN,	img_page_align },
	{ HLL_EXTRA,	img_page_extra },
	{ HLL_NULL }
}; /* img_tag_dlg[] */

static html_page_t tbl_tag_dlg[] = {
	{ HLL_MAIN,	tbl_page_main },
	{ HLL_CORE,	tbl_page_core },
	{ HLL_EVENTS,	tbl_page_events },
	{ HLL_NULL }
}; /* tbl_tag_dlg[] */

static html_page_t input_tag_dlg[] = {
	{ HLL_MAIN,	input_page_main },
	{ HLL_CORE,	input_page_core },
	{ HLL_EVENTS,	input_page_events },
	{ HLL_MISC,	input_page_misc },
	{ HLL_NULL }
}; /* input_tag_dlg[] */

static html_page_t form_tag_dlg[] = {
	{ HLL_MAIN,	form_page_main },
	{ HLL_CORE,	form_page_core },
	{ HLL_EVENTS,	form_page_events },
	{ HLL_NULL }
}; /* form_tag_dlg[] */

static html_page_t textarea_tag_dlg[] = {
	{ HLL_MAIN,	textarea_page_main },
	{ HLL_CORE,	textarea_page_core },
	{ HLL_EVENTS,	textarea_page_events },
	{ HLL_NULL }
}; /* textarea_tag_dlg */

static html_page_t select_tag_dlg[] = {
	{ HLL_MAIN,	select_page_main },
	{ HLL_CORE,	select_page_core },
	{ HLL_EVENTS,	select_page_events },
	{ HLL_NULL }
}; /* select_tag_dlg */

static html_page_t option_tag_dlg[] = {
	{ HLL_MAIN,	option_page_main },
	{ HLL_CORE,	option_page_core },
	{ HLL_EVENTS,	option_page_events },
	{ HLL_NULL }
}; /* option_tag_dlg */

static html_page_t framenew_tag_dlg[] = {
	{ HLL_MAIN,	framenew_page_main },
	{ HLL_NULL }
}; /* input_tag_dlg[] */

static html_page_t frameset_tag_dlg[] = {
	{ HLL_MAIN,	frameset_page_main },
	{ HLL_NULL }
}; /* input_tag_dlg[] */

/*** END: here are the actual dialogs */


#ifdef USE_TOOLBARS
/*
 * for toolbars, note that the tooltip private name/data must match with the
 * value of the tagname field in htmltags_t.
 */

/*
 * This is the "main" html toolbar, where all available buttons go, though only
 * the most commonly used ones will be configured to be shown.
 */
toolbar_data_t html_tbdata[] = {
	{
		N_("Title"),
		N_("Title"),
		HTS_TTL,
		"tb_title.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_TTL
	},
	{
		N_("Link"),
		N_("Link"),
		HTS_LINK,
		"tb_link.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TB_STR_LINK
	},
	{
		N_("Target"),
		N_("Target"),
		HTS_TARG,
		"tb_target.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TB_STR_TARG
	},
	{
		N_("Image"),
		N_("Image"),
		HTS_IMG,
		"tb_image.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TB_STR_IMG
	},
	{
		N_("Comment"),
		N_("Comment"),
		HTS_CMNT,
		"tb_comment.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TB_STR_CMNT
	},
	{
		"<H1>",
		N_("Header1"),
		HTS_HDR1,
		"tb_h1.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_H1
	},
	{
		"<H2>",
		N_("Header2"),
		HTS_HDR2,
		"tb_h2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_H2
	},
	{
		"<H3>",
		N_("Header3"),
		HTS_HDR3,
		"tb_h3.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_H3
	},
	{
		"<H4>",
		N_("Header4"),
		HTS_HDR4,
		"tb_h4.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_H4
	},
	{
		"<H5>",
		N_("Header5"),
		HTS_HDR5,
		"tb_h5.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_H5
	},
	{
		"<H6>",
		N_("Header6"),
		HTS_HDR6,
		"tb_h6.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_H6
	},
	{
		N_("Left"),
		N_("Align Left"),
		HTS_LEFT,
		"tb_leftjust.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_LEFT
	},
	{
		N_("Center"),
		N_("Center"),
		HTS_CENT,
		"tb_centjust.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_CENT
	},
	{
		N_("Right"),
		N_("Align Right"),
		HTS_RGHT, "tb_rightjust.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_RGHT
	},
	{
		"<B>",
		N_("Bold"),
		HTS_BOLD,
		"tb_bold.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_BOLD
	},
	{
		"<I>",
		N_("Italic"),
		HTS_ITAL,
		"tb_italic.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_ITAL
	},
	{
		"<U>",
		N_("Underline"),
		HTS_ULNE,
		"tb_underline.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_ULNE
	},
	{
		"<TT>",
		N_("Typewriter"),
		HTS_TTY,
		"tb_typewriter.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_TTY
	},
	{
		"<S>",
		N_("Strikeout"),
		HTS_STRK,
		"tb_strikeout.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_STRK
	},
	{
		"<EM>",
		N_("Emphasis"),
		HTS_EMPH,
		"tb_emphasis.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_EMPH
	},
	{
		N_("Strong"),
		N_("Strong"),
		HTS_STRG,
		"tb_strong.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_STRG
	},
	{
		N_("Font +1"),
		N_("Font Size +1"),
		HTS_FONTP,
		"tb_fontp1.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_FONTP1
	},
	{
		N_("Font -1"),
		N_("Font Size -1"),
		HTS_FONTM,
		"tb_fontm1.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_FONTM1
	},
	{
		"<PRE>",
		N_("Preformatted"),
		HTS_PRE,
		"tb_pre.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_PRE
	},
	{
		"<SUP>",
		N_("Superscript"),
		HTS_SUP,
		"tb_superscript.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_SUP
	},
	{
		"<SUB>",
		N_("Subscript"),
		HTS_SUB,
		"tb_subscript.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_SUB
	},
	{
		"<HR>",
		N_("Separator"),
		HTS_SEP,
		"tb_separator.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_SEP
	},
	{
		"<P>",
		N_("Paragraph"),
		HTS_PARA,
		"tb_paragraph.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_PARA
	},
	{
		"<BR>",
		N_("Line Break"),
		HTS_LBRK,
		"tb_linebreak.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_LBRK
	},
	{
		"<BIG>",
		N_("Big"),
		HTS_BIG,
		"tb_big.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_BIG
	},
	{
		N_("Small"),
		N_("Small"),
		HTS_SMLL,
		"tb_small.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_SMLL
	},
	{
		"<OL>",
		N_("Ordered List"),
		HTS_ORDR,
		"tb_ordered.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_ORDR
	},
	{
		"<UL>",
		N_("Unordered List"),
		HTS_UORD,
		"tb_unordered.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_UORD
	},
	{
		"<LI>",
		N_("List Item"),
		HTS_LITM,
		"tb_listitem.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_LITM
	},
	{
		"<DL>",
		N_("Definition List"),
		HTS_DEFLIST,
		"tb_listdl.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_DLIST
	},
	{
		"<DT>",
		N_("Definition Term"),
		HTS_DEFTERM,
		"tb_listdt.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_DTERM
	},
	{
		"<DD>",
		N_("Definition"),
		HTS_DEF,
		"tb_listdd.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_DEF
	},
	{
		"<MENU>",
		N_("Menu"),
		HTS_MENU,
		"tb_menu.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TB_STR_MENU
	},
	{ NULL }
}; /* html_tbdata */


/*
 * html toolbar for only font related items
 */
static toolbar_data_t html_font_tbdata[] = {
	{
		"<H1>",
		N_("Header1"),
		HTS_HDR1,
		"tb_h1.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_H1
	},
	{
		"<H2>",
		N_("Header2"),
		HTS_HDR2,
		"tb_h2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_H2
	},
	{
		"<H3>",
		N_("Header3"),
		HTS_HDR3,
		"tb_h3.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_H3
	},
	{
		"<H4>",
		N_("Header4"),
		HTS_HDR4,
		"tb_h4.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_H4
	},
	{
		"<H5>",
		N_("Header5"),
		HTS_HDR5,
		"tb_h5.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_H5
	},
	{
		"<H6>",
		N_("Header6"),
		HTS_HDR6,
		"tb_h6.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_H6
	},
	{
		" SPACE "
	},
	{
		"<B>",
		N_("Bold"),
		HTS_BOLD,
		"tb_bold.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_BOLD
	},
	{
		"<I>",
		N_("Italic"),
		HTS_ITAL,
		"tb_italic.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_ITAL
	},
	{
		"<U>",
		N_("Underline"),
		HTS_ULNE,
		"tb_underline.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_ULNE
	},
	{
		" SPACE "
	},
	{ 
		"<TT>",
		N_("Typewriter"),
		HTS_TTY,
		"tb_typewriter.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_TTY
	},
	{
		"<S>",
		N_("Strikeout"),
		HTS_STRK,
		"tb_strikeout.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_STRK
	},
	{
		"<EM>",
		N_("Emphasis"),
		HTS_EMPH,
		"tb_emphasis.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_EMPH
	},
	{
		N_("Strong"),
		N_("Strong"),
		HTS_STRG,
		"tb_strong.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_STRG
	},
	{
		" SPACE "
	},
	{
		N_("Font +1"),
		N_("Font Size +1"),
		HTS_FONTP,
		"tb_fontp1.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_FONTP1
	},
	{
		N_("Font -1"),
		N_("Font Size -1"),
		HTS_FONTM,
		"tb_fontm1.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_FONTM1
	},
	{
		" SPACE "
	},
	{
		"<PRE>",
		N_("Preformatted"),
		HTS_PRE,
		"tb_pre.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_PRE
	},
	{
		"<SUP>",
		N_("Superscript"),
		HTS_SUP,
		"tb_superscript.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_SUP
	},
	{
		"<SUB>",
		N_("Subscript"),
		HTS_SUB,
		"tb_subscript.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_SUB
	},
	{
		" SPACE "
	},
	{
		"<BIG>",
		N_("Big"),
		HTS_BIG,
		"tb_big.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_BIG
	},
	{
		N_("Small"),
		N_("Small"),
		HTS_SMLL,
		"tb_small.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FONT_TB_STR_SMLL
	},
	{ NULL }
}; /* html_font_tbdata */


static toolbar_data_t html_list_tbdata[] = {
	{
		"<OL>",
		N_("Ordered List"),
		HTS_ORDR,
		"tb_ordered.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_LIST_TB_STR_ORDR
	},
	{
		"<UL>",
		N_("Unordered List"),
		HTS_UORD,
		"tb_unordered.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_LIST_TB_STR_UORD
	},
	{
		"<LI>",
		N_("List Item"),
		HTS_LITM,
		"tb_listitem.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_LIST_TB_STR_LITM
	},
	{
		" SPACE "
	},
	{
		"<DL>",
		N_("Definition List"),
		HTS_DEFLIST,
		"tb_listdl.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_LIST_TB_STR_DLIST
	},
	{
		"<DT>",
		N_("Definition Term"),
		HTS_DEFTERM,
		"tb_listdt.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_LIST_TB_STR_DTERM
	},
	{
		"<DD>",
		N_("Definition"),
		HTS_DEF,
		"tb_listdd.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_LIST_TB_STR_DEF
	},
	{
		" SPACE "
	},
	{
		"<MENU>",
		N_("Menu"),
		HTS_MENU,
		"tb_menu.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_LIST_TB_STR_MENU
	},
	{
		NULL }
}; /* html_list_tbdata */


static toolbar_data_t html_tbl_tbdata[] = {
#if 0
	{
		N_("Wizard"),
		N_("Table Wizard"),
		HTS_TBLWIZ,
		"tb_tablewiz.xpm",
		GTK_SIGNAL_FUNC(html_table_wiz_cb),
		NULL,
		HTML_TBL_TB_STR_WIZ
	},
	{
		" SPACE "
	},
#endif
	{
		"<TABLE>",
		N_("New Table"),
		HTS_TBLNEW,
		"tb_table.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TBL_TB_STR_NEW
	},
	{
		"<TR>",
		N_("Table Row"),
		HTS_TBLROW,
		"tb_tr.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TBL_TB_STR_ROW
	},
	{
		"<TH>",
		N_("Table Header"),
		HTS_TBLHDR,
		"tb_th.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TBL_TB_STR_HDR
	},
	{
		"<TD>",
		N_("Table Data"),
		HTS_TBLDATA,
		"tb_td.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_TBL_TB_STR_DATA
	},
	{
		" SPACE "
	},
	{
		"<TABLE>",
		N_("New Table"),
		HTS_TBLNEW,
		"tb_table2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TBL_TB_STR_NEW2
	},
	{
		"<TR>",
		N_("Table Row"),
		HTS_TBLROW,
		"tb_tr2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TBL_TB_STR_ROW2
	},
	{
		"<TH>",
		N_("Table Header"),
		HTS_TBLHDR,
		"tb_th2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TBL_TB_STR_HDR2
	},
	{
		"<TD>",
		N_("Table Data"),
		HTS_TBLDATA,
		"tb_td2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_TBL_TB_STR_DATA2
	},
	{ NULL }
}; /* html_tbl_tbdata */


static toolbar_data_t html_frame_tbdata[] = {
#if 0
	{
		N_("Wizard"),
		N_("Frame Wizard"),
		HTS_FRAMEWIZ,
		"tb_framewiz.xpm",
		GTK_SIGNAL_FUNC(html_wizard_cb),
		NULL,
		HTML_FRAME_STR_WIZ
	},
	{
		" SPACE "
	},
#endif
	{
		"<FRAME>",
		N_("New Frame"),
		HTS_FRAMENEW,
		"tb_frame.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FRAME_STR_NEW
	},
	{
		"<SET>",
		N_("Frame Set"),
		HTS_FRAMESET,
		"tb_frameset.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FRAME_STR_SET
	},
	{
		" SPACE "
	},
	{
		"<FRAME>",
		N_("New Frame"),
		HTS_FRAMENEW,
		"tb_frame2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FRAME_STR_NEW2
	},
	{
		"<SET>",
		N_("Frame Set"),
		HTS_FRAMESET,
		"tb_frameset2.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FRAME_STR_SET2
	},
	{
		" SPACE "
	},
	{
		"<BASE>",
		N_("Frame Base"),
		HTS_FRAMEBASE,
		"tb_framebase.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FRAME_STR_BASE
	},
	{
		"<NOFRAME>",
		N_("No Frames"),
		HTS_FRAMENO,
		"tb_frameno.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FRAME_STR_NO
	},
	{ NULL }
}; /* html_frame_tbdata */


static toolbar_data_t html_form_tbdata[] = {
	{
		"<FORM>",
		N_("New Form"),
		HTS_FORMNEW,
		"tb_form.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_NEW
	},
	{
		" SPACE "
	},
	{
		N_("Text"),
		N_("Text Field"),
		HTS_FORMTFIELD,
		"tb_inputtext.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_TFIELD
	},
	{
		N_("Password"),
		N_("Passwd Field"),
		HTS_FORMPASSWD,
		"tb_inputpass.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_PASS
	},
	{
		N_("TextArea"),
		N_("Text Area"),
		HTS_FORMTAREA,
		"tb_inputtextarea.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_TAREA
	},
	{
		" SPACE "
	},
	{
		"<SELECT>",
		N_("List Box"),
		HTS_FORMSELECT,
		"tb_formselect.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_SEL
	},
	{
		"<OPTION>",
		N_("List Item"),
		HTS_FORMOPTION,
		"tb_formoption.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_OPT
	},
	{
		" SPACE "
	},
	{
		N_("Radio"),
		N_("Radio Button"),
		HTS_FORMRADIO,
		"tb_inputradio.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_RADIO
	},
	{
		N_("Checkbox"),
		N_("Checkbox"),
		HTS_FORMCHECK,
		"tb_inputcheckbox.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_CHECK
	},
	{
		" SPACE "
	},
	{
		N_("Button"),
		N_("Button"),
		HTS_FORMBUTTON,
		"tb_inputbutton.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_BUTTON
	},
	{
		N_("Image"),
		N_("Image"),
		HTS_FORMIMG,
		"tb_inputimage.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_IMAGE
	},
	{
		N_("Submit"),
		N_("Submit"),
		HTS_FORMSUBMIT,
		"tb_inputsubmit.xpm",
		GTK_SIGNAL_FUNC(html_tag_simple_cb),
		NULL,
		HTML_FORM_STR_SUBMIT
	},
	{
		N_("Reset"),
		N_("Reset"),
		HTS_FORMRESET,
		"tb_inputreset.xpm",
		GTK_SIGNAL_FUNC(html_tag_complex_cb),
		NULL,
		HTML_FORM_STR_RESET
	},
	{ NULL }
}; /* html_form_tbdata */


/* these are the toolbars which comprise the Advanced HTML toolbar */
static tblist_data_t html_tblist_data[] = {
	{ N_("Main"),	N_("HtmlMain"),	html_tbdata },
	{ N_("Fonts"),	N_("HtmlFonts"),	html_font_tbdata },
	{ N_("Lists"),	N_("HtmlLists"),	html_list_tbdata },
	{ N_("Tables"),	N_("HtmlTables"),	html_tbl_tbdata },
	{ N_("Frames"),	N_("HtmlFrames"),	html_frame_tbdata },
	{ N_("Forms"),	N_("HtmlForms"),	html_form_tbdata },
	{ NULL }
};
#endif	/* USE_TOOLBARS */


/*** global function defintions ***/
void
html_tag_chooser_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	GtkWidget *tmp, *taglist;
	int i, j, num_html_tags;
	char *clistnfo[1];
	int *grpcnts;
#ifdef USE_LIBGLADE
	GladeXML *xml;
#else
	GtkWidget *mainvbox, *label, *action;
#endif

	if (misc_show_and_raise(w->tagchooser))
		return;

	/*
	 * scan through all the html tags and count up the number of tags per
	 * group type.
	 */
	grpcnts = g_new0(int, TgCOUNT + 1);
	num_html_tags = (sizeof(htmltags) / sizeof(htmltags[0])) - 1;
	for (i = 0; i < num_html_tags; i++) {
		if (htmltags[i].action != HtmlNONE)
			grpcnts[htmltags[i].taggrp]++;
	}
	GNPDBG_HTML(("html_tag_chooser_cb: num entries in htmltags[] : %d\n",
		     num_html_tags));

	/* now create the tag chooser dialog */
#ifdef USE_LIBGLADE
	xml = my_glade_xml_get("HTML Tag Chooser");
	g_assert(xml);
	w->tagchooser = my_glade_widget_get(xml, "HTML Tag Chooser", 0);
	g_assert(w->tagchooser);
	w->tc_nb = my_glade_widget_get(xml, "tc_nb", 0);
	g_assert(w->tc_nb);
#else
	w->tagchooser = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(w->tagchooser), _("HTML Tag Chooser"));
	gtk_container_set_border_width(GTK_CONTAINER(w->tagchooser), 5);
	mainvbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(w->tagchooser), mainvbox);

	/* Tag Group descriptions */
	w->tc_nb = gtk_notebook_new();
	gtk_box_pack_start(GTK_BOX(mainvbox), w->tc_nb, TRUE, TRUE, 0);
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(w->tc_nb), GTK_POS_LEFT);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(w->tc_nb), TRUE);
	gtk_notebook_popup_enable(GTK_NOTEBOOK(w->tc_nb));
#endif

	for (i = 0; i < (sizeof(tagdesc) / sizeof(tagdesc[0])); i++) {
#ifdef USE_LIBGLADE
		char name[8];
		g_snprintf(name, 8, "cl%d", i);
		taglist = my_glade_widget_get(xml, name, 0);
		g_assert(taglist);
#else
		clistnfo[0] = _("Tag Name");
		taglist = gtk_clist_new_with_titles(1, clistnfo);
# ifdef GTK_HAVE_FEATURES_1_1_0
		gtk_clist_column_titles_passive(GTK_CLIST(taglist));
		(void)gtk_clist_columns_autosize(GTK_CLIST(taglist));
# endif
#endif
		(void)gtk_signal_connect(GTK_OBJECT(taglist), "select_row",
				 (GtkSignalFunc)html_tag_chooser_selrow, w);

		/*
		 * allocate array to hold all the html id's for these tags.
		 * memory is freed when the tag chooser dialog is destroyed.
		 */
		if (!tagdesc[i].html_id_list)
			tagdesc[i].html_id_list = g_new(html_action_t,
							grpcnts[i]);

		/*
		 * now scan through htmltags[] to create the clist.  store each
		 * tag's html id in the html_id_list array.  this is how we
		 * associate in the clist's "select_row" callback exactly which
		 * html tag was selected without having to do a strcmp()
		 * through htmltags[].
		 */
		for (j = 0; j < num_html_tags; j++) {
			if (htmltags[j].taggrp != i)
				continue;

			if (htmltags[j].data) {
#ifdef GTK_HAVE_FEATURES_1_1_0
				clistnfo[0] = g_strdup_printf("%s %s",
							htmltags[j].prefix,
							htmltags[j].data);
#else
				clistnfo[0] =
					g_new(char,
					      strlen(htmltags[j].data) +
					      strlen(htmltags[j].prefix) + 4);
				sprintf(clistnfo[0], "%s %s",
					htmltags[j].prefix, htmltags[j].data);
#endif
				(void)gtk_clist_append(GTK_CLIST(taglist),
						       clistnfo);
				g_free(clistnfo[0]);
			} else {
				clistnfo[0] = htmltags[j].prefix;
				(void)gtk_clist_append(GTK_CLIST(taglist),
						       clistnfo);
			}

			/*
			 * 'listcnt' is reset to zero when the tag chooser
			 * dialog is destroyed.
			 */
			tagdesc[i].html_id_list[tagdesc[i].listcnt] =
							htmltags[j].action;
			tagdesc[i].listcnt++;
		} /* for each html tag */

#ifndef USE_LIBGLADE
		/* put clist in a scrolled window and stuff into notebook */
		tmp = gtk_scrolled_window_new(NULL, NULL);
		gtk_container_add(GTK_CONTAINER(tmp), taglist);		
		gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(tmp),
					       GTK_POLICY_AUTOMATIC,
					       GTK_POLICY_AUTOMATIC);
		gtk_container_set_border_width(GTK_CONTAINER(tmp), 5);
		label = gtk_label_new(gettext(tagdesc[i].grpname));
		gtk_notebook_append_page(GTK_NOTEBOOK(w->tc_nb), tmp, label);
#endif
	} /* for each HTML tag group */

	g_free(grpcnts);

#ifdef USE_LIBGLADE
	tmp = my_glade_widget_get(xml, "tc_insert_tag_but", 0);
	g_assert(tmp);
	(void)gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
				 GTK_SIGNAL_FUNC(html_tag_chooser_insert), w);
	tmp = my_glade_widget_get(xml, "tc_close_but", 0);
	g_assert(tmp);
	(void)gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
				 GTK_SIGNAL_FUNC(html_tag_chooser_close), w);
	gtk_object_destroy(GTK_OBJECT(xml));
#else
	/* lastly, the button s*/
	gtk_box_pack_start(GTK_BOX(mainvbox), gtk_hseparator_new(),
			   FALSE, FALSE, 5);
	action = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_end(GTK_BOX(mainvbox), action, FALSE, FALSE, 5);

	(void)misc_button_new_w_label(_(" Close Dialog "),
				      GNOME_STOCK_BUTTON_CLOSE,
				      GTK_SIGNAL_FUNC(html_tag_chooser_close),
				      w, action, PACK_END | PACK_FILL |
				      CANCEL_DEFAULT | GRAB_DEFAULT, 2);

	(void)misc_button_new_w_label(_(" Insert Tag "), NULL,
				      GTK_SIGNAL_FUNC(html_tag_chooser_insert),
				      w, action, PACK_END | PACK_FILL |
				      CANCEL_DEFAULT, 2);
#endif

	(void)gtk_signal_connect(GTK_OBJECT(w->tagchooser), "delete_event",
				 GTK_SIGNAL_FUNC(gtk_true), NULL);
	gtk_window_set_default_size(GTK_WINDOW(w->tagchooser),
				    prefs.tagchooser_w, prefs.tagchooser_h);
	gtk_widget_show_all(w->tagchooser);
} /* html_tag_chooser_cb */


void
html_tag_lower(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_HTML_TAG_LOWER();
	CLEAR_HTML_TAG_UPPER();
	menu_main_state_set(w);
	msgbar_printf(w, _("Insert HTML tags in LOWER case"));
} /* html_tag_lower */


void
html_tag_upper(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	SET_HTML_TAG_UPPER();
	CLEAR_HTML_TAG_LOWER();
	menu_main_state_set(w);
	msgbar_printf(w, _("Insert HTML tags in UPPER case"));
} /* html_tag_upper */


#ifdef USE_TOOLBARS
/*
 * PUBLIC: html_tb_create
 *
 * in:
 *	w: current window
 *
 * out:
 *	if the Advanced HTML toolbar is used, returns a pointer to the notebook
 *	widget, which contains all the individual toolbars.
 *	if the regular HTML toolbar is used, returns a pointer to the toolbar
 *	widget.
 */
GtkWidget *
html_tb_create(win_t *w)
{
	GtkWidget *tmp, *tnb;
	tblist_data_t *tbldp, *newtbldp;

	if (IS_ADVANCED_HTML_TB()) {
		tnb = gtk_notebook_new();
		gtk_widget_show(tnb);
		gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tnb), GTK_POS_TOP);
		gtk_notebook_set_scrollable(GTK_NOTEBOOK(tnb), TRUE);
		gtk_notebook_popup_enable(GTK_NOTEBOOK(tnb));

		for (tbldp = html_tblist_data; tbldp->name; tbldp++) {
			newtbldp = g_new(tblist_data_t, 1);
			newtbldp->name = g_strdup(tbldp->name);
			newtbldp->private_name = g_strdup(tbldp->private_name);
			newtbldp->tbdata = tbldp->tbdata;
			newtbldp->tb = toolbar_create_start(w, tbldp->tbdata);
			gtk_widget_show(newtbldp->tb);
			tmp = gtk_label_new(tbldp->name);
			gtk_notebook_append_page(GTK_NOTEBOOK(tnb),
						 newtbldp->tb, tmp);

			/* add to HTML tblist also */
			w->html_tblist = g_slist_prepend(w->html_tblist,
							 newtbldp);
		}

		tmp = tnb;
	} else {
		tbldp = html_tblist_data;
		newtbldp = g_new(tblist_data_t, 1);
		newtbldp->name = g_strdup(tbldp->name);
		newtbldp->private_name = g_strdup(tbldp->private_name);
		newtbldp->tbdata = tbldp->tbdata;
		newtbldp->tb = toolbar_create_start(w, tbldp->tbdata);
		gtk_widget_show(newtbldp->tb);

		/* add to HTML tblist also */
		w->html_tblist = g_slist_prepend(w->html_tblist, newtbldp);

		tmp = newtbldp->tb;
	}

	return tmp;
} /* html_tb_create */
#endif	/* USE_TOOLBARS */



#if defined(GTK_HAVE_FEATURES_1_1_0) && !defined(USE_GNOME)
/*
 * PUBLIC: html_menu_cb
 *
 * all HTML insertions from the main menu use this callback.  see menu.c
 */
void
html_menu_cb(GtkWidget *wgt, gpointer cbdata, guint cbaction)
{
	html_menu_cb_common((html_action_t)cbaction, cbdata);
} /* html_menu_cb */
#endif


#if defined(USE_GNOME) || !defined(GTK_HAVE_FEATURES_1_1_0)
/*
 * the next bunch of callbacks is kind of inefficient since we already have
 * html_tag_simple_cb() and html_tag_complex_cb() because the MenuFactory is a
 * bit limited (and because the GNOME menu item callbacks are setup this way).
 */
void
html_fontp1_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFontP, cbdata);
}

void
html_fontm1_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFontM, cbdata);
}

void
html_preformatted_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlPre, cbdata);
}

void
html_deflist_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlDefList, cbdata);
}

void
html_defterm_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlDefTerm, cbdata);
}

void
html_def_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlDef, cbdata);
}

void
html_menu_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlMenu, cbdata);
}

void
html_tablenew_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTblNew, cbdata);
}

void
html_tablerow_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTblRow, cbdata);
}

void
html_tablehdr_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTblHdr, cbdata);
}

void
html_tabledata_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTblData, cbdata);
}

void
html_framenew_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFrameNew, cbdata);
}

void
html_frameset_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFrameSet, cbdata);
}

void
html_framebase_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFrameBase, cbdata);
}

void
html_frameno_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFrameNo, cbdata);
}

void
html_formnew_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormNew, cbdata);
}

void
html_formtfield_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormTfield, cbdata);
}

void
html_formpasswd_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormPass, cbdata);
}

void
html_formtarea_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormTarea, cbdata);
}

void
html_formselect_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormSel, cbdata);
}

void
html_formoption_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormOpt, cbdata);
}

void
html_formradio_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormRadio, cbdata);
}

void
html_formchkbox_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormChk, cbdata);
}

void
html_formbutton_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormBtn, cbdata);
}

void
html_formimage_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormImg, cbdata);
}

void
html_formsubmit_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlFormSubmit, cbdata);
}

void
html_title_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTtl, cbdata);
}

void
html_comment_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlCmnt, cbdata);
}

void
html_link_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlLink, cbdata);
}

void
html_target_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTarg, cbdata);
}

void
html_image_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlImg, cbdata);
}

void
html_h1_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH1, cbdata);
}

void
html_h2_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH2, cbdata);
}

void
html_h3_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH3, cbdata);
}

void
html_h4_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH4, cbdata);
}

void
html_h5_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH5, cbdata);
}

void
html_h6_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlH6, cbdata);
}

void
html_left_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlLeft, cbdata);
}

void
html_center_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlCent, cbdata);
}

void
html_right_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlRght, cbdata);
}

void
html_paragraph_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlPara, cbdata);
}

void
html_linebreak_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlLbrk, cbdata);
}

void
html_separator_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSep, cbdata);
}

void
html_bold_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlBold, cbdata);
}

void
html_italic_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlItal, cbdata);
}

void
html_underline_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlUlne, cbdata);
}

void
html_typewriter_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlTty, cbdata);
}

void
html_strikeout_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlStrk, cbdata);
}

void
html_emphasis_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlEmph, cbdata);
}

void
html_strong_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlStrg, cbdata);
}

void
html_subscript_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSub, cbdata);
}

void
html_superscript_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSup, cbdata);
}

void
html_big_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlBig, cbdata);
}

void
html_small_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlSmll, cbdata);
}

void
html_ordered_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlOrdr, cbdata);
}

void
html_unordered_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlUord, cbdata);
}

void
html_listitem_cb(GtkWidget *widget, gpointer cbdata)
{
	html_menu_cb_common(HtmlLitm, cbdata);
}
#endif	/* defined(USE_GNOME) || !defined(GTK_HAVE_FEATURES_1_1_0) */


/*** local function defintions ***/
/*
 * PRIVATE: html_menu_cb_common
 *
 * common routine called by all the braindead menu callbacks for inserting html
 * tags.
 */
static void
html_menu_cb_common(html_action_t action, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	htmltags_t *htp = html_find_tag_by_action(action);

	if (htp->opt & HT_COMPLEX)
		html_tag_complex_common(htp, DOC_CURRENT(w), htp->tagname);
	else
		html_tag_simple_common(htp, w);
} /* html_menu_cb_common */


/*
 * PRIVATE: html_find_tag_by_action
 *
 * finds the HTML tag name by its action enumeration.
 *
 * XXX - FIXME - TODO - this search should be binary, not linear.
 */
static htmltags_t *
html_find_tag_by_action(html_action_t action)
{
	htmltags_t *htp;

	for (htp = htmltags; htp->action != HtmlNONE; htp++) {
		if (htp->action == action)
			break;
	}
	g_assert(htp->action != HtmlNONE);

	GNPDBG_HTML(("html_find_tag_by_action: action = %d, taggrp = %d, "
		     "opt = x%x, tagname = '%s', prefix = '%s'\n",
		     htp->action, htp->taggrp, htp->opt, htp->tagname,
		     htp->prefix));
	return htp;
} /* html_htp_by_action */


/*
 * PRIVATE: html_tag_simple_cb
 *
 * handles simple tags.  e.g., tags that are symmetrical such as "<B></B>"
 *
 * more complex tags that are asymetrical are handled elsewhere.
 *
 * NOTE: this is **only** used from the toolbar, hence the necessity of using
 * a piece of toolbar data (the tooltip_private text) to find out which tag
 * is being inserted.
 */
static void
html_tag_simple_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	GtkTooltipsData *tooltipsdata;
	htmltags_t *htp;

	/* find the html tag info in the table.  the method used is a little */
	tooltipsdata = gtk_tooltips_data_get(wgt);
	g_assert(tooltipsdata != NULL);
	htp = html_find_tag_by_name(tooltipsdata->tip_private);

	html_tag_simple_common(htp, w);
} /* html_tag_simple_cb */


/*
 * PRIVATE: html_find_tag_by_name
 *
 * finds the entry in the htmltags table by tagname id.  This is only used from
 * html_tag_simple_cb() html_tag_complex_cb().
 */
static htmltags_t *
html_find_tag_by_name(char *tagname)
{
	htmltags_t *htp;

	for (htp = htmltags; htp->tagname; htp++) {
		if (strcmp(tagname, htp->tagname) == 0)
			break;
	}

	GNPDBG_HTML(("html_find_tag_by_name: (tagname = '%s') action = %d, "
		     "taggrp = %d, opt = x%x, tagname = '%s', prefix = '%s'\n",
		     tagname, htp->action, htp->taggrp, htp->opt, htp->tagname,
		     htp->prefix));
	return htp;
} /* html_find_tag_by_name */


/*
 * PRIVATE: html_tag_simple_common
 *
 * common routine for simple html tags
 */
static void
html_tag_simple_common(htmltags_t *htp, win_t *w)
{
	char **buf;

	/* build the buffer */
	buf = html_tag_simple_buf_build(htp, w);

	/* insert it into the text widget and adjust cursor position */
	html_tag_insert(DOC_CURRENT(w), buf, htp);

	g_free(buf[0]);
	g_free(buf[1]);
	g_free(buf);
} /* html_tag_simple_common */


/*
 * PRIVATE: html_tag_simple_buf_build
 *
 * builds and returns a text buffer containing a simple/symmetrical HTML tag. 
 * called only from html_tag_simple_cb().
 */
static char **
html_tag_simple_buf_build(htmltags_t *htp, win_t *w)
{
	int len = 0, start = 0, stop = 0;
	char *buf, *sel, **ret;
	doc_t *d;

	ret = g_new0(char *, 2);

	/*
	 * if there is text already highlighted, then we'll put the html tag
	 * around the highlighted text.
	 */
	d = DOC_CURRENT(w);
	if (GTK_EDITABLE(d->data)->has_selection) {
		start = GTK_EDITABLE(d->data)->selection_start_pos;
		stop = GTK_EDITABLE(d->data)->selection_end_pos;
		if (stop < start)
			SWAP_INT(stop, start);
		len += (stop - start);
	}

	/* build a buffer for the html tag text to insert */
	len += strlen(htp->prefix) + 2 +
	      (htp->data ? (strlen(htp->data) + 1) : 0) +
	      (!(htp->opt & HT_NOSUFFIX) ? (1 + strlen(htp->prefix) + 2) : 1) +
	      ((htp->opt & HT_CRBEFORE) ? 1 : 0) +
	      ((htp->opt & HT_CRAFTER) ? 1 : 0) + 3;
	buf = g_new(char, len);
	/* insert prefix */
	if (htp->opt & HT_BEFORE) {
		sprintf(buf, "\n<%s", htp->prefix);
	} else
		sprintf(buf, "<%s", htp->prefix);
	if (htp->data) {
		strcat(buf, " ");
		strcat(buf, htp->data);
	}
	if (IS_HTML_TAG_LOWER())
		g_strdown(buf);
	strcat(buf, ">");

	if (start != stop) {
		sel = gtk_editable_get_chars(GTK_EDITABLE(d->data),
					     start, stop);
		strcat(buf, sel);
		ret[1] = sel;
		gtk_text_set_point(GTK_TEXT(d->data), start);
		gtk_editable_set_position(GTK_EDITABLE(d->data), start);
		gtk_text_freeze(GTK_TEXT(d->data));
		gtk_text_forward_delete(GTK_TEXT(d->data), stop - start);
		gtk_text_thaw(GTK_TEXT(d->data));
	}

	/* add suffix */
	if (!(htp->opt & HT_NOSUFFIX)) {
		strcat(buf, "</");
		strcat(buf, htp->prefix);
		if (IS_HTML_TAG_LOWER()) {
			int plen = strlen(htp->prefix);
			g_strdown(buf + strlen(buf) - plen);
		}
		strcat(buf, ">");
	}
	if (htp->opt & HT_CRAFTER)
		strcat(buf, "\n");

	GNPDBG_HTML(("html_tag_simple_buf_build: inserting buf (len = %d,"
		     "strlen = %ld) '%s'\n", len, (long)strlen(buf), buf));
	g_assert((size_t)strlen(buf) <= (size_t)len);

	ret[0] = buf;
	return ret;
} /* html_tag_simple_buf_build */


#if 0
static void
html_wizard_cb(GtkWidget *wgt, gpointer cbdata)
{
	printf("html_wizard_cb() entered\n");
} /* html_wizard_cb */
#endif


/*
 * PRIVATE: html_tag_complex_cb
 *
 * toplevel callback to create a dialog box to let the user enter information
 * for a complex/asymmetrical HTML tag. 
 *
 * NOTE: this is **only** used from the toolbar, hence the necessity of using
 * a piece of toolbar data (the tooltip_private text) to find out which tag
 * is being inserted.
 */
static void
html_tag_complex_cb(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	doc_t *d = (doc_t *)w->curdoc;
	GtkTooltipsData *tooltipsdata;
	htmltags_t *htp;

	/* find the html tag info in the table */
	tooltipsdata = gtk_tooltips_data_get(wgt);
	g_assert(tooltipsdata != NULL);
	htp = html_find_tag_by_name(tooltipsdata->tip_private);

	html_tag_complex_common(htp, d, tooltipsdata->tip_text);
} /* html_tag_complex_cb */


/*
 * PRIVATE: html_tag_complex_common
 *
 * common routine for all complex html tags
 *
 * 'opttxt' is optional text that is expected to be used in the dialog title.
 */
static void
html_tag_complex_common(htmltags_t *htp, doc_t *d, char *opttxt)
{
	char *buf;
#ifndef GTK_HAVE_FEATURES_1_1_0
	int len;
#endif

	switch (htp->action) {
	case HtmlTarg:
		html_entry_dlg(d, htp,
			       _("Insert HTML Target"), _("Target Name:"),
			       GTK_SIGNAL_FUNC(html_target_exec));
		break;
	case HtmlCmnt:
		html_entry_dlg(d, htp, _("Insert Comment"), _("Comment Text:"),
			       GTK_SIGNAL_FUNC(html_comment_exec));
		break;
	case HtmlLink:
		html_complex_dlg(link_tag_dlg, d, htp, _("HTML Link Tag"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	case HtmlImg:
		html_complex_dlg(img_tag_dlg, d, htp, _("HTML Image Tag"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	case HtmlTblData:
	case HtmlTblHdr:
	case HtmlTblNew:
	case HtmlTblRow:
#ifdef GTK_HAVE_FEATURES_1_1_0
		buf = g_strdup_printf(_("HTML Table Tag '%s'"), htp->tagname);
#else
		len = 18 + strlen(htp->tagname);
		buf = g_new(char, len);
		g_snprintf(buf, len, _("HTML Table Tag '%s'"), htp->tagname);
#endif
		html_complex_dlg(tbl_tag_dlg, d, htp, buf,
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		g_free(buf);
		break;
	case HtmlFormNew:
		html_complex_dlg(form_tag_dlg, d, htp, _("HTML Form"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	case HtmlFormTarea:
		html_complex_dlg(textarea_tag_dlg, d, htp,
				 _("HTML Form TEXTAREA"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	case HtmlFormSel:
		html_complex_dlg(select_tag_dlg, d, htp, _("HTML Form SELECT"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	case HtmlFormOpt:
		html_complex_dlg(option_tag_dlg, d, htp, _("HTML Form OPTION"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	case HtmlFormBtn:
	case HtmlFormChk:
	case HtmlFormFile:
	case HtmlFormHid:
	case HtmlFormImg:
	case HtmlFormPass:
	case HtmlFormRadio:
	case HtmlFormReset:
	case HtmlFormSubmit:
	case HtmlFormTfield:
#ifdef GTK_HAVE_FEATURES_1_1_0
		if (opttxt)
			buf = g_strdup_printf(_("HTML Form Input: %s"), opttxt);
#else
		len = 17;
		if (opttxt) {
			len += (strlen(opttxt) + 2);
			buf = g_new(char, len);
			g_snprintf(buf, len, _("HTML Form Input: %s"), opttxt);
		}
#endif
		else
			buf = g_strdup(_("HTML Form Input"));

		html_complex_dlg(input_tag_dlg, d, htp, buf,
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		g_free(buf);
		break;
	case HtmlFrameNew:
		html_complex_dlg(framenew_tag_dlg, d, htp, _("HTML New Frame"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	case HtmlFrameSet:
		html_complex_dlg(frameset_tag_dlg, d, htp,
				 _("HTML New Frame Set"),
				 GTK_SIGNAL_FUNC(html_complex_exec),
				 GTK_SIGNAL_FUNC(html_complex_cancel));
		break;
	default:
		g_warning("html_tag_complex_common: unknown action type = %d"
			  " (tagname='%s')\n", htp->action, htp->tagname);
		break;
	}
} /* html_tag_complex_common */


/*
 * PRIVATE: html_complex_dlg
 *
 * routine to create a dialog box for either the link tag or the image tag.
 * called only from html_tag_complex_cb().
 *
 * dlg_pages: points to either link_tag_dlg[] or img_tag_dlg[]
 */
static void
html_complex_dlg(html_page_t *dlg_pages, doc_t *d, htmltags_t *htp,
		  char *title, GtkSignalFunc execute_cb,
		  GtkSignalFunc cancel_cb)
{
	int maxent;
	html_page_t *hpp;
	html_wgt_info_t *hwip;
	html_entry_dlg_t *hedp;
	GtkWidget *dlg, *nb;

	if (misc_show_and_raise(d->w->html_dlg))
		return;

	/* create top-level window and notebook */
	dlg = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(dlg), (title) ? title : _("Edit Tag"));
	nb = gtk_notebook_new();
	gtk_container_border_width(GTK_CONTAINER(nb), 6);
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(nb), GTK_POS_TOP);
	gtk_notebook_set_scrollable(GTK_NOTEBOOK(nb), TRUE);
	gtk_notebook_popup_enable(GTK_NOTEBOOK(nb));
	gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dlg)->vbox), nb, TRUE, TRUE, 0);

	/* need to pass cur doc and htp ptr as callback data */
	hedp = g_new0(html_entry_dlg_t, 1);
	hedp->d = d;
	hedp->htp = htp;
	hedp->dlg_pages = dlg_pages;

	/* check if there is text already selected */
	if (GTK_EDITABLE(d->data)->has_selection) {
		hedp->start = GTK_EDITABLE(d->data)->selection_start_pos;
		hedp->stop = GTK_EDITABLE(d->data)->selection_end_pos;
		if (hedp->stop < hedp->start)
			SWAP_INT(hedp->stop, hedp->start);
	}

	(void)gtk_signal_connect(GTK_OBJECT(dlg), "destroy",
				 GTK_SIGNAL_FUNC(html_dlg_destroy),
				 (gpointer)hedp);

	/* track which dialog window it is */
	d->w->html_dlg = dlg;
	hedp->edp.toplev = d->w->html_dlg;

	/* count max number of entries across all pages */
	for (maxent = 0, hpp = dlg_pages; hpp->hll_id != HLL_NULL; hpp++) {
		int count = 0;

		for (hwip = hpp->hwip; hwip->hll_id != HLL_NULL; hwip++)
			count++;
		if (count > maxent)
			maxent = count;
	}

	/* add pages */
	html_complex_dlg_pages_add(d->w->toplev, dlg_pages, maxent,
				   GTK_NOTEBOOK(nb));

	/* now the buttons */
	html_complex_dlg_buttons_add(dlg, execute_cb, cancel_cb, hedp);

	gtk_widget_show_all(dlg);
} /* html_complex_dlg */


/*
 * PRIVATE: html_complex_dlg_buttons_add
 *
 * creates the "Ok" and "Cancel" buttons for the link/img dialog box.  called
 * only from html_complex_dlg().
 */
static void
html_complex_dlg_buttons_add(GtkWidget *toplev, GtkSignalFunc execute_cb,
			      GtkSignalFunc cancel_cb, html_entry_dlg_t *hedp)
{
	GtkWidget *hbox;

	hbox = GTK_DIALOG(toplev)->action_area;
	gtk_container_border_width(GTK_CONTAINER(hbox), 5);

	(void)misc_button_new_w_label(_(" Ok "), GNOME_STOCK_BUTTON_OK,
				      GTK_SIGNAL_FUNC(execute_cb), hedp, hbox,
				      PACK_START | PACK_EXPAND | PACK_FILL |
				      CANCEL_DEFAULT, 0);

	(void)misc_button_new_w_label(_(" Cancel "), GNOME_STOCK_BUTTON_CANCEL,
				      GTK_SIGNAL_FUNC(cancel_cb), hedp, hbox,
				      PACK_START | PACK_EXPAND | PACK_FILL |
				      CANCEL_DEFAULT | GRAB_DEFAULT, 0);
} /* html_complex_dlg_buttons_add */


/*
 * PRIVATE: html_complex_dlg_pages_add
 *
 * adds pages into the notebook for the link/img dialog box.  called only from
 * html_complex_dlg().
 */
static void
html_complex_dlg_pages_add(GtkWidget *toplev, html_page_t *dlg_pages,
	int maxent, GtkNotebook *nb)
{
	GtkWidget *frame, *table, *tmp;
	html_page_t *hpp;

	for (hpp = dlg_pages; hpp->hll_id != HLL_NULL; hpp++) {
		GNPDBG_HTML(("html_complex_dlg: page '%s'\n",
			     html_label_list[hpp->hll_id]));

		/* it's a new page, so create a frame and table */
		frame = gtk_frame_new(NULL);
		gtk_container_border_width(GTK_CONTAINER(frame), 4);
		table = gtk_table_new(maxent, 2, TRUE);
		gtk_container_border_width(GTK_CONTAINER(table), 4);
		gtk_container_add(GTK_CONTAINER(frame), table);
		tmp = gtk_label_new(gettext(html_label_list[hpp->hll_id]));
		gtk_notebook_append_page(GTK_NOTEBOOK(nb), frame, tmp);

		/* add each item for current page */
		html_complex_dlg_items_add(toplev, GTK_TABLE(table), hpp);
	} /* for each page */
} /* html_complex_dlg_pages_add */


/*
 * PRIVATE: html_complex_dlg_items_add
 *
 * adds each item/entry into a page of a notebook.  called only from
 * html_complex_dlg_pages_add().
 */
static void
html_complex_dlg_items_add(GtkWidget *toplev, GtkTable *table, html_page_t *hpp)
{
	html_wgt_info_t *hwip;
	GtkWidget *hb, *pic;
	GtkWidget *tmp = NULL;
	GtkObject *adj;
	float *spindata;
	char buf[8];
	int row;

	for (row = 0, hwip = hpp->hwip;
	     hwip->hll_id != HLL_NULL;
	     row++, hwip++) {

		/* one of the following three must be present */
		if (hwip->opt & IS_PDOWN) {
			g_assert(hwip->data);
			tmp = html_tag_pulldown_create((char **)hwip->data);
		} else if (hwip->opt & IS_ENTRY) {
			tmp = gtk_entry_new();
		} else if (hwip->opt & IS_CHKBOX) {
			tmp = gtk_check_button_new();
		} else if (hwip->opt & IS_SPINBUT) {
			g_assert(hwip->data);
			spindata = (float *)hwip->data;

			adj = gtk_adjustment_new(0, spindata[0], spindata[1],
						 1, 1, 0);
			tmp = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 1.0, 0);
			if (spindata[2] == SPIN_BUTTON_NO_VALUE)
				gtk_entry_set_text(GTK_ENTRY(tmp), "");
			else {
				g_snprintf(buf, 8, "%f", spindata[2]);
				gtk_entry_set_text(GTK_ENTRY(tmp), buf);
			}
		} else if (hwip->opt & IS_VSPACE) {
			continue;
		} else {
			g_warning("html_complex_dlg_items_add: unknown opt "
				  "(x%x, itemname='%s')\n",
				  hwip->opt, html_label_list[hwip->hll_id]);
			continue;
		}

		g_assert(tmp != NULL);

		/* need to store (keep track of) the widget */
		hwip->wgt = tmp;

		if ((hwip->opt & IS_FILESEL) || (hwip->opt & IS_COLSEL)) {
			hb = gtk_hbox_new(FALSE, 3);
			gtk_box_pack_start(GTK_BOX(hb), tmp, TRUE, TRUE, 0);
			tmp = gtk_button_new();
			if (hwip->opt & IS_FILESEL)
				pic = misc_new_pixmap("filesel.xpm", toplev,
						      NULL, NULL);
			else
				pic = misc_new_pixmap("colsel.xpm", toplev,
						      NULL, NULL);
			gtk_container_add(GTK_CONTAINER(tmp), pic);
			gtk_signal_connect(GTK_OBJECT(tmp), "clicked",
					   GTK_SIGNAL_FUNC(html_sel_dlg_cb),
					   hwip);
			gtk_box_pack_end(GTK_BOX(hb), tmp, TRUE, TRUE, 0);
			tmp = hb;
		}
		if (hwip->opt & IS_CHKBOX)
			gtk_table_attach_defaults(table, tmp, 0, 1, row, row+1);
		else
			gtk_table_attach_defaults(table, tmp, 1, 2, row, row+1);

		/* add label with text */
		tmp = gtk_label_new(gettext(html_label_list[hwip->hll_id]));
		gtk_misc_set_alignment(GTK_MISC(tmp), 0, 0.5);
		if (hwip->opt & IS_CHKBOX)
			gtk_container_add(GTK_CONTAINER(hwip->wgt), tmp);
		else
			gtk_table_attach_defaults(table, tmp, 0, 1, row, row+1);
#if 0
  gtk_table_attach(table, tmp,
		    0, 1,
		    row, row+1,
		    GTK_FILL,
		    GTK_EXPAND | GTK_FILL,
		    0, 0);
#endif

		GNPDBG_HTML(("html_complex_dlg: added item '%s'\n",
			     html_label_list[hwip->hll_id]));
	} /* for each item on current page */
} /* html_complex_dlg_items_add */


/*
 * For the (file|color) selection dialogs, we need all the information
 * contained in "hwip" as well as the widget for the selection dialog itself.
 */
typedef struct html_sel_dlg_data {
	html_wgt_info_t *hwip;
	GtkWidget *sel;
} html_sel_dlg_data_t;


/*
 * PRIVATE: html_sel_dlg_cb
 *
 * Callback for the file|color selection button.
 */
static void
html_sel_dlg_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_wgt_info_t *hwip = (html_wgt_info_t *)cbdata;
	html_sel_dlg_data_t *hsddp;

	hsddp = g_new(html_sel_dlg_data_t, 1);
	hsddp->hwip = hwip;

	if (hwip->opt & IS_FILESEL) {
		hsddp->sel = gtk_efilesel_new(
					gettext(html_label_list[hwip->hll_id]));
		gtk_efilesel_set_mode(GTK_EFILESEL(hsddp->sel),
				      GTK_SELECTION_SINGLE);
		gtk_efilesel_hide_fileop_buttons(GTK_EFILESEL(hsddp->sel));

		(void)gtk_signal_connect(GTK_OBJECT(
			GTK_EFILESEL(hsddp->sel)->ok_button), "clicked",
			GTK_SIGNAL_FUNC(html_sel_dlg_ok_cb), hsddp);
		(void)gtk_signal_connect(GTK_OBJECT(
			GTK_EFILESEL(hsddp->sel)->cancel_button), "clicked",
			GTK_SIGNAL_FUNC(html_sel_dlg_cancel_cb), hsddp);
	} else {
		hsddp->sel = gtk_color_selection_dialog_new(
					gettext(html_label_list[hwip->hll_id]));
		gtk_color_selection_set_update_policy(GTK_COLOR_SELECTION(
			GTK_COLOR_SELECTION_DIALOG(hsddp->sel)->colorsel),
			GTK_UPDATE_CONTINUOUS);

		(void)gtk_signal_connect(GTK_OBJECT(
			GTK_COLOR_SELECTION_DIALOG(hsddp->sel)->ok_button),
			"clicked", GTK_SIGNAL_FUNC(html_sel_dlg_ok_cb), hsddp);
		(void)gtk_signal_connect(GTK_OBJECT(
			GTK_COLOR_SELECTION_DIALOG(hsddp->sel)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(html_sel_dlg_cancel_cb),
			hsddp);
	}

	(void)gtk_signal_connect(GTK_OBJECT(hsddp->sel), "destroy",
				 GTK_SIGNAL_FUNC(html_sel_dlg_destroy_cb),
				 hsddp);
	gtk_grab_add(hsddp->sel);
	gtk_widget_show(hsddp->sel);
} /* html_sel_dlg_cb */


/*
 * PRIVATE: html_sel_dlg_ok_cb
 *
 * Callback for the "Ok" button on the file|color selection dialog
 */
static void
html_sel_dlg_ok_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_sel_dlg_data_t *hsddp = (html_sel_dlg_data_t *)cbdata;

	if (hsddp->hwip->opt & IS_FILESEL) {
		char *txt = gtk_efilesel_get_filename(GTK_EFILESEL(hsddp->sel));
		if (txt && txt != "")
			gtk_entry_set_text(GTK_ENTRY(hsddp->hwip->wgt), txt);
	} else {
		gdouble color[4];
		char buf[MAX_RGB_STR];

		gtk_color_selection_get_color(GTK_COLOR_SELECTION(
			GTK_COLOR_SELECTION_DIALOG(hsddp->sel)->colorsel),
			color);
		g_snprintf(buf, MAX_RGB_STR, "#%x%x%x",
					(unsigned)(color[0] * COL_VAL_HTML),
					(unsigned)(color[2] * COL_VAL_HTML),
					(unsigned)(color[1] * COL_VAL_HTML));
		gtk_entry_set_text(GTK_ENTRY(hsddp->hwip->wgt), buf);
	}

	gtk_widget_destroy(hsddp->sel);
} /* html_sel_dlg_ok_cb */


/*
 * PRIVATE: html_sel_dlg_cancel_cb
 *
 * Callback for the "Cancel" button on the file|color selection dialog.
 */
static void
html_sel_dlg_cancel_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_sel_dlg_data_t *hsddp = (html_sel_dlg_data_t *)cbdata;

	gtk_widget_destroy(hsddp->sel);
} /* html_sel_dlg_cancel_cb */


/*
 * PRIVATE: html_sel_dlg_destroy_cb
 *
 * Callback for the "destroy" event on the file|color selection dialog.
 */
static void
html_sel_dlg_destroy_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_sel_dlg_data_t *hsddp = (html_sel_dlg_data_t *)cbdata;;

	hsddp->hwip = NULL;
	hsddp->sel = NULL;
	g_free(hsddp);
} /* html_sel_dlg_destroy_cb */


/*
 * PRIVATE: html_tag_pulldown_create
 *
 * given itemlist, creates a combo box (a pulldown list).  called only from
 * html_complex_dlg_items_add().
 */
static GtkWidget *
html_tag_pulldown_create(char **itemlist)
{
	GtkWidget *combo;
	GList *list;
	char **item;

	list = NULL;
	for (item = itemlist; *item; item++)
		list = g_list_append(list, (gpointer)(*item));

	combo = gtk_combo_new();
	gtk_combo_set_popdown_strings(GTK_COMBO(combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(combo)->entry),
			   (char *)g_list_nth_data(list, 0));
	return combo;
} /* html_tag_pulldown_create */


/*
 * PRIVATE: html_complex_exec
 *
 * for the Link and Image tag dialogs, this is the callback for the "Ok"
 * button.
 */
static void
html_complex_exec(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;
	char *buf[2];
	int len;

	/* count up the total length needed */
	len = html_complex_calc_len(hedp);
	g_assert(len > 0);

	/* build the buffer */
	buf[0] = html_complex_buf_build(len, hedp);
	buf[1] = NULL;

	/* and away it goes! */
	html_tag_insert(hedp->d, buf, hedp->htp);

	g_free(buf[0]);
	gtk_widget_destroy(hedp->edp.toplev);
} /* html_complex_exec */


/*
 * PRIVATE: html_complex_buf_build
 *
 * builds and returns a text buffer which will be inserted into the current
 * document.  called only from html_complex_exec().
 */
static char *
html_complex_buf_build(int len, html_entry_dlg_t *hedp)
{
	char *buf, *sel;

	buf = g_new(char, len);

	/* stuff in prefix */
	if (hedp->htp->opt & HT_CRBEFORE) {
		sprintf(buf, "\n<%s ", hedp->htp->prefix);
	} else
		sprintf(buf, "<%s ", hedp->htp->prefix);

	/* add each tag attribute */
	html_complex_attr_add(hedp->dlg_pages, buf);

	buf[strlen(buf) - 1] = '\0';	/* remove trailing space */
	if (IS_HTML_TAG_LOWER())
		g_strdown(buf);
	strcat(buf, ">");

	/* if there was text selected, grab it and stick it between the prefix
	 * and the suffix */
	if (hedp->start != hedp->stop) {
		sel = gtk_editable_get_chars(GTK_EDITABLE(hedp->d->data),
					     hedp->start, hedp->stop);
		strcat(buf, sel);
		g_free(sel);
	}

	/* append the suffix, if any */
	if (!(hedp->htp->opt & HT_NOSUFFIX)) {
		sprintf(buf + strlen(buf), "</%s>", hedp->htp->prefix);
		if (IS_HTML_TAG_LOWER()) {
			int plen = strlen(hedp->htp->prefix);
			g_strdown(buf + strlen(buf) - plen - 1);
		}
	}
	if (hedp->htp->opt & HT_CRAFTER)
		strcat(buf, "\n");

	/* delete the selected text, because it's going to be reinserted */
	html_complex_exec_pre(hedp);

	GNPDBG_HTML(("html_complex_buf_build: inserting buf (len = %d," 
		     "strlen = %ld) = '%s'\n", len, (long)strlen(buf), buf));
	return buf;
} /* html_complex_buf_build */


/*
 * PRIVATE: html_complex_attr_add
 *
 * appends each optional attribute for the link/img tag into the text buffer
 * that is to be inserted into the current document.  called only from
 * html_complex_buf_build().
 */
static void
html_complex_attr_add(html_page_t *dlg_pages, char *buf)
{
	html_page_t *hpp;
	html_wgt_info_t *hwip;
	char *tmp = NULL;

	for (hpp = dlg_pages; hpp->hll_id != HLL_NULL; hpp++) {
		for (hwip = hpp->hwip; hwip->hll_id != HLL_NULL; hwip++) {
			if (hwip->opt & IS_VSPACE)
				continue;

			if (hwip->opt & IS_ENTRY || hwip->opt & IS_SPINBUT) {
				tmp = gtk_entry_get_text(GTK_ENTRY(hwip->wgt));
			} else if (hwip->opt & IS_PDOWN) {
				tmp = gtk_entry_get_text(GTK_ENTRY(
						GTK_COMBO(hwip->wgt)->entry));
			} else if (hwip->opt & IS_CHKBOX) {
				if (GTK_TOGGLE_BUTTON(GTK_BUTTON(
							hwip->wgt))->active) {
					tmp = html_tag_list[hwip->htl_id];
				} else
					tmp = NULL;
			}

			if (tmp && tmp[0] != '\0') {
				if ((hwip->opt & IS_PDOWN) && 
					(g_strcasecmp(tmp, "auto") == 0 ||
					 g_strcasecmp(tmp, "none") == 0))
					break;

				strcat(buf, html_tag_list[hwip->htl_id]);
				if (hwip->opt & IS_ASSIGN)
					strcat(buf, "=");
				if (!(hwip->opt & IS_NOQUOTE))
					strcat(buf, "\"");

				if (hwip->opt & IS_ASSIGN)
					strcat(buf, tmp);
				if (!(hwip->opt & IS_NOQUOTE))
					strcat(buf, "\"");
				strcat(buf, " ");
			} /* widget exists */
		} /* for each item on page */
	} /* for each page */
} /* html_complex_attr_add */


/*
 * PRIVATE: html_complex_calc_len
 *
 * calculates an upper bound on the length of the text buffer to be inserted
 * for the link/img tag.  this is done by scanning through the tables, and if
 * an entry widget is non-empty, adding its length and the length of the
 * supporting characters.
 */
static int
html_complex_calc_len(html_entry_dlg_t *hedp)
{
	html_page_t *hpp;
	html_wgt_info_t *hwip;
	char *tmp = NULL;
	int len = 0;

	/* if there is text selected, add its length to the total length */
	if (hedp->start != hedp->stop)
		len += (hedp->stop - hedp->start);

	for (hpp = hedp->dlg_pages; hpp->hll_id != HLL_NULL; hpp++) {
		for (hwip = hpp->hwip; hwip->hll_id != HLL_NULL; hwip++) {
			if (hwip->opt & IS_VSPACE)
				continue;

			if (hwip->opt & IS_ENTRY || hwip->opt & IS_SPINBUT) {
				tmp = gtk_entry_get_text(GTK_ENTRY(hwip->wgt));
			} else if (hwip->opt & IS_PDOWN) {
				tmp = gtk_entry_get_text(
					GTK_ENTRY(GTK_COMBO(hwip->wgt)->entry));
			} else if (hwip->opt & IS_CHKBOX) {
				if (GTK_TOGGLE_BUTTON(GTK_BUTTON(
							hwip->wgt))->active) {
					tmp = html_tag_list[hwip->htl_id];
				} else
					tmp = NULL;
			}

			if (tmp && tmp[0] != '\0') {
				/* if "auto" or "none", don't add this attr */
				if ((hwip->opt & IS_PDOWN) && 
					(g_strcasecmp(tmp, "auto") == 0 ||
					 g_strcasecmp(tmp, "none") == 0))
					break;

				g_assert(html_tag_list[hwip->htl_id]);
				g_assert(hwip->wgt);

				len += (strlen(tmp) +
					strlen(html_tag_list[hwip->htl_id]) +
					((hwip->opt & IS_ASSIGN) ? 1 : 0) +
					(!(hwip->opt & IS_NOQUOTE) ? 2 : 0) +
					1);	/* +1 for space character */
			} /* widget exists */
		} /* for each item on page */
	} /* for each page */

	/* add on prefix and suffix lengths */
	len += strlen(hedp->htp->prefix) + 4 +	/* "<%s " and "</%s>" */
		(!(hedp->htp->opt & HT_NOSUFFIX) ?
			(strlen(hedp->htp->prefix) + 3) : 1) +
		((hedp->htp->opt & HT_CRBEFORE) ? 1 : 0) +
		((hedp->htp->opt & HT_CRAFTER) ? 1 : 0) +
		3;	/* 1 for \0, +2 extra just in case i can't count */

	return len;
} /* html_complex_calc_len */


/*
 * PRIVATE: html_tag_insert
 *
 * insert buf into the text widget and position cursor wrt to inserted tag
 */
static void
html_tag_insert(doc_t *d, char **buf, htmltags_t *htp)
{
	int pt, start, len;

	len = strlen(buf[0]);
	gtk_text_freeze(GTK_TEXT(d->data));

	/* sync up insertion point with current cursor position */
	gtk_text_set_point(GTK_TEXT(d->data),
			   gtk_editable_get_position(GTK_EDITABLE(d->data)));
	start = gtk_editable_get_position(GTK_EDITABLE(d->data));
	gtk_text_insert(GTK_TEXT(d->data), NULL, NULL, NULL, buf[0], len);

	/* update cursor position */
	pt = gtk_text_get_point(GTK_TEXT(d->data));
	if (htp->opt & HT_BETWEEN) {
		gtk_editable_set_position(GTK_EDITABLE(d->data),
			pt - strlen(htp->prefix) -
			((htp->opt & HT_CRAFTER) ? 4 : 3));
	} else if (htp->opt & HT_AFTER) {
		gtk_editable_set_position(GTK_EDITABLE(d->data), pt);
	}
#if 0
	else {	/* for HT_BEFORE, don't do anything */
	case Before:
		g_assert(htp->opt & HT_BEFORE);
		gtk_editable_set_position(GTK_EDITABLE(d->data), pt - len - 1);
	}
#endif

	undo_list_add(d, buf[1] ? g_strdup(buf[1]) : g_strdup(buf[0]),
		      start, start + len, buf[1] ? UndoReplace : UndoInsert);
	gtk_text_thaw(GTK_TEXT(d->data));
	gtk_signal_emit_by_name(GTK_OBJECT(d->data), "changed", d);
} /* html_tag_insert */


/*
 * PRIVATE: html_entry_dlg
 *
 * creates a simply entry dialog box to get input for a tag.  called only from
 * html_tag_complex_cb().
 */
static void
html_entry_dlg(doc_t *d, htmltags_t *htp, char *title, char *msg,
	       GtkSignalFunc execute_cb)
{
	html_entry_dlg_t *hedp;
	char *entrytxt = NULL;

	if (misc_show_and_raise(d->w->html_dlg))
		return;

	hedp = g_new0(html_entry_dlg_t, 1);
	hedp->d = d;
	hedp->htp = htp;

	/* check if there is text already selected */
	if (GTK_EDITABLE(d->data)->has_selection) {
		hedp->start = GTK_EDITABLE(d->data)->selection_start_pos;
		hedp->stop = GTK_EDITABLE(d->data)->selection_end_pos;
		if (hedp->stop < hedp->start)
			SWAP_INT(hedp->stop, hedp->start);
		entrytxt = gtk_editable_get_chars(GTK_EDITABLE(d->data),
						  hedp->start, hedp->stop);
	}

	get_dialog_entry(title, NULL, msg, 128, entrytxt,
			 GTK_SIGNAL_FUNC(execute_cb),       (gpointer)hedp,
			 GTK_SIGNAL_FUNC(html_complex_cancel),  (gpointer)hedp,
			 GTK_SIGNAL_FUNC(html_dlg_destroy), (gpointer)hedp,
			 &hedp->edp.toplev, &hedp->edp.entry);

	d->w->html_dlg = hedp->edp.toplev;

	if (entrytxt)
		g_free(entrytxt);
} /* html_entry_dlg */


/*
 * PRIVATE: html_dlg_destroy
 *
 * common callback to destroy a dialog
 */
static void
html_dlg_destroy(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;

	if (hedp->edp.toplev)
		hedp->d->w->html_dlg = NULL;

	g_free(hedp);
	hedp = NULL;
} /* html_dlg_destroy */


/*
 * PRIVATE: html_complex_cancel
 *
 * common callback for the "Cancel" button.  simply calls gtk_widget_destroy()
 * on the top level widget.
 */
static void
html_complex_cancel(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;

	gtk_widget_destroy(hedp->edp.toplev);
} /* html_complex_cancel */


/*
 * PRIVATE: html_complex_exec_pre
 *
 * Before executing a complex html tag insertion, this routine checks if there
 * is any text selected.  If so, then it will delete the selected text.
 */
static void
html_complex_exec_pre(html_entry_dlg_t *hedp)
{
	if (hedp->start != hedp->stop) {
		gtk_text_set_point(GTK_TEXT(hedp->d->data), hedp->start);
		gtk_editable_set_position(GTK_EDITABLE(hedp->d->data),
					  hedp->start);
		gtk_text_freeze(GTK_TEXT(hedp->d->data));
		gtk_text_forward_delete(GTK_TEXT(hedp->d->data),
					hedp->stop - hedp->start);
		gtk_text_thaw(GTK_TEXT(hedp->d->data));
	}
} /* html_complex_exec_pre */


/*
 * PRIVATE: html_target_exec
 *
 * inserts the text buffer for the target HTML tag.
 */
static void
html_target_exec(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;
	char *buf[2], *entrytext;


	html_complex_exec_pre(hedp);
	entrytext = gtk_entry_get_text(GTK_ENTRY(hedp->edp.entry));
	if (strlen(entrytext) > 0) {
#ifdef GTK_HAVE_FEATURES_1_1_0
		if (IS_HTML_TAG_LOWER()) {
			buf[0] = g_strdup_printf("<a name=\"%s\"></a>",
						 entrytext);
		} else {
			buf[0] = g_strdup_printf("<A NAME=\"%s\"></A>",
						 entrytext);
		}
#else
		buf[0] = g_new(char, strlen(entrytext) + 20);
		if (IS_HTML_TAG_LOWER()) {
			g_snprintf(buf[0], strlen(entrytext) + 20,
				   "<a name=\"%s\"></a>", entrytext);
		} else {
			g_snprintf(buf[0], strlen(entrytext) + 20,
				   "<A NAME=\"%s\"></A>", entrytext);
		}
#endif
		buf[1] = NULL;
		html_tag_insert(hedp->d, buf, hedp->htp);
		g_free(buf[0]);
	}

	gtk_widget_destroy(hedp->edp.toplev);
} /* html_target_exec */


/*
 * PRIVATE: html_comment_exec
 *
 * inserts the text buffer for the HTML comment.
 */
static void
html_comment_exec(GtkWidget *wgt, gpointer cbdata)
{
	html_entry_dlg_t *hedp = (html_entry_dlg_t *)cbdata;
	char *buf[2], *cmt;

	html_complex_exec_pre(hedp);
	cmt = gtk_entry_get_text(GTK_ENTRY(hedp->edp.entry));
	if (strlen(cmt) > 0) {
#ifdef GTK_HAVE_FEATURES_1_1_0
		buf[0] = g_strdup_printf("<!-- %s -->", cmt);
#else
		buf[0] = g_new(char, strlen(cmt) + 12);
		g_snprintf(buf[0], strlen(cmt) + 12, "<!-- %s -->", cmt);
#endif
		buf[1] = NULL;
		html_tag_insert(hedp->d, buf, hedp->htp);
		g_free(buf[0]);
	}

	gtk_widget_destroy(hedp->edp.toplev);
} /* html_comment_exec */


static void
html_tag_chooser_close(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	int i;

	for (i = 0; i < (sizeof(tagdesc) / sizeof(tagdesc[0])); i++) {
		g_free(tagdesc[i].html_id_list);
		tagdesc[i].html_id_list = NULL;
		tagdesc[i].listcnt = 0;
	}

	if (w->tagchooser->window) {
		gdk_window_get_size(w->tagchooser->window,
				    &prefs.tagchooser_w, &prefs.tagchooser_h);
	}
	gtk_widget_destroy(w->tagchooser);
	w->tagchooser = NULL;
	w->tc_row = -1;
} /* html_tag_chooser_close */


/*
 * Callback for the selection of a row in the tag chooser's currently visibile
 * clist.  The only thing that needs to be done here is to save the row number.
 */
static void
html_tag_chooser_selrow(GtkWidget *wgt, int row, int col,
			GdkEventButton *bevent, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;

	w->tc_row = row;

#if 0
	GtkNotebookPage *curpage;
	GtkScrolledWindow *swin;
	GtkCList *clist;
	/*
	 * look at this nonsense we have to do to get to the CList widget.  I
	 * suppose we could save each CList widget in the win_t structure, but
	 * then that's a bit wasteful.
	 */
	curpage = GTK_NOTEBOOK_PAGE(GTK_NOTEBOOK(w->tc_nb)->cur_page);
	swin = GTK_SCROLLED_WINDOW(curpage->child);
	clist = GTK_CLIST(swin->container.child);
#endif
} /* html_tag_chooser_selrow */


/*
 * Callback for the "Insert Tag" button on the tag chooser.  The row number of
 * the currently selected row in the clist is saved during the "select_row"
 * signal, back in html_tag_chooser_selrow().  At this point, we just grab the
 * current notebook's page to get the HTML group id.
 */
static void
html_tag_chooser_insert(GtkWidget *wgt, gpointer cbdata)
{
	win_t *w = (win_t *)cbdata;
	int grpid;

	if (w->tc_row < 0)
		return;

	/*
	 * The notebook's current pagenum also corresponds with the taggrp_id's
	 * because the pages were added in the order of the taggrp_id's.
	 */
	grpid = gtk_notebook_get_current_page(GTK_NOTEBOOK(w->tc_nb));

	/*
	 * Now that we have the taggrp_id, we can look inside tagdesc[] for a
	 * list of html_id's which belong in the group.  The 'row' parameter
	 * represents which row in the clist that was selected, and since that
	 * was added in the order found in htmltags[], it also corresponds
	 * correctly with the html_id stored in tagdesc[] !
	 */

	GNPDBG_HTML(("html_tag_chooser_insert: selected tag id = %d, "
		     "prefix+data = '%s %s'\n",
		     tagdesc[grpid].html_id_list[w->tc_row],
		     htmltags[tagdesc[grpid].html_id_list[w->tc_row]].prefix,
		     htmltags[tagdesc[grpid].html_id_list[w->tc_row]].data));

	html_menu_cb_common(
		htmltags[tagdesc[grpid].html_id_list[w->tc_row]].action,
		cbdata);
} /* html_tag_chooser_insert */

#endif	/* USE_HTMLTAGS */

/* the end */
