/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * gnome-print-compress.c : Compression rutines for gnome-print's native drivers
 *
 * Author:
 *   Chema Celorio <chema@celorio.com>
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <gnome.h>
#include <libgnomeprint/gnome-print-compress.h>


int gnome_print_compress_blank (guchar *in, gint in_size);
int gnome_print_compress_rlc   (guchar *in, guchar *out, gint in_size);
int gnome_print_compress_tiff  (guchar *in, guchar *out, gint in_size);
int gnome_print_compress_drow  (guchar *in, guchar *out, gint in_size, guchar *seed);

/* TODO : add a parameter to the compression methods called "size to beat" so that
   we stop compressing a row that we know is too large to be chosen. We only need
   to return in_size+1 if the length of the current buffer beeing compressed
   is greater than the "size to beat". */ 
 

/**
 * gnome_print_compress_blank :
 * @in: the row to analiyze
 * @out: not used.
 * @in_size: the size of the row
 * 
 * Scan the row and determine if it is a row with no dots.
 * 
 * Return Value: TRUE if it contains only 0's FALSE otherwise
 **/
int
gnome_print_compress_blank (guchar *in, gint in_size)
{
	gint p2; /* Point to where we are analizing chars from the in buffer */
	
	gint retval;
	
	p2=0;

	retval = TRUE; /* This row is inocent until proven guilty */
	while ( p2 < in_size - 1)
	{
		if (in[p2]!=0)
		{
			retval = FALSE;
			break;
		}
		p2++;
	}

	return retval;
}

/**
 * gnome_print_compress_rlc:
 * @in: the buffer to compress
 * @out: the compressed buffer
 * @in_size: size of the in buffer
 * 
 * Implements RunLengthCoding compression
 * 
 * Return Value: size of the compressed buffer
 **/
#define _GNOME_PRINT_MAX_RUN_LENGTH_SIZE 256 /* Usefull for testing purposes */

int
gnome_print_compress_rlc (guchar *in, guchar *out, gint in_size)
{
	gint p1; /* Point to where we are adding into the out buffer */
	gint p2; /* Points to where we are taking chars from the in buffer */
	
	gint run_length;
	
	p1=0;
	p2=1;
	out[1]=in[0];    /* We load the first byte */
	run_length = 0;

	while ( p2 < in_size)
	{

		if (in[p2]==in[p2-1])
		{
			if (run_length == _GNOME_PRINT_MAX_RUN_LENGTH_SIZE - 1)
			{
				/* max number of repetitions have been reached
				   so write the repetition count and start a new one */
				out[p1]=run_length; 
				run_length=0;
				p1+=2;
				out[p1+1]=in[p2];

			}
			else
			{
				run_length++;
			}
		}
		else
		{
			/* The run has been broken, so write the
			   run info and start another one */
			out[p1]=run_length;
			run_length=0;
			p1+=2;
			out[p1+1]=in[p2];
		}
		p2++;
	}

	/* We always need to write the last run */ 
	out[p1]=run_length;
	p1+=2;
	
	return p1;
}



/**
 * gnome_print_compres_tiff : Implements TIFF compression method for
 *                              PCL printers. Reference : [1]Page15-17.
 *             ( it should be usefull for other Printer Languajes too)
 * @in: the array containing the buffer of bytes tha will be converted
 * @out: the output array in TIFF "Packbits"
 * @in_size: the size of the input buffer
 * 
 * returns : the size of the TIFF compressed string
 **/
int
gnome_print_compress_tiff (guchar *in, guchar *out, gint in_size)
{
	
	int p1;         /* Points to the control byte in the out buffer */
	int p2;         /* Points to the next available char in the out buffer */
	int p3;         /* Points to where we are taking chars from the in buffer */ 
	int run_length; /* The number of identical bytes that will be repeated */
	int literal_length; 

	run_length = 0;
	literal_length = 1;

	out[0]=0;
	out[1]=in[0];
	p1=0;
	p3=1;
	p2=2;
	
	while ( p3 < in_size)
	{
		if (in[p3-1]==in[p3])
		{
			if (run_length>0)
			{
				run_length++;
				if ( run_length==128+1)
				{
					out[p1]=-1*(run_length-2);
					p1=p1+2;
					p2=p2+2;
					out[p2-1]=in[p3];
					run_length=0;
					literal_length=1;
				}
			}
			else
			{
				out[p1]=literal_length-2;
				if (literal_length>1)
				{
					p1=p2-1;
					out[p2]=in[p3];
					p2++;
				}
				else
				{
					out[p2]=in[p3];
				}
				literal_length=0;
				run_length=2;
			}
		}
		else
		{
			literal_length++;
			if (literal_length == 1)
			{
				out[p1]=-1*(run_length-1);
				p1=p1+2;
				out[p1]=77;
				out[p2+1]=in[p3];
				p2=p2+2;
				run_length = 0;
			}
			else
			{
				out[p2++]=in[p3];
				if ( literal_length==128+1)
				{
					out[p1]=literal_length-2;
					p1=p1+literal_length;
					p2=p2+1;
					out[p2-1]=in[p3];
					literal_length=1;
				}
			}
		}
		p3++;
	}
	
	if (run_length>0)
		out[p1]=-1*(run_length-1);
	else
		out[p1]=literal_length-1;

	return p2;
}

/**
 * gnome_print_compress_deltarow:
 * @in: the row that we need to code/compress
 * @seed: the row that will be used as a seed
 * @out: the buffer returned with the compressed data
 * @in_size: the size of the incoming buffer
 * 
 *
 * Return Value: the size of the compressed (out) buffer.
 **/
int
gnome_print_compress_drow (guchar *in, guchar *out, gint in_size, guchar *seed)
{
	gint p1; /* Pointer to the control byte in the out buffer */
	gint p2; /* Pointer to where we are adding into the out buffer */
	gint p3; /* Points to where we are taking chars from the in buffer */

	gint offset;
	gint changed_bytes;

	gint i,j;

	p1=0;
	p2=1;
	p3=0;
	changed_bytes = 0;
	offset=0;
	out[p1]=0;
	
	while ( p3 < in_size)
	{
		if (in[p3]==seed[p3])
		{
			if (changed_bytes>0)
			{
				/* If we are here it means that we have been generating
				   literal data and that we just came across a non changing
				   byte, so we need to add the control byte.*/

				/* If out[p1], which is the control byte has a value of 31
				   this means that we used more than one control byte because
				   the offset was bigger than 31 and posible bigger than 286,541,
				   796... etc. "31 + 255 * n" for n=0,1,2...,inf */
				if (out[p1]==31)
				{
					/* We need more than 1 ctrl byte */
					/* we add the upper 3 bits to the control byte*/
					out[p1]=(changed_bytes-1)*32+31;
					/* calculate how many control bytes of 255 we used*/ 
					j=(int)((offset-31)/255);
					for (i=1;i<=j;i++)
					{
						out[p1+i]=255;
						offset-=255;
					}
					/* Now add the last control btye */
					out[p1+i]=offset-31;
				}
				else
				{
					/* We only need 1 control byte */ 
					out[p1]=(changed_bytes-1)*32+offset;
					if (offset==31)
					{
						out[p1]=(changed_bytes-1)*32+offset;
 						out[p1+1]=0; 
					}
				}
				p1=p2;
				p2=p1+1;
				out[p1]=0;
				offset=0;
			}
			offset++;
			/* This is true for 31,31+255,31+255*2,31+255*inf ....*/ 
			if (((offset-31)%255)==0)
			{
				if (offset==31)
					out[p1]=31;
				p2++;
			}
			changed_bytes=0;
		}
		else
		{
			changed_bytes++;
			if (changed_bytes==9)
			{
				if (out[p1]==31)
				{
					/* Shakira really rules, so does the -- operator
					 that's why we use it in the next line */ 
					changed_bytes--;
					/* Since we used more that 1 ctrl byte we know
					   that the 1st ctrl byte has the 5 lower bits
					   set ( that's where the +31 comes from. The
					   *32 is equal to << 5 */ 
					out[p1]=(changed_bytes-1)*32+31;
					j=(int)((offset-31)/255);
					for (i=1;i<=j;i++)
					{
						out[p1+i]=225;
						offset-=255;
					}
					out[p1+i]=offset-31;
				}				
				else
				{
					out[p1]=(changed_bytes-2)*32+offset;
				}
				/* I have no idea what the fuck is this for. je je
				   I just like commenting code when I don't wanna think
				   This song is great... Chema */ 
				p2++;
				p1=p2-1;
				out[p1]=123;
				changed_bytes=1;
				offset=0;
			}
			out[p2++]=in[p3];
		}
		p3++;
	}

	if (changed_bytes==1)
	{
		p2=p1+1; /* we add 1 since we have p2-- latter */ 
		p1=0; /* To clean the debug display */
	}
	else
	{
		/* We need to update the control byte */
		if (out[p1]==31)
		{
			out[p1]=(changed_bytes-1)*32+31;
			j=(int)((offset-31)/255);
			for (i=1;i<=j;i++)
			{
				out[p1+i]=255;
				offset-=255;
			}
			out[p1+i]=offset-31;
		}
		else
		{
			out[p1]=(changed_bytes-1)*32+offset;
		}
	}

	p2--;

	return p2;
}

