/* 
 *  Copyright (C) 1999-2000 Bernd Gehrmann
 *                          bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */

#include <kstddirs.h>

#include "misc.h"
#include "ldapsite.h"
#include "peoplesearch.h"


struct LdapInterpretRec
{
    QString name;
    enum { Name, Email, Phone, Unknown } prop;
};

class PeopleSite : public LdapSite
{
public:
    PeopleSite() : LdapSite()
        { interprets.setAutoDelete(true); }
    ~PeopleSite() {}

    QList<LdapInterpretRec> interprets;
};


class PeopleSiteFactory : public LdapSiteFactory
{
public:
    virtual void interpret(LdapSite *site, const QDomElement &el);
    virtual LdapSite *newLdapSite();
};


LdapSite *PeopleSiteFactory::newLdapSite()
{
    return new PeopleSite();
}


void PeopleSiteFactory::interpret(LdapSite *site, const QDomElement &el)
{
    LdapSiteFactory::interpret(site, el);
    
    PeopleSite *psite = static_cast<PeopleSite*>(site);
    LdapInterpretRec *rec = new LdapInterpretRec();
    rec->name = el.attribute("name");
    rec->prop = LdapInterpretRec::Unknown;
    QString prop = el.attribute("prop");
    if (prop == "name")
        rec->prop = LdapInterpretRec::Name;
    else if (prop == "email")
        rec->prop = LdapInterpretRec::Email;
    else if (prop == "telephonenumber")
        rec->prop = LdapInterpretRec::Phone;
    psite->interprets.append(rec);
}


PeopleSiteManager::PeopleSiteManager()
{
    PeopleSiteFactory factory;
    d = new LdapSiteManager("people-sites", factory);
}


PeopleSiteManager::~PeopleSiteManager()
{
    delete d;
}


PeopleSite *PeopleSiteManager::find(const QString &name)
{
    return static_cast<PeopleSite*>(d->dict.find(name));
}


QStringList PeopleSiteManager::siteList()
{
    QStringList list;
    QDictIterator<LdapSite> it(d->dict);
    for (;  it.current(); ++it)
        list.append(it.currentKey());
    return list;
}


PeopleSearch::PeopleSearch(PeopleSite *site, const QString &searchtext)
    : LdapSearchJob(site, searchtext)
{}


void PeopleSearch::parseItem(const LdapItem &item)
{
    PeopleSite *psite = static_cast<PeopleSite*>(m_site);
    
    qDebug("Entry:");
    QString name;
    QString email;
    QString phone;
    LdapItemConstIterator it;
    for (it = item.begin(); it != item.end(); ++it)
        {
            qDebug("Attribute: %s, Value: %s", (*it).key.latin1(), (*it).value.latin1());
            QListIterator<LdapInterpretRec> it2(psite->interprets);
            for (; it2.current(); ++it2)
                if ((*it).key == it2.current()->name)
                    switch (it2.current()->prop)
                        {
                        case LdapInterpretRec::Name:  name = (*it).value; break;
                        case LdapInterpretRec::Email: email = (*it).value; break;
                        case LdapInterpretRec::Phone: phone = (*it).value; break;
                        default: ;
                        }
        }
    emit foundItem(psite->name, name, email, phone, item);
    qDebug("/Entry:");
    qDebug(" ");
}

