/*
 *   mylistmanager.cpp - Widget for managing a list
 * 
 *   part of knu: KDE network utilities
 *   (will perhaps be part of kdelibs)
 *
 *   Copyright (C) 1997-1999  Bertrand Leconte <b.leconte@mail.dotcom.fd>
 *   Copyright (c) 1998 Cristian Tibirna <ctibirna@gch.ulaval.ca>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
/*
 * $Id: mylistmanager.cpp,v 1.2 1999/06/25 21:29:25 leconte Exp $
 */

#define i18n(s) s

#include "mylistmanager.h"
#include "mylistmanager.moc"

#define WIDGET_INTERVAL 10

/*
 * Constructor with title (and GroupBox)
 */
myListManager::myListManager(QString title, 
			     QWidget *parent, 
			     const char *name, Features f)
  : QWidget(parent, name) 
{
  debug("myListManager::myListManager(\"%s\")", (const char *)title);

  QBoxLayout *topLay = new QVBoxLayout(this, 0, 0, "myListManager::topLay");
  myGroupBox = new QGroupBox(title, this, name);
  
  realConstructor(myGroupBox, true, f);
  
  topLay->addWidget(myGroupBox);
  topLay->activate();
}

/*
 * Constructor
 */
myListManager::myListManager(QWidget *parent, 
			     const char *name, Features f)
  : QWidget(parent, name) 
{
  debug("myListManager::myListManager()");

  realConstructor(this, false, f);
}

/*
 * Real constructor
 */
void
myListManager::realConstructor(QWidget *topWidget, 
			       bool inGroupBox, Features f)
{
  QGridLayout *lay;
  debug("myListManager::realConstructor(QWidget=\"%s\")",
	topWidget->name());

  features = f;
  
  if (inGroupBox) {
    lay = new QGridLayout(topWidget, 5, 2, WIDGET_INTERVAL, -1,
			  "myListManager::lay");
    lay->addRowSpacing(0, WIDGET_INTERVAL/2);
  } else {
    lay = new QGridLayout(topWidget, 5, 2, 0, WIDGET_INTERVAL,
			  "myListManager::lay");
  }

  lay->setRowStretch(0, 0);
  lay->setRowStretch(1, 0);
  lay->setRowStretch(2, 10);
  lay->setRowStretch(3, 0);
  lay->setRowStretch(4, 0);

  lay->setColStretch(0, 10);
  lay->setColStretch(1, 0);

  myLineEdit = new QLineEdit(topWidget, "lineedit");
  myLineEdit->adjustSize();
  myLineEdit->setMinimumSize(myLineEdit->size());
  lay->addWidget(myLineEdit, 1, 0);

  connect(myLineEdit, SIGNAL(textChanged(const QString &)),
	  this,         SLOT(textChanged(const QString &)));
  connect(myLineEdit, SIGNAL(returnPressed()), this, SLOT(addIt()));

  myAddButton = new QPushButton(topWidget, "add_button");
  if (features & ShortLabels) {
    myAddButton->setText(i18n("+"));
  } else {
    myAddButton->setText(i18n("Add"));
  }
  myAddButton->adjustSize();
  myAddButton->setFixedSize(myAddButton->sizeHint());
  myAddButton->setEnabled(false);
  myAddButton->setFocusPolicy(QWidget::TabFocus);
  lay->addWidget(myAddButton, 1, 1);
  connect(myAddButton, SIGNAL(clicked()), this, SLOT(addIt()));

  myRemoveButton = new QPushButton(topWidget, "remove_button");
  if (features & ShortLabels) {
    myRemoveButton->setText(i18n("-"));
  } else {
    myRemoveButton->setText(i18n("Remove"));
  }

  myRemoveButton->adjustSize();
  myRemoveButton->setFixedSize(myRemoveButton->sizeHint());
  myRemoveButton->setEnabled(false);
  myRemoveButton->setFocusPolicy(QWidget::TabFocus);
  lay->addWidget(myRemoveButton, 2, 1, AlignTop);
  connect(myRemoveButton, SIGNAL(clicked()), this, SLOT(removeIt()));

  if (features & UpDownButtons) {
    myUpButton = new QPushButton(i18n("Up"), topWidget, "up_button");
    myUpButton->adjustSize();
    myUpButton->setFixedSize(myUpButton->sizeHint());
    myUpButton->setEnabled(false);
    myUpButton->setFocusPolicy(QWidget::TabFocus);
    lay->addWidget(myUpButton, 3, 1, AlignTop);
    connect(myUpButton, SIGNAL(clicked()), this, SLOT(moveUp()));
    
    myDownButton = new QPushButton(i18n("Down"), topWidget, "down_button");
    myDownButton->adjustSize();
    myDownButton->setFixedSize(myDownButton->sizeHint());
    myDownButton->setEnabled(false);
    myDownButton->setFocusPolicy(QWidget::TabFocus);
    lay->addWidget(myDownButton, 4, 1, AlignTop);
    connect(myDownButton, SIGNAL(clicked()), this, SLOT(moveDown()));
  } else {
    myUpButton = 0;
    myDownButton = 0;
  }

  myListBox = new QListBox(topWidget);
  myListBox->adjustSize();
  myListBox->setMinimumSize(myListBox->size());
  myListBox->setFocusPolicy(QWidget::TabFocus);
  lay->addMultiCellWidget(myListBox, 2, 4, 0, 0);
  //lay->addWidget(myListBox, 2, 0);

  myListBox->setMultiSelection(FALSE);
  myListBox->clearSelection();

  connect(myListBox, SIGNAL(highlighted(int)), 
	  this, SLOT(itemHilited()));

  lay->activate();

  setFocusPolicy(QWidget::StrongFocus);
  setFocusProxy(myLineEdit);

  myStringList = 0;
}

/*
 * Destructor
 */
myListManager::~myListManager() 
{
  debug("myListManager::~myListManager(%s)", name());
}


/*
 * setEnabled()
 */
void 
myListManager::setEnabled(bool a) {
  //debug("myListManager::setEnabled(%s)", (a ? "true": "false"));

  if (myGroupBox) {
    myGroupBox->setEnabled(a);
  }

  myLineEdit->setEnabled(a);
  myListBox->setEnabled(a);

  /* If we have to enable, we make a little more cleaning */
  if (a) {
     myLineEdit->setText("");
     myListBox->clearSelection();
  }

  manageButtons();
}

/*
 * setStringList
 */
void 
myListManager::setStringList(const QStringList &newList) 
{
  //  debug("myListManager::setStringList(const QStringList &) [%s]", name());

  myListBox->clear();
  myLineEdit->setText("");

  for (QStringList::ConstIterator it = newList.begin(); 
       it != newList.end(); ++it) {
    myListBox->insertItem((*it).stripWhiteSpace());
  }
  myStringList = 0;
}  

/*
 * setStringList
 */
void 
myListManager::setStringList(QStringList *newList) 
{
  debug("myListManager::setStringList(QStringList *) [%s]", name());

  myListBox->clear();
  myLineEdit->setText("");

  for (QStringList::ConstIterator it = newList->begin(); 
       it != newList->end(); ++it) {
    myListBox->insertItem((*it).stripWhiteSpace());
  }
  myStringList = newList;
}  

/*
 * stringList
 */
QStringList
myListManager::stringList() const
{
  QStringList sl;
  uint i;

  //debug("myListManager::stringList()");
  if (myStringList) {
    sl = *myStringList;
  } else {
    // Take the value in the listbox
    for (i=0; i < myListBox->count(); i++) {
      sl.append(myListBox->text(i));
    }
  }
  return sl;
}  

/*
 * addIt() - SLOT
 */
void 
myListManager::addIt() 
{
  int c = myListBox->currentItem();
  QString s = myLineEdit->text();

  if (! s.isEmpty()) {
    myListBox->insertItem(s, c);
    //    myListBox->clearSelection();
    myLineEdit->setText("");
  }
  if (myStringList) {
    QStringList::Iterator it;

    //debug("adding \"%s\" at position %d in myStringList [%s]", (const char *)s, c, name());
    if (c>=0) {
      it = myStringList->at(c);
      myStringList->insert(it, s);
    } else {
      myStringList->append(s);
    }
  }
  manageButtons();
}

/*
 * removeIt() - SLOT
 */
void 
myListManager::removeIt() 
{
  int c = myListBox->currentItem();

  if (c >= 0) {
    myListBox->removeItem(c);
    myListBox->clearSelection();
    myListBox->setCurrentItem(c);
    if (myStringList) {
      //debug("removing position %d in myStringList [%s]", c, name());
      QStringList::Iterator it = myStringList->at(c);
      myStringList->remove(it);
    }    
  }
  manageButtons();
}

/*
 * itemHilited() - SLOT
 */
void 
myListManager::itemHilited() 
{
  manageButtons();
}

/*
 * textChanged() - SLOT
 */
void
myListManager::textChanged(const QString &s)
{
  if (s.isEmpty()) {
    myAddButton->setEnabled(false);
  } else {
    myAddButton->setEnabled(true);
  }
}


/*
 * moveUp() - SLOT
 */
void
myListManager::moveUp()
{
  QString string;
  int c = myListBox->currentItem();

  if (c >= 1) {
    string = myListBox->text(c);
    myListBox->setUpdatesEnabled(false);
    myListBox->removeItem(c);
    myListBox->insertItem(string, c-1);
    myListBox->setCurrentItem(c-1); 
   myListBox->setUpdatesEnabled(true);
#warning "gestion de myStringList dans moveUp"
  }
}

/*
 * moveDown() - SLOT
 */
void
myListManager::moveDown()
{
  QString string;
  int c = myListBox->currentItem();
  int max = myListBox->count();

  if (c < max-1) {
    string = myListBox->text(c);
    myListBox->setUpdatesEnabled(false);
    myListBox->removeItem(c);
    myListBox->insertItem(string, c+1);
    myListBox->setCurrentItem(c+1);
    myListBox->setUpdatesEnabled(true);
#warning "gestion de myStringList dans moveDown"
  }
}

/*
 * manageButtons()
 * Set enable for buttons if we have to
 */
void 
myListManager::manageButtons() 
{
  int c;
  
  c = myListBox->currentItem();

  // Remove button
  if (c >= 0) {
    myRemoveButton->setEnabled(true);
  } else {
    myRemoveButton->setEnabled(false);
  }    

  // Add button
  if ((myLineEdit->text()).isEmpty()) {
    myAddButton->setEnabled(false);
  } else {
    myAddButton->setEnabled(true);
  }

  if (myUpButton) {
    // Up button
    if (c >= 1) {
      myUpButton->setEnabled(true);
    } else {
      myUpButton->setEnabled(false);
    }
  }
   
  if (myDownButton) {
    // Down button
    if (c < (int)myListBox->count()-1) {
      myDownButton->setEnabled(true);
    } else {
      myDownButton->setEnabled(false);
    }
  }
}
