/* 
 *  Copyright (C) 1999 Bernd Gehrmann
 *                     bernd@physik.hu-berlin.de
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the "Artistic License" which comes with this Kit. 
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * Artistic License for more details.
 *
 * You should have received a copy of the Artistic License with 
 * this Kit, in the file named "Artistic".  If not, I'll be glad
 * to provide one.
 */

#include <qcombobox.h>
#include <qlineedit.h>
#include <qcheckbox.h>
#include <qlabel.h>
#include <qlayout.h>

#include <kapp.h>
#include <kdatepik.h>
#include <klocale.h>

#include "criterion.h"


CriterionName::CriterionName(QWidget *parent) {
  combo = new QComboBox(parent, "namecombo");
  combo->insertItem(i18n("is"));
  combo->insertItem(i18n("is not"));
  //    combo->insertItem(i18n("contains"));
  //    combo->insertItem(i18n("does not contain"));
  //    combo->insertItem(i18n("matches expression"));
  //    combo->insertItem(i18n("does not match expression"));
  combo->setFixedSize(combo->sizeHint());

  line = new QLineEdit(parent, "nameline");
  line->adjustSize();
  line->setMinimumSize(line->size());
}

    
CriterionName::~CriterionName() {
  delete combo;
  delete line;
}


int CriterionName::type() {
  return Criterion::Name;
}


QString CriterionName::findString() {
  QString pattern;
  if (combo->currentText() == i18n("is not")) {
    pattern = "-not -name ";
  } else {
    pattern = "-name ";
  }
  pattern += "'";
  pattern += line->text();
  pattern += "'";
  return pattern;
}


void CriterionName::addToLayout(QBoxLayout *lay) {
  lay->addWidget(combo, 0);
  combo->show();
  lay->addWidget(line, 1);
  line->show();
}


CriterionTime::CriterionTime(QWidget *parent) {
  combo = new QComboBox(parent, "timecombo");
  combo->insertItem(i18n("today"));
  combo->insertItem(i18n("during the last hour"));
  combo->insertItem(i18n("during the last day"));
  combo->insertItem(i18n("during the last week"));
  combo->insertItem(i18n("during the last month"));
  combo->insertItem(i18n("during the last year"));
  combo->insertItem(i18n("since"));
  combo->insertItem(i18n("before"));
  connect(combo, SIGNAL(activated(int)), SIGNAL(wantLayout()));
  combo->setFixedSize(combo->sizeHint());

  picker = new KDatePicker(parent);
  connect( picker, SIGNAL(dateSelected(QDate)),
	   SLOT(datePicked(QDate)) );
  picker->setMinimumSize(picker->sizeHint());
}

    
CriterionTime::~CriterionTime() {
  delete combo;
  delete picker;
}


void CriterionTime::datePicked(QDate d) {
  date = d;
}


int CriterionTime::type() {
  return Criterion::Time;
}


QString CriterionTime::findString() {
  QString ci = combo->currentText();
  QString pattern;
  if (ci == i18n("today"))
    pattern = "-daystart -mtime 0";
  else if (ci == i18n("during the last hour"))
    pattern = "-mmin -60";
  else if (ci == i18n("during the last day"))
    pattern = "-mtime -1";
  else if (ci == i18n("during the last week"))
    pattern = "-mtime -7";
  else if (ci == i18n("during the last month"))
    pattern = "-mtime -31";
  else if (ci == i18n("during the last year"))
    pattern = "-mtime -365";
  else if (ci == i18n("since"))
    {
      pattern.setNum(date.daysTo(QDate::currentDate()));
      pattern.prepend("-mtime -");
    }
  else if (ci == i18n("before"))
    {
      pattern.setNum(date.daysTo(QDate::currentDate()));
      pattern.prepend(" -mtime +");
    }
  return pattern;
}


void CriterionTime::addToLayout(QBoxLayout *lay) {
  lay->addWidget(combo, 0);
  combo->show();

  if ( (combo->currentText() == i18n("since"))
       || (combo->currentText() == i18n("before")) )
    {
      lay->addWidget(picker, 1);
      picker->show();
    }
  else
    picker->hide();
}


CriterionSize::CriterionSize(QWidget *parent) {
  combo = new QComboBox(parent, "sizecombo");
  combo->insertItem(i18n("is (in kB)"));
  combo->insertItem(i18n("is smaller than (in kB)"));
  combo->insertItem(i18n("is greater than (in kB)"));
  combo->setFixedSize(combo->sizeHint());

  line = new QLineEdit(parent, "sizeline");
  line->adjustSize();
  line->setMinimumSize(line->size());
}

    
CriterionSize::~CriterionSize() {
  delete combo;
  delete line;
}


int CriterionSize::type() {
  return Criterion::Size;
}


QString CriterionSize::findString() {
  QString ci = combo->currentText();
  QString pattern;
  pattern = "-size '";
  if (ci == i18n("is smaller than (in kB)"))
    pattern += "+";
  else if (ci == i18n("is greater than (in kB)"))
    pattern += "-";
  pattern += line->text();
  pattern += "k'";
  return pattern;
}


void CriterionSize::addToLayout(QBoxLayout *lay) {
  lay->addWidget(combo, 0);
  combo->show();
  lay->addWidget(line, 1);
  line->show();
}


CriterionType::CriterionType(QWidget *parent) {
  combo = new QComboBox(parent, "typecombo");
  combo->insertItem(i18n("is a regular file"));
  combo->insertItem(i18n("is a symbolic link"));
  combo->insertItem(i18n("is an executable"));
  combo->insertItem(i18n("is an suid executable"));
  //    combo->insertItem(i18n("is a hard link"));
  combo->setFixedSize(combo->sizeHint());
}

    
CriterionType::~CriterionType() {
  delete combo;
  // delete line;
}


int CriterionType::type() {
  return Criterion::Type;
}


QString CriterionType::findString() {
  QString ci = combo->currentText();
  QString pattern;
  if (ci == i18n("is a regular file"))
    pattern ="-type f";
  else if (ci == i18n("is a symbolic link"))
    pattern ="-type l";
  else if (ci == i18n("is an executable"))
    pattern = "-perm +111 -type f";
  else if (ci == i18n("is an suid executable"))
    pattern = "-perm +6000 -type f";
  return pattern;
}


void CriterionType::addToLayout(QBoxLayout *lay) {
  lay->addWidget(combo, 0);
  combo->show();
}


CriterionOwner::~CriterionOwner() {
  delete combo;
  delete line;
}


CriterionOwner::CriterionOwner(QWidget *parent) {
  combo = new QComboBox(parent);
  combo->insertItem(i18n("is"));
  combo->insertItem(i18n("is not"));
  combo->insertItem(i18n("is invalid"));
  connect(combo, SIGNAL(activated(int)), SIGNAL(wantLayout()));

  line = new QLineEdit(parent);
  line->adjustSize();
  line->setMinimumSize(line->size());
}

    
int CriterionOwner::type() {
  return Criterion::Owner;
}


QString CriterionOwner::findString() {
  QString ci = combo->currentText();
  QString pattern;
  if (ci == i18n("is"))
    pattern = QString("-user ") + line->text();
  if (ci == i18n("is not"))
    pattern = QString("-not -user ") + line->text();
  else if (ci == i18n("is invalid"))
    pattern = "-nouser";
  return pattern;
}


void CriterionOwner::addToLayout(QBoxLayout *lay) {
  lay->addWidget(combo, 0);
  combo->show();
  if (combo->currentText() != i18n("is invalid"))
    {
      lay->addWidget(line, 1);
      line->show();
    }
  else
    line->hide();
}


CriterionGroup::CriterionGroup(QWidget *parent) {
  combo = new QComboBox(parent, "groupcombo");
  combo->insertItem(i18n("is"));
  combo->insertItem(i18n("is not"));
  combo->insertItem(i18n("has GID"));
  combo->insertItem(i18n("does not have GID"));
  combo->insertItem(i18n("is invalid"));
  connect(combo, SIGNAL(activated(int)), SIGNAL(wantLayout()));

  line = new QLineEdit(parent, "groupline");
  line->adjustSize();
  line->setMinimumSize(line->size());
}

    
CriterionGroup::~CriterionGroup() {
  delete combo;
  delete line;
}


int CriterionGroup::type() {
  return Criterion::Group;
}


QString CriterionGroup::findString() {
  QString ci = combo->currentText();
  QString pattern;
  if (ci == i18n("is"))
    pattern = QString("-group ") + line->text();
  if (ci == i18n("is not"))
    pattern = QString("-not -group ") + line->text();
  else if (ci == i18n("is invalid"))
    pattern = "-nogroup";
  return pattern;
}


void CriterionGroup::addToLayout(QBoxLayout *lay) {
  lay->addWidget(combo, 0);
  combo->show();

  if (combo->currentText() != i18n("is invalid"))
    {
      lay->addWidget(line, 1);
      line->show();
    }
  else
    line->hide();
 
}


Criterion::Criterion(QWidget *parent, const char *name )
  : QWidget(parent, name) {
  typecombo = new QComboBox(this);
  typecombo->insertItem(i18n("Name"));
  typecombo->insertItem(i18n("Last Modified"));
  typecombo->insertItem(i18n("Size"));
  typecombo->insertItem(i18n("Type"));
  typecombo->insertItem(i18n("Owner"));
  typecombo->insertItem(i18n("Owning group"));
  typecombo->setFixedSize(typecombo->sizeHint());

  connect( typecombo, SIGNAL( activated( int ) ),
	   SLOT( typeActivated( int ) ) );

  lay = 0;
  spec = new CriterionName(this);
  connect(spec, SIGNAL(wantLayout()), SLOT(layout()));
  layout();
}


QString Criterion::findString() {
  return spec->findString();
}


void Criterion::layout() {
  delete lay;
  lay = new QHBoxLayout(this, 4);

  lay->addWidget(typecombo, 0);
  spec->addToLayout(lay);
  lay->addStretch();
		   
  resize(sizeHint());
}


void Criterion::typeActivated( int type ) {
  if (spec->type() != type) {
    delete spec;
	
    switch ( type ) {
    case Criterion::Name:
      spec = new CriterionName(this);
      break;

    case Criterion::Time:
      spec = new CriterionTime(this);
      break;

    case Criterion::Size:
      spec = new CriterionSize(this);
      break;

    case Criterion::Type:
      spec = new CriterionType(this);
      break;

    case Criterion::Owner:
      spec = new CriterionOwner(this);
      break;

    case Criterion::Group:
      spec = new CriterionGroup(this);
      break;
    }

    connect(spec, SIGNAL(wantLayout()), SLOT(layout()));
    layout();
  }
}
