/*
 *  This file is part of the KDE libraries
 *  Copyright (C) 1999-2000 Harri Porten (porten@kde.org)
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#include <stdio.h>

#include "kjs.h"
#include "object.h"
#include "operations.h"
#include "internal.h"
#include "types.h"
#include "lexer.h"
#include "nodes.h"

#include "object_object.h"
#include "function_object.h"
#include "array_object.h"
#include "bool_object.h"
#include "string_object.h"
#include "number_object.h"
#include "math_object.h"
#include "date_object.h"
#include "regexp_object.h"
#include "error_object.h"

extern int kjsyyparse();

using namespace KJS;

class GlobalFunc : public InternalFunctionImp {
public:
  GlobalFunc(int i) : id(i) { }
  Completion execute(const List &c);
  enum { Eval, ParseInt, ParseFloat };
private:
  int id;
};

Global::Global()
  : Object(new GlobalImp())
{
}

Global::~Global()
{
}

Global Global::current()
{
  return KJScriptImp::current()->glob;
}

KJSO Global::objectPrototype() const
{
  assert(rep);
  return ((GlobalImp*)rep)->objProto;
}

KJSO Global::functionPrototype() const
{
  assert(rep);
  return ((GlobalImp*)rep)->funcProto;
}

GlobalImp::GlobalImp()
  : ObjectImp(ObjectClass)
{
  // constructor properties. prototypes as Global's member variables first.
  objProto = Object(new ObjectPrototype());
  funcProto = Object(new FunctionPrototype());
  arrayProto = Object(new ArrayPrototype(objProto));
  stringProto = Object(new StringPrototype(objProto));
  booleanProto = Object(new BooleanPrototype(objProto));
  numberProto = Object(new NumberPrototype(objProto));
  dateProto = Object(new DatePrototype(objProto));
  regexpProto = Object(new RegExpPrototype(objProto));
  errorProto = Object(new ErrorPrototype(objProto));

  Object objectObj(new ObjectObject(objProto));
  Object arrayObj(new ArrayObject(funcProto));
  Object boolObj(new BooleanObject(funcProto));
  Object stringObj(new StringObject(funcProto));
  Object numObj(new NumberObject(funcProto));
  Object dateObj(new DateObject(dateProto));
  Object regObj(new RegExpObject(regexpProto));
  Object errObj(new ErrorObject(errorProto));

  put("Object", objectObj, DontEnum);
  put("Array", arrayObj, DontEnum);
  put("Boolean", boolObj, DontEnum);
  put("String", stringObj, DontEnum);
  put("Number", numObj, DontEnum);
  put("Date", dateObj, DontEnum);
  put("RegExp", regObj, DontEnum);
  put("Error", errObj, DontEnum);

  objProto.setConstructor(objectObj);
  arrayProto.setConstructor(arrayObj);
  booleanProto.setConstructor(boolObj);
  stringProto.setConstructor(stringObj);
  numberProto.setConstructor(numObj);
  dateProto.setConstructor(dateObj);
  regexpProto.setConstructor(regObj);
  errorProto.setConstructor(errObj);

  // value properties
  put("NaN", Number(NaN), DontEnum | DontDelete);
  put("Infinity", Number(Inf), DontEnum | DontDelete);
  put("undefined", Undefined(), DontEnum | DontDelete);
  put("eval", Function(new GlobalFunc(GlobalFunc::Eval)));
  put("parseInt", Function(new GlobalFunc(GlobalFunc::ParseInt)));
  put("parseFloat", Function(new GlobalFunc(GlobalFunc::ParseFloat)));

  // other properties
  put("Math", Object(new Math()), DontEnum);
}

GlobalImp::~GlobalImp() { }

Completion GlobalFunc::execute(const List &args)
{
  KJSO res;

  if (id == Eval) { // eval()
    KJSO x = args[0];
    if (x.type() != StringType)
      res = x;
    else {
      String s = x.toString();
      Lexer::curr()->setCode(s.value().data(), s.value().size());
      if (kjsyyparse()) {
	KJS::Node::deleteAllNodes();
	return Completion(Normal, Error::create(SyntaxError));
      }

      res = KJS::Node::progNode()->evaluate();
      /* TODO: analyse completion value */

//       if (error())
// 	error()->deref();

      //      if (KJS::Node::progNode())
      //	KJS::Node::progNode()->deleteStatements();

      res = Undefined();
    }
  } else if (id == ParseInt) {
    String str = args[0].toString();
    int radix = args[1].toInt32();
    if (radix == 0)
      radix = 10;
    else if (radix < 2 || radix > 36) {
      res = Number(NaN);
      return Completion(Normal, res);
    }
    /* TODO: use radix */
    int i = 0;
    sscanf(str.value().ascii(), "%d", &i);
    res = Number(i);
  } else if (id == ParseFloat) {
    String str = args[0].toString();
    double d = 0.0;
    sscanf(str.value().ascii(), "%lf", &d);
    res = Number(d);
  }

  return Completion(Normal, res);
}

