/*
 *  LinKT - the Linux Kde pr-Terminal
 *  Copyright (C) 1997-1999 Jochen Sarrazin, DG6VJ. All rights reserved.
 *  
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "didadit.h"
#include "didadit.moc"

#include <kwmmapp.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>

#include "toolbox.h"
#include "channel.h"
#include "global.h"
#include "main.h"
#include <errno.h>
#include <zlib.h>

#include "crc.h"


extern TopLevel *toplevel;



#define LASTBLOCK_INFO 1
#define LASTBLOCK_START 2
#define LASTBLOCK_REQ 3
#define LASTBLOCK_FIN 4
#define LASTBLOCK_ECHOREQUEST 5


// Fehlercodes, die in LinKT verwendet werden.
char error_wrongreq[] = "101 REQ block with data out of range\n";
char error_wrongmd5[] = "102 MD5 validation error\r";
char error_wronginfoline[] = "200 Illegal line in INFO block\r";
char error_nomd5hash[] = "202 INFO block without MD5 info\r";
char error_nosize[] = "203 Empty file or missing size info in INFO block\r";
char error_noblocksize[] = "204 INFO block without BLOCKSIZE info\r";
char error_nofilename[] = "205 INFO block without FILENAME info\r";
char error_cannotopen[] = "300 Could not create file\r";

//   DIDADIT::DIDADIT(char *_filename, bool senden)
//
// Konstruktor der Klasse DIDADIT.
//
// Parameter: _filename: Den Dateinamen mitsamt Pfad
//            senden:    true, wenn die Datei ausgesendet werden soll, sonst
//                       false
DIDADIT::DIDADIT(QWidget *_chan, char *_filename, bool senden) : QObject()
{
   struct stat stat_data;
   int i,len;
   char *tmp;

   chan = _chan;
   blocksize = 2048;
   fd_data = -1;
   fd_r = -1;
   rxlistroot = NULL;
   reqlistroot = NULL;
   win = NULL;
   transInfo = NULL;
   rxbytes = 0;
   txbytes = 0;
   reqCount = 0;
   reqLength = 0;
   lastBlockSent = -1;
   gzipfilesize = -1;


   if (senden)
   {
      longname = (char *) strdup(_filename);
      // Shortname (den eigentlichen Dateinamen) 'rausfinden
      tmp = (char *) strdup(_filename);
      if ((i = lPOS('/',_filename)) != -1)
      {
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }
      shortname = (char *) strdup(tmp);

      free(tmp);

      stat(longname, &stat_data);

      size = stat_data.st_size;
      modtime = stat_data.st_mtime;
      if (!calc_MD5_wholefile(longname, md5))
      {
         status = -1;
         return;
      }

//		compressFile();
   }
   else
   {
      shortname = NULL;
      longname = NULL;
      size = -1;
      modtime = -1;
      offset = 0;
   }

   rxData = NULL;
   rxDataLen = 0;
   starttime = time(NULL);

   if (senden)
   {
      status = 1;
      firstblock = false;
      ((Channel *)chan)->sendString( "#DIDADIT#\r" );
      sendInfoBlock();
      waitForOkAbort = true;
      toplevel->chanListTable->setMode( chan, MODE_DIDADITTX );
      ((Channel *)chan)->updateStatusBar();


      // DIDADIT-Sende-Fenster erstellen
      if (win == NULL && transInfo == NULL)
      {
         // Entsprechendes Fenster oeffnen
         extraTransferWin = config->extraTransferWin;
         if (extraTransferWin)
         {
            win = new TransferWin( chan, TRANSART_DIDADITTX, shortname, size );
            win->show();
            win->setReceivedBytes( 0 );
            if (toplevel->currentChannel != NULL)
               if (toplevel->currentChannel->isActiveWindow())
               {
                  KWM::activate(toplevel->currentChannel->winId());
                  toplevel->currentChannel->setFocus();
               }
         }
         else
         {
            transInfo = new TransferInfo( ((Channel *)chan)->pannerHome, TRANSART_DIDADITTX, shortname, size );
            ((Channel *)chan)->withTransInfo( transInfo );
            transInfo->setReceivedBytes( 0 );
         }
      }
   }
   else
   {
      status = 100;
      firstblock = true;
      waitForOkAbort = false;
      toplevel->chanListTable->setMode( chan, MODE_DIDADITRX );
      ((Channel *)chan)->updateStatusBar();
   }

   timer = new QTimer( this );
   connect(timer, SIGNAL(timeout()), this, SLOT(sendBlock()));
}


DIDADIT::~DIDADIT()
{
   if (!allReceived())
      writeRFile();
   else
      deleteRFile();

   if (rxData != NULL) free(rxData);
   if (longname != NULL) free(longname);
   if (shortname != NULL) free(shortname);

   if (fd_data != -1)
   {
     ::close(fd_data);
     fd_data = -1;
   }

   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }

   delete timer;

   toplevel->chanListTable->setMode( chan, MODE_INFO );
   ((Channel *)chan)->updateStatusBar();
}


void EscapeIt (char *dest, int & dlen, char ch)
{
  if (ch == FEND)
  {
     dest[dlen] = FESC;
     dest[dlen+1] = TFEND;
     dlen += 2;
  }
  else if (ch == FESC)
       {
          dest[dlen] = FESC;
          dest[dlen+1] = TFESC;
          dlen += 2;
       }
       else
       {
          dest[dlen] = ch;
          dlen++;
       }
}



void DIDADIT::SendBlock( char *data, int datalen, unsigned short blockid, int when )
{
   char sSend[MAXROHBLOCKSIZE];
   int slen;
   unsigned short CRC=0;
   int i;


   if ((blockid == KBLOCK_ERR) ||
       (blockid == KBLOCK_DATA) ||
       (blockid == KBLOCK_FIN) ||
       (blockid == KBLOCK_REQ) ||
       (blockid == KBLOCK_FINACK))
   {
      // Bei diesen Blocks muss vor den Daten 16 Bytes MD5-Hash davor
      memmove(data+16, data, datalen);
      datalen += 16;
      memcpy(data, md5, 16);
   }

   // CRC berechnen
   crcthp(lo(blockid),&CRC);
   crcthp(hi(blockid),&CRC);
   CRC = calc_crcthp_str(data, datalen, CRC);

   sSend[0] = FEND;
   memcpy(sSend+1, &blockid, sizeof(blockid));
   slen = 3;

   // Die CRCs ans Ende des Frames packen, damit sie mit "escaped" werden
   memcpy(data+datalen, &CRC, sizeof(CRC));
   datalen += sizeof(CRC);

   for (i=0; i<datalen; i++)
   {
      EscapeIt(sSend, slen, data[i]);
/*      if (slen > 750)
      {
         SendBytes( sSend, slen);
         slen = 0;
      }*/
   }

   sSend[slen] = FEND;
   slen++;

   switch (when)
   {
   	case WHEN_NOW:
			SendBytesNow( sSend, slen );
         break;
   	default:
		   SendBytes( sSend, slen);
	}
}


//   void DIDADIT::SendBytes(char *data, int len)
// Eine bestimmte Anzahl Bytes soll ausgesendet werden (nicht anzeigen).
void DIDADIT::SendBytes(char *data, int len)
{
   ((Channel *)chan)->sendString( len, data, false );
}


//   void DIDADIT::SendBytesNow( char *data, int len )
// Eine bestimmte Anzahl Bytes soll JETZT ausgesendet werden (nicht
// anzeigen).
void DIDADIT::SendBytesNow( char *data, int len )
{
   ((Channel *)chan)->sendStringNow( len, data, false );
}


/*
 * void didadit::DecodeBlock(char *data, int len, t_KBlock *block)
 *
 * Den Block in data dekodieren und in die t_KBlock-Struktur
 * packen.
 */
void DIDADIT::DecodeBlock(char *data, int len, t_KBlock *block)
{
   int i;
   unsigned short crc=0;

   block->error = false;


   block->data = (char *) malloc(len);

   block->len = 0;
   i = 1;
   while (i < len)
   {
      if (data[i] == FESC)
      {
         i++;
         switch (data[i])
         {
            case TFEND: block->data[block->len] = FEND; break;
            case TFESC: block->data[block->len] = FESC; break;
            default: block->error = true;
         }
      }
      else block->data[block->len] = data[i];

      i++;
      block->len++;
   }

   // Block-ID
   memcpy(&block->id,block->data,sizeof(block->id));

   // CRC
   memcpy(&block->crc, block->data+block->len-2, sizeof(block->crc));

   // CRC der Daten und der Block-ID errechnen (ohne CRC)
   crc = calc_crcthp_str(block->data, block->len-2, crc);

   // Die Laenge der CRC und die der ID wieder abziehen
   block->len -= 4;
   if (block->len < 0)
   {
   	block->error = true;
   	return;
   }
   memmove(block->data, block->data+2, block->len);


   if (crc != block->crc) block->error = true;

   if (block->error) return;

   // Wenn in diesem Block ein MD5-Hash drin ist, wird er aus dem
   // daten-Feld in das Hash-Feld gepackt
   if ((block->id == KBLOCK_ERR) ||
       (block->id == KBLOCK_DATA) ||
       (block->id == KBLOCK_FIN) ||
       (block->id == KBLOCK_REQ) ||
       (block->id == KBLOCK_FINACK))
   {
      // Ja, MD5-Hash
      memcpy(block->md5, block->data, 16);
      memmove(block->data, block->data+16, block->len-16);
      block->len -= 16;
	}
	block->data[block->len] = '\0';
}


//   void DIDIADIT::sendMessage(char *str)
// Eine Nachricht (Statusmeldung) auf dem Bildschirm anzeigen und, wenn
// moeglich (Gegenstation keine automatische Station) auch aussenden.
void DIDADIT::sendMessage(char *str)
{
   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( str );
   else
      ((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );
}


// void DIDADIT::checkBlock(char *data, int & len)
//
// Diese Funktion wird immer dann aufgerufen, wenn Daten angekommen
// sind.
//
// Rueckgabe: 0, wenn die Uebertragung fertig ist,
//            1, wenn der Block fehlerhaft ist
//            2, wenn der Block OK ist und verarbeitet werden soll
//            3, wenn der Block noch nicht vollstaendig empfangen wurde
//				  4, wenn der Block uebersprungen wurde und die diese Routine
//					  nochmal aufgerufen werden will
int DIDADIT::checkBlock(char *data, int & len)
{
   int i;

   // Erstmal warten, bis dieser Block komplett empfangen wurde
   if (rxData == NULL)
   {
      // Speicher reservieren
      rxData = (char *) malloc(MAXROHBLOCKSIZE*2);
      rxDataLen = 0;
   }

   // Die jetzt gerade empfangenen Daten an rxData anhaengen
   memcpy(rxData+rxDataLen,data,len);
   rxDataLen += len;
   len = 0;

	// Alles vor dem ersten Auftreten von FEND loeschen
	if ((i = nPOS(FEND, rxData, rxDataLen)) == -1)
   {
      // FEND ist garnicht vorhanden. rxData loeschen
      free(rxData);
      rxData = NULL;
      rxDataLen = 0;
      return 3;
   }
   else
   {
      memmove(rxData,rxData+i,rxDataLen-i);
		rxDataLen -= i;
		rxData[rxDataLen] = '\0';
	}

	// Gucken, ob FEND nochmal vorhanden ist. Wenn ja, ist der ganze
	// Block empfangen worden. In diesem Fall wird der Block dekodiert
	// und in rxData geloescht. Ansonsten wars das erstmal.
	if ((i = nPOS(FEND, rxData+1, rxDataLen-1)) == -1)
	{
		return 3;
	}

	if (i == 0)
	{
		// Erstes FEND entfernen und wieder nach oben in der Schleife
		rxDataLen--;
		memcpy(rxData, rxData+1, rxDataLen);
		return 4;
	}

	// Das uebersprunge erste FEND dazu
	i++;

	// Den Block dekodieren
	DecodeBlock(rxData,i,&block);

	memcpy(rxData,rxData+i,rxDataLen-i);
	rxDataLen -= i;

	// Wenn der Block ungueltig ist, wird abgebrochen.
	if (block.error)
	{
printf("block.error\n");
		return 1;
   }

   return 2;
}


//   void DIDADIT::sendInfoBlock()
//
// Sendet ein Info-Frame mit Informationen zu dieser Datei aus
void DIDADIT::sendInfoBlock()
{
   char tmp[1000];
   char hexmd5[33];


   getHexMD5(md5, hexmd5);

   sprintf(tmp,"FILENAME=%s\rSIZE=%li\rMD5=%s\rTIME=%li\rBLOCKSIZE=%i\rVERSION=%s\r",
               shortname, size, hexmd5, modtime, blocksize, DIDADIT_VERSION);

/*   if (gzipfilesize < size)
   	sprintf(tmp+strlen(tmp), "COMPRESS=gzip,%i\r",gzipfilesize);*/

   SendBlock (tmp, strlen(tmp), KBLOCK_INFO );

/*   lastBlockSent = LASTBLOCK_INFO;
   timer->start(*/
}


//   bool DIDADIT::readInfoBlock()
//
// "Packt" den Info-Block "aus" und speichert alle Infos ab.
//
// Rueckgabe: true, wenn alles OK ist, false wenn ein Fehler aufgetreten
//            ist. Bei false wird die DIDADIT-Uebertragung abgebrochen.
bool DIDADIT::readInfoBlock()
{
   char tmp[500], tmp2[500], *data;
   int i,len;
   char id[200];

   md5[0] = '\0';
   blocksize = -1;
	size = -1;
   if (longname != NULL)
   	free(longname);
	longname = NULL;
	if (shortname != NULL)
   	free(shortname);
	shortname = NULL;


   ((Channel *)chan)->sendString( 5, "#OK#\r", true );

   data = (char *) strdup(block.data);

   while (data[0] != '\0')
   {
      if ((i = POS('\r', data)) == -1)
      {
         strcpy(tmp, data);
         data[0] = '\0';
      }
      else
      {
         memcpy(tmp, data, i);
         tmp[i] = '\0';
         len = strlen(data)-i-1;
         memmove(data, data+i+1, len);
         data[len] = '\0';
      }

      // Die ID=WERT - Paare auseinander spliten
      if ((i = POS('=', tmp)) > -1)
      {
         memcpy(id, tmp, i);
         id[i] = '\0';
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }
      else
      {
		  	strcpy(tmp, error_wronginfoline);
			SendBlock( tmp, strlen(tmp), KBLOCK_ERR );
			showError( error_wronginfoline );
         free(data);
      	return false;
      }

      if (!strcmp(id, "FILENAME"))
      {
         if (longname != NULL) free(longname);
         longname = (char *) strdup(tmp);
         if ((i = POS('/', longname)) == -1)
         {
            if (shortname == NULL)
               shortname = (char *) strdup(longname);
            else
               strcpy(shortname, longname);
         }
         else
         {
            len = strlen(longname)-i-1;
            memcpy(tmp2, longname+i+1, len);
            tmp2[len] = '\0';
            shortname = (char *) strdup(tmp2);
         }
      }
      if (!strcmp(id, "SIZE"))
      {
         i = atoi(tmp);
         if (i != 0) size = i;
      }
      if (!strcmp(id, "MD5"))
      {
         tmp[33] = '\0';
         md5HexToBin( tmp, md5);
      }
      if (!strcmp(id, "TIME"))
      {
         i = atoi(tmp);
         if (i != 0) modtime = i;
      }
      if (!strcmp(id, "BLOCKSIZE"))
      {
         i = atoi(tmp);
         if (i != 0) blocksize = i;
      }
/*      if (!strcmp(id, "COMPRESS"))
      {
      	if ((i = POS(',', tmp)) != -1)
         {
            memcpy(tmp2, tmp, i);
            tmp2[i] = '\0';
            if (!strcmp(tmp2, "gzip"))
            {
            	len = strlen(tmp)-i-1;
	            memmove(tmp, tmp+i+1, len);
               tmp[len] = '\0';
               if ((gzipfilesize = atoi(tmp)) == 0) gzipfilesize = -1;
				}
         }
      }*/
   }

   free(data);


   // Fehlermeldungen, wenn irgendwelche Felder fehlen
   if (md5[0] == '\0')
   {
   	strcpy(tmp, error_nomd5hash);
		SendBlock( tmp, strlen(tmp), KBLOCK_ERR );
		showError( error_nomd5hash );
      return false;
   }

   if (blocksize == -1)
   {
   	strcpy(tmp, error_noblocksize);
		SendBlock( tmp, strlen(tmp), KBLOCK_ERR );
		showError( error_noblocksize );
      return false;
   }

   if (shortname == NULL)
   {
   	strcpy(tmp, error_nofilename);
		SendBlock( tmp, strlen(tmp), KBLOCK_ERR );
		showError( error_nofilename );
      return false;
   }

   if (size == -1)
   {
   	strcpy(tmp, error_nosize);
		SendBlock( tmp, strlen(tmp), KBLOCK_ERR );
		showError( error_nosize );
      return false;
   }


   // Gucken, ob ein .r - File existiert. Wenn ja wird da der zu benutzende
   // Offset und die Liste der schon empfangenen Bytes eingelesen, wenn der
   // MD5-Hash stimmt.
   offset = lookForRFile();

   // Wenn die beantragte Blocksize groesser ist als die maximale, die wir
   // unterstuetzen, verlangen wir eine kleinere.
   if (blocksize > MAXBLOCKSIZE)
   	blocksize = MAXBLOCKSIZE;

   return true;
}


//   void DIDADIT::sendStartBlock()
//
// Sendet den zu dieser Datei gehoerenden Startblock aus.
void DIDADIT::sendStartBlock()
{
	char tmp[1000];


	sprintf(tmp,"OFFSET=%li\rBLOCKSIZE=%i\rVERSION=%s\r",
					offset, blocksize, DIDADIT_VERSION);

	SendBlock (tmp, strlen(tmp), KBLOCK_START );

	// DIDADIT-Empfangs-Fenster erstellen
	if (win == NULL && transInfo == NULL)
	{
		// Entsprechendes Fenster oeffnen
		extraTransferWin = config->extraTransferWin;
		if (extraTransferWin)
		{
			win = new TransferWin( chan, TRANSART_DIDADITRX, shortname, size );
			win->show();
			win->setStartBytes( offset );
			win->setReceivedBytes( 0 );
			if (toplevel->currentChannel != NULL)
				if (toplevel->currentChannel->isActiveWindow())
				{
					KWM::activate(toplevel->currentChannel->winId());
					toplevel->currentChannel->setFocus();
				}
		}
		else
		{
			transInfo = new TransferInfo( ((Channel *)chan)->pannerHome, TRANSART_DIDADITRX, shortname, size );
			((Channel *)chan)->withTransInfo( transInfo );
			transInfo->setStartBytes( offset );
			transInfo->setReceivedBytes( 0 );
		}
	}

	sprintf(tmp, "<LinKT>: DIDADIT-Download started at offset %li.\r", offset);
	((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
}


//   void DIDADIT::readStartBlock()
//
// "Packt" den Start-Blocks "aus" und speichert alle Infos ab.
void DIDADIT::readStartBlock()
{
   char tmp[200],*data;
   int i,len,partoffset=-1;
   char id[200];
   unsigned char partmd5[16], calcpartmd5[16];

   partmd5[0] = '\0';
//   blocksize = -1;

   data = (char *) strdup(block.data);

   while (data[0] != '\0')
   {
      if ((i = POS('\r', data)) == -1)
      {
         strcpy(tmp, data);
         data[0] = '\0';
      }
      else
      {
         memcpy(tmp, data, i);
         tmp[i] = '\0';
         len = strlen(data)-i-1;
         memmove(data, data+i+1, len);
         data[len] = '\0';
      }

      // Die ID=WERT - Paare auseinander spliten
      if ((i = POS('=', tmp)) > -1)
      {
         memcpy(id, tmp, i);
         id[i] = '\0';
         len = strlen(tmp)-i-1;
         memmove(tmp, tmp+i+1, len);
         tmp[len] = '\0';
      }

      if (!strcmp(id, "OFFSET"))
      {
         offset = atoi(tmp);
      }
      if (!strcmp(id, "PARTMD5"))
      {
         tmp[33] = '\0';
         md5HexToBin( tmp, partmd5);
      }
      if (!strcmp(id, "BLOCKSIZE"))
      {
         i = atoi(tmp);
         if (i != 0) blocksize = i;
      }
   }

   // Gucken, ob partmd5 korrekt ist, wenn er uebergeben wurde
   if (partmd5[0] != '\0' && offset != 0)
   {
   	partoffset = offset;
      if (!calc_MD5_partfile( longname, calcpartmd5, offset ))
      	partoffset = 0;
      else
	      if (!memcmp(partmd5, calcpartmd5, 16))
	      	partoffset = 0;
   }

   if (partoffset != -1) offset = partoffset;


	sprintf(tmp, "<LinKT>: DIDADIT-Upload started at offset %li.\r", offset);
	((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
	if (extraTransferWin)
		win->setStartBytes( offset );
	else
		transInfo->setStartBytes( offset );

   free(data);
}


//   int DIDADIT::proceed(char *data, int len)
//
// Diese Funktion wird vom Terminalprogramm mit allen Didadit-Daten
// aufgerufen und wickelt eigentlich alles ab.
//
// Rueckgabe: 0, wenn die Uebertragung fertig ist;
//            1, wenn dies der erste Block ist, der fehlerhaft ist.
//            2, wenn alles OK ist und nix gemacht werden muss
//
// Ist der allererste Block fehlerhaft (im Empfangsfall), scheint die
// Uebertragung nicht mit Didadit zu erfolgen -> fallback auf AutoBIN.
int DIDADIT::proceed(char *data, int len)
{
   char tmp[100];
   int errcode, i;


   if (waitForOkAbort)
   {
      if (!strncmp(data, "#ABORT#", 7))
      {
         strcpy(tmp, "<LinKT>: DIDADIT-Transmission aborted by receiver.\r");
         ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
         return 0;
      }
      if (strncmp(data, "#OK#", 4)) return 2;
      waitForOkAbort = false;
      ((Channel *)chan)->flags &= ~CH_LINEMODE;
   }

   for (;;)
   {
	   switch (checkBlock(data,len))
	   {
	      case 0: return 0;
	      case 1: if (firstblock)
	                 return 1;
	              else
	                 continue;
	      case 3: return 2;
         case 4: continue;
	   }
	   firstblock = false;


      // Soll die Uebertragung abgebrochen werden?
      if (block.id == KBLOCK_ABORT)
      {
		   // Ergebnis der Uebertragung ausgeben
         strcpy(tmp, "<LinKT>: Transfer aborted by peer\r");
         ((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

		   if (win != NULL)
		   {
		      delete win;
		      win = NULL;
		   }
		   if (transInfo != NULL)
		   {
		      delete transInfo;
		      transInfo = NULL;
		      ((Channel *)chan)->withoutTransInfo();
		   }

         return 0;
      }

      if (block.id == KBLOCK_ERR)
		{
      	// Ein Fehler ist aufgetreten. Fehlerinfo ausgeben und eventuell
         // abbrechen.
         if ((i = POS(' ', block.data)) != -1)
         {
            memcpy(tmp, block.data, i);
            tmp[i] = '\0';
            block.len -= (i+1);
            memmove( block.data, block.data+i+1, block.len);
            block.data[block.len] = '\0';
            errcode = atoi(tmp);

            if (errcode > 0)
            {
					sprintf(tmp, "<LinKT>: DIDADIT-Error:\r         %s\r", block.data);
					((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );

					if (errcode == 101 || errcode == 102 || errcode == 200 ||
               	 errcode == 202 || errcode == 203 || errcode == 204 ||
               	 errcode == 205 || errcode == 300)
						return 0;
				}
         }
      }

	   switch (status)
	   {
	      case 1: // Erwarte START-Block
	              readStartBlock();
	              if (offset != (unsigned long)size)
	                 ((Channel *)chan)->sendDidadit();
	              SendBlock( tmp, 0, KBLOCK_FIN );
	              status = 2;
	              break;
	      case 2: switch (block.id)
	              {
	                 case KBLOCK_FINACK:
	                      readFinAck();
	                      return 0;
	                 case KBLOCK_REQ:
	                      if (!readReqBlock()) return 0;
	                      break;
	              }
	      // RX //
	      case 100: // Erwarte INFO-Block - alles andere wird ignoriert
	                if (block.id == KBLOCK_INFO)
	                {
	                   if (!readInfoBlock()) return 0;
	                   if (!createDataFiles()) return 0;
	                   sendStartBlock();
	                   status = 101;
	                }
	                break;
	      case 101: // 102 = 101, KBLOCK_FIN wurde schon empfangen
	      case 102: switch (block.id)
	                {
	                   case KBLOCK_DATA:
                      		if (!checkMD5()) return 0;
	                        if (readDataBlock())
	                           return 0;
	                        break;
	                   case KBLOCK_FIN:
	                        if (readFinBlock())
	                           return 0;
	                        status = 102;
	                        break;
	                }
	                break;
	   }
   }
}


//   void DIDADIT::readDataBlock()
//
// Ein Datenblock wird empfangen. Abspeichern und merken.
//
// Rueckgabe: true, wenn die komplette Uebertragung fertig ist,
//            false wenn nicht.
bool DIDADIT::readDataBlock()
{
   unsigned long offset;
   unsigned short blocklen;
   char tmp[50];

   // Die ersten 4 Bytes ist der Offset, danach kommt die Blocklaenge
   // (2 Bytes)
   memcpy(&offset, block.data, 4);
   memcpy(&blocklen, block.data+4, 2);
   memmove(block.data, block.data+6, block.len-6);
   block.len -= 6;

   write(fd_data, block.data, block.len);

   addRXListe( offset, offset+blocklen-1 );

   rxbytes += blocklen;
   if (extraTransferWin)
      win->setReceivedBytes( rxbytes );
   else
      transInfo->setReceivedBytes( rxbytes );


   if (status == 102)
      if (allReceived())
      {
         SendBlock( tmp, 0, KBLOCK_FINACK );

         // Ergebnis der Uebertragung ausgeben
         showEndText( true );

         // .r - File loeschen
         deleteRFile();

         if (win != NULL)
         {
            delete win;
            win = NULL;
         }
         if (transInfo != NULL)
         {
            delete transInfo;
            transInfo = NULL;
            ((Channel *)chan)->withoutTransInfo();
         }
         return true;
      }

   return false;
}


//   bool DIDADIT::createDataFiles()
//
// Erzeugt ein File mit dem Namen <filename>.r im abin-Verzeichnis, in dem
// alle moeglichen Informationen abgespeichert werden sowie das eigentliche
// Datenfile (<filename>).
//
// Rueckgabe: true, wenn ok, false wenn ein Fehler aufgetreten ist.
bool DIDADIT::createDataFiles()
{
	char tmp[500];

/*	if (gzipfilesize > -1)
	{
		sprintf(tmp, "/tmp/lktdwn.%s.z", shortname);
      strcpy(compfilename, tmp);
	}
   else*/
		sprintf(tmp, "%s/%s", config->dirABin, shortname);

   if ((fd_data = open(tmp, O_WRONLY|O_CREAT)) == -1)
   {
   	strcpy(tmp, error_cannotopen);
		SendBlock( tmp, strlen(tmp), KBLOCK_ERR );
		showError( error_cannotopen );
      return false;
   }
   // Zugriffsrechte einstellen
   fchmod(fd_data, S_IRUSR|S_IWUSR);

   return true;
}


bool DIDADIT::getNextPacket( char *txdata, int & len )
{
   char sSend[MAXROHBLOCKSIZE];
   int slen;
   unsigned short CRC=0;
   int i;
   unsigned short blockid=KBLOCK_DATA;
   long datalen;
   unsigned short dlen;
   bool ready=false;
   char data[MAXROHBLOCKSIZE];
   unsigned long loffset;
   s_reqliste *tmp;
   char str[100];


   // Weitere Daten aus dem Sourcefile holen
   if (fd_data == -1)
   {
      if ((fd_data = open(longname, O_RDONLY)) == -1)
      {
printf("cannot open sourcefile: %s\n", strerror(errno));
	   	strcpy(str, error_cannotopen);
			SendBlock( str, strlen(str), KBLOCK_ERR );
			showError( error_cannotopen );
         len = 0;
         return false;
      }
      lseek(fd_data, offset, SEEK_CUR);
   }

   if (reqlistroot != NULL)
   {
      loffset = lseek( fd_data, reqlistroot->offset, SEEK_SET );
      datalen = read( fd_data, data, reqlistroot->len );

      tmp = reqlistroot;
      reqlistroot = reqlistroot->next;
      free(tmp);

//      if (reqlistroot == NULL) ready = true;
   }
   else
   {
      loffset = lseek(fd_data, 0, SEEK_CUR);
      if ((datalen = read(fd_data, data, MAXBLOCKSIZE)) != MAXBLOCKSIZE)
      {
         if (datalen == -1)
         {
            len = 0;
            printf("read()-error: %s\n", strerror(errno));
            return true;
         }
         // Das File ist komplett ausgesendet worden.
         ready = true;
      }
   }

   // Bei diesen Blocks muss vor den Daten 16 Bytes MD5-Hash davor
   dlen = (unsigned short) datalen;
   memmove(data+22, data, datalen);
   datalen += 22;
   // Offset und Blocklaenge dazu
   memcpy(data, md5, 16);
   memcpy(data+16, &loffset, 4);
   memcpy(data+20, &dlen, 2);


   // CRC berechnen
   crcthp(lo(blockid),&CRC);
   crcthp(hi(blockid),&CRC);
   CRC = calc_crcthp_str(data, datalen, CRC);

   sSend[0] = FEND;
   memcpy(sSend+1, &blockid, sizeof(blockid));
   slen = 3;

   // Die CRCs ans Ende des Frames packen, damit sie mit "escaped" werden
   memcpy(data+datalen, &CRC, sizeof(CRC));
   datalen += sizeof(CRC);

   for (i=0; i<datalen; i++)
   {
      EscapeIt(sSend, slen, data[i]);
/*      if (slen > 750)
      {
         SendBytes( sSend, slen);
         slen = 0;
      }*/
   }

   sSend[slen] = FEND;
   slen++;


/////////////////////////// DEBUGGING //////////////////////////////////
/*if (loffset == 0)
   sSend[100] = 'A';*/
/////////////////////////// DEBUGGING //////////////////////////////////


   txbytes += dlen;

   if (extraTransferWin)
      win->setReceivedBytes( txbytes );
   else
      transInfo->setReceivedBytes( txbytes );


   memcpy( txdata, sSend, slen );
   len = slen;

   return ready;
}


//   void DIDADIT::addRXListe( long start, long ende )
//
// Ein neuer Block wurde empfangen - Eintragen in die Liste im Speicher
void DIDADIT::addRXListe( unsigned long start, unsigned long ende )
{
   s_rxliste *tmp, *neu, *next, *last;
   bool ready;


   // Allererster Eintrag: Eintragen und 'raus
   if (rxlistroot == NULL)
   {
      rxlistroot = (s_rxliste *) malloc(sizeof(s_rxliste));
      rxlistroot->next = NULL;
      rxlistroot->start = start;
      rxlistroot->ende = ende;
      return;
   }

   // Gucken, ob das im Augenblick empfangene Frame das naechste in
   // der Folge ist
   tmp = rxlistroot;
   ready = false;
   while (tmp != NULL && !ready)
   {
      if (tmp->ende == start-1)
      {
         tmp->ende = ende;
         ready = true;
      }
      tmp = tmp->next;
   }

   if (!ready)
   {
      // Irgendwo war ein Loch. Gucken, an welcher Stelle dieser Eintrag
      // 'rein muss und einen neuen Eintrag erzeugen
      tmp = rxlistroot;
      ready = false;
      last = NULL;
      while (tmp != NULL && !ready)
      {
         if (tmp->ende < start)
         {
            neu = (s_rxliste *) malloc(sizeof(s_rxliste));
            neu->next = tmp->next;
            neu->start = start;
            neu->ende = ende;
            tmp->next = neu;
            ready = true;
         }
         else
            if (tmp->start > start)
            {
               if (tmp->ende-1 > ende)
               {
                  // Neuer Eintrag
                  neu = (s_rxliste *) malloc(sizeof(s_rxliste));
                  if (last == NULL)
                  {
                     rxlistroot = neu;
                     rxlistroot->next = tmp;
                  }
                  else
                  {
                     last->next = neu;
                     neu->next = tmp;
                  }
                  neu->start = start;
                  neu->ende = ende;
               }
               else
                  tmp->start = start;
               ready = true;
            }
            else tmp = tmp->next;
         last = tmp;
      }
   }


   // Gucken, ob irgendwelche Bereiche doppelt abgedeckt sind, Eintraege
   // also ueberlappen
   tmp = rxlistroot;
   while (tmp != NULL)
   {
      next = tmp->next;
      if (next != NULL)
         while (tmp->ende >= next->start-1)
         {
            // next reicht in den Bereich von tmp 'rein
            tmp->ende = next->ende;
            // next loeschen
            tmp->next = next->next;
            free(next);
            next = tmp->next;
            if (next == NULL) break;
         }
      tmp = tmp->next;
   }
}


//   void DIDADIT::showEndText( bool rx )
//
// Gibt Statusinformationen am Ende der Uebertragungen zurueck.
// Parameter rx ist true, wenn das File empfangen wurde, sonst false.
void DIDADIT::showEndText( bool rx )
{
   time_t timediff;
   char *timeptr;
   char tmp2[500];
   char text[500];

   timediff = time(NULL) - starttime;
   if (timediff == 0) timediff = 1;
   timeptr = (char *)spec_time(timediff);

   if (rx)
      strcpy(text,"<LinKT>: DIDADIT-RX OK. (time: %s, %li baud)\xD" \
      				"         filename: %s, bytes: %i\xD" \
                  "         Requested Blocks: %li (%li bytes)\xD");
   else
      strcpy(text,"<LinKT>: DIDADIT-TX OK. (time: %s, %li baud)\xD" \
      				"         filename: %s, bytes: %i\xD" \
                  "         Retransmitted Blocks: %li (%li bytes)\xD");
   sprintf(tmp2, text,
                 timeptr,(size*8)/timediff, shortname, size, reqCount, reqLength);


   if ((((Channel *)chan)->userinfo->getType() & TYPE_TERMINAL) == TYPE_TERMINAL)
      ((Channel *)chan)->sendString( tmp2 );
   else
      ((Channel *)chan)->outText( tmp2, strlen(tmp2), OUTCOLOR_TXTEXT );

   free(timeptr);
}


//   void DIDADIT::readFinBlock()
//
// Ein FIN-Block gucken, ob alles da ist. Wenn nicht werden die fehlenden
// Teile neu angefordert.
bool DIDADIT::readFinBlock()
{
   char tmp[100];
   s_rxliste *ltmp;


   if (rxlistroot == NULL)
   {
      // Alles neu, es wurde garnichts empfangen
      sendReqBlock( 0, size );
      return false;
   }


   if ((rxlistroot->start == 0) && (rxlistroot->ende == (unsigned long)size-1))
   {
      // Alles wurde empfangen. FINACK-Block senden; Feierabend
      SendBlock( tmp, 0, KBLOCK_FINACK );

      // Ergebnis der Uebertragung ausgeben
      showEndText( true );

      if (win != NULL)
      {
         delete win;
         win = NULL;
      }
      if (transInfo != NULL)
      {
         delete transInfo;
         transInfo = NULL;
         ((Channel *)chan)->withoutTransInfo();
      }

      return true;
   }


   // Teile wurden nicht empfangen
   ltmp = rxlistroot;

   if (rxlistroot != NULL)
      if (rxlistroot->start != 0)
         sendReqBlock( 0, ltmp->start-1 );

   while (ltmp != NULL)
   {
      if (ltmp->next != NULL)
         sendReqBlock( ltmp->ende+1, ltmp->next->start-1 );
      else
         if (ltmp->ende+1 < (unsigned long)size)
            sendReqBlock( ltmp->ende+1, size-ltmp->ende );

      ltmp = ltmp->next;
   }

   return false;
}


void DIDADIT::readFinAck()
{
   // Ergebnis der Uebertragung ausgeben
   showEndText( false );

   if (win != NULL)
   {
      delete win;
      win = NULL;
   }
   if (transInfo != NULL)
   {
      delete transInfo;
      transInfo = NULL;
      ((Channel *)chan)->withoutTransInfo();
   }
}


//   void DIDADIT::sendReqBlock( unsigned long offset, unsigned char blocklen )
//
// Fordert einen Bereich des Files neu an
void DIDADIT::sendReqBlock( unsigned long start, unsigned long stop )
{
   char data[50];
   unsigned long len=stop-start+1;

   memcpy(data, &start, 4);
   memcpy(data+4, &len, 4);

   reqCount++;
   reqLength += len;

   SendBlock( data, 8, KBLOCK_REQ );
}


//   bool DIDADOT::readReqBlock()
//
// Ein REQ-Block wird empfangen. Daten in die entsprechende verkettete
// Liste eintragen und das Aussenden anfordern, wenn gerade keine REQ-
// Bloecke empfangen werden.
bool DIDADIT::readReqBlock()
{
	unsigned long offset, len;
	s_reqliste *tmp;
	char str[100];


	memcpy( &offset, block.data, 4 );
	memcpy( &len, block.data+4, 4 );

   if ((int)offset > size)
   {
   	strcpy(str, error_wrongreq);
		SendBlock( str, strlen(str), KBLOCK_ERR );
		showError( error_wrongreq );
   	return false;
   }

	reqCount++;
	reqLength += len;

	if (reqlistroot == NULL)
	{
		// Erster Eintrag
		reqlistroot = (s_reqliste *)malloc(sizeof(s_reqliste));
		tmp = reqlistroot;
	}
	else
	{
		tmp = reqlistroot;
		while (tmp->next != NULL) tmp = tmp->next;

		tmp->next = (s_reqliste *)malloc(sizeof(s_reqliste));
		tmp = tmp->next;
	}

	tmp->next = NULL;
	tmp->offset = offset;
	tmp->len = len;

	sprintf(str, "<LinKT>: Received REQ-Block. Offset=%li, len=%li.\xD", offset, len);
	((Channel *)chan)->outText( str, strlen(str), OUTCOLOR_TXTEXT );

	((Channel *)chan)->sendDidadit();

   return true;
}


bool DIDADIT::allReceived()
{
   if (rxlistroot == NULL)
      return false;
   return (rxlistroot->start == 0 && rxlistroot->ende == (unsigned long)size-1);
}


//   void DIDADIT::writeRFile()
//
// Die Infos zu diesem Transfer werden unter <filename>.r abgespeichert
// (Fuer den Resume-Mode)
void DIDADIT::writeRFile()
{
   char tmp[500];
   s_rxliste *rxlist;
   unsigned char hash[16];


   sprintf(tmp, "%s/%s.r", config->dirABin, shortname);

   if ((fd_r = open(tmp, O_RDWR|O_CREAT)) == -1)
      return;

   // Zugriffsrechte einstellen
   fchmod(fd_r,S_IRUSR|S_IWUSR);

   sprintf(tmp, "%s/%s", config->dirABin, shortname);
   if (!calc_MD5_wholefile(tmp, hash))
   {
   }

   // Die ersten 16 Bytes ist der MD5-Hash des fertigen Files
   write(fd_r, md5, 16);
   // Danach kommt der Hash des schon empfangenen Files
   write(fd_r, hash, 16);

   // Dann kommt die Liste der empfangenen Bytes
   rxlist = rxlistroot;
   while (rxlist != NULL)
   {
      sprintf(tmp,"%li %li\xA", rxlist->start, rxlist->ende);
      write(fd_r, tmp, strlen(tmp));

      rxlist = rxlist->next;
   }

   ::close(fd_r);
}


//   void DIDADIT::deleteRFile()
//
// Wenn das R-File (<filename>.r) existiert, wird es geloescht
void DIDADIT::deleteRFile()
{
   char tmp[500];

   sprintf(tmp, "%s/%s.r", config->dirABin, shortname);

   if (file_exist(tmp))
      unlink(tmp);
}


//   unsigned long DIDADIT::lookForRFile()
//
// Guckt, ob ein .r-File fuer diesen Dateinamen vorhanden ist. Wenn
// ja wird es eingelesen und der zu verwendende Offset zurueckgegeben.
unsigned long DIDADIT::lookForRFile()
{
   int fd;
   char tmp[500];
   int i;
   unsigned char hash[16];
   FILE *f;
   unsigned long start, ende;
   s_rxliste *rxliste, *last=NULL;


   sprintf(tmp, "%s/%s.r", config->dirABin, shortname);

   if ((fd = open(tmp, O_RDONLY)) == -1)
      return 0;

   sprintf(tmp, "%s/%s", config->dirABin, shortname);
   if (!calc_MD5_wholefile(tmp, hash))
   {
   	return 0;
   }

   // Der Hash des schon gespeicherten Files
   if ((i = read(fd, tmp, 16)) != 16)
      return 0;
   tmp[16] = '\0';
   if (memcmp(tmp, md5, 16) != 0)
      return 0;

   // Der Hash des kompletten Files
   if ((i = read(fd, tmp, 16)) != 16)
      return 0;
   tmp[16] = '\0';
   if (memcmp(tmp, hash, 16) != 0)
      return 0;

   if ((f = fdopen(fd, "r")) == NULL)
      return 0;

   while (fgets(tmp, 499, f) != NULL)
   {
      if ((i = POS('\xA', tmp)) != -1) tmp[i] = '\0';

      if ((i = POS(' ',tmp)) != -1)
      {
         if (sscanf(tmp, "%li %li", &start, &ende) == 2)
         {
            // Neuer Eintrag
            if (rxlistroot == NULL)
            {
               // Erster Eintrag
               rxlistroot = (s_rxliste *)malloc(sizeof(s_rxliste));
               rxliste = rxlistroot;
            }
            else
            {
               rxliste = rxlistroot;
               while (rxliste->next != NULL) rxliste = rxliste->next;

               rxliste->next = (s_rxliste *)malloc(sizeof(s_rxliste));
               rxliste = rxliste->next;
            }

            rxliste->next = NULL;
            rxliste->start = start;
            rxliste->ende = ende;
            last = rxliste;
         }
      }
   }

   if (last == NULL) return 0;

   return (last->ende+1);

   ::close(fd);
}


void DIDADIT::getHexMD5( unsigned char *md5, char *hexmd5 )
{
   sprintf(hexmd5, "%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X%.2X",
                   md5[0],md5[1],md5[2],md5[3],md5[4],md5[5],md5[6],md5[7],
                   md5[8],md5[9],md5[10],md5[11],md5[12],md5[13],md5[14],md5[15]);
}


void DIDADIT::md5HexToBin( char *hex, unsigned char *bin )
{
   int i;
   char tmp[3];

   tmp[3] = '\0';

   for (i=0; i<16; i++)
   {
      tmp[0] = hex[i*2];
      tmp[1] = hex[(i*2)+1];
      bin[i] = get_hex (tmp);
   }
}


//   void DIDADIT::abortIt()
//
// Sendet einen ABORT-Block an die Gegenseite
void DIDADIT::abortIt()
{
	char tmp[2];
   SendBlock( tmp, 0, KBLOCK_ABORT, WHEN_NOW );
}


//   bool DIDADIT::didaditTx()
//
// Gibt true zurueck, wenn wir etwas DIDADIT-maessiges aussenden, ansonsten
// false (wenn etwas empfangen wird).
bool DIDADIT::didaditTx()
{
   if (status >= 100)
      return false;
   return true;
}


void DIDADIT::sendBlock()
{
}


int DIDADIT::getStatus()
{
   return status;
}


bool DIDADIT::checkMD5()
{
	char tmp[100];


	if (!memcmp(md5, block.md5, 16)) return true;

  	strcpy(tmp, error_wrongmd5);
	SendBlock( tmp, strlen(tmp), KBLOCK_ERR );
	showError( error_wrongmd5 );
   return false;
}


void DIDADIT::showError( const char *text )
{
	char tmp[100];
   int i, len;


   if ((i = POS(' ', text)) == -1)
   	return;

	len = strlen(text)-i-1;
	memcpy(tmp, text+i+1, len);
   tmp[len] = '\0';

	sprintf(tmp, "<LinKT>: Sent DIDADIT-Error:\r         %s\r", block.data);
	((Channel *)chan)->outText( tmp, strlen(tmp), OUTCOLOR_TXTEXT );
}


//   void DIDADIT::compressFile()
//
// Packt die Datei zusammen und schreibt das Ergebnis nach /tmp
void DIDADIT::compressFile()
{
	unsigned char src[30000], dest[50000];
   int sfd, dfd, i;
   long readcount;
   unsigned long writecount;


   gzipfilesize = -1;
	if ((sfd = open(longname, O_RDONLY)) == -1) return;

   sprintf(compfilename, "/tmp/lktup.%s.z", shortname);
   if ((dfd = open(compfilename, O_WRONLY|O_CREAT)) == -1)
   {
   	::close(sfd);
      return;
   }
   fchmod(dfd, S_IRUSR|S_IWUSR);

   gzipfilesize = 0;
   while ((readcount = read(sfd, src, 30000)) > 0)
   {
   	writecount = 50000;
      if (( i = compress(dest, &writecount, src, (unsigned long)readcount)) != Z_OK)
      {
      	::close(sfd);
         ::close(dfd);
         unlink(compfilename);
	      gzipfilesize = -1;
         return;
      }
	   gzipfilesize += writecount;

      write (dfd, dest, writecount);
   }

   ::close(sfd);
   ::close(dfd);
}


