/*
Donut Bump Mapping Demo
This demo shows how to use a bump mapping technique using Glide(tm)
Copyright (C) 1999  3Dfx Interactive, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

#include "convexhull.h"


// takes an array of num_verts pointers, each of which points to a structure
// where the first 2 elements of the structure are the X and Y components
// the verts pointer array gets changed to contain only those vertices that
// are in the convex hull of the original (concave) polygon
// the return value is the number of vertices in the convex hull
// NOTE: the number of vertices passed in cannot exceet MAX_NUM_VERTS!!!!
int ConvexHull0(float **verts, int num_verts)
{
	const int MAX_NUM_VERTS = 256;
	static struct _vert_info_
	{
		short vert_index;
		short out;
	} vert_info[MAX_NUM_VERTS];
	int i, num_verts_in_convex_hull;
	float *curr, *prev, *next;
	float e0x, e0y, e1x, e1y, crossz;

	if (num_verts > MAX_NUM_VERTS)
	{
		return 0;
	}

	// mark all vertices that are outside of the convex hull
	for (i=0; i<num_verts; i++)
	{
		curr = verts[i];
		prev = verts[(i-1+num_verts)%num_verts];
		next = verts[(i+1)%num_verts];

		// get the edge vectors
		e0x = next[0] - curr[0];
		e0y = next[1] - curr[1];

		e1x = prev[0] - curr[0];
		e1y = prev[1] - curr[1];

		// find the z component of the cross product
		crossz = e0x*e1y - e0y*e1x;

		// store the index of this vert
		vert_info[i].vert_index = i;

		// if negative crossz, then this vertex is not in the convex hull
//		vert_info[i].out = (*(int *)&crossz)>>31;
		if (crossz < 0)
		{
			vert_info[i].out = 1;
		}
		else
		{
			vert_info[i].out = 0;
		}
	}

	num_verts_in_convex_hull = 0;
	for (i=0; i<num_verts; i++)
	{
		if (!vert_info[i].out)
		{
			verts[num_verts_in_convex_hull] = verts[vert_info[i].vert_index];
			num_verts_in_convex_hull++;
		}
	}

	return num_verts_in_convex_hull;
}
int ConvexHull(float **verts, int num_verts)
{
	const int MAX_NUM_VERTS = 256;
	float *dst_verts[MAX_NUM_VERTS], *src_verts[MAX_NUM_VERTS];
	int i, vert_added, curr_num_verts;
	float *curr, *prev, *next;
	float e0x, e0y, e1x, e1y, crossz;

	if (num_verts > MAX_NUM_VERTS)
	{
		return 0;
	}

	// copy all the vertices into the src_verts array
	for (i=0; i<num_verts; i++)
	{
		src_verts[i] = verts[i];
	}
	curr_num_verts = num_verts;

	while (1)
	{
		vert_added = 0;
		for (i=0; i<curr_num_verts; i++)
		{
			curr = src_verts[i];
			prev = src_verts[(i-1+curr_num_verts)%curr_num_verts];
			next = src_verts[(i+1)%curr_num_verts];

			// get the edge vectors
			e0x = next[0] - curr[0];
			e0y = next[1] - curr[1];

			e1x = prev[0] - curr[0];
			e1y = prev[1] - curr[1];

			// find the z component of the cross product
			crossz = e0x*e1y - e0y*e1x;

			// if negative crossz, then this vertex is not in the convex hull
			// we need to remove it, and run this loop again, because removing
			// this vert could've created another vertex that's not in the convex hull
			if (crossz < 0)
			{
				// add all remaining verts
				// we need to do this loop again after removing this vert
				i++; // skip this one
				while (i < curr_num_verts)
				{
					dst_verts[vert_added] = src_verts[i];
					vert_added++;
					i++;
				}
			}
			else
			{
				// add this vert
				dst_verts[vert_added] = src_verts[i];
				vert_added++;
			}
		}

		// if no verts were removed, we're done
		if (vert_added == curr_num_verts)
		{
			break;
		}

		// copy all the dst_verts into src_verts for the next loop
		curr_num_verts = vert_added;
		for (i=0; i<curr_num_verts; i++)
		{
			src_verts[i] = dst_verts[i];
		}
	}

	// copy back the convex hull vertices into verts
	for (i=0; i<curr_num_verts; i++)
	{
		verts[i] = dst_verts[i];
	}

	return curr_num_verts;
}
