/*****************************************************************************/
/*  ftp.h - include file for the whole ftp program                           */
/*  Copyright (C) 1998 Brian Masney <masneyb@seul.org>                       */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.                */
/*****************************************************************************/

#ifndef __FTP_H
#define __FTP_H

#define _REENTRANT
#include <pthread.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <ctype.h>
#include <pwd.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/utsname.h>
#include <signal.h>
#include <setjmp.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include "config.h"

#if defined(HAVE_GETTEXT)
 #if !defined(_)
  #define _(String) gettext (String)
 #endif
 #include <libintl.h>
#else if !defined(_)
 #define _(String) String
#endif

#include "rfc959.h"

/* Some general settings */
#define BASE_CONF_DIR		"~/.gftp"
#define CONFIG_FILE		BASE_CONF_DIR "/gftprc"
#define MAXSTR			255
#define MAX_HIST_LEN		10
#define ANON_LOGIN		"anonymous"

#define FILE_ISDIR		1 << 0 /* Is this a dir? */
#define FILE_ISEXE		1 << 1 /* Is this executable? */
#define FILE_ISLINK		1 << 2 /* Is this a symlink? */
#define FILE_RESTART		1 << 3 /* Are we going to restart this file
                                          transfer */
#define FILE_SHOWN		1 << 4 /* Is this file shown in the listbox */
#define FILE_SELECTED		1 << 5 /* Is this file selected */
#define FILE_ASCII		1 << 6 /* Download this as ascii. If this
                                          isn't set, it will download it as
                                          binary  */
#define FILE_TRANS_DONE		1 << 7 /* This file is done transferring */
#define FILE_TRANS_DONE_VIEW	1 << 8 /* View this file when it is done */
#define FILE_TRANS_DONE_EDIT	1 << 9 /* Edit this file when it is done */
#define FILE_TRANS_DONE_RM	1 << 10 /* rm this file when it is done */
#define FILE_TRANS_EXISTS	1 << 11 /* This file exists on both sides */
#define FILE_TRANS_SKIP		1 << 12 /* Skip the file in the transfer */

#define WINDOW_CACHED		1 << 0 /* Is this directory listing cached */
#define WINDOW_SORTASDS		1 << 1 /* If this is set, it is sorted ascending */
#define WINDOW_SORTED 		1 << 2 /* Is this listing sorted yet? */

#define TRANSFER_DIRECTION      1 << 0 /* Are we dling (1) or uploading (0) */
#define TRANSFER_CREATED	1 << 1 /* This will be set when an entry is made 
                                          in the listbox for this file transfer. */
#define TRANSFER_SELECTED	1 << 2 /* Whether this item is selected in the 
                                          listbox or not */
#define TRANSFER_STARTED	1 << 3 /* Are we transfering these files? */
#define TRANSFER_NEW		1 << 4 /* We create a new structure...don't show or
                                          create it yet */
#define TRANSFER_SHOW		1 << 5 /* We can now show the transfer and start it
                                          if we can */
#define TRANSFER_UPDATE		1 << 6 /* Just update the status of the file 
                                          transfer */
#define TRANSFER_CANCEL		1 << 7 /* Cancel this transfer */
#define TRANSFER_DONE		1 << 8 /* Done with this transfer */
#define TRANSFER_NEED_UPDATED	1 << 9 /* This will be set when we update the
                                           transfer status */
#define TRANSFER_ON_NEXT_FILE	1 << 10 /* This is set when the file is set to
                                           the next one */
#define TRANSFER_AUTO_RESTART	1 << 11 /* This transfer was automatically 
                                           restarted */

#define FIREWALL_LOGIN_SITE	1 /* We have to issue the SITE cmd to
                                     login the firewall so we can
                                     login into the ftp server */
#define FIREWALL_LOGIN_USERPASS 2 /* We have to issue USER and PASS cmd
                                     to login the firewall so we can
                                     login into the ftp server */
#define FIREWALL_LOGIN_NO_AUTH	3 /* We issue USER ftp user@host:port and PASS */
#define HTTP_PROXY		100

#define VIEWEDIT_RM_FILE	1 << 0 /* Delete the file after we are done
                                          viewing it */
#define VIEWEDIT_VIEW_FILE	1 << 1 /* If this is set, the file will be
                                          viewed. Otherwise, it will be 
                                          edited */

struct ftp_file_data {
   /* It is important to keep these structure members in THE SAME ORDER! */
   char *file,
        *user,
        *group,
        *attribs;
   time_t datetime;
   unsigned long size;

   /* These structure members can be in any order, and new members can be
      added at the bottom */
   char *remote_file;
   unsigned long remote_size;
   unsigned long flags;
   int clist_num;
   struct ftp_file_data *next;
};

typedef enum protocol_type_tag {
   ftp,
   http,
   none
} protocol_type;

/* This structure will be contained in the ftp_window_data structure below.
   This should contain all of the information that will be needed about a 
   remote site. This structure will also be contained inside the 
   ftp_transfer_data structure for the multiple downloads. */
   
struct ftp_host_data {
   gftp_request *ftpdata; /* Information about the current FTP host */
   int totalfiles; /* How many files are here */
   unsigned int wait : 1, /* Shall we wait for the connection to finish? */
                stopable : 1, /* Can we stop? */
                getdir : 1; /* Do we want to get the dir listing? */
   protocol_type protocol;
   struct ftp_file_data *files, /* All of the files in the current directory */
                        *last; /* This is the last file. This makes it easier
                                  for adding files to the end of the list */
   struct ftp_window_data *wdata;
   pthread_t tid;
   int connection_number; /* The number of times we have tried to connect thus far */
};

/* There will be 2 of these created...one for the local side and one for the 
   remote side. When all of the callbacks are created, the proper structure
   will be passed to the function */
   
struct ftp_window_data {
   GtkWidget *combo, /* Entry widget/history for the user to enter a directory */
             *hoststxt, /* Show which directory we're in */
             *listbox; /* Our listbox showing the files */
   int local, /* -1 = Not Connected, 0 = Remote, 1 = Local */
       totalitems,  /* How many items are in our list */
       numselected, /* How many items are selected in the list */
       sortcol, /* Which column we are sorting by */
       histlen; /* The number of entries in the history list */
   unsigned long flags; /* Our flags. See WINDOW_* flags above */
   char *filespec;
   struct ftp_host_data *hdata; /* The host that we are connected to */
   GList *history; /* The history of the directories in the combo widget */
   GtkItemFactory *ifactory; /* The menus that will come up when you right click */
};         

/* This will be the structure used for each thread in transfering files */

struct ftp_transfer_data {
   struct ftp_host_data *hdata; /* The host that we are connecting to */
   struct ftp_file_data *curfle, /* The current file that we are transferring */
                        *prevfle, /* The previous file in the list */
                        *updfle; /* The last file that was updated on the screen */
   struct ftp_transfer_data *next;
   unsigned long curtrans, /* How much of the file we actually transferred */
                 sttrans, /* Where we started at on the file */
                 flags; /* Our flags. See the TRANSFER_* flags above */
   struct timeval starttime, /* The time that we started the transfer */
                  lasttime; /* The beginning of the last read() */
   char progressstr[MAXSTR], /* String to be displayed in the listbox under
                                the progress column */
        **dirs_to_be_made; /* Pointer to strings of directories to be made */
   int num_dirs_to_be_made, /* Number of directories that need to be made */
       current_file_number, /* The current number of the file we are transfering */
       current_file_retries; /* The number of times we tried to transfer this file */
   GtkWidget *clist; /* CList widget for the dialog box when a file exists */
   pthread_mutex_t mutex; /* Mutex protecting this struct */
   pthread_t tid; /* This thread id */
   float kbs; /* Current KB/s we are getting */
};

/* This will be for queueing messages to the logging window. This will be
   used for the threads, since gtk is not thread safe, we cannot update the
   display inside the threads */
   
struct ftp_log_queue {
   char *msg; /* Our message */
   int type; /* Logging type. LOG_MISC, LOG_SEND, or LOG_RECV */
};

/* This will hold information about currently open files being view or edited */

struct viewedit_data {
   char *filename, /* File we are viewing/editing currently */
        *remote_filename; /* The filename on the remote computer */
   unsigned int flags; /* Our flags. See the VIEWEDIT_* defines above */
   struct stat st; /* Vital file statistics */
   pid_t pid; /* Our process id */
   char **argv; /* Our arguments we passed to execvp. We will free it when the
                   process terminates. This is the safest place to free this */
};

/* For registering file formats */

struct pix_ext {
   char *ext, /* The file extension to register */
        *filename, /* The xpm file to display */
        *view_program, /* The program used to view this file */
        *ascii_binary; /* Is this a ASCII transfer A or a BINARY transfer B */
   int stlen; /* How long is the file extension. */
   GdkPixmap *pixmap; /* Our pixmap */
   GdkBitmap *mask; /* Mask for the pixmap */
   struct pix_ext *next; /* Next file extension */
};

struct conn_categories { /* Each category will have one of these nodes */
   char *path; /* Path */
   GtkCTreeNode *cnode; /* In the edit bookmarks menu, the node */
   char *hostname, /* Our actual internet hostname */
        *dir, /* Initial directory */
        *user, /* Username to log in as */
        *pass, /* Our password */
        *acct, /* Account */
        *port, /* Port to connect to */
        *firewall; /* Connect through the firewall? */
        
        /* You may be asking why are some of the above fields char * instead
           of integers. This makes it easier for me to parse data from the
           config file */

   unsigned int isfolder : 1, /* If this is set, then the children field can
                                 be non-NULL */
                save_password : 1; /* Save this password */
   struct conn_categories *children, /* The children of this node. */
                          *prev, /* The parent of this node */
                          *next; /* The next sibling of this node */
};

/* Passed to callback functions by MakeEditDialog and MakeYesNoDialog */
struct dialog_data {
   GtkWidget *dialog,
             *edit;
   GList *all_buttons;
   gpointer data;
};

/* All these variables are defined in options.h */
extern GdkPixmap *dotdot_pixmap, *dir_pixmap, *linkdir_pixmap, *linkfile_pixmap,
   *exe_pixmap, *doc_pixmap, *open_dir_pixmap;
extern GdkBitmap *dotdot_mask, *dir_mask, *linkdir_mask, *linkfile_mask, 
   *exe_mask, *doc_mask, *open_dir_mask;

extern GtkItemFactory *factory;
extern GHashTable *bookmarks_htable;
extern struct pix_ext *registered_exts;
extern struct conn_categories *hostcat;
extern struct ftp_transfer_data *file_transfers;
extern pthread_mutex_t transfer_mutex, log_mutex;
extern GList *viewedit_processes, *file_transfer_logs;
extern GtkWidget *logwdw, *dlwdw, *local_frame, *remote_frame, *log_table, 
   *transfer_scroll, *firewall_btn, *openurl_btn, *stop_btn, *hostedit, 
   *useredit, *passedit, *portedit;
struct ftp_window_data window1, window2;

extern char *emailaddr, *edit_program, *view_program,
   *firewall_host, *firewall_username, *firewall_password,
   *firewall_account, version[], *proxy_config;
extern int firewall_port; /* What port to connect to? */

extern int do_one_transfer_at_a_time, start_file_transfers, 
    transfer_in_progress, use_cache, passive_transfer, sort_dirs_first,
    confirm_delete, use_default_dl_types, show_hidden_files,
    reconnect_diag, preserve_attribs, refresh_files, save_geometry,
    listbox_local_width, listbox_remote_width, listbox_file_height, 
    transfer_height, log_height, timeout, retries, sleep_time,
    listbox_filename_width, listbox_size_width, listbox_user_width,
    listbox_group_width, listbox_date_width, listbox_attribs_width;
extern float maxkbs;

/* bookmarks.c */
void run_bookmark (gpointer data);
void add_bookmark (gpointer data);
void edit_bookmarks (gpointer data);
void build_bookmarks_menu (void);

/* cache.c */
FILE *new_cache_entry (char *description);
FILE *find_cache_entry (char *description);
void clear_cache_files (void);
void delete_cache_entry (char *description);

/* chmod_dialog.c */
void chmod_dialog (gpointer data);

/* config_file.c */
void options_dialog (gpointer data);
void read_config_file (void);
void write_config_file (void);
int parse_args (char *str, int numargs, int lineno, char **first, ...);
GHashTable *build_hash_table (struct conn_categories *entry);

/* delete_dialog.c */
void delete_dialog (gpointer data);

/* file_transfer.c */
void retrCB (GtkWidget *widget, gpointer data);
void putCB (GtkWidget *widget, gpointer data);
void dotrans (struct ftp_transfer_data *tdata);
void *ftp_get_files (void *ptr);
void *ftp_put_files (void *ptr);

/* ftp.c */
int ftp_list_files (struct ftp_window_data *wdata, int usecache);
struct ftp_file_data *get_remote_files (struct ftp_host_data *hdata, char *path, int *total, int load_from_cache, GtkLabel *up_wid);
int http_get_next_file (gftp_request *request, gftp_file *fle, FILE *cachefd);
int http_connect (struct ftp_host_data *hdata);
int http_get_file (struct ftp_host_data *hdata, char *filename);
int parse_html_line (char *tempstr, gftp_file *fle);
void add_file_transfer (struct ftp_transfer_data *tdata);
struct ftp_transfer_data *transfer_one_file (char *local_file, char *remote_file, int dl_or_up);
int ftp_connect (struct ftp_host_data *hdata, int getdir);
void stop_button (GtkWidget *widget, struct ftp_window_data *wdata);
void disconnect (struct ftp_window_data *wdata);
unsigned long get_file_transfer_mode (char *filename);

/* gftp.c */
void sortrows (GtkCList *clist, gint column, gpointer data);
void delete_ftp_file_info (struct ftp_window_data *wdata);
void queue_log (gftp_logging_type level, void *ptr, const char *string, ...);
void ftp_log (gftp_logging_type level, void *ptr, const char *string, ...);
void toolbar_hostedit (GtkWidget *widget, gpointer data);
void update_ftp_info (struct ftp_window_data *wdata);
void refresh (struct ftp_window_data *wdata);
void fix_display (void);

/* misc.c */
char *insert_commas (unsigned long number, char *dest, int size);
void add_local_files (struct ftp_window_data *win);
struct ftp_file_data *get_local_files (char *path, int *total);
struct ftp_file_data *parse_local_file (FILE *fd, int local, int *total, protocol_type proto);
void add_file_listbox (struct ftp_window_data *wdata, struct ftp_file_data *fle);
long file_countlf (int filefd, long endpos);
char *alltrim (char *str);
int expand_path (char *src, char *dest, size_t size);
void remove_double_slashes (char *string);
char *make_temp_filename (char *destbuf, char *ext, int size);
void free_file_list (struct ftp_file_data *filelist);
void free_fdata (struct ftp_file_data *fle);
void free_hdata (struct ftp_host_data *hdata);
struct ftp_host_data *new_hdata (void);
void free_tdata (struct ftp_transfer_data *tdata);
struct ftp_transfer_data *new_tdata (void);
void copy_hdata_struct (struct ftp_host_data *hdata, struct ftp_host_data *newhdata);
void copy_fdata_struct (struct ftp_file_data *fle, struct ftp_file_data *newfle);
int compare_hdata_structs (gftp_request *request1, gftp_request *request2, int compare_dirs);
struct ftp_file_data *get_next_selected_filename (struct ftp_file_data *filelist);
void open_xpm (char *filename, GtkWidget *parent, GdkPixmap **pixmap, GdkBitmap **mask, int quit_on_err);
GtkWidget *toolbar_pixmap (GtkWidget *widget, char **xpmdata);
int check_status(char *name, struct ftp_window_data *wdata, int only_one, int at_least_one, int use_ftp_protocol);
void make_nonnull (char **str);
void create_item_factory (GtkItemFactory *ifactory, guint n_entries, GtkItemFactoryEntry *entries, gpointer callback_data);

/* misc_dialogs.c */
void about_dialog (gpointer data);
void MakeEditDialog (char *diagtxt, char *infotxt, char *deftext, int edit_shown, int erase,
	char *oktxt, void (*okfunc)(), void *okptr,
	char *canceltxt, void (*cancelfunc)(), void *cancelptr);
void MakeYesNoDialog (char *diagtxt, char *infotxt, int erase, int num, ...);
void destroy_dialog_data (struct dialog_data *data);
void site_dialog (struct ftp_window_data *wdata);
void change_filespec (struct ftp_window_data *wdata);
void openurl_dialog (gpointer data);
void openurl_get_drag_data (GtkWidget *widget, GdkDragContext *context, gint x, 
	gint y, GtkSelectionData *selection_data, guint info, guint32 clk_time, 
	struct ftp_window_data *wdata);
void reconnect_dialog (struct ftp_host_data *hdata);
void viewlog (gpointer data);
void savelog (gpointer data);
void clearlog (gpointer data);

/* mkdir_dialog.c */
void mkdir_dialog (gpointer data);

/* rename_dialog.c */
void rename_dialog (gpointer data);

/* view_dialog.c */
void view_dialog (gpointer data);
void edit_dialog (gpointer data);
void view_file (char *filename, int viewedit, int del_file, int start_pos, char *remote_file);

#endif
