/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#include "icewm.h"

YFrameButton::YFrameButton(YWindow *parent,
                           YFrameWindow *frame,
                           WMCommand command,
                           WMCommand command2):
    YFrameControl(parent, frame)
{
    fCommand = command;
    fCommand2 = command2;
    isDown = isActive = wasPopupActive = isHilited = 0;
    fPopup = 0;

    int w = minimizePixmap->width(), h = minimizePixmap->height();
    switch (wmLook) {
#ifdef CONFIG_LOOK_WARP3
    case lookWarp3:
        setSize(w + 2, h + 2);
        break;
#endif
#ifdef CONFIG_LOOK_WARP4
    case lookWarp4:
        setSize(w + 0, h / 2 + 0);
        break;
#endif
#if defined(CONFIG_LOOK_NICE) || defined(CONFIG_LOOK_WIN95)
#ifdef CONFIG_LOOK_NICE
    case lookNice:
#endif
#ifdef CONFIG_LOOK_WIN95
    case lookWin95:
#endif
        setSize(w + 3, h + 3);
        break;
#endif
#ifdef CONFIG_LOOK_MOTIF
    case lookMotif:
        setSize(w + 2, h + 2);
        break;
#endif
    default:
        break;
    }
}

YFrameButton::~YFrameButton() {
    if (fPopup)
        fPopup->popdown();
}

void YFrameButton::activate() {
    if (!isActive && isDown) {
        isActive = 1;
        repaint();
    }
}

void YFrameButton::deactivate() {
    if (isActive) {
        isActive = 0;
        repaint();
    }
}

void YFrameButton::handleButton(const XButtonEvent &button) {
    if (button.type == ButtonPress && button.button == 1) {
        if (fPopup == 0) {
            if (!(button.state & ControlMask) && raiseOnClickButton) {
                frame()->activate();
                if (raiseOnClickButton)
                    frame()->wmRaise();
            }
        }
        isDown = 1;
        if (fCommand == cmdSysMenu) {
            wasPopupActive = 0;
            if (fPopup) {
                wasPopupActive = 1;
            } else {
                activate();
                fPopup = frame()->windowMenu();
                frame()->updateMenu();
                fPopup->popup(this, frame(),
                              x() + frame()->titlebar()->x() + frame()->x(),
                              height() + y() + frame()->titlebar()->y() + frame()->y(),
                              width(), height(),
                              YPopupWindow::pfButtonDown |
                              YPopupWindow::pfCanFlipVertical);
            }
        } else {
            activate();
        }
    } else if (button.type == ButtonRelease) {
        int inWindow = (button.x >= 0 &&
                        button.y >= 0 &&
                        button.x < int (width()) &&
                        button.y < int (height()));
        int down = isDown;

        isDown = 0;
        if (fCommand == cmdSysMenu) {
            if (wasPopupActive || !inWindow) {
                if (fPopup) {
                    donePopup(fPopup);
                    deactivate();
                }
            }
        } else {
            deactivate();

            if (inWindow && down)
                if ((button.state & ShiftMask) && fCommand2 != cmdNone)
                    frame()->handleCommand(fCommand2, 0);
                else
                    frame()->handleCommand(fCommand, 0);

            return ;
        }
    }
    YWindow::handleButton(button);
}

void YFrameButton::handleClick(const XButtonEvent &down, const XButtonEvent &up, int count) {
    if (fCommand == cmdSysMenu && down.button == 1 && up.button == 1) {
        if (count == 2) {
            frame()->wmClose();
            if (fPopup) {
                donePopup(fPopup);
                deactivate();
            }
        }
    }
}

void YFrameButton::handleMotion(const XMotionEvent &motion) {
#if 0
    if (isActive && fCommand == cmdSysMenu && (motion.state & Button1Mask)) {
    } else if (isDown) {
        if (motion.x >= 0 && motion.y >= 0 &&
            motion.x < int(width()) && motion.y < int(height()))
            activate();
        else
            deactivate();
    }
#endif
    YWindow::handleMotion(motion);
}

void YFrameButton::handleCrossing(const XCrossingEvent &crossing) {
    if (crossing.type == EnterNotify) {
        isHilited = 1;
        if (isDown)
            activate();
        else
            repaint();
    } else if (crossing.type == LeaveNotify) {
        isHilited = 0;
        if (isDown)
            deactivate();
        else
            repaint();
    }
}

void YFrameButton::setCommand(WMCommand command, WMCommand command2) {
    fCommand2 = command2;
    if (command != fCommand) {
        fCommand = command;
        repaint();
    }
}

void YFrameButton::donePopup(YPopupWindow *popup) {
    if (popup != fPopup) {
        MSG(("popup differs for button??"));
        return ;
    }
    popdownMenu();
}

void YFrameButton::popupMenu() {
    assert(fPopup == 0);
    if (fCommand == cmdSysMenu) {
        isDown = 1;
        isHilited = 1;
        wasPopupActive = 0;
        activate();
        fPopup = frame()->windowMenu();
        frame()->updateMenu();
        windowMenu->popup(this, frame(),
                          x() + frame()->titlebar()->x() + frame()->x(),
                          height() + y() + frame()->titlebar()->y() + frame()->y(),
                          width(), height(),
                          YPopupWindow::pfCanFlipVertical);
        isDown = 0;
    }
}

void YFrameButton::popdownMenu() {
    if (fPopup) {
        MSG(("done popup in frame button"));
        //isHilited = 0;
        fPopup->popdown();
        isDown = 0;
        isHilited = 0;
        isActive = 0;
        repaint();
        fPopup = 0;
    }
}

void YFrameButton::paint(Graphics &g, int , int , unsigned int , unsigned int ) {
    int xPos = 1, yPos = 1;
    YPixmap *pixmap = 0;
    YMaskPixmap *icon = 0;

    g.setColor(buttonBg);

    switch (fCommand) {
    case cmdMaximize: pixmap = maximizePixmap; break;
    case cmdMinimize: pixmap = minimizePixmap; break;
    case cmdRestore: pixmap = restorePixmap; break;
    case cmdClose: pixmap = closePixmap; break;
    case cmdSysMenu: icon = frame()->clientIcon()->small(); break;
    default:
        pixmap = 0;
    }

    switch (wmLook) {
#ifdef CONFIG_LOOK_WARP4
    case lookWarp4:
        if (fCommand == cmdSysMenu) {
            g.fillRect(0, 0, width(), height());

            if (isActive)
                g.setColor(activeTitleBarBg);

            g.fillRect(1, 1, width() - 2, height() - 2);
            
            if (icon)
                g.drawMaskPixmap(icon,
                                 (width() - icon->width()) / 2,
                                 (height() - icon->height()) / 2);
        } else {
            int picYpos = isActive ? 20 : 0;

            g.fillRect(0, 0, width(), height());
            
            if (pixmap)
                XCopyArea(app->display(),
                          pixmap->pixmap(), handle(),
                          g.handle(),
                          0, picYpos,
                          pixmap->width(), pixmap->height() / 2,
                          (width() - pixmap->width()) / 2,
                          (height() - pixmap->height() / 2) / 2);
        }
        break;
#endif
#if defined(CONFIG_LOOK_MOTIF) || defined(CONFIG_LOOK_WARP3) || defined(CONFIG_LOOK_NICE)
#ifdef CONFIG_LOOK_MOTIF
    case lookMotif:
#endif
#ifdef CONFIG_LOOK_WARP3
    case lookWarp3:
#endif
#ifdef CONFIG_LOOK_NICE
    case lookNice:
#endif
        if (wmLook == lookNice && !isHilited)
            g.drawRect(0, 0, width() - 1, height() - 1);
        else
            g.draw3DRect(0, 0, width() - 1, height() - 1,
                         isActive ? false : true);
#ifdef CONFIG_LOOK_MOTIF
        if (wmLook != lookMotif)
#endif
        {
            if (isActive) {
                xPos = 3;
                yPos = 3;
                g.drawLine(1, 1, width() - 2, 1);
                g.drawLine(1, 1, 1, height() - 2);
                g.drawLine(2, 2, width() - 3, 2);
                g.drawLine(2, 2, 2, height() - 3);
            } else {
                xPos = 2;
                yPos = 2;
                g.drawRect(1, 1, width() - 3, width() - 3);
            }
        }
        /* else if (wmLook == lookWin95) {
            xPos = 2;
            yPos = 2;
        }*/

        int xW, yW;
        
#ifdef CONFIG_LOOK_MOTIF
        if (wmLook == lookMotif)
        {
            xW = width() - 2;
            yW = height() - 2;
        } else
#endif
        {
            xW = width() - 4;
            yW = height() - 4;
        }
        if (fCommand == cmdSysMenu) {
            g.fillRect(xPos, yPos, xW, yW);
            
            if (icon)
                g.drawMaskPixmap(icon,
                                 xPos + (xW - icon->width()) / 2,
                                 yPos + (yW - icon->height()) / 2);
        } else {
            g.drawCenteredPixmap(xPos, yPos, xW, yW, pixmap);
        }
        break;
#endif
#ifdef CONFIG_LOOK_WIN95
    case lookWin95:
        if (fCommand == cmdSysMenu) {
            if (!isActive) {
                YColor *bg = frame()->titlebar()->active()
                    ? activeTitleBarBg
                    : inactiveTitleBarBg;
                g.setColor(bg);
            }

            g.fillRect(0, 0, width(), height());
            if (icon)
                g.drawMaskPixmap(icon,
                                 (width() - icon->width()) / 2,
                                 (height() - icon->height()) / 2);
        } else {
            g.drawBorderW(0, 0, width() - 1, height() - 1,
                          isActive ? false: true);

            if (isActive)
                xPos = yPos = 2;

            g.drawCenteredPixmap(xPos, yPos, width() - 3, height() - 3,
                                 pixmap);
        }
        break;
#endif
    default:
        break;
    }
}

YButton::YButton(YWindow *parent, WMCommand command, void *context): YWindow(parent) {
    selected = 0;
    isActive = 0;
    fPopup = 0;
    fCommand = command;
    fContext = context;
    wasPopupActive = 0;
    fPixmap = 0;
    fPressed = 0;
    fListener = 0;
}

YButton::YButton(YWindow *parent, YPopupWindow *popup): YWindow(parent) {
    selected = 0;
    isActive = 0;
    fPopup = popup;
    fCommand = cmdNone;
    wasPopupActive = 0;
    fPixmap = 0;
    fText = 0;
    fContext = 0;
    fPressed = 0;
    fListener = 0;
}

YButton::~YButton() {
    if (isActive)
        popdown();
}

void YButton::paint(Graphics &g, int /*x*/, int /*y*/, unsigned int /*width*/, unsigned int /*height*/) {
    int d = (fPressed || isActive) ? 1 : 0;

    g.setColor(taskBarBg);
    g.drawBorderW(0, 0, width() - 1, height() - 1, d ? false : true);

    if (fPixmap) {
        g.drawPixmap(fPixmap, 1 + d, 1 + d);
    } else {
        g.fillRect(1 + d, 1 + d, width() - 3, height() - 3);

        YFont *font;
        
        if (fPressed)
            font = activeTaskBarFont;
        else
            font = normalTaskBarFont;

        if (fText) {
            int w = XTextWidth(font->getFontStruct(), fText, strlen(fText));
            int p = (width() - w) / 2;
            int yp =  (height() - 1 - font->height()) / 2
                + font->ascent() + d;

            g.setColor(normalTaskBarAppFg);
            g.setFont(font);
            g.drawChars(fText, 0, strlen(fText), d + p, yp);
        }
    }
}

void YButton::handleButton(const XButtonEvent &button) {
    if (button.type == ButtonPress && button.button == 1) {
        selected = 2;
        if (fPopup) {
            wasPopupActive = isActive;
            if (!isActive)
                popup();
        } else {
        }
        isActive = 1;
        repaint();
    } else if (button.type == ButtonRelease) {
        int inWindow = (button.x >= 0 &&
                        button.y >= 0 &&
                        button.x < int (width()) &&
                        button.y < int (height()));
        selected = 0;
        if (fPopup) {
            if ((!inWindow || wasPopupActive) && isActive) {
                popdown();
                isActive = 0;
                repaint();
            }
        } else {
            int wasActive = isActive;
            if (isActive) {
                isActive = 0;
                repaint();
            }
            if (inWindow && wasActive) {
                if (fListener)
                    fListener->ButtonClick(this);
                else
                    app->handleCommand(fCommand, fContext);
            }
        }
    }
}

void YButton::handleCrossing(const XCrossingEvent &crossing) {
    if (selected > 0) {
        if (crossing.type == EnterNotify) {
            selected = 2;
            if (!fPopup) { 
                isActive = 1;
                repaint();
            }
        } else if (crossing.type == LeaveNotify) {
            selected = 1;
            if (!fPopup) { 
                isActive = 0;
                repaint();
            }
        }
    }
}

void YButton::setCommand(WMCommand command, void *context) {
    fCommand = command;
    fContext = context;
}

void YButton::setPixmap(YPixmap *pixmap) {
    fPixmap = pixmap;
    if (pixmap) {
        setSize(pixmap->width() + 3,
                pixmap->height() + 3);
    }
}

void YButton::setPressed(int pressed) {
    if (fPressed != pressed) {
        fPressed = pressed;
        repaint();
    }
}

void YButton::setText(char *str) {
    fText = str ? strdup(str) : 0;
    if (fText) {
        int w = XTextWidth(activeTaskBarFont->getFontStruct(), fText, strlen(fText));
        int h = statusFont->ascent();
        
        setSize(3 + w + 4, 3 + h + 4);
    }
}

void YButton::setPopup(YPopupWindow *popup) {
    if (fPopup) {
        popdown();
        delete fPopup;
    }
    fPopup = popup;
}

void YButton::donePopup(YPopupWindow *popup) {
    if (popup != fPopup) {
        MSG(("popup different?"));
        return ;
    }
    popdown();
    isActive = 0;
    selected = 0;
    repaint();
}

void YButton::popup() {
    if (fPopup) {
        fPopup->popup(this, 0,
                      x() + parent()->x() - 2,
                      y() + parent()->y() + height(),
                      0,
                      height(),
                      YPopupWindow::pfCanFlipVertical | 
                      YPopupWindow::pfButtonDown);
    }
}

void YButton::popdown() {
    if (isActive && fPopup) {
        fPopup->popdown();
        isActive = 0;
    }
}
