/*
 *  Apple II emulator 6502 instruction table and routines.
 *
 *  The routines in this file were moved from the apple2.S file,
 *  originally by Alexander Jean-Claude Bottema.
 *
 *  $Id: 6502.h,v 1.2 1998/02/07 21:21:34 chernabog Exp $
 *
 *  MODIFICATION HISTORY
 *   v0.5 by Aaron Culliney <chernabog@baldmountain.bbn.com>, Feb 1998.
 *     This code has nothing to do with my employer, GTE Internetworking,
 *     BBN Technologies.  It was written completely on my own time and on
 *     my own machine.
 *
 **/

#ifndef __ASSEMBLY__
#define table_6502(nme,fmt,ins,val,args) \
	{ nme, fmt, val, args },
#else
#define table_6502(nme,fmt,ins,val,args) \
	.long		ins

	.align 4
table_6502_opcodes:
#endif
table_6502("BRK", "BRK",		op_BRK,		0x00, 0)
table_6502("ORA", "ORA ($%02X,X)",	op_ORA_ind_x,	0x01, 1)
table_6502("???", "???",		op_UNK,		0x02, 0)
table_6502("???", "???",		op_UNK,		0x03, 0)
table_6502("???", "???",		op_UNK,		0x04, 0)
table_6502("ORA", "ORA $%02X",		op_ORA_zpage,	0x05, 1)
table_6502("ASL", "ASL $%02X",		op_ASL_zpage,	0x06, 1)
table_6502("???", "???",		op_UNK,		0x07, 0)
table_6502("PHP", "PHP",		op_PHP,		0x08, 0)
table_6502("ORA", "ORA #$%02X",		op_ORA_imm,	0x09, 1)
table_6502("ASL", "ASL",		op_ASL_acc,	0x0A, 0)
table_6502("???", "???",		op_UNK,		0x0B, 0)
table_6502("???", "???",		op_UNK,		0x0C, 0)
table_6502("ORA", "ORA $%02X%02X",	op_ORA_abs,	0x0D, 2)
table_6502("ASL", "ASL $%02X%02X",	op_ASL_abs,	0x0E, 2)
table_6502("???", "???",		op_UNK,		0x0F, 0)
table_6502("BPL", "BPL $%04X (_$%02X)",	op_BPL,		0x10, 3)
table_6502("ORA", "ORA ($%02X),Y",	op_ORA_ind_y,	0x11, 1)
table_6502("???", "???",		op_UNK,		0x12, 0)
table_6502("???", "???",		op_UNK,		0x13, 0)
table_6502("???", "???",		op_UNK,		0x14, 0)
table_6502("ORA", "ORA $%02X,X",	op_ORA_zpage_x,	0x15, 1)
table_6502("ASL", "ASL $%02X,X",	op_ASL_zpage_x,	0x16, 1)
table_6502("???", "???",		op_UNK,		0x17, 0)
table_6502("CLC", "CLC",		op_CLC,		0x18, 0)
table_6502("ORA", "ORA $%02X%02X,Y",	op_ORA_abs_y,	0x19, 2)
table_6502("???", "???",		op_UNK,		0x1A, 0)
table_6502("???", "???",		op_UNK,		0x1B, 0)
table_6502("???", "???",		op_UNK,		0x1C, 0)
table_6502("ORA", "ORA $%02X%02X,X",	op_ORA_abs_x,	0x1D, 2)
table_6502("ASL", "ASL $%02X%02X,X",	op_ASL_abs_x,	0x1E, 2)
table_6502("???", "???",		op_UNK,		0x1F, 0)
table_6502("JSR", "JSR %02X%02X",	op_JSR,		0x20, 2)
table_6502("AND", "AND ($%02X,X)",	op_AND_ind_x,	0x21, 1)
table_6502("???", "???",		op_UNK,		0x22, 0)
table_6502("???", "???",		op_UNK,		0x23, 0)
table_6502("BIT", "BIT $%02X",		op_BIT_zpage,	0x24, 1)
table_6502("AND", "AND $%02X",		op_AND_zpage,	0x25, 1)
table_6502("ROL", "ROL $%02X",		op_ROL_zpage,	0x26, 1)
table_6502("???", "???",		op_UNK,		0x27, 0)
table_6502("PLP", "PLP",		op_PLP,		0x28, 0)
table_6502("AND", "AND #$%02X",		op_AND_imm,	0x29, 1)
table_6502("ROL", "ROL",		op_ROL_acc,	0x2A, 0)
table_6502("???", "???",		op_UNK,		0x2B, 0)
table_6502("BIT", "BIT $%02X%02X",	op_BIT_abs,	0x2C, 2)
table_6502("AND", "AND $%02X%02X",	op_AND_abs,	0x2D, 2)
table_6502("ROL", "ROL $%02X%02X",	op_ROL_abs,	0x2E, 2)
table_6502("???", "???",		op_UNK,		0x2F, 0)
table_6502("BMI", "BMI $%04X (_$%02X)",	op_BMI,		0x30, 3)
table_6502("AND", "AND ($%02X),Y",	op_AND_ind_y,	0x31, 1)
table_6502("???", "???",		op_UNK,		0x32, 0)
table_6502("???", "???",		op_UNK,		0x33, 0)
table_6502("???", "???",		op_UNK,		0x34, 0)
table_6502("AND", "AND $%02X,X",	op_AND_zpage_x,	0x35, 1)
table_6502("ROL", "ROL $%02X,X",	op_ROL_zpage_x,	0x36, 1)
table_6502("???", "???",		op_UNK,		0x37, 0)
table_6502("SEC", "SEC",		op_SEC,		0x38, 0)
table_6502("AND", "AND $%02X%02X,Y",	op_AND_abs_y,	0x39, 2)
table_6502("???", "???",		op_UNK,		0x3A, 0)
table_6502("???", "???",		op_UNK,		0x3B, 0)
table_6502("???", "???",		op_UNK,		0x3C, 0)
table_6502("AND", "AND $%02X%02X,X",	op_AND_abs_x,	0x3D, 2)
table_6502("ROL", "ROL $%02X%02X,X",	op_ROL_abs_x,	0x3E, 2)
table_6502("???", "???",		op_UNK,		0x3F, 0)
table_6502("RTI", "RTI",		op_RTI,		0x40, 0)
table_6502("EOR", "EOR ($%02X,X)",	op_EOR_ind_x,	0x41, 1)
table_6502("???", "???",		op_UNK,		0x42, 0)
table_6502("???", "???",		op_UNK,		0x43, 0)
table_6502("???", "???",		op_UNK,		0x44, 0)
table_6502("EOR", "EOR $%02X",		op_EOR_zpage,	0x45, 1)
table_6502("LSR", "LSR $%02X",		op_LSR_zpage,	0x46, 1)
table_6502("???", "???",		op_UNK,		0x47, 0)
table_6502("PHA", "PHA",		op_PHA,		0x48, 0)
table_6502("EOR", "EOR #$%02X",		op_EOR_imm,	0x49, 1)
table_6502("LSR", "LSR",		op_LSR_acc,	0x4A, 0)
table_6502("???", "???",		op_UNK,		0x4B, 0)
table_6502("JMP", "JMP %02X%02X",	op_JMP_abs,	0x4C, 2)
table_6502("EOR", "EOR %02X%02X",	op_EOR_abs,	0x4D, 2)
table_6502("LSR", "LSR %02X%02X",	op_LSR_abs,	0x4E, 2)
table_6502("???", "???",		op_UNK,		0x4F, 0)
table_6502("BVC", "BVC $%04X (_$%02X)",	op_BVC,		0x50, 3)
table_6502("EOR", "EOR ($%02X),Y",	op_EOR_ind_y,	0x51, 1)
table_6502("???", "???",		op_UNK,		0x52, 0)
table_6502("???", "???",		op_UNK,		0x53, 0)
table_6502("???", "???",		op_UNK,		0x54, 0)
table_6502("EOR", "EOR $%02X,X",	op_EOR_zpage_x,	0x55, 1)
table_6502("LSR", "LSR $%02X,X",	op_LSR_zpage_x,	0x56, 1)
table_6502("???", "???",		op_UNK,		0x57, 0)
table_6502("CLI", "CLI",		op_CLI,		0x58, 0)
table_6502("EOR", "EOR $%02X%02X,Y",	op_EOR_abs_y,	0x59, 2)
table_6502("???", "???",		op_UNK,		0x5A, 0)
table_6502("???", "???",		op_UNK,		0x5B, 0)
table_6502("???", "???",		op_UNK,		0x5C, 0)
table_6502("EOR", "EOR $%02X%02X,X",	op_EOR_abs_x,	0x5D, 2)
table_6502("LSR", "LSR $%02X%02X,X",	op_LSR_abs_x,	0x5E, 2)
table_6502("???", "???",		op_UNK,		0x5F, 0)
table_6502("RTS", "RTS",		op_RTS,		0x60, 0)
table_6502("ADC", "ADC ($%02X,X)",	op_ADC_ind_x,	0x61, 1)
table_6502("???", "???",		op_UNK,		0x62, 0)
table_6502("???", "???",		op_UNK,		0x63, 0)
table_6502("???", "???",		op_UNK,		0x64, 0)
table_6502("ADC", "ADC $%02X",		op_ADC_zpage,	0x65, 1)
table_6502("ROR", "ROR $%02X",		op_ROR_zpage,	0x66, 1)
table_6502("???", "???",		op_UNK,		0x67, 0)
table_6502("PLA", "PLA",		op_PLA,		0x68, 0)
table_6502("ADC", "ADC #$%02X",		op_ADC_imm,	0x69, 1)
table_6502("ROR", "ROR",		op_ROR_acc,	0x6A, 0)
table_6502("???", "???",		op_UNK,		0x6B, 0)
table_6502("JMP", "JMP ($%02X%02X)",	op_JMP_ind,	0x6C, 2)
table_6502("ADC", "ADC $%02X%02X",	op_ADC_abs,	0x6D, 2)
table_6502("ROR", "ROR $%02X%02X",	op_ROR_abs,	0x6E, 2)
table_6502("???", "???",		op_UNK,		0x6F, 0)
table_6502("BVS", "BVS $%04X (_$%02X)",	op_BVS,		0x70, 3)
table_6502("ADC", "ADC ($%02X),Y",	op_ADC_ind_y,	0x71, 1)
table_6502("???", "???",		op_UNK,		0x72, 0)
table_6502("???", "???",		op_UNK,		0x73, 0)
table_6502("???", "???",		op_UNK,		0x74, 0)
table_6502("ADC", "ADC $%02X,X",	op_ADC_zpage_x,	0x75, 1)
table_6502("ROR", "ROR $%02X,X",	op_ROR_zpage_x,	0x76, 1)
table_6502("???", "???",		op_UNK,		0x77, 0)
table_6502("SEI", "SEI",		op_SEI,		0x78, 0)
table_6502("ADC", "ADC $%02X%02X,Y",	op_ADC_abs_y,	0x79, 2)
table_6502("???", "???",		op_UNK,		0x7A, 0)
table_6502("???", "???",		op_UNK,		0x7B, 0)
table_6502("???", "???",		op_UNK,		0x7C, 0)
table_6502("ADC", "ADC $%02X%02X,X",	op_ADC_abs_x,	0x7D, 2)
table_6502("ROR", "ROR $%02X%02X,X",	op_ROR_abs_x,	0x7E, 2)
table_6502("???", "???",		op_UNK,		0x7F, 0)
table_6502("???", "???",		op_UNK,		0x80, 0)
table_6502("STA", "STA ($%02X,X)",	op_STA_ind_x,	0x81, 1)
table_6502("???", "???",		op_UNK,		0x82, 0)
table_6502("???", "???",		op_UNK,		0x83, 0)
table_6502("STY", "STY $%02X",		op_STY_zpage,	0x84, 1)
table_6502("STA", "STA $%02X",		op_STA_zpage,	0x85, 1)
table_6502("STX", "STX $%02X",		op_STX_zpage,	0x86, 1)
table_6502("???", "???",		op_UNK,		0x87, 0)
table_6502("DEY", "DEY",		op_DEY,		0x88, 0)
table_6502("???", "???",		op_UNK,		0x89, 0)
table_6502("TXA", "TXA",		op_TXA,		0x8A, 0)
table_6502("???", "???",		op_UNK,		0x8B, 0)
table_6502("STY", "STY $%02X%02X",	op_STY_abs,	0x8C, 2)
table_6502("STA", "STA $%02X%02X",	op_STA_abs,	0x8D, 2)
table_6502("STX", "STX $%02X%02X",	op_STX_abs,	0x8E, 2)
table_6502("???", "???",		op_UNK,		0x8F, 0)
table_6502("BCC", "BCC $%04X (_$%02X)",	op_BCC,		0x90, 3)
table_6502("STA", "STA ($%02X),Y",	op_STA_ind_y,	0x91, 1)
table_6502("???", "???",		op_UNK,		0x92, 0)
table_6502("???", "???",		op_UNK,		0x93, 0)
table_6502("STY", "STY $%02X,X",	op_STY_zpage_x,	0x94, 1)
table_6502("STA", "STA $%02X,X",	op_STA_zpage_x,	0x95, 1)
table_6502("STX", "STX $%02X,Y",	op_STX_zpage_y,	0x96, 1)
table_6502("???", "???",		op_UNK,		0x97, 0)
table_6502("TYA", "TYA",		op_TYA,		0x98, 0)
table_6502("STA", "STA $%02X%02X,Y",	op_STA_abs_y,	0x99, 2)
table_6502("TXS", "TXS",		op_TXS,		0x9A, 0)
table_6502("???", "???",		op_UNK,		0x9B, 0)
table_6502("???", "???",		op_UNK,		0x9C, 0)
table_6502("STA", "STA $%02X%02X,X",	op_STA_abs_x,	0x9D, 2)
table_6502("???", "???",		op_UNK,		0x9E, 0)
table_6502("???", "???",		op_UNK,		0x9F, 0)
table_6502("LDY", "LDY #$%02X",		op_LDY_imm,	0xA0, 1)
table_6502("LDA", "LDA ($%02X,X)",	op_LDA_ind_x,	0xA1, 1)
table_6502("LDX", "LDX #$%02X",		op_LDX_imm,	0xA2, 1)
table_6502("???", "???",		op_UNK,		0xA3, 0)
table_6502("LDY", "LDY $%02X",		op_LDY_zpage,	0xA4, 1)
table_6502("LDA", "LDA $%02X",		op_LDA_zpage,	0xA5, 1)
table_6502("LDX", "LDX $%02X",		op_LDX_zpage,	0xA6, 1)
table_6502("???", "???",		op_UNK,		0xA7, 0)
table_6502("TAY", "TAY",		op_TAY,		0xA8, 0)
table_6502("LDA", "LDA #$%02X",		op_LDA_imm,	0xA9, 1)
table_6502("TAX", "TAX",		op_TAX,		0xAA, 0)
table_6502("???", "???",		op_UNK,		0xAB, 0)
table_6502("LDY", "LDY $%02X%02X",	op_LDY_abs,	0xAC, 2)
table_6502("LDA", "LDA $%02X%02X",	op_LDA_abs,	0xAD, 2)
table_6502("LDX", "LDX $%02X%02X",	op_LDX_abs,	0xAE, 2)
table_6502("???", "???",		op_UNK,		0xAF, 0)
table_6502("BCS", "BCS $%04X (_$%02X)",	op_BCS,		0xB0, 3)
table_6502("LDA", "LDA ($%02X),Y",	op_LDA_ind_y,	0xB1, 1)
table_6502("???", "???",		op_UNK,		0xB2, 0)
table_6502("???", "???",		op_UNK,		0xB3, 0)
table_6502("LDY", "LDY $%02X,X",	op_LDY_zpage_x,	0xB4, 1)
table_6502("LDA", "LDA $%02X,X",	op_LDA_zpage_x,	0xB5, 1)
table_6502("LDX", "LDX $%02X,Y",	op_LDX_zpage_y,	0xB6, 1)
table_6502("???", "???",		op_UNK,		0xB7, 0)
table_6502("CLV", "CLV",		op_CLV,		0xB8, 0)
table_6502("LDA", "LDA $%02X%02X,Y",	op_LDA_abs_y,	0xB9, 2)
table_6502("TSX", "TSX",		op_TSX,		0xBA, 0)
table_6502("???", "???",		op_UNK,		0xBB, 0)
table_6502("LDY", "LDY $%02X%02X,X",	op_LDY_abs_x,	0xBC, 2)
table_6502("LDA", "LDA $%02X%02X,X",	op_LDA_abs_x,	0xBD, 2)
table_6502("LDX", "LDX $%02X%02X,Y",	op_LDX_abs_y,	0xBE, 2)
table_6502("???", "???",		op_UNK,		0xBF, 0)
table_6502("CPY", "CPY #$%02X",		op_CPY_imm,	0xC0, 1)
table_6502("CMP", "CMP ($%02X,X)",	op_CMP_ind_x,	0xC1, 1)
table_6502("???", "???",		op_UNK,		0xC2, 0)
table_6502("???", "???",		op_UNK,		0xC3, 0)
table_6502("CPY", "CPY $%02X",		op_CPY_zpage,	0xC4, 1)
table_6502("CMP", "CMP $%02X",		op_CMP_zpage,	0xC5, 1)
table_6502("DEC", "DEC $%02X",		op_DEC_zpage,	0xC6, 1)
table_6502("???", "???",		op_UNK,		0xC7, 0)
table_6502("INY", "INY",		op_INY,		0xC8, 0)
table_6502("CMP", "CMP #$%02X",		op_CMP_imm,	0xC9, 1)
table_6502("DEX", "DEX",		op_DEX,		0xCA, 0)
table_6502("???", "???",		op_UNK,		0xCB, 0)
table_6502("CPY", "CPY $%02X%02X",	op_CPY_abs,	0xCC, 2)
table_6502("CMP", "CMP $%02X%02X",	op_CMP_abs,	0xCD, 2)
table_6502("DEC", "DEC $%02X%02X",	op_DEC_abs,	0xCE, 2)
table_6502("???", "???",		op_UNK,		0xCF, 0)
table_6502("BNE", "BNE $%04X (_$%02X)",	op_BNE,		0xD0, 3)
table_6502("CMP", "CMP ($%02X),Y",	op_CMP_ind_y,	0xD1, 1)
table_6502("???", "???",		op_UNK,		0xD2, 0)
table_6502("???", "???",		op_UNK,		0xD3, 0)
table_6502("???", "???",		op_UNK,		0xD4, 0)
table_6502("CMP", "CMP $%02X,X",	op_CMP_zpage_x,	0xD5, 1)
table_6502("DEC", "DEC $%02X,X",	op_DEC_zpage_x,	0xD6, 1)
table_6502("???", "???",		op_UNK,		0xD7, 0)
table_6502("CLD", "CLD",		op_CLD,		0xD8, 0)
table_6502("CMP", "CMP $%02X%02X,Y",	op_CMP_abs_y,	0xD9, 2)
table_6502("???", "???",		op_UNK,		0xDA, 0)
table_6502("???", "???",		op_UNK,		0xDB, 0)
table_6502("???", "???",		op_UNK,		0xDC, 0)
table_6502("CMP", "CMP $%02X%02X,X",	op_CMP_abs_x,	0xDD, 2)
table_6502("DEC", "DEC $%02X%02X,X",	op_DEC_abs_x,	0xDE, 2)
table_6502("???", "???",		op_UNK,		0xDF, 0)
table_6502("CPX", "CPX #$%02X",		op_CPX_imm,	0xE0, 1)
table_6502("SBC", "SBC ($%02X,X)",	op_SBC_ind_x,	0xE1, 0)
table_6502("???", "???",		op_UNK,		0xE2, 0)
table_6502("???", "???",		op_UNK,		0xE3, 0)
table_6502("CPX", "CPX $%02X",		op_CPX_zpage,	0xE4, 1)
table_6502("SBC", "SBC $%02X",		op_SBC_zpage,	0xE5, 1)
table_6502("INC", "INC $%02X",		op_INC_zpage,	0xE6, 1)
table_6502("???", "???",		op_UNK,		0xE7, 0)
table_6502("INX", "INX",		op_INX,		0xE8, 0)
table_6502("SBC", "SBC #$%02X",		op_SBC_imm,	0xE9, 1)
table_6502("NOP", "NOP",		op_NOP,		0xEA, 0)
table_6502("???", "???",		op_UNK,		0xEB, 0)
table_6502("CPX", "CPX $%02X%02X",	op_CPX_abs,	0xEC, 2)
table_6502("SBC", "SBC $%02X%02X",	op_SBC_abs,	0xED, 2)
table_6502("INC", "INC $%02X%02X",	op_INC_abs,	0xEE, 2)
table_6502("???", "???",		op_UNK,		0xEF, 0)
table_6502("BEQ", "BEQ $%04X (_$%02X)",	op_BEQ,		0xF0, 3)
table_6502("SBC", "SBC ($%02X),Y",	op_SBC_ind_y,	0xF1, 1)
table_6502("???", "???",		op_UNK,		0xF2, 0)
table_6502("???", "???",		op_UNK,		0xF3, 0)
table_6502("???", "???",		op_UNK,		0xF4, 0)
table_6502("SBC", "SBC $%02X,X",	op_SBC_zpage_x,	0xF5, 1)
table_6502("INC", "INC $%02X,X",	op_INC_zpage_x,	0xF6, 1)
table_6502("???", "???",		op_UNK,		0xF7, 0)
table_6502("SED", "SED",		op_SED,		0xF8, 0)
table_6502("SBC", "SBC $%02X%02X,Y",	op_SBC_abs_y,	0xF9, 2)
table_6502("???", "???",		op_UNK,		0xFA, 0)
table_6502("???", "???",		op_UNK,		0xFB, 0)
table_6502("???", "???",		op_UNK,		0xFC, 0)
table_6502("SBC", "SBC $%02X%02X,X",	op_SBC_abs_x,	0xFD, 2)
table_6502("INC", "INC $%02X%02X,X",	op_INC_abs_x,	0xFE, 2)
table_6502("???", "???",		op_UNK,		0xFF, 0)


#ifdef __ASSEMBLY__

   /* ----------------------------------------------------------------------
       6502 routines and instructions
      ---------------------------------------------------------------------- */

ENTRY(assign_6502_table)
		pushl	%edi
		pushl	%eax
		movl	$0, %edi
	1:	movl	SYMBOL_NAME(table_6502_opcodes)(,%edi,4), %eax
		movl	%eax, SYMBOL_NAME(table_opcodes)(,%edi,4)
		incl	%edi
		cmpl	$0x100, %edi
		jl	1b
		popl	%eax
		popl	%edi
		ret

   /* ----------------------------------
       ADC instructions
      ---------------------------------- */

op_ADC_imm:
		GetImm
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_imm_dec		# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_imm_dec:
		DoADC_d
		Continue

op_ADC_zpage:
		GetZPage
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_zpage_dec	# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_zpage_dec:
		DoADC_d
		Continue
op_ADC_zpage_x:
		GetZPage_X
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_zpage_x_dec	# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_zpage_x_dec:
		DoADC_d
		Continue

op_ADC_abs:
		GetAbs
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_abs_dec		# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_abs_dec:
		DoADC_d
		Continue

op_ADC_abs_x:
		GetAbs_X
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_abs_x_dec	# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_abs_x_dec:
		DoADC_d
		Continue

op_ADC_abs_y:
		GetAbs_Y
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_abs_y_dec	# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_abs_y_dec:
		DoADC_d
		Continue

op_ADC_ind_x:
		GetIndZPage_X
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_ind_x_dec	# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_ind_x_dec:
		DoADC_d
		Continue

op_ADC_ind_y:
		GetIndZPage_Y
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_ADC_ind_y_dec	# Yes, jump to decimal version
		DoADC_b
		Continue
op_ADC_ind_y_dec:
		DoADC_d
		Continue

   /* ----------------------------------
       AND instructions
      ---------------------------------- */

op_AND_imm:
		GetImm
		DoAND
		Continue

op_AND_zpage:
		GetZPage
		DoAND
		Continue

op_AND_zpage_x:
		GetZPage_X
		DoAND
		Continue

op_AND_abs:
		GetAbs
		DoAND
		Continue

op_AND_abs_x:
		GetAbs_X
		DoAND
		Continue

op_AND_abs_y:
		GetAbs_Y
		DoAND
		Continue

op_AND_ind_x:
		GetIndZPage_X
		DoAND
		Continue

op_AND_ind_y:
		GetIndZPage_Y
		DoAND
		Continue

   /* ----------------------------------
       ASL instructions
      ---------------------------------- */

op_ASL_acc:
		shlb	$1, A_Reg
		FlagNZC
		Continue

op_ASL_zpage:
		GetZPage
		DoASL
		Continue

op_ASL_zpage_x:
		GetZPage_X
		DoASL
		Continue

op_ASL_abs:
		GetAbs
		DoASL
		Continue

op_ASL_abs_x:
		GetAbs_X
		DoASL
		Continue

   /* ----------------------------------
       BCC instruction
      ---------------------------------- */

op_BCC:
		GetFromPC_B
		bt	C_Flag_Bit, FF_Reg
		jc	op_BCC_not
		cbw
		addw	%ax, PC_Reg
op_BCC_not:
		Continue

   /* ----------------------------------
       BCS instruction
      ---------------------------------- */

op_BCS:
		GetFromPC_B
		bt	C_Flag_Bit, FF_Reg
		jnc	op_BCS_not
		cbw
		addw	%ax, PC_Reg
op_BCS_not:
		Continue

   /* ----------------------------------
       BEQ instruction
      ---------------------------------- */

op_BEQ:
		GetFromPC_B
		bt	Z_Flag_Bit, FF_Reg
		jnc	op_BEQ_not
		cbw
		addw	%ax, PC_Reg
op_BEQ_not:
		Continue

   /* ----------------------------------
       BIT instructions
      ---------------------------------- */

op_BIT_zpage:
		GetZPage
		DoBIT
		Continue

op_BIT_abs:
		GetAbs
		DoBIT
		Continue

   /* ----------------------------------
       BMI instruction
      ---------------------------------- */

op_BMI:
		GetFromPC_B
		bt	N_Flag_Bit, FF_Reg
		jnc	op_BMI_not
		cbw
		addw	%ax, PC_Reg
op_BMI_not:
		Continue

   /* ----------------------------------
       BNE instruction
      ---------------------------------- */

op_BNE:
		GetFromPC_B
		bt	Z_Flag_Bit, FF_Reg
		jc	op_BNE_not
		cbw
		addw	%ax, PC_Reg
op_BNE_not:
		Continue

   /* ----------------------------------
       BPL instruction
      ---------------------------------- */

op_BPL:
		GetFromPC_B
		bt	N_Flag_Bit, FF_Reg
		jc	op_BPL_not
		cbw
		addw	%ax, PC_Reg
op_BPL_not:
		Continue

   /* ----------------------------------
       BRK instruction
      ---------------------------------- */

op_BRK:
		incw	PC_Reg
		movw	PC_Reg, %ax
		Push(%ah)
		Push(%al)
		movb	F_Reg, %al
		bts	B_Flag_Bit, %eax
		bts	X_Flag_Bit, %eax
		xorb	%ah, %ah
		movb	SYMBOL_NAME(table_encode_flags)(,%eax,1), %al
		Push(%al)
		bts	I_Flag_Bit, FF_Reg
		movw	$0xFFFE, EffectiveAddr
		call	read_memory_word	/* JMP($FFFE) */
		movw	%ax, PC_Reg
		Continue

   /* ----------------------------------
       BVC instruction
      ---------------------------------- */

op_BVC:
		GetFromPC_B
		bt	V_Flag_Bit, FF_Reg
		jc	op_BVC_not
		cbw
		addw	%ax, PC_Reg
op_BVC_not:
		Continue

   /* ----------------------------------
       BVS instruction
      ---------------------------------- */

op_BVS:
		GetFromPC_B
		bt	V_Flag_Bit, FF_Reg
		jnc	op_BVS_not
		cbw
		addw	%ax, PC_Reg
op_BVS_not:
		Continue

   /* ----------------------------------
       CLC instruction
      ---------------------------------- */

op_CLC:
		btr	C_Flag_Bit, FF_Reg
		Continue

   /* ----------------------------------
       CLD instruction
      ---------------------------------- */

op_CLD:
		btr	D_Flag_Bit, FF_Reg
		Continue

   /* ----------------------------------
       CLI instruction
      ---------------------------------- */

op_CLI:
		btr	I_Flag_Bit, FF_Reg
		Continue

   /* ----------------------------------
       CLV instruction
      ---------------------------------- */

op_CLV:
		btr	V_Flag_Bit, FF_Reg
		Continue

   /* ----------------------------------
       CMP instructions
      ---------------------------------- */

op_CMP_imm:
		GetImm
		DoCMP
		Continue

op_CMP_zpage:
		GetZPage
		DoCMP
		Continue

op_CMP_zpage_x:
		GetZPage_X
		DoCMP
		Continue

op_CMP_abs:
		GetAbs
		DoCMP
		Continue

op_CMP_abs_x:
		GetAbs_X
		DoCMP
		Continue

op_CMP_abs_y:
		GetAbs_Y
		DoCMP
		Continue

op_CMP_ind_x:
		GetIndZPage_X
		DoCMP
		Continue

op_CMP_ind_y:
		GetIndZPage_Y
		DoCMP
		Continue

   /* ----------------------------------
       CPX instructions
      ---------------------------------- */

op_CPX_imm:
		GetImm
		DoCPX
		Continue

op_CPX_zpage:
		GetZPage
		DoCPX
		Continue

op_CPX_abs:
		GetAbs
		DoCPX
		Continue

   /* ----------------------------------
       CPY instructions
      ---------------------------------- */

op_CPY_imm:
		GetImm
		DoCPY
		Continue

op_CPY_zpage:
		GetZPage
		DoCPY
		Continue

op_CPY_abs:
		GetAbs
		DoCPY
		Continue

   /* ----------------------------------
       DEC instructions
      ---------------------------------- */

op_DEC_zpage:
		GetZPage
		DoDEC
		Continue

op_DEC_zpage_x:
		GetZPage_X
		DoDEC
		Continue

op_DEC_abs:
		GetAbs
		DoDEC
		Continue

op_DEC_abs_x:
		GetAbs_X
		DoDEC
		Continue

   /* ----------------------------------
       DEX instruction
      ---------------------------------- */

op_DEX:
		decb	X_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       DEY instruction
      ---------------------------------- */

op_DEY:
		decb	Y_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       EOR instructions
      ---------------------------------- */

op_EOR_imm:
		GetImm
		DoEOR
		Continue

op_EOR_zpage:
		GetZPage
		DoEOR
		Continue

op_EOR_zpage_x:
		GetZPage_X
		DoEOR
		Continue

op_EOR_abs:
		GetAbs
		DoEOR
		Continue

op_EOR_abs_x:
		GetAbs_X
		DoEOR
		Continue

op_EOR_abs_y:
		GetAbs_Y
		DoEOR
		Continue

op_EOR_ind_x:
		GetIndZPage_X
		DoEOR
		Continue

op_EOR_ind_y:
		GetIndZPage_Y
		DoEOR
		Continue

   /* ----------------------------------
       INC instructions
      ---------------------------------- */

op_INC_zpage:
		GetZPage
		DoINC
		Continue

op_INC_zpage_x:
		GetZPage_X
		DoINC
		Continue

op_INC_abs:
		GetAbs
		DoINC
		Continue

op_INC_abs_x:
		GetAbs_X
		DoINC
		Continue

   /* ----------------------------------
       INX instruction
      ---------------------------------- */

op_INX:
		incb	X_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       INY instruction
      ---------------------------------- */

op_INY:
		incb	Y_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       JMP instructions
      ---------------------------------- */

op_JMP_abs:
		GetAbs
		DoJMP
		Continue

op_JMP_ind:
		xorl	%eax, %eax
		GetFromMem_B(PC_Reg_E)
		xchgb	%al, %ah
		cmpb	$0xFF, %ah
		je	special_case
		incw	PC_Reg
		GetFromMem_B(PC_Reg_E)
		xchgb	%al, %ah
		GetFromMem_W(%eax)
		movw	%ax, PC_Reg
		Continue
special_case:			/*?*/
		incw	PC_Reg
		subw	$0x100, PC_Reg
		GetFromMem_B(PC_Reg_E)
		xchgb	%al, %ah
		GetFromMem_W(%eax)
		movw	%ax, PC_Reg
		Continue

   /* ----------------------------------
       JSR instruction
      ---------------------------------- */

op_JSR:
		GetAbs
		DoJSR
		Continue

   /* ----------------------------------
       LDA instructions
      ---------------------------------- */

op_LDA_imm:
		GetImm
		DoLDA
		Continue

op_LDA_zpage:
		GetZPage
		DoLDA
		Continue

op_LDA_zpage_x:
		GetZPage_X
		DoLDA
		Continue

op_LDA_abs:
		GetAbs
		DoLDA
		Continue

op_LDA_abs_x:
		GetAbs_X
		DoLDA
		Continue

op_LDA_abs_y:
		GetAbs_Y
		DoLDA
		Continue

op_LDA_ind_x:
		GetIndZPage_X
		DoLDA
		Continue

op_LDA_ind_y:
		GetIndZPage_Y
		DoLDA
		Continue

   /* ----------------------------------
       LDX instructions
      ---------------------------------- */

op_LDX_imm:
		GetImm
		DoLDX
		Continue

op_LDX_zpage:
		GetZPage
		DoLDX
		Continue

op_LDX_zpage_y:
		GetZPage_Y
		DoLDX
		Continue

op_LDX_abs:
		GetAbs
		DoLDX
		Continue

op_LDX_abs_y:
		GetAbs_Y
		DoLDX
		Continue

   /* ----------------------------------
       LDY instructions
      ---------------------------------- */

op_LDY_imm:
		GetImm
		DoLDY
		Continue

op_LDY_zpage:
		GetZPage
		DoLDY
		Continue

op_LDY_zpage_x:
		GetZPage_X
		DoLDY
		Continue

op_LDY_abs:
		GetAbs
		DoLDY
		Continue

op_LDY_abs_x:
		GetAbs_X
		DoLDY
		Continue

   /* ----------------------------------
       LSR instructions
      ---------------------------------- */

op_LSR_acc:
		shrb	$1, A_Reg
		FlagNZC
		Continue

op_LSR_zpage:
		GetZPage
		DoLSR
		Continue

op_LSR_zpage_x:
		GetZPage_X
		DoLSR
		Continue

op_LSR_abs:
		GetAbs
		DoLSR
		Continue

op_LSR_abs_x:
		GetAbs_X
		DoLSR
		Continue

   /* ----------------------------------
       NOP instruction
      ---------------------------------- */

op_NOP:
		Continue

   /* ----------------------------------
       ORA instructions
      ---------------------------------- */

op_ORA_imm:
		GetImm
		DoORA
		Continue

op_ORA_zpage:
		GetZPage
		DoORA
		Continue

op_ORA_zpage_x:
		GetZPage_X
		DoORA
		Continue

op_ORA_abs:
		GetAbs
		DoORA
		Continue

op_ORA_abs_x:
		GetAbs_X
		DoORA
		Continue

op_ORA_abs_y:
		GetAbs_Y
		DoORA
		Continue

op_ORA_ind_x:
		GetIndZPage_X
		DoORA
		Continue

op_ORA_ind_y:
		GetIndZPage_Y
		DoORA
		Continue

   /* ----------------------------------
       PHA instruction
      ---------------------------------- */

op_PHA:
		Push(A_Reg)
		Continue

   /* ----------------------------------
       PHP instruction
      ---------------------------------- */

op_PHP:
		movb	F_Reg, %al
		bts	X_Flag_Bit, %eax
		xorb	%ah, %ah
		movb	SYMBOL_NAME(table_encode_flags)(,%eax,1), %al
		Push(%al)
		Continue

   /* ----------------------------------
       PLA instruction
      ---------------------------------- */

op_PLA:
		Pop(A_Reg)
		orb	A_Reg, A_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       PLP instruction
      ---------------------------------- */
		
op_PLP:
		Pop(%al)
		btr	X_Flag_Bit, %eax
		xorb	%ah, %ah
		movb	SYMBOL_NAME(table_decode_flags)(,%eax,1), F_Reg
		Continue

   /* ----------------------------------
       ROL instructions
      ---------------------------------- */

op_ROL_acc:
		bt	C_Flag_Bit, FF_Reg
		rclb	$1, A_Reg
		jc	op_ROL_acc_carry
		orb	A_Reg, A_Reg
		clc
		FlagNZC
		Continue
op_ROL_acc_carry:
		orb	A_Reg, A_Reg
		stc
		FlagNZC
		Continue

op_ROL_zpage:
		GetZPage
		DoROL(rol_zpage, Continue)

op_ROL_zpage_x:
		GetZPage_X
		DoROL(rol_zpage_x, Continue)

op_ROL_abs:
		GetAbs
		DoROL(rol_abs, Continue)

op_ROL_abs_x:
		GetAbs_X
		DoROL(rol_abs_x, Continue)

   /* ----------------------------------
       ROR instructions
      ---------------------------------- */

op_ROR_acc:
		bt	C_Flag_Bit, FF_Reg
		rcrb	$1, A_Reg
		jc	op_ROR_acc_carry
		orb	A_Reg, A_Reg
		clc
		FlagNZC
		Continue
op_ROR_acc_carry:
		orb	A_Reg, A_Reg
		stc
		FlagNZC
		Continue

op_ROR_zpage:
		GetZPage
		DoROR(ror_zpage, Continue)

op_ROR_zpage_x:
		GetZPage_X
		DoROR(ror_zpage_x, Continue)

op_ROR_abs:
		GetAbs
		DoROR(ror_abs, Continue)

op_ROR_abs_x:
		GetAbs_X
		DoROR(ror_abs_x, Continue)

   /* ----------------------------------
       RTI instruction
      ---------------------------------- */

op_RTI:
		Pop(%al)
		btr	X_Flag_Bit, %eax
		xorb	%ah, %ah
		movb	SYMBOL_NAME(table_decode_flags)(,%eax,1), %al
		movb	%al, F_Reg
		Pop(%al)
		Pop(%ah)
		movw	%ax, PC_Reg
		Continue

   /* ----------------------------------
       RTS instruction
      ---------------------------------- */

op_RTS:
		Pop(%al)
		Pop(%ah)
		incw	%ax
		movw	%ax, PC_Reg
		Continue

   /* ----------------------------------
       SBC instructions
      ---------------------------------- */

op_SBC_imm:
		GetImm
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_imm_dec		# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_imm_dec:
		DoSBC_d
		Continue

op_SBC_zpage:
		GetZPage
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_zpage_dec	# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_zpage_dec:
		DoSBC_d
		Continue

op_SBC_zpage_x:
		GetZPage_X
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_zpage_x_dec	# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_zpage_x_dec:
		DoSBC_d
		Continue

op_SBC_abs:
		GetAbs
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_abs_dec		# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_abs_dec:
		DoSBC_d
		Continue

op_SBC_abs_x:
		GetAbs_X
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_abs_x_dec	# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_abs_x_dec:
		DoSBC_d
		Continue

op_SBC_abs_y:
		GetAbs_Y
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_abs_y_dec	# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_abs_y_dec:
		DoSBC_d
		Continue

op_SBC_ind_x:
		GetIndZPage_X
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_ind_x_dec	# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_ind_x_dec:
		DoSBC_d
		Continue

op_SBC_ind_y:
		GetIndZPage_Y
		bt	D_Flag_Bit, FF_Reg	# Decimal mode?
		jc	op_SBC_ind_y_dec	# Yes, jump to decimal version
		DoSBC_b
		Continue
op_SBC_ind_y_dec:
		DoSBC_d
		Continue

   /* ----------------------------------
       SEC instruction
      ---------------------------------- */

op_SEC:
		bts	C_Flag_Bit, FF_Reg
		Continue

   /* ----------------------------------
       SED instruction
      ---------------------------------- */

op_SED:
		bts	D_Flag_Bit, FF_Reg
		Continue

   /* ----------------------------------
       SEI instruction
      ---------------------------------- */

op_SEI:
		bts	I_Flag_Bit, FF_Reg
		Continue

   /* ----------------------------------
       STA instructions
      ---------------------------------- */

op_STA_imm:
		GetImm
		DoSTA
		Continue

op_STA_zpage:
		GetZPage
		DoSTA
		Continue

op_STA_zpage_x:
		GetZPage_X
		DoSTA
		Continue

op_STA_abs:
		GetAbs
		DoSTA
		Continue

op_STA_abs_x:
		GetAbs_X
		DoSTA
		Continue

op_STA_abs_y:
		GetAbs_Y
		DoSTA
		Continue

op_STA_ind_x:
		GetIndZPage_X
		DoSTA
		Continue

op_STA_ind_y:
		GetIndZPage_Y
		DoSTA
		Continue

   /* ----------------------------------
       STX instructions
      ---------------------------------- */

op_STX_zpage:
		GetZPage
		DoSTX
		Continue

op_STX_zpage_y:
		GetZPage_Y
		DoSTX
		Continue

op_STX_abs:
		GetAbs
		DoSTX
		Continue

   /* ----------------------------------
       STY instructions
      ---------------------------------- */

op_STY_zpage:
		GetZPage
		DoSTY
		Continue

op_STY_zpage_x:
		GetZPage_X
		DoSTY
		Continue

op_STY_abs:
		GetAbs
		DoSTY
		Continue

   /* ----------------------------------
       TAX instruction
      ---------------------------------- */

op_TAX:
		movb	A_Reg, X_Reg
		orb	X_Reg, X_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       TAY instruction
      ---------------------------------- */

op_TAY:
		movb	A_Reg, Y_Reg
		orb	Y_Reg, Y_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       TSX instruction
      ---------------------------------- */

op_TSX:
		movb	SP_Reg_L, X_Reg
		orb	X_Reg, X_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       TXA instruction
      ---------------------------------- */

op_TXA:
		movb	X_Reg, A_Reg
		orb	A_Reg, A_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       TXS instruction
      ---------------------------------- */

op_TXS:
		movb	X_Reg, SP_Reg_L
		Continue

   /* ----------------------------------
       TYA instruction
      ---------------------------------- */

op_TYA:
		movb	Y_Reg, A_Reg
		orb	A_Reg, A_Reg
		FlagNZ
		Continue

   /* ----------------------------------
       ??? instruction
      ---------------------------------- */

op_UNK:
		Continue


#endif	/* __ASSEMBLY__ */
