# =============================================================================
#
# File:		util.tcl
# Project:	TkDesk
#
# Started:	11.10.94
# Changed:	11.10.94
# Author:	cb
#
# Description:	Misc. utility procs.
#
# Copyright (C) 1996  Christian Bolik
# 
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
# 
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
# 
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
# See the file "COPYING" in the base directory of this distribution
# for more.
#
# -----------------------------------------------------------------------------
#
# Sections:
#	proc dsk_debug {str}
#	proc dsk_busy {}
#	proc dsk_lazy {}
#	proc dsk_status {str}
#	proc dsk_logname {}
#
# =============================================================================

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_debug
# Args:		args		Optional -nonewline and string to print.
# Returns: 	""
# Desc:		Prints an arbitrary string on stderr if tkdesk(debug) is != 0.
# Side-FX:	none
#

proc dsk_debug {args} {
    global tkdesk

    if !$tkdesk(debug) return

    if {[llength $args] < 1} {
	error "too few arguments to dsk_debug"
    }

    set str [join $args]
    set nonewline 0
    if [string match "-nonew*" [lindex $args 0]] {
	set nonewline 1
	set str [lreplace $str 0 0]
    }

    if $nonewline {
    	puts -nonewline stderr $str
    } else {
    	puts stderr $str
    }
    return
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_busy
# Args:		what		(opt.) list of busy itcl classes 
# Returns: 	""
# Desc:		Displays the cursor in busy state.
# Side-FX:	none
#

set dsk_busy(still_busy) 0

proc dsk_busy {{what ""}} {
    global cb_tools tkdesk dsk_busy

    incr dsk_busy(still_busy)
    
    #if {$tkdesk(status) == "busy"} {
	#return
    #}
    set tkdesk(status) busy

    if {$what != ""} {
    	foreach class $what {
	    foreach obj [itcl_info objects -class $class] {
	    	catch "blt_busy hold $obj \
			-cursor \"@$cb_tools(path)/bitmaps/timer.xbm \
			$cb_tools(path)/bitmaps/timer.mask.xbm black white\" "
	    }
    	}
    } else {
    	foreach class {dsk_FileViewer dsk_FileList dsk_DiskUsage dsk_FileInfo
		dsk_Periodic dsk_Editor} {
	    foreach obj [itcl_info objects -class $class] {
	    	catch "blt_busy hold $obj \
			-cursor \"@$cb_tools(path)/bitmaps/timer.xbm \
			$cb_tools(path)/bitmaps/timer.mask.xbm black white\" "
	    }
    	}
    	foreach w [winfo children .] {
	    if {[winfo class $w] == "Toplevel"} {
	    	catch "blt_busy hold $w -cursor watch \
			-cursor \"@$cb_tools(path)/bitmaps/timer.xbm \
			$cb_tools(path)/bitmaps/timer.mask.xbm black white\" "
	    }
    	}
    }

    update idletasks
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_lazy
# Args:		what		(opt.) list of now lazy itcl classes 
# Returns: 	""
# Desc:		Displays the cursor in normal state.
# Side-FX:	none
#

proc dsk_lazy {{what ""} {force 0}} {
    global tkdesk dsk_busy

    if {$tkdesk(status) == "lazy"} {
	return
    }

    if $force {
	set dsk_busy(still_busy) 0
    } else {
	incr dsk_busy(still_busy) -1
	if {$dsk_busy(still_busy) > 0} {
	    return
	}
    }
    
    set tkdesk(status) lazy

    update idletasks

    if {$what != ""} {
    	foreach class $what {
	    foreach obj [itcl_info objects -class $class] {
	    	catch "blt_busy release $obj"
	    }
    	}
    } else {
    	foreach class {dsk_FileViewer dsk_FileList dsk_DiskUsage dsk_FileInfo
		dsk_Periodic dsk_Editor} {
	    foreach obj [itcl_info objects -class $class] {
	    	catch "blt_busy release $obj"
	    }
    	}
    	foreach w [winfo children .] {
	    if {[winfo class $w] == "Toplevel"} {
	    	catch "blt_busy release $w"
	    }
    	}
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_status
# Args:		str	string to display
# Returns: 	""
# Desc:		Displays $str in the status bar of all file viewers.
# Side-FX:	none
#

proc dsk_status {str} {

    foreach fv [itcl_info objects -class dsk_FileViewer] {
	$fv status $str
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_fs_status
# Args:		file - opt. name of file
# Returns: 	A string that contains the available disk space.
# Desc:		...
# Side-FX:	none
#

proc dsk_fs_status {{file ""}} {
    global tkdesk

    if {$file == ""} {
	set file [$tkdesk(active_viewer) curdir]
    }

    ot_maplist [dsk_statfs $file] cap used avail
    if {$cap > 0} {
	return "[format %.1f [expr $avail./1000]] MB available."
    } else {
	return ""
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_logname
# Args:		none
# Returns: 	The user's login name.
# Desc:		
# Side-FX:	none
#

proc dsk_logname {} {
    global tkdesk

    return [exec $tkdesk(cmd,whoami)]
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dd_handle_text
# Args:		w		widget name
#		replace 	(opt.) set to 1 if ex. text should be deleted
# Returns: 	""
# Desc:		Drop handler for entry and text widgets.
# Side-FX:	none
#

proc dd_handle_text {w {replace 0}} {
    global DragDrop

    if !$replace {
    	$w insert insert $DragDrop(text)
    } else {
	$w delete 0 end
	$w insert end $DragDrop(text)
    }
}

#
# -----------------------------------------------------------------------------
#
# Proc:		_make_path_valid
# Args:		path		path to shorten to a valid path
#		fallback	fallback path if $path is totally broken
# Returns: 	a valid path
# Desc:		see above
# Side-FX:	none
#

proc _make_path_valid {path {fallback "/"}} {
    global tkdesk

    catch "set path \[cb_tilde $path collapse\]"
    if {[string index $path 0] != "/"} {
	set path [$tkdesk(active_viewer) curdir]/$path
    }

    if [file isdirectory $path] {
	return $path
    }

    set vpath ""
    for {set i 1} {$i < [string length $path]} {incr i} {
	if {[string index $path $i] == "/"} {
	    set tpath [string range $path 0 [expr $i - 1]]
	    if [file isdirectory $tpath] {
		set vpath $tpath
	    } else {
		break
	    }
	}
    }

    if {$vpath == ""} {
	set vpath $fallback
    }

    return $vpath
}

#
# -----------------------------------------------------------------------------
#
# Proc:		dsk_maplist
# Args:		list - a tcl list
#               args - a list of variable name that will be set to the elements
#                      of $list
# Returns: 	""
# Desc:		see Args
# Side-FX:	none
#

proc dsk_maplist {list args} {

    set i 0
    foreach var $args {
	upvar $var v
	set v [lindex $list $i]
	incr i
    }

    return ""
}

