------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ A G G R                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.69 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Expander; use Expander;
with Features; use Features;
with Itypes;   use Itypes;
with Namet;    use Namet;
with Nmake;    use Nmake;
with Nlists;   use Nlists;
with Opt;      use Opt;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch5;  use Sem_Ch5;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sem_Type; use Sem_Type;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stringt;  use Stringt;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Uintp;    use Uintp;

with System.Parameters;

package body Sem_Aggr is

   ------------------------------------------------------
   -- Subprogram Specs for RECORD AGGREGATE Processing --
   ------------------------------------------------------

   procedure Resolve_Record_Aggregate (N : Node_Id; Typ : Entity_Id);
   --  This procedure performs all the semantic checks required for record
   --  aggregates. This procedure assumes that the others choice is by
   --  itself and appears last in the aggregate, if it is present. This
   --  test was previously done is Resolve_Aggregate.
   --
   --    N is the N_Aggregate node.
   --    Typ is the record type for the aggregate resolution
   --
   --  While performing the semantic checks, this procedure
   --  builds a new Component_Association_List where each record field
   --  appears alone in a Component_Choice_List along with its corresponding
   --  expression. The record fields in the Component_Association_List
   --  appear in the same order in which they appear in the record type Typ.
   --
   --  Once this new Component_Association_List is built and all the
   --  semantic checks performed, the original aggregate subtree is replaced
   --  with the new named record aggregate just built. Note that the subtree
   --  substitution is performed with Rewrite_Substitute_Tree so as to be
   --  able to retrieve the original aggregate.
   --
   --  The aggregate subtree manipulation performed by Resolve_Record_Aggregate
   --  yields the aggregate format expected by Gigi. Typically, this kind of
   --  tree manipulations are done in the expander. However, because the
   --  semantic checks that need to be performed on record aggregates really
   --  go hand in hand with the record aggreagate normalization, the aggregate
   --  subtree transformation is performed during resolution rather than
   --  expansion. Had we decided otherwise we would have had to duplicate
   --  most of the code in the expansion procedure Expand_Record_Aggregate.
   --  Note, however, that all the expansion concerning aggegates for tagged
   --  records is done in Expand_Record_Aggregate.
   --
   --  The algorithm of Resolve_Record_Aggregate proceeds as follows:
   --
   --  1. Make sure that the record type against which the record aggregate
   --     has to be resolved is not abstract. Furthermore if the type is
   --     a null aggregate make sure the input aggregate N is also null.
   --
   --  2. Verify that the structure of the aggregate is that of a record
   --     aggregate. Specifically, look for component associations and ensure
   --     that each choice list only has identifiers or the N_Others_Choice
   --     node. Note that at this point Analyze_Aggregate has already made
   --     sure that the N_Others_Choice occurs last and by itself.
   --
   --  3. If Typ contains discriminants, the values for each discriminant
   --     is looked for. If the record type Typ has variants, we check
   --     that the expressions corresponding to each discriminant ruling
   --     the (possibly nested) variant parts of Typ, are static. This
   --     allows us to determine the variant parts to which the rest of
   --     the aggregate must conform. The names of discriminants with their
   --     values are saved in a new association list, New_Assoc_List which
   --     is later augmented with the names and values of the remaining
   --     components in the record type.
   --
   --     During this phase we also make sure that every discriminant is
   --     assigned exactly one value. Note that when several values
   --     for a given discriminant are found, semantic processing continues
   --     looking for further errors. In this case it's the first
   --     discriminant value found which we will be recorded.
   --
   --     IMPORTANT NOTE: For derived tagged types this procedure expects
   --     First_Discriminant and Next_Discriminant to give the correct list
   --     of discriminants, in the correct order.
   --
   --  4. After all the discriminant values have been gathered, we can
   --     set the Etype of the record aggregate. If Typ contains no
   --     discriminants this is straightforward: the Etype of N is just
   --     Typ, otherwise a new implicit constrained subtype of Typ is
   --     built to be the Etype of N.
   --
   --  5. Gather the remaining record components according to the discriminant
   --     values. This involves recursively traversing the record type
   --     structure to see what variants are selected by the given discriminant
   --     values. This processing is a little more convoluted if Typ is a
   --     derived tagged types since we need to retrieve the record structure
   --     of all the ancestors of Typ.
   --
   --  6. After gathering the record components we look for their values
   --     in the record aggregate and emit appropriate error messages
   --     should we not find such values or should they be duplicated.
   --
   --  7. We then make sure no illegal component names appear in the
   --     record aggegate and make sure that the type of the record
   --     components appearing in a same choice list is the same.
   --     Finally we ensure that the others choice, if present, is
   --     used to provide the value of at least a record component.
   --
   --  8. The original aggregate node is replaced with the new named
   --     aggregate built in steps 3 through 6, as explained earlier.
   --
   --  Given the complexity of record aggregate resolution, the primary
   --  goal of this routine is clarity and simplicity rather than execution
   --  and storage efficiency. If there are only positional components in the
   --  aggregate the running time is linear. If there are associations
   --  the running time is still linear as long as the order of the
   --  associations is not too far off the order of the components in the
   --  record type. If this is not the case the running time is at worst
   --  quadratic in the size of the association list.

   procedure Gather_Components
     (Comp_List     : Node_Id;
      Governed_By   : List_Id;
      Into          : Elist_Id;
      Report_Errors : out Boolean);
   --  The purpose of this procedure is to gather the valid components
   --  in a record type according to the values of its discriminants.
   --  Specifically:
   --
   --    Comp_List is an N_Component_List node.
   --
   --    Governed_By is a list of N_Component_Association nodes,
   --     where each choice list contains the name of a discriminant and
   --     the expression field gives its value. The values of the
   --     discriminants governing the (possibly nested) variant parts in
   --     Comp_List are found in this Component_Association List.
   --
   --    Into is the list where the valid components are appended.
   --     Note that Into need not be an Empty list. If it's not, components
   --     are attached to its tail.
   --
   --    Report_Errors is set to True if the values of the discriminants
   --     are non-static.

   -----------------------------------------------------
   -- Subprogram Specs for ARRAY AGGREGATE Processing --
   -----------------------------------------------------

   procedure Resolve_Array_Aggregate
     (N              : Node_Id;
      Index          : Node_Id;
      Component_Typ  : Entity_Id;
      Others_Allowed : Boolean);
   --  This procedure performs the semantic checks for an array aggregate.
   --  The procedure works by recursively checking each nested aggregate.
   --  Specifically, after checking a sub-aggreate nested at the i-th level
   --  we recursively check all the subaggregates at the i+1-st level
   --  (if any).
   --
   --    N is the current N_Aggregate node to be checked.
   --
   --    Index is the index node (not type) corresponding to the array
   --    sub-aggregate that we are currently checking. Subsequent indices
   --    are obtained from Index using Next_Index.
   --    To be more specific two concepts are important when checking
   --    aggregates:
   --
   --      * The applicable index constraint.
   --        It "is a constraint provided by certain contexts [...] that
   --        can be used to determine the bounds of the array value
   --        specified by the aggregate" (ARM 4.3.3 (10)).
   --        If Others_Allowed is False there is no applicable index
   --        constraint. Otherwise the lower and upper bounds of the constraint
   --        are obtained by invoking Get_Index_Bounds on Index.
   --
   --      * The index subtype, i.e. the subtype of the index corresponding to
   --        the array sub-aggregate.
   --        This subtype is the Base_Type (Etype (Index)).
   --
   --    Component_Typ is the array component type.
   --
   --    Others_Allowed indicates whether an others choice is allowed
   --    in the context where the top-level aggregate appeared.
   --
   --  The algorithm of Resolve_Array_Aggregate proceeds as follows:
   --
   --  1. Make sure that the others choice, if present, is by itself
   --     and appears last in the aggregate. This test has already been
   --     performed during the analysis phase. If the aggregate format was
   --     found to be incorrect during analysis, the Etype of the aggregate
   --     is set to Any_Type. Also check that we do not have positional and
   --     named components in the array aggregate (unless the named
   --     association is an others choice). Finally if an others choice is
   --     present, make sure it is allowed in the aggregate contex.
   --
   --  2. Verify the validity of the discrete_choices inside the aggregate if
   --     the aggregate contains named components. Specifically verify that:
   --
   --     (a) If a null range is present it must be the only possible choice
   --         in the array aggregate.
   --
   --     (b) Ditto for a non static range.
   --
   --     (c) Ditto for a non static expression.
   --
   --     In addition this step analyzes and resolves each discrete_choice,
   --     making sure that its type is the type of the corresponding Index.
   --     If we are not at the lowest array aggregate level (in the case of
   --     multi-dimensional aggregates) then invoke Resolve_Array_Aggregate
   --     recursively on each component expression. Otherwise resolve
   --     the bottom level component expressions against the expected
   --     component type.
   --
   --     Finally, we examine the aggregate and index constraints to
   --     determine whether any constraint errors can be raised at run-time.
   --     If this is the case, we emit warning messages (not errors) and
   --     replace the aggregate with the statement raise Constraint_Error.
   --     The precise checks are the following:
   --
   --     (a) Check that the index range defined by aggregate bounds is
   --         compatible with corresponding index subtype.
   --
   --     (b) If an others choice is present check that no aggregate index is
   --         outside the bounds of the index constraint
   --
   --     (c) Check that the others choice specifies at least one value
   --         (if it specifies zero, that is ok, but emit a gentle warning,
   --         if it specifies less than zero still a warning but warn about
   --         constraint error being raises at run time).
   --
   --     (d) If there is no others choice but the context provides the bounds
   --         of the aggregate, check that aggregate length matches that of
   --         the index constraint
   --
   --  3. For positional aggregates we loop over the component expressions
   --     either recursively invoking Resolve_Array_Aggregate on each of these
   --     for multi-dimensional array aggregates or resolving the bottom level
   --     component expressions against the expected component type.
   --
   --     Like in the named aggregate case, we examine the aggregate and index
   --     constraints to determine whether any constraint errors can be
   --     raised at run-time. If this is the case, we emit warning messages
   --     and replace the aggregate with a raise Constraint_Error.
   --
   --  ??? Note that we do not perform any static checks for the equality of
   --  bounds of sub-aggregates that correspond to a same index.

   procedure Make_String_Into_Aggregate (N : Node_Id);
   --  A string literal can appear in  a context in  which a one dimensional
   --  array of characters is expected. This procedure simply rewrites the
   --  string as an aggregate, prior to resolution.

   -----------------------
   -- Resolve_Aggregate --
   -----------------------

   procedure Resolve_Aggregate (N : Node_Id; Typ : Entity_Id) is
      Pkind : constant Node_Kind := Nkind (Parent (N));

   begin
      --  Make sure that the others choice is by itself and appears
      --  last in the aggregate, if it's present. This test is actually
      --  not performed here. The code for this is in Analyze_Aggregate.
      --  What Analyze_Aggregate does is to set the Etype of the record
      --  aggregate to be Any_Type to signal that there was a problem
      --  with an others choice.

      if Etype (N) = Any_Type then
         Set_Etype (N, Any_Composite);

      elsif Is_Limited_Type (Typ) then
         Error_Msg_N ("aggregate type cannot be limited", N);

      elsif Is_Class_Wide_Type (Typ) then
         Error_Msg_N ("aggregate cannot be of a class-wide type", N);

      elsif Is_Record_Type (Typ) then
         Resolve_Record_Aggregate (N, Typ);

      elsif Is_Array_Type (Typ) then
         --  In the following we determine whether an others choice is allowed
         --  inside the array aggregate. The test checks the context
         --  in which the array aggregate occurs. If the context does not
         --  permit it, or the aggregate type is unconstrained, an others
         --  choice is not allowed.
         --
         --  Note that there is no node for Explicit_Actual_Parameter.
         --  To test for this context we therefore have to test for node
         --  N_Parameter_Association which itself appears only if there is a
         --  formal parameter. Consequently we also need to test for
         --  N_Procedure_Call_Statement or N_Function_Call.

         if Is_Constrained (Typ) and then
           (Pkind = N_Assignment_Statement      or else
            Pkind = N_Parameter_Association     or else
            Pkind = N_Function_Call             or else
            Pkind = N_Procedure_Call_Statement  or else
            Pkind = N_Generic_Association       or else
            Pkind = N_Formal_Object_Declaration or else
            Pkind = N_Return_Statement          or else
            Pkind = N_Object_Declaration        or else
            Pkind = N_Component_Declaration     or else
            Pkind = N_Parameter_Specification   or else
            Pkind = N_Qualified_Expression      or else
            Pkind = N_Aggregate                 or else
            Pkind = N_Component_Association)
         then
            Resolve_Array_Aggregate
              (N,
               First_Index (Typ),
               Component_Type (Typ),
               Others_Allowed => True);

         else
            Resolve_Array_Aggregate
              (N,
               First_Index (Typ),
               Component_Type (Typ),
               Others_Allowed => False);
         end if;

         --  Aggregate type is the constrained type imposed by the context.

         Set_Etype (N, Typ);

      else
         Error_Msg_N ("illegal context for aggregate", N);

      end if;
   end Resolve_Aggregate;

   ---------------------------------
   -- Resolve_Extension_Aggregate --
   ---------------------------------

   --  There are two cases to consider:

   --  a) If the ancestor part is a type mark, the components needed are
   --  the difference between the components of the expected type and the
   --  components of the given type mark.

   --  b) If the ancestor part is an expression, it must be unambiguous,
   --  and once we have its type we can also compute the needed  components
   --  as in the previous case. In both cases, if the ancestor type is not
   --  the immediate ancestor, we have to build this ancestor recursively.

   --  In both cases discriminants of the ancestor type do not play a
   --  role in the resolution of the needed components, because inherited
   --  discriminants cannot be used in a type extension. As a result we can
   --  compute independently the list of components of the ancestor type and
   --  of the expected type.

   procedure Resolve_Extension_Aggregate (N : Node_Id; Typ : Entity_Id) is
      A        :  constant Node_Id := Ancestor_Part (N);
      A_Type   : Entity_Id;
      I        : Interp_Index;
      It       : Interp;
      Imm_Type : Entity_Id;

      function Valid_Ancestor_Type return Boolean;
      --  Verify that the type of the ancestor part is a non-private ancestor
      --  of the expected type.

      function Valid_Ancestor_Type return Boolean is
         Imm_Type : Entity_Id := Base_Type (Typ);
      begin
         while Is_Derived_Type (Imm_Type)
           and then Etype (Imm_Type) /= Base_Type (A_Type)
         loop
            Imm_Type := Etype (Base_Type (Imm_Type));
         end loop;

         if Etype (Imm_Type) /= Base_Type (A_Type) then
            Error_Msg_NE ("expect ancestor type of &", A, Typ);
            return false;
         else
            return true;
         end if;
      end Valid_Ancestor_Type;

   begin
      if not Is_Tagged_Type (Typ) then
         Error_Msg_N ("type of extension aggregate must be tagged", N);
         return;

      elsif Is_Limited_Type (Typ) then
         Error_Msg_N ("aggregate type cannot be limited", N);
         return;

      elsif Is_Class_Wide_Type (Typ) then
         Error_Msg_N ("aggregate cannot be of a class-wide type", N);
         return;
      end if;

      if Is_Entity_Name (A)
        and then Is_Type (Entity (A))
      then
         A_Type   := Entity (A);
         Imm_Type := Base_Type (Typ);

         if Valid_Ancestor_Type then
            Resolve_Record_Aggregate (N, Typ);
         end if;

      elsif Nkind (A) /= N_Aggregate then
         if Is_Overloaded (A) then
            A_Type := Any_Type;
            Get_First_Interp (A, I, It);

            while Present (It.Typ) loop

               if Is_Tagged_Type (It.Typ)
                  and then not Is_Limited_Type (It.Typ)
               then
                  if A_Type /= Any_Type then
                     Error_Msg_N ("cannot resolve expression", A);
                     return;
                  else
                     A_Type := It.Typ;
                  end if;
               end if;

               Get_Next_Interp (I, It);
            end loop;

            if A_Type = Any_Type then
               Error_Msg_N
                 ("ancestor part must be non-limited tagged type", A);
               return;
            end if;

         else
            A_Type := Etype (A);
         end if;

         if Valid_Ancestor_Type then
            Resolve_Record_Aggregate (N, Typ);
         end if;

      else
         Error_Msg_N (" No unique type for this aggregate",  A);
      end if;

   end Resolve_Extension_Aggregate;

   ------------------------------
   -- Resolve_Record_Aggregate --
   ------------------------------

   procedure Resolve_Record_Aggregate (N : Node_Id; Typ : Entity_Id) is
      Expr            : Node_Id;
      Record_Def      : Node_Id;
      Positional_Expr : Node_Id;

      Assoc : Node_Id;
      --  N_Component_Association node belonging to the input aggregate N

      New_Assoc_List  : List_Id := New_List;
      New_Assoc       : Node_Id;
      --  New_Assoc_List is the newly built list of N_Component_Association
      --  nodes. New_Assoc is one such N_Component_Association node in it.
      --  Please note that while Assoc and New_Assoc contain the same
      --  kind of nodes, they are used to iterate over two different
      --  N_Component_Association lists.

      New_Aggregate  : Node_Id := New_Copy (N);
      Component      : Entity_Id;
      Component_Elmt : Elmt_Id;
      Components     : Elist_Id := New_Elmt_List;
      --  Components is the list of the record components whose value must
      --  be provided in the aggregate. This list does include discriminants.

      Next_Expr : Node_Id;
      Others_Etype : Entity_Id := Empty;
      --  This variable is used to save the Etype of the last record component
      --  that takes its value from the others choice. Its purpose is:
      --
      --    (a) make sure the others choice is useful
      --
      --    (b) make sure the type of all the components whose value is
      --        subsumed by the others choice are the same.
      --
      --  This variable is updated as a side effect of function Get_Value

      procedure Add_Association (Component : Entity_Id; Expr : Node_Id);
      --  Builds a new N_Component_Association node which associates
      --  Component to expression Expr and adds it to the new association
      --  list New_Assoc_List being built.

      function Get_Value
        (Compon                 : Node_Id;
         From                   : List_Id;
         Consider_Others_Choice : Boolean := False)
         return Node_Id;
      --  Given a record component stored in parameter Compon, the
      --  following function returns its value as it appears in the list
      --  From, which is a list of N_Component_Association nodes. If no
      --  component association has a choice for the searched component,
      --  the value provided by the others choice is returned, if there
      --  is  one and Consider_Others_Choice is set to true. Otherwise
      --  Empty is returned. If there is more than one component association
      --  giving a value for the searched record component, an error message
      --  is emitted and the first found value is returned.
      --
      --  If Consider_Others_Choice is set and the returned expression comes
      --  from the others choice, then Others_Etype is set as a side effect.
      --  An error message is emitted if the components taking their value
      --  from the others choice do not have same type.

      function Replace_Discriminants (In_Type : Entity_Id) return Entity_Id;
      --  In_Type is a type or subtype. If In_Type is a record subtype or an
      --  array subtype, a new Itype is created and returned. This Itype is
      --  attached to the aggregate node N. It is a copy of In_Type except
      --  that every occurrence of a record discriminant of the original
      --  record aggregate type Typ is replaced with its corresponding value
      --  as given by the record aggregate.
      --
      --  Note that the Itype is created only if there is at least one such
      --  discriminant in subtype In_Type. Otherwise In_Type is returned.
      --  For example consider the following code:
      --
      --    type rec (D : integer) is record
      --       F : String (1..D);
      --    end record;
      --
      --    X : rec := (D => 3, "abc");
      --
      --  When analyzing the record aggregate, the Etype initially found for
      --  field F is String (1 .. D). However the Etype of "abc" must be
      --  String  (1..3). Replace_Discriminants carries out precisely this
      --  transformation, i.e. in this case
      --
      --    Replace_Discriminants (String (1..D))
      --
      --  returns
      --
      --    String (1..3).
      --
      --  Note that this function begins creating an Itype, before it knows
      --  whether it will be useful or not. If the Itype is not needed, that
      --  storage is wasted.

      ---------------------
      -- Add_Association --
      ---------------------

      procedure Add_Association (Component : Entity_Id; Expr : Node_Id) is
         New_Assoc   : Node_Id;
         Choice_List : List_Id := New_List;

      begin
         Append (New_Occurrence_Of (Component, Sloc (Expr)), Choice_List);
         New_Assoc :=
           Make_Component_Association (Sloc (Expr),
             Choices    => Choice_List,
             Expression => Expr);
         Append (New_Assoc, New_Assoc_List);
      end Add_Association;

      ---------------
      -- Get_Value --
      ---------------

      function Get_Value
        (Compon                 : Node_Id;
         From                   : List_Id;
         Consider_Others_Choice : Boolean := False)
         return                   Node_Id
      is
         Assoc         : Node_Id;
         Expr          : Node_Id := Empty;
         Selector_Name : Node_Id;
         New_Expr      : Node_Id;

      begin
         if Present (From) then
            Assoc := First (From);
         else
            return Empty;
         end if;

         while Present (Assoc) loop
            Selector_Name := First (Choices (Assoc));
            while Present (Selector_Name) loop
               if Nkind (Selector_Name) = N_Others_Choice then
                  if Consider_Others_Choice and then No (Expr) then
                     if Present (Others_Etype) and then
                        Base_Type (Others_Etype) /= Base_Type (Etype (Compon))
                     then
                        Error_Msg_N ("components in OTHERS choice must " &
                                     "have same type", Selector_Name);
                     end if;

                     Others_Etype := Etype (Compon);
                     New_Expr := New_Copy_Tree (Expression (Assoc));
                     Save_Interps (Expression (Assoc), New_Expr);
                     return New_Expr;
                  end if;

               elsif Chars (Compon) = Chars (Selector_Name) then
                  if No (Expr) then
                     Expr := Expression (Assoc);
                  else
                     Error_Msg_NE
                       ("more than one value supplied for &",
                        Selector_Name, Compon);
                  end if;
               end if;

               Selector_Name := Next (Selector_Name);
            end loop;
            Assoc := Next (Assoc);
         end loop;

         return Expr;
      end Get_Value;

      ---------------------------
      -- Replace_Discriminants --
      ---------------------------

      function Replace_Discriminants (In_Type : Entity_Id) return Entity_Id is
         New_Expr_List      : Elist_Id;
         Discrim_Constraint : Elmt_Id;

         Index_Type     : Entity_Id;
         Old_Index      : Node_Id;
         New_Index      : Node_Id;
         New_Index_List : List_Id;

         Old_Expr  : Node_Id;
         New_Expr  : Node_Id;
         Low_Expr  : Node_Id;
         High_Expr : Node_Id;

         Itype  : Entity_Id;
         Need_To_Create_Itype : Boolean := False;

      begin
         --  In the following code N refers to the input aggregate node and Typ
         --  its type. These are the parameters of Resolve_Record_Aggregate.

         if not Has_Discriminants (Typ)
           or else (Ekind (In_Type) /= E_Record_Subtype
                      and then
                    Ekind (In_Type) /= E_Array_Subtype)
         then
            return In_Type;
         end if;

         if Ekind (In_Type) = E_Record_Subtype and then
            Has_Discriminants (In_Type)
         then
            New_Expr_List := New_Elmt_List;

            Discrim_Constraint :=
              First_Elmt (Discriminant_Constraint (In_Type));

            while Present (Discrim_Constraint) loop
               Old_Expr := Node (Discrim_Constraint);

               if Nkind (Old_Expr)          = N_Identifier   and then
                  Ekind (Entity (Old_Expr)) = E_Discriminant
               then
                  Need_To_Create_Itype := True;
                  New_Expr := Get_Value (Old_Expr, From => New_Assoc_List);
                  pragma Assert (Present (New_Expr));
                  Append_Elmt (New_Expr, New_Expr_List);

               else
                  Append_Elmt (Old_Expr, New_Expr_List);
               end if;

               Discrim_Constraint := Next_Elmt (Discrim_Constraint);
            end loop;

            if not Need_To_Create_Itype then
               return In_Type;
            end if;

            Itype := New_Itype (E_Record_Subtype, N);

            Set_Etype                   (Itype, Base_Type (In_Type));
            Set_Esize                   (Itype, Esize (In_Type));
            Set_Discriminant_Constraint (Itype, New_Expr_List);
            Set_Is_Tagged_Type          (Itype, Is_Tagged_Type (In_Type));
            Set_Has_Discriminants       (Itype);
            Set_Is_Constrained          (Itype);
            Set_First_Entity            (Itype, First_Entity (In_Type));
            Set_Last_Entity             (Itype, Last_Entity (In_Type));
            Set_Has_Tasks               (Itype, Has_Tasks (In_Type));
            Set_Depends_On_Private      (Itype, Depends_On_Private (In_Type));

            if Is_Tagged_Type (In_Type) then
               Set_Access_Disp_Table (Itype, Access_Disp_Table (In_Type));
            end if;

            return Itype;

         elsif Ekind (In_Type) = E_Array_Subtype then
            New_Index_List := New_List;

            Old_Index := First_Index (In_Type);
            while Present (Old_Index) loop
               New_Index := New_Copy_Tree (Old_Index);

               if Nkind (New_Index) = N_Range then
                  Set_Etype (New_Index, Base_Type (Etype (Old_Index)));

                  Get_Index_Bounds (New_Index, Low_Expr, High_Expr);

                  Old_Expr := Low_Expr;
                  for J in 1 .. 2 loop
                     if Nkind (Old_Expr)         = N_Identifier   and then
                       Ekind (Entity (Old_Expr)) = E_Discriminant
                     then
                        Need_To_Create_Itype := True;
                        New_Expr :=
                          Get_Value (Old_Expr, From => New_Assoc_List);
                        pragma Assert (Present (New_Expr));

                        if J = 1 then
                           Set_Low_Bound
                             (New_Index, New_Copy_Tree (New_Expr));
                        else
                           Set_High_Bound
                             (New_Index, New_Copy_Tree (New_Expr));
                        end if;
                     end if;

                     Old_Expr := High_Expr;
                  end loop;

                  --  Create anonymous index type for range.

                  Index_Type :=
                    New_Itype (Subtype_Kind (Ekind (Etype (New_Index))), N);

                  Set_Etype        (Index_Type, Etype (New_Index));
                  Set_Esize        (Index_Type, Esize (Etype (New_Index)));
                  Set_Scalar_Range (Index_Type, New_Index);

                  Set_Etype (New_Index, Index_Type);
               end if;

               Append (New_Index, To => New_Index_List);
               Old_Index := Next_Index (Old_Index);
            end loop;

            if not Need_To_Create_Itype then
               return In_Type;
            end if;

            Itype := New_Itype (E_Array_Subtype, N);

            Set_Is_Constrained (Itype);
            Set_Etype          (Itype, Base_Type (In_Type));
            Set_Esize          (Itype, Esize (In_Type));
            Set_First_Index    (Itype, First (New_Index_List));
            Set_Component_Type (Itype,
                                  Replace_Discriminants
                                    (In_Type => Component_Type (In_Type)));
            Set_Has_Tasks      (Itype, Has_Tasks (In_Type));
            Set_Has_Controlled (Itype, Has_Controlled (In_Type));
            Set_Depends_On_Private (Itype, Depends_On_Private (In_Type));

            return Itype;
         end if;

         return In_Type;
      end Replace_Discriminants;

   --  Start processing for Resolve_Record_Aggregate

   begin

      --  New Aggregate eventually replaces the aggregate being resolved. It
      --  is initialized here, and attached to the tree explicitly to enforce
      --  the rule that a tree fragment should never be analyzed or resolved
      --  unless it is attached to the current compilation unit.

      Set_Component_Associations (New_Aggregate, New_Assoc_List);
      Set_Parent (New_Aggregate, Parent (N));

      --  STEP 1: abstract type and null record verification

      if Is_Abstract (Typ) then
         Error_Msg_N ("type of aggregate cannot be abstract",  N);
      end if;

      if No (First_Entity (Typ)) and then Null_Record_Present (N) then
         Set_Etype (N, Typ);
         return;

      elsif Present (First_Entity (Typ))
        and then Null_Record_Present (N)
        and then not Is_Tagged_Type (Typ)
      then
         Error_Msg_N ("record aggregate cannot be null", N);
         return;

      elsif No (First_Entity (Typ)) then
         Error_Msg_N ("record aggregate must be null", N);
         return;
      end if;

      --  STEP 2: Verify aggregate structure

      Step_2 : declare
         Selector_Name : Node_Id;
         Bad_Aggregate : Boolean := False;

      begin
         if Present (Component_Associations (N)) then
            Assoc := First (Component_Associations (N));
         else
            Assoc := Empty;
         end if;

         while Present (Assoc) loop
            Selector_Name := First (Choices (Assoc));
            while Present (Selector_Name) loop
               if Nkind (Selector_Name) /= N_Identifier and then
                  Nkind (Selector_Name) /= N_Others_Choice
               then
                  Error_Msg_N
                    ("selector name should be identifier or OTHERS",
                     Selector_Name);
                  Bad_Aggregate := True;
               end if;

               Selector_Name := Next (Selector_Name);
            end loop;

            Assoc := Next (Assoc);
         end loop;

         if Bad_Aggregate then
            return;
         end if;
      end Step_2;

      --  STEP 3: Find discriminant Values

      Step_3 : declare
         Discrim               : Entity_Id;
         Missing_Discriminants : Boolean := False;

      begin
         if Present (Expressions (N)) then
            Positional_Expr := First (Expressions (N));
         else
            Positional_Expr := Empty;
         end if;

         if Has_Discriminants (Typ) then
            Discrim := First_Discriminant (Typ);
         else
            Discrim := Empty;
         end if;

         --  First find the discriminant values in the positional components

         while Present (Discrim) and then Present (Positional_Expr) loop
            Next_Expr := Next (Positional_Expr);
            Remove (Positional_Expr);
            Add_Association (Discrim, Positional_Expr);
            Resolve (Positional_Expr, Etype (Discrim));
            Check_Non_Static_Context (Positional_Expr);

            if Present (Get_Value (Discrim,
                                   From => Component_Associations (N)))
            then
               Error_Msg_NE
                 ("more than one value supplied for discriminant&",
                  N, Discrim);
            end if;

            Positional_Expr := Next_Expr;
            Discrim         := Next_Discriminant (Discrim);
         end loop;

         --  Find remaining discriminant values, if any, among named components

         while Present (Discrim) loop
            Expr :=
              Get_Value
                (Discrim,
                 From => Component_Associations (N),
                 Consider_Others_Choice => True);

            if No (Expr) then
               Error_Msg_NE
                 ("no value supplied for discriminant &", N, Discrim);
               Missing_Discriminants := True;

            else
               Add_Association (Discrim, Expr);
               Resolve (Expr, Etype (Discrim));
               Check_Non_Static_Context (Expr);
            end if;

            Discrim := Next_Discriminant (Discrim);
         end loop;

         if Missing_Discriminants then
            return;
         end if;

         --  At this point and until the beginning of STEP 6, New_Assoc_List
         --  contains only the discriminants and their values.

      end Step_3;

      --  STEP 4: Set the Etype of the record aggregate

      if Has_Discriminants (Typ) then
         Build_Constrained_Itype : declare
            Discrim_Exprs : Elist_Id  := New_Elmt_List;
            Constr_Itype  : Entity_Id := New_Itype (E_Record_Subtype, N);

         begin
            New_Assoc  := First (New_Assoc_List);
            while Present (New_Assoc) loop
               Append_Elmt (Expression (New_Assoc), Discrim_Exprs);
               New_Assoc  := Next (New_Assoc);
            end loop;

            Set_Etype                   (Constr_Itype, Base_Type (Typ));
            Set_Esize                   (Constr_Itype, Esize (Typ));
            Set_Is_Tagged_Type          (Constr_Itype, Is_Tagged_Type (Typ));
            Set_Has_Discriminants       (Constr_Itype);
            Set_Is_Constrained          (Constr_Itype);
            Set_First_Entity            (Constr_Itype, First_Entity (Typ));
            Set_Last_Entity             (Constr_Itype, Last_Entity (Typ));
            Set_Discriminant_Constraint (Constr_Itype, Discrim_Exprs);
            Set_Has_Tasks               (Constr_Itype, Has_Tasks (Typ));

            if Is_Tagged_Type (Typ) then
               Set_Access_Disp_Table
                 (Constr_Itype, Access_Disp_Table (Typ));
            end if;

            Set_Etype (N, Constr_Itype);
         end Build_Constrained_Itype;

      else
         Set_Etype (N, Typ);
      end if;

      --  STEP 5: Get remaining components according to discriminant values

      Step_5 : declare
         Parent_Typ      : Entity_Id;
         Root_Typ        : Entity_Id;
         Parent_Typ_List : Elist_Id;
         Parent_Elmt     : Elmt_Id;
         Errors_Found    : Boolean := False;


      begin
         if Is_Derived_Type (Typ) and then Is_Tagged_Type (Typ) then
            Parent_Typ_List := New_Elmt_List;

            --  If this is an extension aggregate, the component list must
            --  include all components that are not in the given ancestor
            --  type. Otherwise,  the component list must include components
            --  of all ancestors.

            if Nkind (N) = N_Extension_Aggregate then
               Root_Typ := Etype (Ancestor_Part (N));
            else
               Root_Typ := Root_Type (Typ);

               if Nkind (Parent (Base_Type (Root_Typ)))
                    = N_Private_Type_Declaration
               then
                  Error_Msg_NE
                    ("type of aggregate has private ancestor&!",
                     N, Root_Typ);
                  Error_Msg_N  ("must use extension aggregate!", N);
                  return;
               end if;

               Record_Def := Type_Definition (Parent (Base_Type (Root_Typ)));

               Gather_Components
                 (Component_List (Record_Def),
                  Governed_By   => New_Assoc_List,
                  Into          => Components,
                  Report_Errors => Errors_Found);
            end if;

            Parent_Typ  := Typ;
            while Parent_Typ /= Root_Typ loop

               Prepend_Elmt (Parent_Typ, To => Parent_Typ_List);
               Parent_Typ := Etype (Parent_Typ);

               if Nkind (Parent (Base_Type (Parent_Typ))) =
                                        N_Private_Type_Declaration
                 and then Nkind (N) /= N_Extension_Aggregate
               then
                  Error_Msg_NE
                    ("type of aggregate has private ancestor&!",
                     N, Parent_Typ);
                  Error_Msg_N  ("must use extension aggregate!", N);
                  return;
               end if;
            end loop;

            --  Now collect components from all other ancestors.

            Parent_Elmt := First_Elmt (Parent_Typ_List);

            while Present (Parent_Elmt) loop
               Parent_Typ := Node (Parent_Elmt);
               Record_Def := Type_Definition (Parent (Base_Type (Parent_Typ)));
               Gather_Components
                 (Component_List (Record_Extension_Part (Record_Def)),
                  Governed_By   => New_Assoc_List,
                  Into          => Components,
                  Report_Errors => Errors_Found);
               Parent_Elmt := Next_Elmt (Parent_Elmt);
            end loop;

         else
            Record_Def := Type_Definition (Parent (Base_Type (Typ)));

            if Null_Present (Record_Def) then
               null;
            else
               Gather_Components
                 (Component_List (Record_Def),
                  Governed_By   => New_Assoc_List,
                  Into          => Components,
                  Report_Errors => Errors_Found);
            end if;
         end if;

         if Errors_Found then
            return;
         end if;
      end Step_5;

      --  STEP 6: Find component Values

      Component_Elmt := First_Elmt (Components);

      --  First scan the remaining positional associations in the aggregate.
      --  Remember that at this point Positional_Expr contains the current
      --  positional association if any is left after looking for discriminant
      --  values in step 3.

      while Present (Positional_Expr) and then Present (Component_Elmt) loop
         Next_Expr := Next (Positional_Expr);
         Component := Node (Component_Elmt);
         Remove (Positional_Expr);
         Add_Association (Component, Positional_Expr);
         Resolve
           (Positional_Expr,
            Replace_Discriminants (In_Type => Etype (Component)));
         Check_Non_Static_Context (Positional_Expr);

         if Present
           (Get_Value (Component, From => Component_Associations (N)))
         then
            Error_Msg_NE
              ("more than one value supplied for Component &", N, Component);
         end if;

         Positional_Expr := Next_Expr;
         Component_Elmt  := Next_Elmt (Component_Elmt);
      end loop;

      if Present (Positional_Expr) then
         Error_Msg_N
           ("too many components for record aggregate", Positional_Expr);
      end if;

      --  Now scan for the named arguments of the aggregate

      while Present (Component_Elmt) loop
         Component := Node (Component_Elmt);
         Expr :=
           Get_Value (Component,
                      From => Component_Associations (N),
                      Consider_Others_Choice => True);

         if No (Expr) then
            Error_Msg_NE ("no value supplied for component &", N, Component);
         else
            Add_Association (Component, Expr);
            Resolve
              (Expr, Replace_Discriminants (In_Type => Etype (Component)));
            Check_Non_Static_Context (Expr);
         end if;

         Component_Elmt := Next_Elmt (Component_Elmt);
      end loop;

      --  STEP 7: check for invalid components + check type in choice list

      Step_7 : declare
         Selectr : Node_Id;
         --  Selector name

         Typech  : Entity_Id;
         --  Type of first component in choice list

      begin
         if Present (Component_Associations (N)) then
            Assoc := First (Component_Associations (N));
         else
            Assoc := Empty;
         end if;

         Verification : while Present (Assoc) loop
            Selectr := First (Choices (Assoc));
            Typech := Empty;

            if Nkind (Selectr) = N_Others_Choice then
               if No (Others_Etype) then
                  Error_Msg_N
                    ("OTHERS must represent at least one component", Selectr);
               end if;

               exit Verification;
            end if;

            while Present (Selectr) loop
               New_Assoc := First (New_Assoc_List);
               while Present (New_Assoc) loop
                  Component := First (Choices (New_Assoc));
                  exit when Chars (Selectr) = Chars (Component);
                  New_Assoc := Next (New_Assoc);
               end loop;

               --  If no association, this is not a a legal component of
               --  of the type in question,  except if this is an internal
               --  component supplied by a previous expansion.

               if No (New_Assoc) then

                  if Chars (Selectr) /= Name_uTag
                    and then Chars (Selectr) /= Name_uParent
                    and then Chars (Selectr) /= Name_uController
                  then
                     Error_Msg_N ("component & is undefined", Selectr);
                  end if;

               elsif No (Typech) then
                  Typech := Base_Type (Etype (Component));

               elsif Typech /= Base_Type (Etype (Component)) then
                  Error_Msg_N
                    ("components in choice list must have same type", Selectr);
               end if;

               Selectr := Next (Selectr);
            end loop;

            Assoc := Next (Assoc);
         end loop Verification;
      end Step_7;

      --  STEP 8: replace the original aggregate

      Step_8 : declare
         New_Aggregate : Node_Id := New_Copy (N);

      begin
         Set_Expressions            (New_Aggregate, No_List);
         Set_Etype                  (New_Aggregate, Etype (N));
         Set_Component_Associations (New_Aggregate, New_Assoc_List);
         Rewrite_Substitute_Tree (N, New_Aggregate);
      end Step_8;
   end Resolve_Record_Aggregate;

   -----------------------
   -- Gather_Components --
   -----------------------

   procedure Gather_Components
     (Comp_List     : Node_Id;
      Governed_By   : List_Id;
      Into          : Elist_Id;
      Report_Errors : out Boolean)
   is
      Assoc           : Node_Id;
      Variant         : Node_Id;
      Discrete_Choice : Node_Id;
      Comp_Item       : Node_Id;

      Discrim         : Entity_Id;
      Discrim_Name    : Node_Id;
      Discrim_Value   : Node_Id;

   begin
      Report_Errors := False;

      if Null_Present (Comp_List) then
         return;
      elsif Present (Component_Items (Comp_List)) then
         Comp_Item := First (Component_Items (Comp_List));
      else
         Comp_Item := Empty;
      end if;

      while Present (Comp_Item) loop

         --  Skip the tag of a tagged record, as well as all items
         --  that are not user components (anonymous types, rep clauses,
         --  Parent field, controller field).

         if Nkind (Comp_Item) = N_Component_Declaration
           and then Chars (Defining_Identifier (Comp_Item)) /= Name_uTag
           and then Chars (Defining_Identifier (Comp_Item)) /= Name_uParent
           and then Chars (Defining_Identifier (Comp_Item)) /= Name_uController
         then
            Append_Elmt (Defining_Identifier (Comp_Item), Into);
         end if;

         Comp_Item := Next (Comp_Item);
      end loop;

      if No (Variant_Part (Comp_List)) then
         return;
      else
         Discrim_Name := Name (Variant_Part (Comp_List));
         Variant      := First (Variants (Variant_Part (Comp_List)));
      end if;

      --  Look for the discriminant that governs this variant part.
      --  The discriminant *must* be in the Governed_By List

      Assoc := First (Governed_By);
      loop
         Discrim := First (Choices (Assoc));
         exit when Chars (Discrim_Name) = Chars (Discrim);
         Assoc := Next (Assoc);
      end loop;

      Discrim_Value := Expression (Assoc);

      if not (Is_Static_Expression (Discrim_Value)
        and then Is_Static_Subtype (Etype (Discrim_Value)))
      then
         Error_Msg_NE
           ("value for discriminant & must be static", Discrim_Value, Discrim);
         Report_Errors := True;
         return;
      end if;

      Search_For_Discriminant_Value : declare
         Low  : Node_Id;
         High : Node_Id;

         UI_High          : Uint;
         UI_Low           : Uint;
         UI_Discrim_Value : constant Uint := Expr_Value (Discrim_Value);

      begin
         Find_Discrete_Value : while Present (Variant) loop
            Discrete_Choice := First (Discrete_Choices (Variant));
            while Present (Discrete_Choice) loop

               exit Find_Discrete_Value when
                 Nkind (Discrete_Choice) = N_Others_Choice;

               Get_Index_Bounds (Discrete_Choice, Low, High);

               UI_Low  := Expr_Value (Low);
               UI_High := Expr_Value (High);

               exit Find_Discrete_Value when
                 UI_Low <= UI_Discrim_Value
                   and then
                 UI_High >= UI_Discrim_Value;

               Discrete_Choice := Next (Discrete_Choice);
            end loop;

            Variant := Next (Variant);
         end loop Find_Discrete_Value;
      end Search_For_Discriminant_Value;

      if No (Variant) then
         Error_Msg_NE
           ("value of discriminant & is out of range", Discrim_Value, Discrim);
         Report_Errors := True;
         return;
      end  if;

      --  If we have found the corresponding choice, recursively add its
      --  components to the Into list.

      Gather_Components
        (Component_List (Variant), Governed_By, Into, Report_Errors);
   end Gather_Components;

   --------------------------------
   -- Make_String_Into_Aggregate --
   --------------------------------

   procedure Make_String_Into_Aggregate (N : Node_Id) is
      C          : Char_Code;
      C_Node     : Node_Id;
      Exprs      : List_Id := New_List;
      Loc        : constant Source_Ptr := Sloc (N);
      New_N      : Node_Id;
      P          : Source_Ptr := Loc + 1;
      Str        : constant String_Id  := Strval (N);
      Strlen     : constant Nat        := String_Length (Str);

   begin
      for J in  1 .. Strlen loop
         C := Get_String_Char (Str, J);
         Set_Character_Literal_Name (C);

         C_Node :=  Make_Character_Literal (P, Name_Find, C);
         Set_Etype (C_Node, Any_Character);
         Set_Analyzed (C_Node);
         Append_To (Exprs, C_Node);

         P := P + 1;
         --  something special for wide strings ?
      end loop;

      New_N := Make_Aggregate (Loc, Expressions => Exprs);
      Set_Analyzed (New_N);
      Set_Etype (New_N, Any_Composite);

      Rewrite_Substitute_Tree (N, New_N);
   end Make_String_Into_Aggregate;

   -----------------------------
   -- Resolve_Array_Aggregate --
   -----------------------------

   procedure Resolve_Array_Aggregate
     (N              : Node_Id;
      Index          : Node_Id;
      Component_Typ  : Entity_Id;
      Others_Allowed : Boolean)
   is
      Assoc  : Node_Id;
      Choice : Node_Id;
      Expr   : Node_Id;

      Index_Subtype : Entity_Id := Base_Type (Etype (Index));
      --  The subtype of the index corresponding to the array sub-aggregate

      Subtype_Low  : Node_Id;
      Subtype_High : Node_Id;
      --  The low and upper bounds of the Index_Subtype

      Index_Low  : Node_Id;
      Index_High : Node_Id;
      --  The low and upper bounds of the applicable index constraint (if any)

      Others_Present : Boolean;

      Nb_Elements : Uint := UI_From_Int (0);
      --  The actual number of elements specified in a positional or named
      --  aggegate

      Nb_Choices : Nat := 0;
      --  Contains the overall number of named choices in this sub-aggregate

      Case_Table_Size : Nat;
      --  Contains the size of the case table needed to sort aggregate choices

      procedure Resolve_Aggr_Expr (Expr : Node_Id; Index : Node_Id);
      --  Resolves an aggregate expression Expr. Index is the index
      --  corresponding to the expresion.

      procedure Replace_Aggr_With_Raise_CE;
      --  Replaces aggregate node with raise Constraint_Error.

      procedure Check_Bounds (L, H : Node_Id; A_L, A_H : Node_Id);
      --  Checks that range A_L and A_H are both in L .. H. Emits a warning
      --  if not and replaces the aggregate node N with raise constraint error.

      procedure Check_Others_Validity (L, H : Node_Id;  Len : Uint);
      --  Checks that an others choice specifies at least one value.
      --  If it specifies zero a gentle warning is emitted. If it specifies
      --  less a more severe warning is emitted and the aggregate node N is
      --  replaced with raise constraint error.
      --  L and H give the lower and upper bounds of the aggreagte index
      --  constraint, Len the actual number of elements specified in the
      --  aggregate.

      procedure Check_Length
        (L, H : Node_Id;
         Len  : Uint;
         Same : Boolean := True);
      --  If Same is True checks that range L .. H contains exactly Len
      --  elements. If Same is False checks that range L .. H contains at least
      --  Len elements. If the check fails, the procedure replaces the
      --  aggregate node N with raise constraint error.

      procedure Resolve_Aggr_Expr (Expr : Node_Id; Index : Node_Id) is
         Nxt_Index : Node_Id := Next_Index (Index);

      begin
         if Present (Nxt_Index) then
            if Nkind (Expr) /= N_Aggregate then

               --  A string literal can appear where a one-dimensional array
               --  of characters is expected.

               if Is_Character_Type (Component_Typ)
                 and then No (Next_Index (Nxt_Index))
                 and then Nkind (Expr) = N_String_Literal
               then
                  Make_String_Into_Aggregate (Expr);
               else
                  Error_Msg_N ("nested array aggregate expected", Expr);
                  return;
               end if;
            end if;

            Resolve_Array_Aggregate
              (Expr, Nxt_Index, Component_Typ, Others_Allowed);

         else
            Resolve (Expr, Component_Typ);
            Check_Non_Static_Context (Expr);

         end if;
      end Resolve_Aggr_Expr;

      procedure Replace_Aggr_With_Raise_CE is
      begin
         Rewrite_Substitute_Tree (N, Make_Raise_Constraint_Error (Sloc (N)));
         Set_Analyzed (N, True);
         Set_Etype (N, Any_Composite);
         Set_Raises_Constraint_Error (N);
      end Replace_Aggr_With_Raise_CE;

      procedure Check_Bounds (L, H : Node_Id; A_L, A_H : Node_Id) is
      begin
         if Is_Static_Expression (L)
           and then Is_Static_Expression (A_L)
           and then Expr_Value (L) > Expr_Value (A_L)
         then
            Compile_Time_Constraint_Error (A_L, "choice out of bouds?");
            Replace_Aggr_With_Raise_CE;
            return;
         end if;

         if Is_Static_Expression (H)
           and then Is_Static_Expression (A_H)
           and then Expr_Value (H) < Expr_Value (A_H)
         then
            Compile_Time_Constraint_Error (A_H, "choice out of bouds?");
            Replace_Aggr_With_Raise_CE;
            return;
         end if;

         --  repeat the test the other way around for null ranges

         if Is_Static_Expression (L)
           and then Is_Static_Expression (A_H)
           and then Expr_Value (L) > Expr_Value (A_H)
         then
            Compile_Time_Constraint_Error (A_H, "choice out of bouds?");
            Replace_Aggr_With_Raise_CE;
            return;
         end if;

         if Is_Static_Expression (H)
           and then Is_Static_Expression (A_L)
           and then Expr_Value (H) < Expr_Value (A_L)
         then
            Compile_Time_Constraint_Error (A_L, "choice out of bouds?");
            Replace_Aggr_With_Raise_CE;
            return;
         end if;
      end Check_Bounds;

      procedure Check_Others_Validity (L, H : Node_Id;  Len : Uint) is
         Nb_Elmt : Uint;
      begin
         if not Is_Static_Expression (L)
           or else not Is_Static_Expression (H)
         then
            return;
         end if;

         --  Check for null ranges

         if Expr_Value (H) >= Expr_Value (L) then
            Nb_Elmt := Expr_Value (H) - Expr_Value (L) + 1;
         else
            Nb_Elmt := UI_From_Int (0);
         end if;

         if Nb_Elmt < Len then
            Compile_Time_Constraint_Error (N, "too many elements?");

         elsif Nb_Elmt = Len then
            Error_Msg_N
              ("OTHERS choice unnecessary, all values explicitely specified?",
               N);

         end if;
      end Check_Others_Validity;

      procedure Check_Length
        (L, H : Node_Id;
         Len  : Uint;
         Same : Boolean := True)
      is
         Nb_Elmt : Uint;
      begin
         if not Is_Static_Expression (L)
           or else not Is_Static_Expression (H)
         then
            return;
         end if;

         --  Check for null ranges

         if Expr_Value (H) >= Expr_Value (L) then
            Nb_Elmt := Expr_Value (H) - Expr_Value (L) + 1;
         else
            Nb_Elmt := UI_From_Int (0);
         end if;

         if Nb_Elmt < Len then
            Compile_Time_Constraint_Error (N, "too many elements?");

         elsif Same and then Nb_Elmt > Len then
            Compile_Time_Constraint_Error (N, "too few elements?");

         end if;
      end Check_Length;

   begin
      --  STEP 1: make sure the aggregate is correctly formatted

      if Etype (N) = Any_Type then
         Set_Etype (N, Any_Composite);
         return;
      end if;

      --  At this point we know that the others choice, if present, is by
      --  itself and appears last in the aggregate.

      if Present (Expressions (N))
        and then Present (Component_Associations (N))
        and then
          Nkind (First (Choices (First
            (Component_Associations (N))))) /= N_Others_Choice
      then
         Error_Msg_N ("mixed positional/named associations", N);
         return;
      end if;

      --  Test for the validity of an others choice if present

      if Present (Component_Associations (N)) then
         Assoc := Last (Component_Associations (N));
         Others_Present := (Nkind (First (Choices (Assoc))) = N_Others_Choice);
      else
         Others_Present := False;
      end if;

      if Others_Present and then (not Others_Allowed) then
         Error_Msg_N ("OTHERS choice not allowed here",
                      First (Choices (Assoc)));
         return;
      end if;

      --  STEP 2: Process named components and verify their semantic validity

      if No (Expressions (N)) then

         --  Count the overall number of choices so that we can allocate array
         --  Table below to contain the discrete choices in the aggregate.

         Assoc := First (Component_Associations (N));
         while Present (Assoc) loop
            Choice := First (Choices (Assoc));
            while Present (Choice) loop
               Nb_Choices := Nb_Choices + 1;
               Choice := Next (Choice);
            end loop;

            Assoc := Next (Assoc);
         end loop;

         if Others_Present then
            Case_Table_Size := Nb_Choices - 1;
         else
            Case_Table_Size := Nb_Choices;
         end if;

         Step_2 : declare
            Low  : Node_Id;
            High : Node_Id;
            --  Denote the lowest and highest values in an aggregate choice

            Choices_Low  : Node_Id;
            Choices_High : Node_Id;
            --  The lowest and highest values in all the aggregate choices

            Aggr_Low  : Node_Id;
            Aggr_High : Node_Id;
            --  The actual low and high bounds of the aggegate

            Table : Case_Table_Type (1 .. Case_Table_Size);
            --  Used to sort all the different choice values

            Nb_Static_Choices : Nat := 0;

         begin
            Assoc := First (Component_Associations (N));
            while Present (Assoc) loop

               Choice := First (Choices (Assoc));
               while Present (Choice) loop
                  Analyze (Choice);

                  if Nkind (Choice) = N_Others_Choice then
                     exit;

                  --  Test for subtype mark without constraint

                  elsif Is_Entity_Name (Choice) and then
                    Is_Type (Entity (Choice))
                  then
                     if Base_Type (Entity (Choice)) /= Index_Subtype then
                        Error_Msg_N
                          ("invalid subtype mark in aggregate choice", Choice);
                        return;
                     end if;

                  elsif Nkind (Choice) = N_Subtype_Indication then
                     Resolve_Discrete_Subtype_Indication (Choice,
                                                          Index_Subtype);

                  else -- Choice is an expression
                     Resolve (Choice, Index_Subtype);
                     Check_Non_Static_Context (Choice);

                  end if;

                  --  If we could not resolve the discrete choice stop here

                  if Etype (Choice) = Any_Type then
                     return;
                  end if;

                  Get_Index_Bounds (Choice, Low, High);

                  if (not Is_Static_Expression (Low))
                    or else (not Is_Static_Expression (High))
                  then
                     if Nb_Choices /= 1 then
                        Error_Msg_N ("non static choice in array aggregate " &
                                     "must be the only choice", Choice);
                        return;
                     end if;

                  elsif Nkind (Low) = N_Raise_Constraint_Error
                    or else Nkind (High) = N_Raise_Constraint_Error
                  then
                     --  Static constraint error. Replace aggregate itself.

                     Replace_Aggr_With_Raise_CE;
                     return;

                  else
                     if Nb_Choices /= 1
                       and then Expr_Value (Low) > Expr_Value (High)
                     then
                        Error_Msg_N ("null range in array aggregate must " &
                                     "be the only choice", Choice);
                        return;
                     end if;

                     Nb_Static_Choices := Nb_Static_Choices + 1;
                     Table (Nb_Static_Choices).Choice_Lo := Low;
                     Table (Nb_Static_Choices).Choice_Hi := High;
                  end if;

                  Choice := Next (Choice);
               end loop;

               Resolve_Aggr_Expr (Expression (Assoc), Index);
               Assoc := Next (Assoc);
            end loop;

            --  If aggregate contained a non static choice or only an "others"
            --  choice nothing more to do.

            if Nb_Static_Choices = 0 then
               return;
            end if;

            Sort_Case_Table (Table);

            for J in 1 .. Nb_Static_Choices - 1 loop
               if Expr_Value (Table (J).Choice_Hi) >=
                  Expr_Value (Table (J + 1).Choice_Lo)
               then
                  Error_Msg_N ("duplicate choice values in array aggregate",
                               Table (J).Choice_Hi);
                  return;

               elsif (not Others_Present)
                 and then
                   (Expr_Value (Table (J + 1).Choice_Lo) -
                    Expr_Value (Table (J).Choice_Hi)) > 1
               then
                  Error_Msg_N ("missing association in array aggregate", N);
                  return;

               end if;
            end loop;

            --  Check if any constraint errors can be raised at run-time.

            Choices_Low  := Table (1).Choice_Lo;
            Choices_High := Table (Nb_Static_Choices).Choice_Hi;

            Get_Index_Bounds (Index, Index_Low, Index_High);

            Subtype_Low  := Type_Low_Bound (Index_Subtype);
            Subtype_High := Type_High_Bound (Index_Subtype);

            if Others_Present then
               Aggr_Low  := Index_Low;
               Aggr_High := Index_High;
            else
               Aggr_Low  := Choices_Low;
               Aggr_High := Choices_High;
            end if;

            --  Check (a)

            Check_Bounds (Subtype_Low, Subtype_High, Aggr_Low, Aggr_High);

            if Others_Present then
               --  Check (b)

               Check_Bounds (Index_Low, Index_High, Choices_Low, Choices_High);

               --  Check (c)

               if Expr_Value (Choices_High) >= Expr_Value (Choices_Low) then
                  Nb_Elements :=
                    (Expr_Value (Choices_High) - Expr_Value (Choices_Low)) + 1;
               else
                  Nb_Elements := UI_From_Int (0);
               end if;

               Check_Others_Validity (Index_Low, Index_High,  Nb_Elements);

            elsif Others_Allowed
              and then Is_Static_Expression (Aggr_High)
              and then Is_Static_Expression (Aggr_Low)
            then
               --  Check (d)

               if Expr_Value (Aggr_High) >= Expr_Value (Aggr_Low) then
                  Nb_Elements :=
                    (Expr_Value (Aggr_High) - Expr_Value (Aggr_Low)) + 1;
               else
                  Nb_Elements := UI_From_Int (0);
               end if;

               Check_Length (Index_Low, Index_High,  Nb_Elements);

            end if;
         end Step_2;

      --  STEP 3: Process positional components

      else
         Expr := First (Expressions (N));
         Nb_Elements := UI_From_Int (0);

         while Present (Expr) loop
            Nb_Elements := Nb_Elements + 1;
            Resolve_Aggr_Expr (Expr, Index);
            Expr := Next (Expr);
         end loop;

         if Others_Present then
            Assoc := Last (Component_Associations (N));
            Resolve_Aggr_Expr (Expression (Assoc), Index);
         end if;

         --  Check if any constraint errors can be raised at run-time.

         Get_Index_Bounds (Index, Index_Low, Index_High);
         Subtype_Low  := Type_Low_Bound (Index_Subtype);
         Subtype_High := Type_High_Bound (Index_Subtype);

         Check_Length (Subtype_Low, Subtype_High, Nb_Elements, Same => False);

         if Others_Present then
            Check_Others_Validity (Index_Low, Index_High, Nb_Elements);

         elsif Others_Allowed then
            Check_Length (Index_Low, Index_High, Nb_Elements);

         end if;

      end if;

   end Resolve_Array_Aggregate;

end Sem_Aggr;
