------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUNTIME COMPONENTS                          --
--                                                                          --
--                     S Y S T E M . D I R E C T _ I O                      --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.6 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- The GNAT library is free software; you can redistribute it and/or modify --
-- it under terms of the GNU Library General Public License as published by --
-- the Free Software  Foundation; either version 2, or (at your option) any --
-- later version.  The GNAT library is distributed in the hope that it will --
-- be useful, but WITHOUT ANY WARRANTY;  without even  the implied warranty --
-- of MERCHANTABILITY  or  FITNESS FOR  A PARTICULAR PURPOSE.  See the  GNU --
-- Library  General  Public  License for  more  details.  You  should  have --
-- received  a copy of the GNU  Library  General Public License  along with --
-- the GNAT library;  see the file  COPYING.LIB.  If not, write to the Free --
-- Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.        --
--                                                                          --
------------------------------------------------------------------------------

--  This package contains the declaration of the control block used for
--  Direct_IO. This must be declared at the outer library level. It also
--  contains code that is shared between instances of Direct_IO.

with Interfaces.C_Streams;
with System.File_Control_Block;

package System.Direct_IO is

   package FCB renames System.File_Control_Block;

   type Operation is (Op_Read, Op_Write, Op_Other);
   --  Type used to record last operation (to optimize sequential operations)

   subtype Count is Interfaces.C_Streams.long;
   --  The Count type in each instantiation is derived from this type

   subtype Positive_Count is Count range 1 .. Count'Last;

   type Direct_AFCB is new FCB.AFCB with record
      Index : Count := 1;
      --  Current Index value

      Bytes : Interfaces.C_Streams.size_t;
      --  Length of item in bytes (set from inside generic template)

      Last_Op : Operation := Op_Other;
      --  Last operation performed on file, used to avoid unnecessary
      --  repositioning between successive read or write operations.
   end record;

   function AFCB_Allocate (Control_Block : Direct_AFCB) return FCB.AFCB_Ptr;

   procedure AFCB_Close (File : access Direct_AFCB);
   procedure AFCB_Free  (File : access Direct_AFCB);

   procedure Read
     (File : in out Direct_AFCB;
      Item : out Ada.Streams.Stream_Element_Array;
      Last : out Ada.Streams.Stream_Element_Offset);
   --  Required overriding of Read, not actually used for Direct_IO

   procedure Write
     (File : in out Direct_AFCB;
      Item : in Ada.Streams.Stream_Element_Array);
   --  Required overriding of Write, not actually used for Direct_IO

   type File_Type is access all Direct_AFCB;
   --  File_Type in individual instantiations is derived from this type

   procedure Create
     (File : in out File_Type;
      Mode : in FCB.File_Mode := FCB.Inout_File;
      Name : in String := "";
      Form : in String := "");

   function End_Of_File (File : in File_Type) return Boolean;

   function Index (File : in File_Type) return Positive_Count;

   procedure Open
     (File : in out File_Type;
      Mode : in FCB.File_Mode;
      Name : in String;
      Form : in String := "");

   procedure Read
     (File : in File_Type;
      Item : System.Address;
      From : in Positive_Count);

   procedure Read
     (File : in File_Type;
      Item : System.Address);

   procedure Reset (File : in out File_Type; Mode : in FCB.File_Mode);

   procedure Reset (File : in out File_Type);

   procedure Set_Index (File : in File_Type; To : in Positive_Count);

   function Size (File : in File_Type) return Count;

   procedure Write
     (File : in File_Type;
      Item : System.Address;
      To   : in Positive_Count);

   procedure Write
     (File : in File_Type;
      Item : System.Address);

end System.Direct_IO;
