------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              S E M _ C H 6                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.171 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Casing;   use Casing;
with Debug;    use Debug;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Exp_Ch7;  use Exp_Ch7;
with Exp_Util; use Exp_Util;
with Lib;      use Lib;
with Namet;    use Namet;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch4;  use Sem_Ch4;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Ch12; use Sem_Ch12;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sem_Disp; use Sem_Disp;
with Sinput;   use Sinput;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Sinfo.CN; use Sinfo.CN;
with Snames;   use Snames;
with Stringt;  use Stringt;
with Tbuild;   use Tbuild;
with Treepr;   use Treepr;

package body Sem_Ch6 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   procedure Analyze_Generic_Subprogram_Body (N : Node_Id; Gen_Id : Entity_Id);
   --  Analyze a generic subprogram body

   procedure Check_Conformance (Old_Id, New_Id : Entity_Id);
   --  Verify that the specifications given in a subprogram declaration
   --  conforms to that given in the body.

   procedure Enter_Overloaded_Entity (S : Entity_Id);
   --  This procedure makes S, a new overloaded entity, into the first
   --  visible entity with that name.

   procedure Install_Entity (E : Entity_Id);
   --  Make single entity visible. Used for generic formals as well.

   procedure Install_Formals (Id : Entity_Id);
   --  On entry to a subprogram body, make the formals visible. Note
   --  that simply placing the subprogram on the scope stack is not
   --  sufficient: the formals must become the current entities for
   --  their names.

   procedure May_Need_Actuals (Fun : Entity_Id);
   --  Flag functions that can be called without parameters, i.e. those that
   --  have no parameters, or those for which defaults exist for all parameters

   procedure Valid_Operator_Definition (Designator : Entity_Id);
   --  Verify that an operator definition has the proper number of formals

   ---------------------------------------------
   -- Analyze_Abstract_Subprogram_Declaration --
   ---------------------------------------------

   procedure Analyze_Abstract_Subprogram_Declaration (N : Node_Id) is
      Designator : constant Entity_Id := Analyze_Spec (Specification (N));

   begin
      New_Overloaded_Entity (Designator);
      Set_Is_Abstract (Designator);
      Check_Delayed_Subprogram (Designator);
   end Analyze_Abstract_Subprogram_Declaration;

   ----------------------------
   -- Analyze_Function_Call  --
   ----------------------------

   procedure Analyze_Function_Call (N : Node_Id) is
      P      : constant Node_Id := Name (N);
      L      : constant List_Id := Parameter_Associations (N);
      Actual : Node_Id;

   begin
      Analyze (P);

      --  If error analyzing name, then set Any_Type as result type and return

      if Etype (P) = Any_Type then
         Set_Etype (N, Any_Type);
         return;
      end if;

      --  Otherwise analyze the parameters

      if Present (L) then
         Actual := First (L);

         while Present (Actual) loop
            Analyze_Expression (Actual);
            Actual := Next (Actual);
         end loop;
      end if;

      Analyze_Call (N);

   end Analyze_Function_Call;

   -------------------------------------
   -- Analyze_Generic_Subprogram_Body --
   -------------------------------------

   procedure Analyze_Generic_Subprogram_Body
     (N      : Node_Id;
      Gen_Id : Entity_Id)
   is
      Gen_Decl : constant Node_Id := Get_Declaration_Node (Gen_Id);
      Spec     : Node_Id;
      Kind     : constant Entity_Kind := Ekind (Gen_Id);
      Nam      : Entity_Id;
      New_N    : Node_Id;

   begin
      --  Copy body, and disable expansion while analyzing the generic.

      New_N := Copy_Generic_Node (N, Empty, Instantiating => False);
      Rewrite_Substitute_Tree (N, New_N);
      Expander_Mode_Save_And_Set (False);

      Spec := Specification (N);
      --  Within the body of the generic, the subprogram is callable, and
      --  behaves like the corresponding non-generic unit.

      Nam := Defining_Unit_Simple_Name (Spec);

      if Kind = E_Generic_Procedure
        and then Nkind (Spec) /= N_Procedure_Specification
      then
         Error_Msg_N ("invalid body for generic procedure ", Nam);
         return;

      elsif Kind = E_Generic_Function
        and then Nkind (Spec) /= N_Function_Specification
      then
         Error_Msg_N ("invalid body for generic function ", Nam);
         return;
      end if;

      Set_Corresponding_Body (Gen_Decl, Nam);
      Set_Corresponding_Spec (N, Gen_Id);
      Set_Has_Completion (Gen_Id);

      if Nkind (N) = N_Subprogram_Body_Stub then
         return;
      end if;

      --  Make generic parameters immediately visible in the body. They are
      --  needed to process the formals declarations. Then make the formals
      --  visible in a separate step.

      New_Scope (Gen_Id);
      declare
         E : Entity_Id;

      begin
         E := First_Entity (Gen_Id);
         while Present (E) and then Ekind (E) not in Formal_Kind loop
            Install_Entity (E);
            E := Next_Entity (E);
         end loop;

         Set_Use (Generic_Formal_Declarations (Gen_Decl));

         --  Now generic formals are visible, and the specification can be
         --  analyzed, for subsequent conformance check.

         Nam := Analyze_Spec (Spec);

         if Present (E) then

            --  E is the first formal parameter, which must be the first
            --  entity in the subprogram body.

            Set_First_Entity (Gen_Id,  E);

            --  Now make formal parameters visible

            while Present (E) loop
               Install_Entity (E);
               E := Next_Formal (E);
            end loop;
         end if;
      end;

      --  Visible generic entity is callable within its own body.

      Set_Ekind (Gen_Id, Ekind (Nam));

      --  Check_Conformance (Gen_Id, Nam);

      Analyze_Declarations (Declarations (N));
      Check_Completion;
      Analyze (Handled_Statement_Sequence (N));
      End_Use (Declarations (N));

      Save_Global_References (Original_Node (N));

      --  Prior to exiting the scope, include generic formals again
      --  in the set of local entities.

      Set_First_Entity (Gen_Id, First_Entity (Gen_Id));

      End_Use (Generic_Formal_Declarations (Gen_Decl));
      End_Scope;

      --  Outside of its body, unit is generic again.

      Set_Ekind (Gen_Id,  Kind);
      Expander_Mode_Restore;

   end Analyze_Generic_Subprogram_Body;

   -----------------------------
   -- Analyze_Operator_Symbol --
   -----------------------------

   --  An operator symbol such as "+" or "and" may appear in context where
   --  the literal denotes an entity name,  such as  "+"(x,  y) or in a
   --  context when it is just a string,  as in  (conjunction = "or"). In
   --  these cases the parser generates this node, and the semantics does
   --  the disambiguation (another such case is an actual in an instantiation).

   procedure Analyze_Operator_Symbol (N : Node_Id) is
      Par : Node_Id := Parent (N);

   begin
      if        (Nkind (Par) = N_Function_Call     and then N = Name (Par))
        or else (Nkind (Par) = N_Indexed_Component and then N = Prefix (Par))
        or else  Nkind (Par) = N_Pragma_Argument_Association
        or else  Nkind (Par) = N_Subprogram_Renaming_Declaration
      then
         Find_Direct_Name (N);

      else
         Change_Operator_Symbol_To_String_Literal (N);
         Set_Etype (N, Any_String);
      end if;
   end Analyze_Operator_Symbol;

   -----------------------------------
   -- Analyze_Parameter_Association --
   -----------------------------------

   procedure Analyze_Parameter_Association (N : Node_Id) is
   begin
      Analyze (Explicit_Actual_Parameter (N));
   end Analyze_Parameter_Association;

   ----------------------------
   -- Analyze_Procedure_Call --
   ----------------------------

   procedure Analyze_Procedure_Call (N : Node_Id) is
      P       : constant Node_Id := Name (N);
      Actuals : constant List_Id := Parameter_Associations (N);
      Actual  : Node_Id;
      Loc     : Source_Ptr := Sloc (N);
      New_N   : Node_Id;
      S       : Entity_Id;

      procedure Analyze_And_Resolve is
      begin
         Analyze_Call (N);
         Resolve (N, Standard_Void_Type);
      end Analyze_And_Resolve;

   --  Start of processing for Analyze_Procedure_Call

   begin
      --  The syntactic construct: PREFIX ACTUAL_PARAMETER_PART can denote
      --  a procedure call or an entry call. The prefix may denote an access
      --  to subprogram type, in which case an implicit dereference applies.
      --  If the prefix is an indexed component (without implicit defererence)
      --  then the construct denotes a call to a member of an entire family.
      --  If the prefix is a simple name, it may still denote a call to a
      --  parameterless member of an entry family. Resolution of these various
      --  interpretations is delicate.

      Analyze (P);

      --  If error analyzing prefix, then set Any_Type as result and return

      if Etype (P) = Any_Type then
         Set_Etype (N, Any_Type);
         return;
      end if;

      --  Otherwise analyze the parameters

      if Present (Actuals) then
         Actual := First (Actuals);

         while Present (Actual) loop
            Analyze_Expression (Actual);
            Actual := Next (Actual);
         end loop;
      end if;

      if Is_Entity_Name (P) then
         Analyze_And_Resolve;

      elsif Nkind (P) = N_Explicit_Dereference then
         if Ekind (Etype (P)) = E_Subprogram_Type then
            Analyze_And_Resolve;
         else
            Error_Msg_N ("expect access to procedure in call", P);
         end if;

      --  The name can be a selected component or an indexed component
      --  that yields an access to subprogram. Such a prefix is legal if
      --  the call has parameter associations.

      elsif Is_Access_Type (Etype (P))
        and then Ekind (Designated_Type (Etype (P))) = E_Subprogram_Type
      then
         if Present (Actuals) then
            Analyze_And_Resolve;
         else
            Error_Msg_N ("missing explicit dereference in call ", N);
         end if;

      --  If not an access to subprogram, then the prefix must resolve to
      --  the name of an entry, entry family, or protected operation.

      --  For the case of a simple entry call, P is a selected component
      --  where the prefix is the task and the selector name is the entry.
      --  A call to a protected procedure will have the same syntax.

      elsif Nkind (P) = N_Selected_Component
        and then (Ekind (Entity (Selector_Name (P))) = E_Entry
        or else   Ekind (Entity (Selector_Name (P))) = E_Procedure)
      then
         Analyze_And_Resolve;

      elsif Nkind (P) = N_Selected_Component
        and then Ekind (Entity (Selector_Name (P))) = E_Entry_Family
        and then Present (Actuals)
        and then No (Next (First (Actuals)))
      then
         --  Can be call to parameterless entry family. What appears to be
         --  the sole argument is in fact the entry index. Rewrite prefix
         --  of node accordingly. Source representation is unchanged by this
         --  transformation.

         New_N :=
           Make_Indexed_Component (Loc,
             Prefix => New_Copy (P),
             Expressions => Actuals);
         Set_Name (N, New_N);
         Set_Etype (New_N, Standard_Void_Type);
         Set_Parameter_Associations (N,  No_List);
         Analyze_And_Resolve;

      --  For the case of a reference to an element of an entry family, P is
      --  an indexed component whose prefix is a selected component (task and
      --  entry family), and whose index is the entry family index.

      elsif Nkind (P) = N_Indexed_Component
        and then Nkind (Prefix (P)) = N_Selected_Component
        and then Ekind (Entity (Selector_Name (Prefix (P)))) = E_Entry_Family
      then
         Analyze_And_Resolve;

      --  If the prefix is the name of an entry family,  it is a call from
      --  within the task body itself.

      elsif Nkind (P) = N_Indexed_Component
        and then Nkind (Prefix (P)) = N_Identifier
        and then Ekind (Entity (Prefix (P))) = E_Entry_Family
      then
         New_N :=
           Make_Selected_Component (Loc,
             Prefix => New_Occurrence_Of (Scope (Entity (Prefix (P))), Loc),
             Selector_Name => New_Occurrence_Of (Entity (Prefix (P)), Loc));
         Rewrite_Substitute_Tree (Prefix (P), New_N);
         Analyze (P);
         Analyze_And_Resolve;

      --  Anything else is an error.

      else
         Error_Msg_N ("Invalid procedure or entry call", N);
      end if;
   end Analyze_Procedure_Call;

   ------------------
   -- Analyze_Spec --
   ------------------

   function Analyze_Spec (N : Node_Id) return Entity_Id is
      Designator : constant Entity_Id := Defining_Unit_Simple_Name (N);
      Formals    : constant List_Id   := Parameter_Specifications (N);

   begin
      if Nkind (N) = N_Function_Specification then
         Set_Ekind (Designator, E_Function);
         Find_Type (Subtype_Mark (N));
         Set_Etype (Designator, Entity (Subtype_Mark (N)));

      else
         Set_Ekind (Designator, E_Procedure);
         Set_Etype (Designator, Standard_Void_Type);
      end if;

      if Present (Formals) then
         Set_Scope (Designator, Current_Scope);
         New_Scope (Designator);
         Process_Formals (Designator, Formals, N);
         End_Scope;
      end if;

      if Nkind (N) = N_Function_Specification then
         if Nkind (Designator) = N_Defining_Operator_Symbol then
            Valid_Operator_Definition (Designator);
         end if;

         May_Need_Actuals (Designator);

         if Is_Abstract (Etype (Designator))
           and then Nkind (Parent (N)) /= N_Abstract_Subprogram_Declaration
         then
            Error_Msg_N
              ("function that returns abstract type must be abstract", N);
         end if;
      end if;

      return Designator;
   end Analyze_Spec;

   -----------------------------
   -- Analyze_Subprogram_Body --
   -----------------------------

   --  This procedure is called for regular subprogram bodies, generic bodies,
   --  and for subprogram stubs of both kinds. In the case of stubs, only the
   --  specification matters, and is used to create a proper declaration for
   --  the subprogram, or to perform conformance checks.

   procedure Analyze_Subprogram_Body (N : Node_Id) is
      Spec        : constant Node_Id    := Specification (N);
      Nam         : constant Entity_Id  := Defining_Unit_Simple_Name (Spec);
      Gen_Id      : constant Entity_Id  := Current_Entity_In_Scope (Nam);
      Decls       : List_Id;
      Loc         : Source_Ptr;
      Subp        : Entity_Id;
      Prev        : Entity_Id;
      Last_Formal : Entity_Id;
      Vsn_Name    : Name_Id;

   begin
      if Debug_Flag_C then
         Write_Str ("====  Compiling subprogram body ");
         Write_Name (Chars (Nam));
         Write_Str (" from ");
         Write_Location (Sloc (N));
         Write_Eol;
      end if;

      Trace_Scope (N, Nam, " Analyze subprogram");
      Set_Ekind (Nam, E_Subprogram_Body);

      --  Generic subprograms are handled separately. They always have a
      --  generic specification. Determine whether  current scope has
      --  a previous declaration.

      if Present (Gen_Id)
        and then not Is_Overloadable (Gen_Id)
      then
         if Ekind (Gen_Id) = E_Generic_Procedure
           or else Ekind (Gen_Id) = E_Generic_Function
         then
            Analyze_Generic_Subprogram_Body (N, Gen_Id);
            return;

         else
            --  Previous entity conflicts with subprogram name.
            --  Attempting to enter name will post error.

            Enter_Name (Nam);
            return;
         end if;

      --  Non-generic case, find the subprogram declaration, if one was
      --  seen, or enter new overloaded entity in the current scope.

      else
         Subp := Analyze_Spec (Spec);

         --  Get corresponding spec if not already set (the latter happens
         --  in the case of a subprogram instantiation, where the field
         --  was set during the instantiation

         if Nkind (N) = N_Subprogram_Body_Stub
           or else No (Corresponding_Spec (N))
         then
            Prev := Find_Corresponding_Spec (N);
         else
            Prev := Corresponding_Spec (N);
         end if;
      end if;

      --  Place subprogram on scope stack, and make formals visible. If there
      --  is a spec, the visible entity remains that of the spec. The defining
      --  entity for the body is entered in the chain of entities in that case,
      --  to insure that it is instantiated if it appears in  a generic unit.

      if Present (Prev) then
         if Is_Abstract (Prev) then
            Error_Msg_N ("an abstract subprogram cannot have a body", N);
         else
            Check_Conformance (Prev, Subp);
         end if;

         if Nkind (N) /= N_Subprogram_Body_Stub then
            Set_Corresponding_Spec (N, Prev);
         end if;

         Set_Corresponding_Body (Get_Declaration_Node (Prev), Subp);
         Install_Formals (Prev);
         Append_Entity (Subp, Current_Scope);
         Last_Formal := Last_Entity (Prev);
         New_Scope (Prev);

      else
         New_Overloaded_Entity (Subp);
         Install_Formals (Subp);
         New_Scope (Subp);
      end if;

      Set_Has_Completion (Subp);

      if Nkind (N) = N_Subprogram_Body_Stub then
         End_Scope; -- nothing else to process
         return;

      else
         Analyze_Declarations (Declarations (N));
         Check_Completion;

         --  Expand cleanup actions if necessary


         Analyze (Handled_Statement_Sequence (N));

         End_Use (Declarations (N));
         End_Scope;

         if Present (Prev) then

            --  Chain the declared entities on the id for the body.
            --  The id for the spec only holds the formals.

            if Present (Last_Formal) then
               Set_Next_Entity
                  (Last_Entity (Subp), Next_Entity (Last_Formal));
               Set_Next_Entity (Last_Formal, Empty);

            else
               Set_First_Entity (Subp, First_Entity (Prev));
               Set_First_Entity (Prev, Empty);
            end if;
         end if;
      end if;

      --  If function, make sure we had at least one return statement

      if Ekind (Nam) = E_Function
        or else Ekind (Nam) = E_Generic_Function
      then
         if (Present (Prev) and then Return_Present (Prev))
           or else (No (Prev) and then Return_Present (Subp))
         then
            null;
         else
            Error_Msg_N ("missing RETURN statement in function body", N);
         end if;
      end if;

   end Analyze_Subprogram_Body;

   ------------------------------------
   -- Analyze_Subprogram_Declaration --
   ------------------------------------

   procedure Analyze_Subprogram_Declaration (N : Node_Id) is
      Designator : constant Entity_Id := Analyze_Spec (Specification (N));
      Param_Spec : Node_Id;

   begin
      --  The visible part of an RCI unit, shall not contain the declaration
      --  of a subprogram to which a pragma Inline applies RM E.2.3(12-13).

      if Inside_Remote_Call_Interface_Unit (N) then

         if Is_Inlined (Defining_Unit_Simple_Name (Specification (N))) then
            Error_Msg_N
              ("?inlined subprogram cannot be declared in rci unit", N);
         end if;

         --  Iterate through the parameter specification list, checking that
         --  no access parameter and no limited type paramter in the list.

         if Present (Parameter_Specifications (Specification (N))) then
            Param_Spec := First (Parameter_Specifications (Specification (N)));

            while Present (Param_Spec) loop

               if Nkind (Parameter_Type (Param_Spec)) =
                 N_Access_Definition
               then
                  Error_Msg_N
                  ("?subprogram in rci unit cannot have an access parameter",
                    N);

               elsif Is_Limited_Type (Entity (Parameter_Type
                 (Param_Spec)))
               then
                  Error_Msg_N
                  ("?subprogram in rci unit cannot have a limited parameter",
                    N);
               end if;

               Param_Spec := Next (Param_Spec);
            end loop;
         end if;
      end if;

      Trace_Scope (N, Defining_Unit_Simple_Name (Specification (N)),
                                         " Analyze subprogram spec. ");

      if Debug_Flag_C then
         Write_Str ("====  Compiling subprogram spec ");
         Write_Name (Chars (Designator));
         Write_Str (" from ");
         Write_Location (Sloc (N));
         Write_Eol;
      end if;

      New_Overloaded_Entity (Designator);
      Check_Delayed_Subprogram (Designator);

   end Analyze_Subprogram_Declaration;

   ------------------------------
   -- Check_Delayed_Subprogram --
   ------------------------------

   procedure Check_Delayed_Subprogram (Designator : Entity_Id) is
      F : Entity_Id;

   begin
      if Has_Private_Component (Etype (Designator)) then
         Set_Is_Delayed (Designator);

      else
         F := First_Formal (Designator);

         while Present (F) loop
            if Has_Private_Component (Etype (F)) then
               Set_Is_Delayed (Designator);
               exit;
            end if;

            F := Next_Formal (F);
         end loop;
      end if;
   end Check_Delayed_Subprogram;

   -----------------------
   -- Check_Conformance --
   -----------------------

   procedure Check_Conformance (Old_Id, New_Id : Entity_Id) is
      Old_Spec : constant List_Id :=
        Parameter_Specifications
               (Specification (Get_Declaration_Node (Old_Id)));

      New_Spec : constant List_Id :=
        Parameter_Specifications
               (Specification (Get_Declaration_Node (New_Id)));

      Old_Param_Spec : Node_Id;
      New_Param_Spec : Node_Id;
      Old_Formal     : Entity_Id;
      New_Formal     : Entity_Id;

      procedure Conformance_Error is
      begin
         Error_Msg_Sloc := Sloc (Old_Id);
         Error_Msg_N ("no conformance with declaration#", New_Id);
      end Conformance_Error;

   --  Start of processing for Check_Conformance

   begin
      if not Present (Old_Spec) then
         if not Present (New_Spec) then
            return;
         else
            Conformance_Error;
         end if;

      else
         Old_Param_Spec := First (Old_Spec);
         New_Param_Spec := First (New_Spec);

         while Present (Old_Param_Spec)
           and then Present (New_Param_Spec)
         loop

            Old_Formal := Defining_Identifier (Old_Param_Spec);
            New_Formal := Defining_Identifier (New_Param_Spec);

            if Chars (Old_Formal) = Chars (New_Formal)
              and then Ekind (Old_Formal) = Ekind (New_Formal)
              and then
                (Etype (Old_Formal) = Etype (New_Formal)
                  or else Full_View (Etype (Old_Formal)) = Etype (New_Formal)
                  or else
                    (Ekind (Etype (Old_Formal)) = E_Anonymous_Access_Type
                      and then
                        Ekind (Etype (New_Formal)) = E_Anonymous_Access_Type
                      and then
                        Directly_Designated_Type (Etype (Old_Formal)) =
                        Directly_Designated_Type (Etype (New_Formal))))
            then
               null;
            else
               Conformance_Error;
               return;
            end if;

            --  Here is where we check the More_Ids and Prev_Ids flags to
            --  make sure they match. This catches a misconformance like

            --    A,B : Integer
            --    A : Integer; B : Integer

            --  which are represented identically in the tree except for the
            --  setting of these flags.

            if More_Ids (Old_Param_Spec) /= More_Ids (New_Param_Spec)
              or else Prev_Ids (Old_Param_Spec) /= Prev_Ids (New_Param_Spec)
            then
               Conformance_Error;
            end if;

            Old_Param_Spec := Next (Old_Param_Spec);
            New_Param_Spec := Next (New_Param_Spec);
         end loop;
      end if;
   end Check_Conformance;

   -----------------------------
   -- Enter_Overloaded_Entity --
   -----------------------------

   procedure Enter_Overloaded_Entity (S : Entity_Id) is
      E : Entity_Id;

   begin
      E := Current_Entity (S);
      Set_Is_Immediately_Visible (S);
      Set_Current_Entity (S);
      Set_Homonym (S, E);

      E := Current_Entity_In_Scope (S);
      if Present (E) then
         Set_Has_Homonym (E);
         Set_Has_Homonym (S);
      end if;

      Append_Entity (S, Current_Scope);
      Set_Public_Status (S);

      if Debug_Flag_E then
         Write_Str ("New overloaded entity chain: ");
         Write_Name (Chars (S));
         E := S;

         while Present (E) loop
            Write_Str (" "); Write_Int (Int (E));
            E := Homonym (E);
         end loop;

         Write_Eol;
      end if;
   end Enter_Overloaded_Entity;

   -----------------------------
   -- Find_Corresponding_Spec --
   -----------------------------

   function Find_Corresponding_Spec (N : Node_Id) return Entity_Id is
      Spec       : constant Node_Id   := Specification (N);
      Designator : constant Entity_Id := Defining_Unit_Simple_Name (Spec);
      E          : Entity_Id;

   begin
      E := Current_Entity (Designator);

      while Present (E) loop

         if Scope (E) = Current_Scope
           and then Type_Conformant (E, Designator)
         then
            if not Has_Completion (E) then

               if Nkind (N) /= N_Subprogram_Body_Stub then
                  Set_Corresponding_Spec (N, E);
               end if;

               Set_Has_Completion (E);
               return E;

            --  If body already exists, this is an error unless the
            --  previous declaration is the implicit declaration of
            --  a derived subprogram.

            elsif No (Alias (E)) and then not Is_Internal (E) then
               Error_Msg_N ("duplicate subprogram body", N);
            end if;
         end if;

         E := Homonym (E);
      end loop;

      --  On exit, we know that no previous declaration of subprogram exists

      return Empty;
   end Find_Corresponding_Spec;

   --------------------
   -- Install_Entity --
   --------------------

   procedure Install_Entity (E : Entity_Id) is
      Prev : constant Entity_Id := Current_Entity (E);

   begin
      Set_Is_Immediately_Visible (E);
      Set_Current_Entity (E);
      Set_Homonym (E, Prev);
   end Install_Entity;

   ---------------------
   -- Install_Formals --
   ---------------------

   procedure Install_Formals (Id : Entity_Id) is
      F : Entity_Id;

   begin
      F := First_Formal (Id);

      while Present (F) loop
         Install_Entity (F);
         F := Next_Formal (F);
      end loop;
   end Install_Formals;

   ----------------------
   -- May_Need_Actuals --
   ----------------------

   procedure May_Need_Actuals (Fun : Entity_Id) is
      F : Entity_Id;
      B : Boolean;

   begin
      F := First_Formal (Fun);
      B := True;

      while Present (F) loop
         if No (Default_Value (F)) then
            B := False;
            exit;
         end if;

         F := Next_Formal (F);
      end loop;

      Set_Needs_No_Actuals (Fun, B);
   end May_Need_Actuals;

   ---------------------
   -- Mode_Conformant --
   ---------------------

   function Mode_Conformant (S1, S2 : Entity_Id) return Boolean is
      P1 : Entity_Id;
      P2 : Entity_Id;

      function Same_Type (T1 : Entity_Id; T2 : Entity_Id) return Boolean;
      --  Check whether T1 and T2 are the same type (for conformance purposes)

      function Same_Type (T1 : Entity_Id; T2 : Entity_Id) return Boolean is
      begin
         --  For a general access type, the designated subtypes must
         --  match statically.

         return Base_Type (T1) = Base_Type (T2)
           or else (Ekind (T1) = E_Anonymous_Access_Type
             and then Ekind (T2) = E_Anonymous_Access_Type
             and then Directly_Designated_Type (T1) =
                      Directly_Designated_Type (T2));
      end Same_Type;

   --  Start of processing for Mode_Conformant

   begin
      P1 := First_Formal (S1);
      P2 := First_Formal (S2);

      if not Same_Type (Etype (S1), Etype (S2)) then
         return False;

      else
         while Present (P1) and then Present (P2) loop
            if not Same_Type (Etype (P1), Etype (P2))
              or else Ekind (P1) /= Ekind (P2)
            then
               return False;
            end if;

            P1 := Next_Formal (P1);
            P2 := Next_Formal (P2);
         end loop;

         --  Both must be empty on exit from list to return True

         return P1 = P2;
      end if;
   end Mode_Conformant;

   ---------------------------
   -- New_Overloaded_Entity --
   ---------------------------

   procedure New_Overloaded_Entity (S : Entity_Id) is
      E        : Entity_Id := Current_Entity_In_Scope (S);
      Prev_Vis : Entity_Id := Empty;

   begin
      if No (E) then
         Enter_Overloaded_Entity (S);
         Check_Dispatching_Operation (S, Empty);

      elsif not Is_Overloadable (E) then
         Error_Msg_N ("duplicate identifier:&", S);

      else
         --  E exists and is overloadable. Determine whether S is the body
         --  of E, a new overloaded entity with a different signature, or
         --  an error altogether.

         while Present (E) and then Scope (E) = Current_Scope loop
            if Type_Conformant (E, S) then

               --  If the old and new entities have the same profile and
               --  one is not the body of the other, then this is an error,
               --  unless one of them is implicitly declared.

               if Present (Alias (S)) then

                  --  When an derived operation is overloaded it may be
                  --  due to the fact that the full view of a private extension
                  --  re-inherits. It has to be dealt with.

                  Check_Operation_From_Private_View (S, E);

                  --  In any case the derived operation remains hidden by
                  --  the existing declaration.

                  return;

               elsif Present (Alias (E)) or else Is_Internal (E) then

                  --  E is a derived operation or an internal operator which
                  --  is being overridden. Remove E from further visibility.
                  --  Furthermore, if E is a dispatching operation, it must be
                  --  replaced in the list of primitive operations of its type

                  declare
                     Prev : Entity_Id;

                  begin
                     Prev := First_Entity (Current_Scope);

                     while Next_Entity (Prev) /= E loop
                        Prev := Next_Entity (Prev);
                     end loop;

                     --  E must be removed both from the entity_list of the
                     --  current scope, and from the visibility chain

                     if Debug_Flag_E then
                        Write_Str ("Override implicit operation ");
                        Write_Int (Int (E));
                        Write_Eol;
                     end if;

                     if Prev_Vis /= Empty then
                        Set_Homonym (Prev_Vis, Homonym (E));
                        --  Skip E in the visibility chain
                     else
                        Set_Name_Entity_Id (Chars (E), Homonym (E));
                     end if;

                     Set_Next_Entity (Prev, Next_Entity (E));

                     if No (Next_Entity (Prev)) then
                        Set_Last_Entity (Current_Scope, Prev);
                     end if;

                     Enter_Overloaded_Entity (S);
                     if Is_Dispatching_Operation (E) then
                        Check_Dispatching_Operation (S, E);
                     else
                        Check_Dispatching_Operation (S, Empty);
                     end if;

                     return;
                  end;

               --  Here we have a real error (identical profile)

               else
                  if Sloc (E) = Standard_Location then
                     Error_Msg_N
                       ("declaration of& conflicts with name in Standard", S);
                  else
                     Error_Msg_Sloc := Sloc (E);
                     Error_Msg_N ("declaration of& conflicts with that#", S);
                  end if;

                  return;
               end if;

            else
               null;
            end if;

            Prev_Vis := E;
            E := Homonym (E);
         end loop;

         --  On exit, we know that S is a new entity

         Enter_Overloaded_Entity (S);
         Check_Dispatching_Operation (S, Empty);
      end if;

   end New_Overloaded_Entity;

   ---------------------
   -- Process_Formals --
   ---------------------

   procedure Process_Formals
     (S           : Entity_Id;
      T           : List_Id;
      Related_Nod : Node_Id)
   is
      Param_Spec  : Node_Id;
      Formal      : Entity_Id;
      Formal_Type : Entity_Id;
      Default     : Node_Id;

   begin
      --  In order to prevent premature use of the formals in the same formal
      --  part, the Ekind is left undefined until all default expressions are
      --  analyzed. The Ekind is established in a separate loop at the end.

      Param_Spec := First (T);

      while Present (Param_Spec) loop

         --  Case of ordinary parameters

         if Nkind (Parameter_Type (Param_Spec)) /= N_Access_Definition then
            Find_Type (Parameter_Type (Param_Spec));
            Formal_Type := Entity (Parameter_Type (Param_Spec));

            if Ekind (Formal_Type) = E_Incomplete_Type
              or else (Is_Class_Wide_Type (Formal_Type)
                and then Ekind (Etype (Formal_Type)) = E_Incomplete_Type)
            then
               Error_Msg_N ("invalid use of incomplete type&",
                 Etype (Parameter_Type (Param_Spec)));
            end if;

         else
            --  An access formal type

            Formal_Type :=
              Access_Definition (Related_Nod, Parameter_Type (Param_Spec));
         end if;

         Formal := Defining_Identifier (Param_Spec);
         Enter_Name (Formal);
         Set_Etype (Formal, Formal_Type);

         Default :=  Expression (Param_Spec);

         if Present (Default) then
            Analyze_Expression (Default);
            Resolve (Default, Formal_Type);

            if Out_Present (Param_Spec) then
               Error_Msg_N
                 ("default initialization only allowed for IN parameters",
                  Param_Spec);
            end if;
         end if;

         Param_Spec := Next (Param_Spec);
      end loop;

      --  Now set the kind (mode) of each formal

      Param_Spec := First (T);

      while Present (Param_Spec) loop
         Formal := Defining_Identifier (Param_Spec);
         Set_Formal_Mode (Formal);

         if Ekind (Formal) = E_In_Parameter then
            Set_Default_Value (Formal, Expression (Param_Spec));
         end if;

         Param_Spec := Next (Param_Spec);
      end loop;

   end Process_Formals;

   ---------------------
   -- Set_Formal_Mode --
   ---------------------

   procedure Set_Formal_Mode (Formal_Id : Entity_Id) is
      Spec : constant Node_Id := Parent (Formal_Id);

   begin
      if Out_Present (Spec) then

         if Ekind (Scope (Formal_Id)) = E_Function
           or else Ekind (Scope (Formal_Id)) = E_Generic_Function
         then
            Error_Msg_N ("functions can only have IN parameters", Spec);
            Set_Ekind (Formal_Id, E_In_Parameter);

         elsif In_Present (Spec) then
            Set_Ekind (Formal_Id, E_In_Out_Parameter);

         else
            Set_Ekind (Formal_Id, E_Out_Parameter);
         end if;

      else
         Set_Ekind (Formal_Id, E_In_Parameter);
      end if;
   end Set_Formal_Mode;

   ---------------------
   -- Type_Conformant --
   ---------------------

   function Type_Conformant (S1, S2 : Entity_Id) return Boolean is
      P1 : Entity_Id;
      P2 : Entity_Id;

      function Same_Type (T1, T2 : Entity_Id) return Boolean;

      function Same_Type (T1, T2 : Entity_Id) return Boolean is
      begin
         return Base_Type (T1) = Base_Type (T2)

           or else (Ekind (T1) = E_Anonymous_Access_Type
           and then Ekind (T2) = E_Anonymous_Access_Type
           and then Same_Type (Directly_Designated_Type (T1),
             Directly_Designated_Type (T2)));

      end Same_Type;

   --  Start of processing for Type_Conformant

   begin
      if not Same_Type (Etype (S1), Etype (S2)) then
         return False;

      else
         P1 := First_Formal (S1);
         P2 := First_Formal (S2);

         while Present (P1) and then Present (P2) loop
            if not Same_Type (Etype (P1), Etype (P2)) then
               return False;
            end if;

            P1 := Next_Formal (P1);
            P2 := Next_Formal (P2);
         end loop;

         --  Both must be empty on exit from list to return True

         return (P1 = P2);
      end if;
   end Type_Conformant;

   -------------------------------
   -- Valid_Operator_Definition --
   -------------------------------

   procedure Valid_Operator_Definition (Designator : Entity_Id) is
      N    : Integer := 0;
      F    : Entity_Id;
      Id   : constant Name_Id := Chars (Designator);
      N_OK : Boolean;

   begin
      F := First_Formal (Designator);

      while Present (F) loop
         N := N + 1;

         if Present (Default_Value (F)) then
            Error_Msg_N
              ("default values not allowed for operator parameters",
               Parent (F));
         end if;

         F := Next_Formal (F);
      end loop;

      --  Verify that user-defined operators have proper number of arguments
      --  First case of operators which can only be unary

      if Id = Name_Op_Not
        or else Id = Name_Op_Abs
      then
         N_OK := (N = 1);

      --  Case of operators which can be unary or binary

      elsif Id = Name_Op_Add
        or Id = Name_Op_Subtract
      then
         N_OK := (N in 1 .. 2);

      --  All other operators can only be binary

      else
         N_OK := (N = 2);
      end if;

      if not N_OK then
         Error_Msg_N
           ("incorrect number of arguments for operator", Designator);
      end if;

      --  TBSL: no explicit definition of inequality can return Boolean ???
   end Valid_Operator_Definition;

end Sem_Ch6;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.169
--  date: Wed Aug 24 19:37:20 1994;  author: dewar
--  Minor reformatting
--  ----------------------------
--  revision 1.170
--  date: Sun Aug 28 21:32:21 1994;  author: comar
--  (Enter_Overloaded_Entity): remove call to Check_Dispatching_Operation
--  (New_Overloaded_Entity): call Check_Dispatching_Operation after entering
--   the entity because only in this context we can be sure of recognizing the
--   overriding case. Don't call directly Override_Disaptching_Operation this
--   is now done in Check_Dispatching_Operation.
--  ----------------------------
--  revision 1.171
--  date: Wed Aug 31 00:05:51 1994;  author: schonber
--  (Enter_Overloaded_Entity): find previous entity in scope to determine
--   whether new entity has a local homonym.
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
