/*
 * @(#)file.c	1.6 91/09/06
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <sys/vfs.h>

#include "defs.h"

#ifdef __linux
#define	minor(x)	((x) & 0xff)
#define	major(x)	((x) >> 8 & 0xff)
#endif

static Xlat *op, openmodes[] = {
	{O_RDONLY, "RDONLY"},	/* read enable */
	{O_WRONLY, "WRONLY"},	/* write enable */
	{O_RDWR, "RDWR"},	/* read/write enable */
#ifdef linux
	{O_NDELAY, "NDELAY"},	/* non blocking I/O (4.2 style) */
	{O_APPEND, "APPEND"},	/* append (writes guaranteed at the end) */
	{O_CREAT, "CREAT"},	/* open with file create */
	{O_TRUNC, "TRUNC"},	/* open with truncation */
	{O_EXCL, "EXCL"},	/* error on open if file exists */
	{O_SYNC, "SYNC"},	/* do all writes synchronously */
	{O_NOCTTY, "NOCTTY"},	/* don't assign a ctty on this open */
#else
	{_FNDELAY, "NDELAY"},	/* non blocking I/O (4.2 style) */
	{_FAPPEND, "APPEND"},	/* append (writes guaranteed at the end) */
	{_FMARK, "MARK"},	/* internal; mark during gc() */
	{_FDEFER, "DEFER"},	/* internal; defer for next gc pass */
	{_FASYNC, "ASYNC"},	/* signal pgrp when data ready */
	{_FSHLOCK, "SHLOCK"},	/* BSD flock() shared lock present */
	{_FEXLOCK, "EXLOCK"},	/* BSD flock() exclusive lock present */
	{_FCREAT, "CREAT"},	/* open with file create */
	{_FTRUNC, "TRUNC"},	/* open with truncation */
	{_FEXCL, "EXCL"},	/* error on open if file exists */
	{_FNBIO, "NBIO"},	/* non blocking I/O (sys5 style) */
	{_FSYNC, "SYNC"},	/* do all writes synchronously */
	{_FNONBLOCK, "NONBLOCK"},	/* non blocking I/O (POSIX style) */
	{_FNOCTTY, "NOCTTY"},	/* don't assign a ctty on this open */
#endif
	{0, NULL},
};

int
sys_open(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
		/* flags */
		printxval(openmodes, tcp->u_args[1] & O_ACCMODE, "O_???");
		addflags(openmodes, tcp->u_args[1] & ~O_ACCMODE);
		if (tcp->u_args[1] & O_CREAT) {
			/* mode */
			fprintf(outf, ", %#o", tcp->u_args[2]);
		}
	}
	return 0;
}

int
sys_creat(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#o", tcp->u_args[1]);
	}
	return 0;
}

int
sys_access(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#o", tcp->u_args[1]);
	}
	return 0;
}

int
sys_umask(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%#o", tcp->u_args[0]);
	}
	return 0;
}

static Xlat whence[] = {
	SEEK_SET,	"SEEK_SET",
	SEEK_CUR,	"SEEK_CUR",
	SEEK_END,	"SEEK_END",
	0,		NULL,
};

int
sys_lseek(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d, %u, ", tcp->u_args[0], tcp->u_args[1]);
		printxval(whence, tcp->u_args[2], "L_???");
	}
	return 0;
}

int
sys_truncate(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %u", tcp->u_args[1]);
	}
	return 0;
}

int
sys_ftruncate(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d, %u", tcp->u_args[0], tcp->u_args[1]);
	}
	return 0;
}


/* several stats */
static void
printstat(pid, addr)
{
	struct stat statbuf;

	if (umove(pid, addr, sizeof statbuf, (char *)&statbuf) < 0) {
		fprintf(outf, "{...}");
		return;
	}
	fprintf(outf,
	"{dev %x %x ino %u mode %#o nlink %d uid %d gid %d size %u ...}",
		major(statbuf.st_dev), minor(statbuf.st_dev),
		statbuf.st_ino, statbuf.st_mode, statbuf.st_nlink,
		statbuf.st_uid, statbuf.st_gid, statbuf.st_size);
}

int
sys_stat(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
	} else {
		if (syserror(tcp))
			fprintf(outf, "%#x", tcp->u_args[1]);
		else
			printstat(tcp->pid, tcp->u_args[1]);
	}
	return 0;
}

int
sys_fstat(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d, ", tcp->u_args[0]);
	} else {
		if (syserror(tcp))
			fprintf(outf, "%#x", tcp->u_args[1]);
		else
			printstat(tcp->pid, tcp->u_args[1]);
	}
	return 0;
}

int
sys_lstat(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
	} else {
		if (syserror(tcp))
			fprintf(outf, "%#x", tcp->u_args[1]);
		else
			printstat(tcp->pid, tcp->u_args[1]);
	}
	return 0;
}

static void
printstatfs(pid, addr)
{
	struct statfs statbuf;

	if (umove(pid, addr, sizeof statbuf, (char *)&statbuf) < 0) {
		fprintf(outf, "[...]");
		return;
	}
	fprintf(outf,
	"[type %#x bsize %u blocks %u free %u files %u ffree %u ..]",
		statbuf.f_type, statbuf.f_bsize, statbuf.f_blocks,
		statbuf.f_bfree, statbuf.f_files, statbuf.f_ffree);
}

int
sys_statfs(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		(void)printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
	} else {
		if (syserror(tcp))
			fprintf(outf, "%#x", tcp->u_args[1]);
		else
			printstatfs(tcp->pid, tcp->u_args[1]);
	}
	return 0;
}

int
sys_fstatfs(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
	} else {
		if (syserror(tcp))
			fprintf(outf, "%#x", tcp->u_args[1]);
		else
			printstatfs(tcp->pid, tcp->u_args[1]);
	}
	return 0;
}

/* directory */
int
sys_chdir(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
	}
	return 0;
}

int
sys_mkdir(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#o", tcp->u_args[1]);
	}
	return 0;
}

int
sys_rmdir(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
	}
	return 0;
}

int
sys_fchdir(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d", tcp->u_args[0]);
	}
	return 0;
}

int
sys_chroot(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
	}
	return 0;
}

int
sys_fchroot(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d", tcp->u_args[0]);
	}
	return 0;
}

int
sys_link(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
		printstr(tcp->pid, tcp->u_args[1], -1);
	}
	return 0;
}

int
sys_unlink(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
	}
	return 0;
}

int
sys_symlink(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
		printstr(tcp->pid, tcp->u_args[1], -1);
	}
	return 0;
}

int
sys_readlink(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
	} else {
		if (syserror(tcp))
			fprintf(outf, "%#x", tcp->u_args[1]);
		else
			printstr(tcp->pid, tcp->u_args[1], tcp->u_rval);
		fprintf(outf, ", %u", tcp->u_args[2]);
	}
	return 0;
}

int
sys_rename(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
		printstr(tcp->pid, tcp->u_args[1], -1);
	}
	return 0;
}

int
sys_chown(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %u, %u", tcp->u_args[1], tcp->u_args[2]);
	}
	return 0;
}

int
sys_fchown(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d, %u, %u",
			tcp->u_args[0], tcp->u_args[1], tcp->u_args[2]);
	}
	return 0;
}

int
sys_chmod(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#o", tcp->u_args[1]);
	}
	return 0;
}

int
sys_fchmod(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%d, %#o", tcp->u_args[0], tcp->u_args[1]);
	}
	return 0;
}

int
sys_utimes(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
		printtv(tcp->pid, tcp->u_args[1]);
	}
	return 0;
}

int
sys_utime(tcp)
struct tcb *tcp;
{
	long ut[2];

	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", ");
		if (!tcp->u_args[1])
			fprintf(outf, "(struct utimbuf *) 0");
		else
			if (umove(tcp->pid, tcp->u_args[1], sizeof ut,
					(char *) ut) < 0)
				fprintf(outf, "{...}");
			else
				fprintf(outf, "{%u, %u}", ut[0], ut[1]);
	}
	return 0;
}

int
sys_mknod(tcp)
struct tcb *tcp;
{
	int mode = tcp->u_args[1];

	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#o(%s)", mode,
			mode==S_IFCHR?"CHR":(
				mode==S_IFBLK?"BLK":(
					mode==S_IFREG?"REG":(
						mode==S_IFIFO?"FIFO":""
					)
				)
			)
		);
		fprintf(outf, ", dev %x %x",
				major(tcp->u_args[2]), minor(tcp->u_args[2]));
	}
	return 0;
}

int
sys_mkfifo(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#o", tcp->u_args[1]);
	}
	return 0;
}

int
sys_fsync(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u", tcp->u_args[0]);
	}
	return 0;
}

int
sys_getdents(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u", tcp->u_args[0]);
		fprintf(outf, ", %#x", tcp->u_args[1]);
		fprintf(outf, ", %u", tcp->u_args[2]);
	}
	return 0;
}

#ifdef linux
#include <sys/dirent.h>

static void
printdir(pid, addr)
{
	struct dirent d;

	if (umove(pid, addr, sizeof d, (char *)&d) < 0) {
		fprintf(outf, "{...}");
		return;
	} else {
		fprintf(outf, "{ino %d off %d reclen %u name ",
			d.d_ino, d.d_off, d.d_reclen);
		printstr(pid, (int)((struct dirent *)addr)->d_name, d.d_reclen);
		fprintf(outf, "}");
	}
}

int
sys_readdir(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		fprintf(outf, "%u, ", tcp->u_args[0]);
	} else {
		if (syserror(tcp))
			fprintf(outf, "%#x", tcp->u_args[1]);
		else
			printdir(tcp->pid, tcp->u_args[1]);
		fprintf(outf, ", %u", tcp->u_args[2]);
	}
	return 0;
}

int
sys_uselib(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
	}
	return 0;
}

int
sys_oldstat(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#x", tcp->u_args[1]);
	}
	return 0;
}

int
sys_oldlstat(tcp)
struct tcb *tcp;
{
	if (entering(tcp)) {
		printstr(tcp->pid, tcp->u_args[0], -1);
		fprintf(outf, ", %#x", tcp->u_args[1]);
	}
	return 0;
}
#endif
