------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ P R A G                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.131 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

--  This unit contains the semantic processing for all pragmas, both language
--  and implementation defined. To add a new pragma, see packages Par.Prag,
--  and Sem_Prag. A certain amount of syntax checking has been done in each
--  case by the routines in Par.Prag. This procedure carries out the remaining
--  semantic checks.

with Atree;    use Atree;
with Einfo;    use Einfo;
with Errout;   use Errout;
with Exp_Util; use Exp_Util;
with Features; use Features;
with Lib;      use Lib;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Eval; use Sem_Eval;
with Sem_Intr; use Sem_Intr;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Stringt;  use Stringt;
with Tbuild;   use Tbuild;
with Ttypes;
with Uintp;    use Uintp;

package body Sem_Prag is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Is_Generic_Subprogram (Id : Entity_Id) return Boolean;
   --  Return True if Id is a generic procedure or a function

   --------------------
   -- Analyze_Pragma --
   --------------------

   procedure Analyze_Pragma (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);

      Nargs : Int;
      --  Number of pragma argument associations

      function Arg1 return Node_Id;
      function Arg2 return Node_Id;
      function Arg3 return Node_Id;
      function Arg4 return Node_Id;
      --  Obtain specified Pragma_Argument_Association. It is allowable to
      --  call the routine for the argument one past the last present argument,
      --  but that is the only case in which a non-present argument can be
      --  referenced.

      procedure Error_Pragma (Msg : String);
      --  Outputs error message for current pragma. The message contains an %
      --  that will be replaced with the pragma name, and the flag is placed
      --  on the pragma itself.

      procedure Error_Pragma_Argument (Msg : String; Arg : Node_Id);
      --  Outputs error message for current pragma. The message contains an %
      --  that will be replaced with the pragma name, and the flag is placed
      --  on the expression of the pragma argument specified by Arg.

      function Find_Lib_Unit_Name return Entity_Id;
      --  Find the defining entity of the spec library unit name.

      procedure Find_Program_Unit_Name (Id : Node_Id);
      --  If the pragma is a compilation unit pragma,  the id must denote the
      --  compilation unit in the same compilation,  and the pragma must appear
      --  in the list of preceding or trailing pragmas. If it is a program
      --  unit pragma that is not a compilation unit pragma, then the
      --  identifier must be visible.

      function Is_Before_First_Decl
        (Pragma_Node : Node_Id;
         Decls       : List_Id)
         return        Boolean;
      --  Return True if Pragma_Node is before the first declarative item in
      --  Decls where Decls is the list of declarative items.

      function Is_Inside_Generic_Instantiation
        (Pragma_Node : Node_Id)
         return        Boolean;
      --  Return True if Pragma_Node is inside a generic instantiation.

      procedure Pragma_Misplaced;
      --  Issue fatal error message for misplaced pragma

      procedure Pragma_Not_Implemented;
      --  Issue warning message for unimplemented pragma

      procedure Process_Interface_Name
        (Subprogram_Def : Entity_Id;
         Ext_Arg        : Node_Id;
         Link_Arg       : Node_Id);
      --  Given the last two arguments of pragma Import, pragma Export, or
      --  pragma Interface_Name, performs validity checks and sets the
      --  Interface_Name field of the given subprogram entity to the
      --  appropriate external or link name, depending on the arguments
      --  given. Ext_Arg is always present, but Link_Arg may be missing.
      --  Note that Ext_Arg may represent the Link_Name if Link_Arg is
      --  missing, and appropriate named notation is used for Ext_Arg.

      function Valid_Unit_Pragma return Boolean;
      --  Legality checks for library unit pragmas (Pure, Preelaborate...).

      ----------
      -- Arg1 --
      ----------

      function Arg1 return Node_Id is
      begin
         return First (Pragma_Argument_Associations (N));
      end Arg1;

      ----------
      -- Arg2 --
      ----------

      function Arg2 return Node_Id is
      begin
         return Next (Arg1);
      end Arg2;

      ----------
      -- Arg3 --
      ----------

      function Arg3 return Node_Id is
      begin
         return Next (Arg2);
      end Arg3;

      ----------
      -- Arg4 --
      ----------

      function Arg4 return Node_Id is
      begin
         return Next (Arg3);
      end Arg4;

      ------------------
      -- Error_Pragma --
      ------------------

      procedure Error_Pragma (Msg : String) is
      begin
         Error_Msg_N (Msg, N);
      end Error_Pragma;

      ---------------------------
      -- Error_Pragma_Argument --
      ---------------------------

      procedure Error_Pragma_Argument (Msg : String; Arg : Node_Id) is
      begin
         Error_Msg_Name_1 := Chars (N);
         Error_Msg_N (Msg, Expression (Arg1));
      end Error_Pragma_Argument;

      ----------------------------
      -- Find_Lib_Unit_Name --
      ----------------------------

      function Find_Lib_Unit_Name return Entity_Id is
         Lib_Unit    : constant Node_Id := Enclosing_Lib_Unit_Node (N);
         Unit_Entity : Entity_Id        := Enclosing_Lib_Unit_Entity (N);
         Unit_Kind   : Node_Kind        := Nkind (Unit (Lib_Unit));

      begin
         if Unit_Kind in N_Generic_Renaming_Declaration
           or else Unit_Kind = N_Package_Renaming_Declaration
           or else Unit_Kind = N_Subprogram_Renaming_Declaration
         then
            --  Library_Unit_Renaming not allowed for Pure, Preelaborate

            Error_Pragma ("pragma% cannot follow library unit renaming");
            Unit_Entity := Empty;
         end if;

         return Unit_Entity;
      end Find_Lib_Unit_Name;

      ----------------------------
      -- Find_Program_Unit_Name --
      ----------------------------

      procedure Find_Program_Unit_Name (Id : Node_Id) is
         Unit_Name : Entity_Id;
         Unit_Kind : Node_Kind;
         P         : constant Node_Id := Parent (N);

      begin
         if Nkind (P) = N_Compilation_Unit then
            Unit_Kind := Nkind (Unit (P));

            if Unit_Kind = N_Subprogram_Declaration
              or else Unit_Kind = N_Package_Declaration
              or else Unit_Kind in N_Generic_Declaration
            then
               Unit_Name :=
                 Defining_Unit_Simple_Name (Specification (Unit (P)));

               if Chars (Id) = Chars (Unit_Name) then
                  Set_Entity (Id, Unit_Name);
                  Set_Etype (Id, Etype (Unit_Name));
               else
                  Error_Pragma
                    ("cannot find program unit referenced by pragma%");
                  Set_Etype (Id, Any_Type);
               end if;

            else
               Error_Pragma ("pragma% inapplicable to this unit");
               Set_Etype (Id, Any_Type);
            end if;

         else
            Analyze (Id);
         end if;

      end Find_Program_Unit_Name;

      --------------------------
      -- Is_Before_First_Decl --
      --------------------------

      function Is_Before_First_Decl
        (Pragma_Node : Node_Id;
         Decls       : List_Id)
         return        Boolean
      is
         Item            : Node_Id := First (Decls);

      begin
         if Is_Inside_Generic_Instantiation (Pragma_Node) then
            return True;
         end if;

         --  Only pragmas can come before this Pragma_Node.

         loop
            if No (Item) or else Nkind (Item) /= N_Pragma then
               return False;

            elsif Item = Pragma_Node then
               return True;
            end if;

            Item := Next (Item);
         end loop;

      end Is_Before_First_Decl;

      -------------------------------------
      -- Is_Inside_Generic_Instantiation --
      -------------------------------------

      function Is_Inside_Generic_Instantiation
        (Pragma_Node     : Node_Id)
         return            Boolean
      is
         Parent_Node : Node_Id   := Parent (Pragma_Node);
         Parent_Kind : Node_Kind := Nkind (Parent_Node);

      begin
         --  Notice that a library unit pragma inside generic body is
         --  misplaced and will be found later.

         if Parent_Kind = N_Package_Specification then
            if Present (Generic_Parent (Parent_Node)) then
               return True;
            end if;

         --  It is impossible to be inside (generic) subprogram_spec

         elsif Parent_Kind = N_Subprogram_Body then
            if Present (Generic_Parent (Parent (Corresponding_Spec (
              Parent (Parent_Node))))) then
               return True;
            end if;
         end if;

         return False;

      end Is_Inside_Generic_Instantiation;

      ----------------------
      -- Pragma_Misplaced --
      ----------------------

      procedure Pragma_Misplaced is
      begin
         Error_Pragma ("incorrect placement of pragma%");
      end Pragma_Misplaced;

      ----------------------------
      -- Pragma_Not_Implemented --
      ----------------------------

      procedure Pragma_Not_Implemented is
      begin
         Error_Pragma ("pragma% not implemented?");
      end Pragma_Not_Implemented;

      ----------------------------
      -- Process_Interface_Name --
      ----------------------------

      procedure Process_Interface_Name
        (Subprogram_Def : Entity_Id;
         Ext_Arg        : Node_Id;
         Link_Arg       : Node_Id)
      is
         Ext_Nam  : Node_Id;
         Link_Nam : Node_Id;

      begin
         if No (Link_Arg) then
            if Chars (Ext_Arg) = No_Name
              or else Chars (Ext_Arg) = Name_External_Name
            then
               Ext_Nam  := Expression (Ext_Arg);
               Link_Nam := Empty;
            else
               Ext_Nam  := Empty;
               Link_Nam := Expression (Ext_Arg);
            end if;

         else
            Ext_Nam  := Expression (Ext_Arg);
            Link_Nam := Expression (Link_Arg);
         end if;

         --  Check expressions for external name and link name are static

         if Present (Ext_Nam) then
            Analyze (Ext_Nam);
            Resolve (Ext_Nam, Standard_String);
            Check_Static_Expression (Ext_Nam);
         end if;

         if Present (Link_Nam) then
            Analyze (Link_Nam);
            Resolve (Link_Nam, Standard_String);
            Check_Static_Expression (Link_Nam);
         end if;

         --  If there is no link name, just set the external name

         if No (Link_Nam) then
            Set_Interface_Name (Subprogram_Def, Ext_Nam);

         --  For the Link_Name case, the given literal is preceded by an
         --  asterisk, which indicates to GCC that the given name should
         --  be taken literally, and in particular that no prepending of
         --  underlines should occur, even in systems where this is the
         --  normal default.

         else
            Start_String;
            Store_String_Char (Get_Char_Code ('*'));

            for J in 1 .. String_Length (Strval (Link_Nam)) loop
               Store_String_Char (Get_String_Char (Strval (Link_Nam), J));
            end loop;

            Link_Nam :=
              Make_String_Literal (Sloc (Link_Nam), End_String);

            Set_Interface_Name (Subprogram_Def, Link_Nam);
         end if;
      end Process_Interface_Name;

      ---------------------------------
      -- Process_Suppress_Unsuppress --
      ---------------------------------

      procedure Process_Suppress_Unsuppress (Sense : Boolean) is
         C         : constant Check_Id :=
                       Get_Check_Id (Chars (Expression (Arg1)));
         E_Id      : Node_Id;
         E         : Entity_Id;
         Effective : Boolean;

         procedure Suppress_Unsuppress_Echeck (E : Entity_Id; C : Check_Id);
         --  Used to suppress a single check on the given entity

         procedure Suppress_Unsuppress_Echeck (E : Entity_Id; C : Check_Id) is
         begin

            --  First set appropriate suppress flags in the entity

            case C is
               when Access_Check =>
                  Effective := Suppress_Access_Checks (E);
                  Set_Suppress_Access_Checks (E, Sense);

               when Accessibility_Check =>
                  Effective := Suppress_Accessibility_Checks (E);
                  Set_Suppress_Accessibility_Checks (E, Sense);

               when Discriminant_Check =>
                  Effective := Suppress_Discriminant_Checks  (E);
                  Set_Suppress_Discriminant_Checks (E, Sense);

               when Division_Check =>
                  Effective := Suppress_Division_Checks (E);
                  Set_Suppress_Division_Checks (E, Sense);

               when Elaboration_Check =>
                  Effective := Suppress_Elaboration_Checks (E);
                  Set_Suppress_Elaboration_Checks (E, Sense);

               when Index_Check =>
                  Effective := Suppress_Index_Checks (E);
                  Set_Suppress_Index_Checks (E, Sense);

               when Length_Check =>
                  Effective := Suppress_Length_Checks (E);
                  Set_Suppress_Length_Checks (E, Sense);

               when Overflow_Check =>
                  Effective := Suppress_Overflow_Checks (E);
                  Set_Suppress_Overflow_Checks (E, Sense);

               when Range_Check =>
                  Effective := Suppress_Range_Checks (E);
                  Set_Suppress_Range_Checks (E, Sense);

               when Storage_Check =>
                  Effective := Suppress_Storage_Checks (E);
                  Set_Suppress_Storage_Checks (E, Sense);

               when Tag_Check =>
                  Effective := Suppress_Tag_Checks (E);
                  Set_Suppress_Tag_Checks (E, Sense);

               when All_Checks =>
                  Suppress_Unsuppress_Echeck (E, Access_Check);
                  Suppress_Unsuppress_Echeck (E, Accessibility_Check);
                  Suppress_Unsuppress_Echeck (E, Discriminant_Check);
                  Suppress_Unsuppress_Echeck (E, Division_Check);
                  Suppress_Unsuppress_Echeck (E, Elaboration_Check);
                  Suppress_Unsuppress_Echeck (E, Index_Check);
                  Suppress_Unsuppress_Echeck (E, Length_Check);
                  Suppress_Unsuppress_Echeck (E, Overflow_Check);
                  Suppress_Unsuppress_Echeck (E, Range_Check);
                  Suppress_Unsuppress_Echeck (E, Storage_Check);
                  Suppress_Unsuppress_Echeck (E, Tag_Check);
            end case;

            --  If the entity is not declared in the current
            --  scope, then we make an entry in the
            --  Entity_Suppress table so that the flag will be
            --  removed on exit. This entry is only made if the
            --  suppress did something (i.e. the flag was not
            --  already set).

            if Effective and then Scope (E) /= Current_Scope then
               Entity_Suppress.Increment_Last;
               Entity_Suppress.Table
                 (Entity_Suppress.Last).Entity := E;
               Entity_Suppress.Table
                 (Entity_Suppress.Last).Check  := C;
            end if;
         end Suppress_Unsuppress_Echeck;

      --  Start of processing for Process_Suppress_Unsuppress

      begin
         if Nargs = 1 then
            case C is
               when Access_Check =>
                  Scope_Suppress.Access_Checks := Sense;

               when Accessibility_Check =>
                  Scope_Suppress.Accessibility_Checks := Sense;

               when Discriminant_Check =>
                  Scope_Suppress.Discriminant_Checks := Sense;

               when Division_Check =>
                  Scope_Suppress.Division_Checks := Sense;

               when Elaboration_Check =>
                  Scope_Suppress.Elaboration_Checks := Sense;

               when Index_Check =>
                  Scope_Suppress.Index_Checks := Sense;

               when Length_Check =>
                  Scope_Suppress.Length_Checks := Sense;

               when Overflow_Check =>
                  Scope_Suppress.Overflow_Checks := Sense;

               when Range_Check =>
                  Scope_Suppress.Range_Checks := Sense;

               when Storage_Check =>
                  Scope_Suppress.Storage_Checks := Sense;

               when Tag_Check =>
                  Scope_Suppress.Tag_Checks := Sense;

               when All_Checks =>
                  Scope_Suppress := (others => Sense);

            end case;

         --  Case of two arguments present, where the check is
         --  suppressed for a specified entity (given as the second
         --  argument of the pragma)

         else
            E_Id := Expression (Arg2);
            Analyze (E_Id);
            E := Entity (E_Id);

            if E = Any_Id then
               return;
            else
               Suppress_Unsuppress_Echeck (E, C);

               while Present (Homonym (E)) loop
                  E := Homonym (E);
                  Suppress_Unsuppress_Echeck (E, C);
               end loop;
            end if;
         end if;

      end Process_Suppress_Unsuppress;

      --------------------------------
      -- Valid_Configuration_Pragma --
      --------------------------------

      function Valid_Configuration_Pragma return Boolean is
      begin
         return True; -- for now ??? need semantic placement check later
      end Valid_Configuration_Pragma;

      -----------------------
      -- Valid_Unit_Pragma --
      -----------------------

      function Valid_Unit_Pragma return Boolean is
         Decl        : Node_Id;
         Plist       : List_Id;
         Parent_Node : Node_Id;
         Unit_Name   : Entity_Id;
         Valid       : Boolean := True;
         Unit_Kind   : Node_Kind;
         Unit_Node   : Node_Id;

      begin
         if not Is_List_Member (N) then
            Pragma_Misplaced;
            Valid := False;

         else
            Plist := List_Containing (N);
            Parent_Node := Parent (Plist);

            if Parent_Node = Empty then
               Pragma_Misplaced;
               Valid := False;

            elsif Nkind (Parent_Node) = N_Compilation_Unit then

               --  Pragma must appear after a compilation_unit, and must have
               --  an argument with the right name.

               if Plist /= Following_Pragmas (Parent_Node) then
                  Pragma_Misplaced;
                  Valid := False;

               elsif Nargs > 0 then

                  Unit_Node  := Unit (Parent_Node);
                  Unit_Kind  := Nkind (Unit_Node);

                  Analyze (Expression (Arg1));

                  if Unit_Kind = N_Generic_Subprogram_Declaration
                    or else Unit_Kind = N_Subprogram_Declaration
                  then
                     Unit_Name :=
                       Defining_Unit_Simple_Name (Specification (Unit_Node));

                  elsif Unit_Kind = N_Function_Instantiation
                    or else Unit_Kind = N_Package_Instantiation
                    or else Unit_Kind = N_Procedure_Instantiation
                  then
                     Unit_Name := Defining_Unit_Simple_Name (Unit_Node);
                  else
                     Pragma_Misplaced;
                     Valid := False;
                  end if;

                  if Valid
                    and then Unit_Name /= Entity (Expression (Arg1))
                  then
                     Error_Pragma_Argument
                       ("pragma% argument is not current unit name", Arg1);
                     Valid := False;
                  end if;

               else
                  Error_Pragma ("missing argument in pragma%");
                  Valid := False;
               end if;

            elsif Is_Before_First_Decl (N, Plist) then

               --  Name is optional, pragma applies to enclosing unit.

               Unit_Node := Get_Declaration_Node (Current_Scope);
               Unit_Kind := Nkind (Unit_Node);

               if (Unit_Kind = N_Package_Declaration
                 and then Present (Generic_Parent (Specification (Unit_Node))))
                   or else Nkind (Original_Node (Unit_Node)) =
                                               N_Formal_Package_Declaration
               then
                  --  The pragma appears in (the equivalent of) an instance.
                  --  validation takes place in the generic itself.

                  Valid := True;

               elsif Nkind (Parent (Unit_Node)) /= N_Compilation_Unit then
                  Pragma_Misplaced;
                  Valid := False;

               elsif Unit_Kind = N_Package_Body
                 or else Unit_Kind = N_Subprogram_Body
               then
                  Pragma_Misplaced;
                  Valid := False;

               elsif Nargs > 0 then
                  Analyze (Expression (Arg1));

                  if Entity (Expression (Arg1)) /= Current_Scope then
                     Error_Pragma_Argument
                       ("name in pragma% must be enclosing unit", Arg1);
                     Valid := False;
                  end if;

               else
                  --  Pragma with no argument is legal here.

                  null;
               end if;

            --  If not first in declarative part, name is required.

            elsif Nargs > 0 then
               Analyze (Expression (Arg1));
               Unit_Name := Entity (Expression (Arg1));
               Unit_Node := Get_Declaration_Node (Unit_Name);

               if Scope (Unit_Name) /= Current_Scope then
                  Error_Pragma_Argument
                    ("argument of pragma% is not in current scope", Arg1);
                  Valid := False;

               elsif Nkind (Unit_Node) not in N_Generic_Instantiation
                 and then Nkind (Unit_Node) /= N_Generic_Subprogram_Declaration
                 and then Nkind (Unit_Node) /= N_Subprogram_Declaration
               then
                  Error_Pragma_Argument ("invalid name in pragma%", Arg1);
                  Valid := False;
               end if;

            else
               Error_Pragma ("missing argument in pragma%");
               Valid := False;
            end if;
         end if;

         return Valid;

      end Valid_Unit_Pragma;

   --------------------------------------------
   -- Start of processing for Analyze_Pragma --
   --------------------------------------------

   begin
      --  Count number of arguments

      declare
         Arg_Node : Node_Id;

      begin
         Nargs := 0;

         if Present (Pragma_Argument_Associations (N)) then
            Arg_Node := Arg1;

            while Arg_Node /= Empty loop
               Nargs := Nargs + 1;
               Arg_Node := Next (Arg_Node);
            end loop;
         end if;
      end;

      --  An enumeration type defines the pragmas that are supported by the
      --  implementation. Get_Pragma_Id (in package Prag) transorms a name
      --  into the corresponding enumeration value for the following case.

      case Get_Pragma_Id (Chars (N)) is

         -----------------
         -- Abort_Defer --
         -----------------

         when Pragma_Abort_Defer =>
            Note_Feature (Implementation_Dependent_Pragmas, Loc);

            --  The only required semantic processing is to check the
            --  placement. This pragma must appear at the start of the
            --  statement sequence of a handled sequence of statements.

            if Nkind (Parent (N)) /= N_Handled_Sequence_Of_Statements
              or else N /= First (Statements (Parent (N)))
            then
               Pragma_Misplaced;
            end if;

         ------------
         -- Ada_83 --
         ------------

         when Pragma_Ada_83 =>
            Note_Feature (Implementation_Dependent_Pragmas, Loc);

            if Valid_Configuration_Pragma then
               Ada_83 := True;
               Ada_9X := False;
            end if;

         ------------
         -- Ada_9X --
         ------------

         when Pragma_Ada_9X =>
            Note_Feature (Implementation_Dependent_Pragmas, Loc);

            if Valid_Configuration_Pragma then
               Ada_83 := False;
               Ada_9X := True;
            end if;

         ----------------------
         -- All_Calls_Remote --
         ----------------------

         when Pragma_All_Calls_Remote =>
            Pragma_Not_Implemented;

         ------------
         -- Assert --
         ------------

         when Pragma_Assert => Assert : begin
            Note_Feature (Implementation_Dependent_Pragmas, Loc);

            --  If we are not in debug mode then rewrite the pragma with
            --  a null statement and do not even analyze the pragma.

            if not Assertions_Enabled then
               Rewrite_Substitute_Tree (N, Make_Null_Statement (Loc));

            --  If we are in debug mode, then rewrite the pragma with its
            --  corresponding if statement, and then analyze the statement
            --  The expansion transforms:

            --    pragma Assert (condition [,procedure_call]);

            --  into

            --    if not condition then
            --       raise System.Assertions.Assert_Failure;
            --    end if;

            else
               declare
                  Stmts : List_Id;

               begin
                  Stmts := New_List;

                  Append_To (Stmts,
                    Make_Raise_Statement (Loc,
                      Name =>
                        New_Reference_To (RTE (RE_Assert_Failure), Loc)));

                  Rewrite_Substitute_Tree (N,
                    Make_If_Statement (Loc,
                      Condition =>
                        Make_Op_Not (Loc,
                          Right_Opnd => Expression (Arg1)),
                      Then_Statements => Stmts));

                  Analyze (N);
               end;
            end if;

         end Assert;

         ------------------
         -- Asynchronous --
         ------------------

         when Pragma_Asynchronous =>
            Note_Feature (New_Representation_Pragmas, Loc);
            Pragma_Not_Implemented;

         ------------
         -- Atomic --
         ------------

         when Pragma_Atomic =>
            Note_Feature (New_Representation_Pragmas, Loc);
            Pragma_Not_Implemented;

         -----------------------
         -- Atomic_Components --
         -----------------------

         when Pragma_Atomic_Components =>
            Note_Feature (New_Representation_Pragmas, Loc);
            Pragma_Not_Implemented;

         --------------------
         -- Attach_Handler --
         --------------------

         when Pragma_Attach_Handler =>
            Pragma_Not_Implemented;

         ----------------
         -- Controlled --
         ----------------

         when Pragma_Controlled =>
            Pragma_Not_Implemented;

         ----------------
         -- Convention --
         ----------------

         when Pragma_Convention => Convention : declare
            Id          : Node_Id;
            Proc_Def_Id : Entity_Id;

         begin
            Note_Feature (New_Representation_Pragmas, Loc);

            if Chars (Expression (Arg1)) = Name_Intrinsic then
               Id := Expression (Arg2);
               Find_Program_Unit_Name (Id);

               --  Remaining processing is needed only if we found the name.

               if Etype (Id) /= Any_Type then
                  Proc_Def_Id := Entity (Id);

                  if not Is_Subprogram (Proc_Def_Id)
                    and then not Is_Generic_Subprogram (Proc_Def_Id)
                  then
                     Error_Pragma_Argument
                       ("second argument of pragma% must be a subprogram",
                        Arg2);

                  elsif Scope (Proc_Def_Id) /= Current_Scope then
                     Error_Pragma_Argument
                       ("pragma% must be in same declarative part", Arg2);
                  else

                     while Present (Proc_Def_Id)
                       and then Scope (Proc_Def_Id) = Current_Scope loop
                        Set_Has_Convention_Intrinsic (Proc_Def_Id);
                        Proc_Def_Id := Homonym (Proc_Def_Id);
                     end loop;

                  end if;
               end if;
            else
               Pragma_Not_Implemented;
            end if;
         end Convention;

         -----------
         -- Debug --
         -----------

         when Pragma_Debug =>
            Note_Feature (Implementation_Dependent_Pragmas, Loc);

            --  If we are not in debug mode then rewrite the pragma with
            --  a null statement and do not even analyze the pragma.

            if not Assertions_Enabled then
               Rewrite_Substitute_Tree (N, Make_Null_Statement (Loc));

            --  If we are in debug mode, then rewrite the pragma with its
            --  corresponding procedure call, and then analyze the call.

            else
               Rewrite_Substitute_Tree (N, New_Copy (Debug_Statement (N)));
               Analyze (N);
            end if;

         -------------------
         -- Discard_Names --
         -------------------

         when Pragma_Discard_Names =>
            Note_Feature (New_Representation_Pragmas, Loc);

            Pragma_Not_Implemented;

         ---------------
         -- Elaborate --
         ---------------

         when Pragma_Elaborate => Elaborate : declare
            Plist       : List_Id;
            Parent_Node : Node_Id;
            Arg         : Node_Id;
            Citem       : Node_Id;

         begin
            --  Pragma must be in context items list of a compilation unit

            if not Is_List_Member (N) then
               Pragma_Misplaced;
               return;

            else
               Plist := List_Containing (N);
               Parent_Node := Parent (Plist);

               if Parent_Node = Empty
                 or else Nkind (Parent_Node) /= N_Compilation_Unit
                 or else Context_Items (Parent_Node) /= Plist
               then
                  Pragma_Misplaced;
                  return;
               end if;
            end if;

            --  In Ada 83 mode, there can be no items following it in the
            --  context list except other pragmas and implicit with clauses
            --  (e.g. those added by use of Rtsfind). In Ada 9X mode, this
            --  placement rule does not apply.

            if Ada_83 then
               Citem := Next (N);

               while Present (Citem) loop
                  if Nkind (Citem) = N_Pragma
                    or else (Nkind (Citem) = N_With_Clause
                              and then Implicit_With (Citem))
                  then
                     null;
                  else
                     Error_Pragma
                       ("(Ada 83) pragma% must be at end of context clause");
                     Pragma_Misplaced;
                     return;
                  end if;

                  Citem := Next (Citem);
               end loop;
            end if;

            --  Finally, the arguments must all be units mentioned in a with
            --  clause in the same context clause. Note we already checked
            --  (in Par.Prag) that the arguments are either identifiers or

            Arg := Arg1;
            Outer : while Present (Arg) loop
               Citem := First (Plist);

               Inner : while Citem /= N loop
                  if Nkind (Citem) = N_With_Clause
                    and then Same_Name (Name (Citem), Expression (Arg))
                  then
                     Set_Elaborate_Present (Citem, True);
                     exit Inner;
                  end if;

                  Citem := Next (Citem);
               end loop Inner;

               if Citem = N then
                  Error_Pragma_Argument
                    ("Argument of pragma% is not with'ed unit", Arg);
               end if;

               Arg := Next (Arg);
            end loop Outer;
         end Elaborate;

         -------------------
         -- Elaborate_All --
         -------------------

         when Pragma_Elaborate_All => Elaborate_All : declare
            Plist       : List_Id;
            Parent_Node : Node_Id;
            Arg         : Node_Id;
            Citem       : Node_Id;

         begin
            --  Pragma must be in context items list of a compilation unit

            if not Is_List_Member (N) then
               Pragma_Misplaced;
               return;

            else
               Plist := List_Containing (N);
               Parent_Node := Parent (Plist);

               if Parent_Node = Empty
                 or else Nkind (Parent_Node) /= N_Compilation_Unit
                 or else Context_Items (Parent_Node) /= Plist
               then
                  Pragma_Misplaced;
                  return;
               end if;
            end if;

            --  Note: unlike pragma Elaborate, pragma Elaborate_All does not
            --  have to appear at the end of the context clause, but may
            --  appear mixed in with other items.

            --  Final check: the arguments must all be units mentioned in
            --  a with clause in the same context clause. Note that we
            --  already checked (in Par.Prag) that all the arguments are
            --  either identifiers or selected components.

            Arg := Arg1;
            Outr : while Present (Arg) loop
               Citem := First (Plist);

               Innr : while Citem /= N loop
                  if Nkind (Citem) = N_With_Clause
                    and then Same_Name (Name (Citem), Expression (Arg))
                  then
                     Set_Elaborate_All_Present (Citem, True);
                     exit Innr;
                  end if;

                  Citem := Next (Citem);
               end loop Innr;

               if Citem = N then
                  Error_Pragma_Argument
                    ("Argument of pragma% is not with'ed unit", Arg);
               end if;

               Arg := Next (Arg);
            end loop Outr;
         end Elaborate_All;

         --------------------
         -- Elaborate_Body --
         --------------------

         when Pragma_Elaborate_Body => Elaborate_Body : declare
            Plist      : List_Id;
            Cunit_Node : Node_Id;

         begin
            if Valid_Unit_Pragma then

               Plist := List_Containing (N);
               Cunit_Node := Parent (Plist);

               --  Case of pragma appearing in declarative part. Only
               --  legal if it is in a package specification.

               if Nkind (Cunit_Node) /= N_Compilation_Unit then
                  if Nkind (Cunit_Node) = N_Package_Specification then
                     Cunit_Node := Parent (Parent (Cunit_Node));
                  else
                     Pragma_Misplaced;
                     return;
                  end if;
               end if;

               Set_Elaborate_Body_Present (Cunit_Node, True);
               Set_Body_Required (Cunit_Node, True);
            end if;
         end Elaborate_Body;

         ------------
         -- Export --
         ------------

         when Pragma_Export =>
            Note_Feature (New_Representation_Pragmas, Loc);

            Pragma_Not_Implemented;

         ------------
         -- Import --
         ------------

         when Pragma_Import | Pragma_Interface => Import : declare
            Id          : Entity_Id;
            Proc_Def_Id : Entity_Id;

         begin
            Note_Feature (New_Representation_Pragmas, Loc);

            Id := Expression (Arg2);
            Find_Program_Unit_Name (Id);

            --  If we did not find the name, we are done

            if Etype (Id) = Any_Type then
               return;
            end if;

            Proc_Def_Id := Entity (Id);

            if not Is_Subprogram (Proc_Def_Id)
              and then not Is_Generic_Subprogram (Proc_Def_Id)
            then
               Error_Pragma_Argument
                 ("second argument of pragma% must be a subprogram", Arg2);

            else
               --  If name is overloaded, pragma applies to all the
               --  denoted entities in the same declarative part.
               --  Ignore inherited subprograms, because the pragma will
               --  apply to the parent operation which is the one called.

               while Present (Proc_Def_Id) loop

                  if Is_Overloadable (Proc_Def_Id)
                    and then Present (Alias (Proc_Def_Id))
                  then
                     null;

                  elsif Parent (Get_Declaration_Node (Proc_Def_Id)) /=
                        Parent (N)
                  then
                     exit;

                  else
                     Set_Is_Imported (Proc_Def_Id);

                     --  If Import intrinsic, set intrinsic flag
                     --  and verify that it is known as such.

                     if Chars (Expression (Arg1)) = Name_Intrinsic then
                        Set_Is_Intrinsic_Subprogram (Proc_Def_Id);
                        Check_Intrinsic_Subprogram
                          (Proc_Def_Id, Expression (Arg2));
                        Set_Has_Convention_Intrinsic (Proc_Def_Id);

                     --  If convention C, set flag

                     elsif Chars (Expression (Arg1)) = Name_C then
                        Set_Has_Convention_C (Proc_Def_Id);
                     end if;

                     --  All interfaced procedures need an external
                     --  symbol created for them since they are
                     --  always referenced from another object file.

                     Set_Is_Public (Proc_Def_Id);
                     Set_Has_Completion (Proc_Def_Id);

                     if Nargs >= 3 then
                        Process_Interface_Name (Proc_Def_Id, Arg3, Arg4);
                     end if;
                  end if;

                  Proc_Def_Id := Homonym (Proc_Def_Id);
               end loop;
            end if;
         end Import;

         ------------
         -- Inline --
         ------------

         when Pragma_Inline => Inline : declare
            Assoc    : Node_Id;
            Decl     : Node_Id;
            Subp_Id  : Node_Id;
            Subp     : Entity_Id;

            procedure Make_Inline (Subp : Entity_Id);
            --  Subp is the defining unit name of the subprogram
            --  declaration. Set the flag, as well as the flag in the
            --  corresponding boy, if there is one present.

            procedure Make_Inline (Subp : Entity_Id) is
               Kind : Entity_Kind := Ekind (Subp);

            begin
               if Etype (Subp) = Any_Type then
                  return;

               --  The referenced entity must either be the enclosing entity,
               --  or an entity declared within the current open scope.

               elsif Present (Scope (Subp))
                 and then Scope (Subp) /= Current_Scope
                 and then Subp /= Current_Scope
               then
                  Pragma_Misplaced;
                  return;
               end if;

               --  Processing for procedure, operator or function

               if Kind = E_Procedure
                 or else Kind = E_Function
                 or else Kind = E_Operator
               then
                  Set_Is_Inlined (Subp, True);

                  Decl := Parent (Parent (Subp));

                  if Nkind (Decl) = N_Subprogram_Declaration
                    and then Present (Corresponding_Body (Decl))
                  then
                     Set_Is_Inlined (Corresponding_Body (Decl), True);
                  end if;

               --  Don't do anything for a generic procedure or generic
               --  function. The instance will be marked inlined as
               --  required during the compilation of the instance.

               elsif Kind = E_Generic_Procedure
                 or else Kind = E_Generic_Function
               then
                  null;

               --  Anything else is an error

               else
                  Error_Pragma_Argument
                    ("expect subprogram name for pragma%", Assoc);
               end if;
            end Make_Inline;

         begin
            Assoc := Arg1;

            while Present (Assoc) loop
               Subp_Id := Expression (Assoc);
               Analyze (Subp_Id);
               Subp := Entity (Subp_Id);

               if Subp = Any_Id then
                  null;
               else
                  Make_Inline (Subp);

                  while Present (Homonym (Subp))
                    and then Scope (Homonym (Subp)) = Current_Scope
                  loop
                     Make_Inline (Homonym (Subp));
                     Subp := Homonym (Subp);
                  end loop;
               end if;

               Assoc := Next (Assoc);
            end loop;
         end Inline;

         ----------------------
         -- Inspection_Point --
         ----------------------

         when Pragma_Inspection_Point =>
            Pragma_Not_Implemented;

         ---------------
         -- Interface --
         ---------------

         --  Pragma Interface is processed by the same circuit as pragma
         --  Import (except that for Interface, the parser has verified
         --  that only two arguments are present, so the processing for
         --  the third and fourth arguments has no effect for Interface).

         --------------------
         -- Interface_Name --
         --------------------

         when Pragma_Interface_Name => Interface_Name : declare
            Id          : constant Node_Id := Expression (Arg1);
            Link_Name   : constant Node_Id := Expression (Arg2);
            Proc_Def_Id : Entity_Id;

         begin
            Note_Feature (Implementation_Dependent_Pragmas, Loc);

            Analyze (Id);

            --  Remaining processing is needed only if we found the name.
            --  Check that name represents a subprogram for which a pragma
            --  Interface has been given. Then process the interface name.

            if Etype (Id) /= Any_Type then
               Proc_Def_Id := Entity (Id);

               if not Is_Subprogram (Proc_Def_Id) then
                  Error_Pragma_Argument
                    ("argument of pragma% is not subprogram", Arg1);

               elsif not Is_Imported (Proc_Def_Id) then
                  Error_Pragma_Argument
                    ("argument of pragma% is not imported subprogram", Arg1);
               else
                  Process_Interface_Name (Proc_Def_Id, Arg2, Arg3);
               end if;
            end if;
         end Interface_Name;

         -----------------------
         -- Interrupt_Handler --
         -----------------------

         when Pragma_Interrupt_Handler =>
            Pragma_Not_Implemented;

         ------------------------
         -- Interrupt_Priority --
         ------------------------

         when Pragma_Interrupt_Priority => Interrupt_Priority : declare
            P : constant Node_Id := Parent (N);

         begin
            if Nargs /= 0 then
               Analyze (Expression (Arg1));
               Resolve (Expression (Arg1), RTE (RE_Priority));
            end if;

            if Nkind (P) /= N_Task_Definition
              and then Nkind (P) /= N_Protected_Definition
            then
               Pragma_Misplaced;
               return;

            elsif Has_Priority_Pragma (P) then
               Error_Pragma ("duplicate pragma% not allowed");

            else
               Set_Has_Priority_Pragma (P, True);
            end if;
         end Interrupt_Priority;

         --------------------
         -- Linker_Options --
         --------------------

         when Pragma_Linker_Options =>
            Pragma_Not_Implemented;

         ----------
         -- List --
         ----------

         --  There is nothing to do here, since we did all the processing
         --  for this pragma in Par.Prag (so that it works properly even in
         --  syntax only mode)

         when Pragma_List =>
            null;

         --------------------
         -- Locking_Policy --
         --------------------

         when Pragma_Locking_Policy =>
            Pragma_Not_Implemented;

         -----------------
         -- Memory_Size --
         -----------------

         when Pragma_Memory_Size =>
            Pragma_Not_Implemented;

         -----------------------
         -- Normalize_Scalars --
         -----------------------

         when Pragma_Normalize_Scalars =>
            Pragma_Not_Implemented;

         --------------
         -- Optimize --
         --------------

         --  Nothing to do, since all checks done in Par.Prag and we don't
         --  actually pay any attention to this pragma (does anyone?)

         when Pragma_Optimize =>
            null;

         ----------
         -- Pack --
         ----------

         when Pragma_Pack => Pack : declare
            Assoc   : Node_Id := Arg1;
            Type_Id : Node_Id := Expression (Assoc);
            Typ     : Entity_Id;

         begin
            Find_Type (Type_Id);
            Typ := Entity (Type_Id);

            if Typ = Any_Type then
               return;

            elsif not Is_Composite_Type (Typ) then
               Error_Pragma ("pragma% does not specify composite type");

            elsif Scope (Typ) /= Current_Scope then
               Error_Pragma
                 ("pragma% does not specify type in same declarative part");

            --  Array, record, or task type. Pragma Pack only works for
            --  records, but it is too annoying to generate warnings for
            --  packed arrays, so don't bother.

            else
               Set_Is_Packed (Typ);
            end if;
         end Pack;

         ----------
         -- Page --
         ----------

         --  There is nothing to do here, since we did all the processing
         --  for this pragma in Par.Prag (so that it works properly even in
         --  syntax only mode)

         when Pragma_Page =>
            null;

         ------------------
         -- Preelaborate --
         ------------------

         --  Set the flag Is_Preelaborated of program unit name entity

         when Pragma_Preelaborate => Preelaborate : declare
            Ent : Entity_Id;

         begin
            if Valid_Unit_Pragma then
               Ent := Find_Lib_Unit_Name;

               if Present (Ent) then
                  Set_Is_Preelaborated (Ent);
               end if;
            end if;
         end Preelaborate;

         --------------
         -- Priority --
         --------------

         when Pragma_Priority => Priority : declare
            P : constant Node_Id := Parent (N);
            V : Uint;

         begin
            Analyze (Expression (Arg1));
            Resolve (Expression (Arg1), Any_Integer);

            if Nkind (P) /= N_Task_Definition
              and then Nkind (P) /= N_Protected_Definition
              and then Nkind (P) /= N_Subprogram_Body
            then
               Pragma_Misplaced;
               return;

            else
               if Nkind (P) = N_Subprogram_Body
                 and then not Is_Static_Expression (Expression (Arg1))
               then
                  Check_Static_Expression (Expression (Arg1));

               else
                  V := Expr_Value (Expression (Arg1));

                  if UI_Is_Negative (V)
                    or else not UI_Is_In_Int_Range (V)
                  then
                     Error_Pragma_Argument
                       ("priority value out of range", Arg1);

                  else
                     Set_Main_Priority
                       (Get_Sloc_Unit_Number (Loc), UI_To_Int (V));
                  end if;

               end if;

               if Has_Priority_Pragma (P) then
                  Error_Pragma ("duplicate pragma% not allowed");
               else
                  Set_Has_Priority_Pragma (P, True);
               end if;

            end if;
         end Priority;

         ----------
         -- Pure --
         ----------

         --  Set the flag Is_Pure of program unit name entity

         when Pragma_Pure => Pure : declare
            Ey : Entity_Id;

         begin
            if Valid_Unit_Pragma then
               Ey := Find_Lib_Unit_Name;

               if Present (Ey) then
                  Set_Is_Pure (Ey);
               end if;
            end if;
         end Pure;

         --------------------
         -- Queuing_Policy --
         --------------------

         when Pragma_Queuing_Policy =>
            Pragma_Not_Implemented;

         ---------------------------
         -- Remote_Call_Interface --
         ---------------------------

         --  Set the flag Is_Remote_Call_Interface of program unit name entity

         when Pragma_Remote_Call_Interface => Remote_Call_Interface : declare
            Ey : Entity_Id;

         begin
            if Valid_Unit_Pragma then
               Ey := Find_Lib_Unit_Name;

               if Present (Ey) then
                  Set_Is_Remote_Call_Interface (Ey);
               end if;
            end if;
         end Remote_Call_Interface;

         ------------------
         -- Remote_Types --
         ------------------

         --  Set the flag Is_Remote_Types of program unit name entity

         when Pragma_Remote_Types => Remote_Types : declare
            Ey : Entity_Id;

         begin
            if Valid_Unit_Pragma then
               Ey := Find_Lib_Unit_Name;

               if Present (Ey) then
                  Set_Is_Remote_Types (Ey);
               end if;
            end if;
         end Remote_Types;

         ------------------
         -- Restrictions --
         ------------------

         when Pragma_Restrictions =>
            Pragma_Not_Implemented;

         ----------------
         -- Reviewable --
         ----------------

         when Pragma_Reviewable =>
            Pragma_Not_Implemented;

         ------------
         -- Shared --
         ------------

         when Pragma_Shared =>
            Pragma_Not_Implemented;

         --------------------
         -- Shared_Passive --
         --------------------

         --  Set the flag Is_Shared_Passive of program unit name entity

         when Pragma_Shared_Passive => Shared_Passive : declare
            Ey : Entity_Id;

         begin
            if Valid_Unit_Pragma then
               Ey := Find_Lib_Unit_Name;

               if Present (Ey) then
                  Set_Is_Shared_Passive (Ey);
               end if;
            end if;
         end Shared_Passive;

         ------------------
         -- Storage_Size --
         ------------------

         when Pragma_Storage_Size => Storage_Size : declare
            P : constant Node_Id := Parent (N);

         begin
            Analyze (Expression (Arg1));
            Resolve (Expression (Arg1), Any_Integer);

            if Nkind (P) /= N_Task_Definition then
               Pragma_Misplaced;
               return;

            else
               if Has_Storage_Size_Pragma (P) then
                  Error_Pragma ("duplicate pragma% not allowed");
               else
                  Set_Has_Storage_Size_Pragma (P, True);
               end if;
            end if;
         end Storage_Size;

         ------------------
         -- Storage_Unit --
         ------------------

         --  Only permitted argument is System'Storage_Unit value

         when Pragma_Storage_Unit =>
            if UI_Ne (Intval (Expression (Arg1)),
                      UI_From_Int (Ttypes.System_Storage_Unit))
            then
               Error_Msg_Uint_1 := Intval (Expression (Arg1));
               Error_Pragma_Argument
                 ("the only allowed argument for pragma% is ^", Arg1);
            end if;

         --------------
         -- Suppress --
         --------------

         when Pragma_Suppress =>
            Process_Suppress_Unsuppress (True);

         -----------------
         -- System_Name --
         -----------------

         --  Nothing to do here, since this pragma was completely
         --  handled in Par

         when Pragma_System_Name =>
            null;

         ------------------------
         -- Unimplemented_Unit --
         ------------------------

         --  Note: this only gives an error if we are generating code,
         --  or if we are in a generic library unit (where the pragma
         --  appears in the body, not in the spec).

         when Pragma_Unimplemented_Unit => Unimplemented_Unit : declare
            Cunitent : Entity_Id := Cunit_Entity (Get_Sloc_Unit_Number (Loc));
            Ent_Kind : Entity_Kind := Ekind (Cunitent);

         begin
            if Operating_Mode = Generate_Code
              or else Ent_Kind = E_Generic_Function
              or else Ent_Kind = E_Generic_Procedure
              or else Ent_Kind = E_Generic_Package
            then
               Error_Msg_N ("& is not implemented", Cunitent);
               raise Unrecoverable_Error;
            end if;
         end Unimplemented_Unit;

         ----------------
         -- Unsuppress --
         ----------------

         when Pragma_Unsuppress =>
            Process_Suppress_Unsuppress (False);

         -----------------------------
         -- Task_Dispatching_Policy --
         -----------------------------

         when Pragma_Task_Dispatching_Policy =>
            Pragma_Not_Implemented;

         --------------
         -- Volatile --
         --------------

         when Pragma_Volatile => Volatile : declare
            E_Id : Node_Id := Expression (Arg1);
            E    : Entity_Id;

         begin
            Note_Feature (New_Representation_Pragmas, Loc);

            Analyze (E_Id);

            if not Is_Entity_Name (E_Id)
              or else Etype (E_Id) = Any_Type
            then
               return;
            else
               E := Entity (E_Id);
            end if;

            if Ekind (E) = E_Variable
              or else Ekind (E) = E_Component
              or else Ekind (E) = E_Constant
            then
               Set_Is_Volatile (E);
            else
               Pragma_Not_Implemented;
            end if;

         end Volatile;

         -------------------------
         -- Volatile_Components --
         -------------------------

         when Pragma_Volatile_Components =>
            Note_Feature (New_Representation_Pragmas, Loc);

            Pragma_Not_Implemented;

      end case;
   end Analyze_Pragma;

   ---------------------------
   -- Is_Generic_Subprogram --
   ---------------------------

   function Is_Generic_Subprogram (Id : Entity_Id) return Boolean is
   begin
      return  Ekind (Id) = E_Generic_Procedure
        or else Ekind (Id) = E_Generic_Function;
   end Is_Generic_Subprogram;

end Sem_Prag;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.129
--  date: Thu Jul 28 08:52:31 1994;  author: schonber
--  Replace Find_Name with Analyze.
--  ----------------------------
--  revision 1.130
--  date: Mon Aug  1 13:14:52 1994;  author: dewar
--  Pragma Inline must allow operators as well as functions and procedures
--  ----------------------------
--  revision 1.131
--  date: Wed Aug 10 05:16:25 1994;  author: dewar
--  Change many uses of Error_Pragma to Error_Pragma_Argument so that flag is
--   posted more accurately (avoids a number of junk cascaded messages)
--  (Error_Pragma_Argument): Post message on expression of given argument
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
