-----------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                             S E M _ C H 1 3                              --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.94 $                             --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Exp_Util; use Exp_Util;
with Features; use Features;
with Itypes;   use Itypes;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Ch3;  use Sem_Ch3;
with Sem_Ch8;  use Sem_Ch8;
with Sem_Eval; use Sem_Eval;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Stand;    use Stand;
with Sinfo;    use Sinfo;
with Snames;   use Snames;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body Sem_Ch13 is

   -----------------------
   -- Local Subprograms --
   -----------------------

   function Already_Frozen (T : Entity_Id; N : Node_Id) return Boolean;
   --  Called at the start of processing a representation clause. Used to
   --  check that type T, referenced by representation clause N, is not
   --  already frozen. If the type is not frozen, then False is returned,
   --  and the caller can proceed. If the type is frozen, then an error
   --  message is issued and True is returned (which is a signal to the
   --  caller to abandon processing of the too late rep clause).

   --------------------
   -- Already_Frozen --
   --------------------

   function Already_Frozen (T : Entity_Id; N : Node_Id) return Boolean is
   begin
      if Is_Frozen (T) then
         Error_Msg_NE
           ("rep clause appears too late, type& already frozen", N, T);
         return True;
      else
         return False;
      end if;
   end Already_Frozen;

   -----------------------
   -- Analyze_At_Clause --
   -----------------------

   --  An at clause is replaced by the corresponding Address attribute
   --  definition clause that is the preferred approach in Ada 9X.

   procedure Analyze_At_Clause (N : Node_Id) is
   begin
      Rewrite_Substitute_Tree (N,
        Make_Attribute_Definition_Clause (Sloc (N),
          Name  => Identifier (N),
          Chars => Name_Address,
          Expression => Expression (N)));
      Analyze (N);
   end Analyze_At_Clause;

   -----------------------------------------
   -- Analyze_Attribute_Definition_Clause --
   -----------------------------------------

   procedure Analyze_Attribute_Definition_Clause (N : Node_Id) is
      Nam  : constant Node_Id := Name (N);
      Attr : constant Name_Id := Chars (N);
      Expr : constant Node_Id := Expression (N);
      Id   : constant Attribute_Id := Get_Attribute_Id (Attr);
      Typ  : Node_Id;
      Ent  : Entity_Id;

   begin
      Analyze (Nam);
      Ent := Entity (Nam);

      --  Ignore rep clauses for junk entities or for frozen types

      if Etype (Nam) = Any_Type then
         return;

      elsif Is_Type (Ent) and then Already_Frozen (Ent, Nam) then
         return;
      end if;

      case Id is

         --  Address attribute definition clause

         when Attribute_Address => Address : begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if Has_Address_Clause (Ent) then
               Error_Msg_N ("address already given for &", Nam);

            elsif Ekind (Ent) not in Subprogram_Kind
              and then Ekind (Ent) /= E_Variable
              and then Ekind (Ent) /= E_Constant
            then
               Error_Msg_N ("address cannot be given for &", Nam);

            else
               Analyze (Expr);

               Typ := RTE (RE_Address);
               Resolve (Expr, Typ);
               Set_Has_Address_Clause (Ent, True);
               Set_Is_Delayed (Ent);
            end if;
         end Address;

         --  Alignment attribute definition clause

         when Attribute_Alignment => Alignment : declare
            Align : constant Uint := Static_Integer (Expr);

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if not Is_Type (Ent)
              and then Ekind (Ent) /= E_Variable
              and then Ekind (Ent) /= E_Constant
            then
               Error_Msg_N ("alignment cannot be given for &", Nam);

            elsif Has_Alignment_Clause (Ent) then
               Error_Msg_Sloc := Sloc (Alignment_Clause (Ent));
               Error_Msg_N ("alignment clause previously given#", N);

            elsif Align /= No_Uint then
               if UI_Is_Negative (Align) then
                  Error_Msg_N ("negative alignment not allowed", Expr);

               else
                  Set_Alignment_Clause (Ent, N);
                  Set_Has_Alignment_Clause (Ent);
               end if;
            end if;
         end Alignment;

         --  Component_Size attribute definition clause

         when Attribute_Component_Size => Component_Size : declare
            Component_Size : constant Uint      := Static_Integer (Expr);
            Btype          : constant Entity_Id := Base_Type (Ent);

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));

            if Has_Component_Size_Clause (Btype) then
               Error_Msg_Sloc := Sloc (Component_Size_Clause (Btype));
               Error_Msg_N
                 ("component size clase for& previously given#", Nam);

            elsif not Is_Array_Type (Ent) then
               Error_Msg_N ("component size requires array type", Nam);

            elsif not Is_First_Named_Subtype (Ent) then
               Error_Msg_N ("cannot give component size for array subtype", N);

            elsif Component_Size /= No_Uint then

               --  Note that Gigi is in charge of checking that the size we
               --  are assigning is acceptable, and will generate the error
               --  message if the size is inappropriate.

               Set_Component_Size_Clause (Btype, N);
               Set_Has_Component_Size_Clause (Btype, True);
            end if;
         end Component_Size;

         --  Size attribute definition clause

         when Attribute_Size => Size : declare
            Size : constant Uint := Static_Integer (Expr);

         begin
            if Has_Size_Clause (Ent) then
               Error_Msg_N ("size already given for &", Nam);

            elsif not Is_Type (Ent)
              and then Ekind (Ent) /= E_Variable
              and then Ekind (Ent) /= E_Constant
            then
               Error_Msg_N ("size cannot be given for &", Nam);

            elsif Size /= No_Uint then

               --  Note that Gigi is in charge of checking that the size we
               --  are assigning is acceptable, and will generate the error
               --  message if the size is inappropriate.

               Set_Esize (Ent, Size);
               Set_Has_Size_Clause (Ent, True);
            end if;
         end Size;

         --  Small attribute definition clause

         when Attribute_Small => Small : declare

            function Can_Derive_From (E : Entity_Id) return Boolean;
            --  Find if given small and range specification of the base type
            --  allow derivation from the specified entity.

            Int_Type      : Entity_Id;
            Implicit_Type : constant Entity_Id := Base_Type (Ent);
            Size_Min      : Uint;
            Small         : constant Ureal     := Expr_Value (Expr);
            Small_Min     : constant Ureal     :=
                              UR_Exponentiate (
                                Ureal_2,
                                UI_Difference (
                                  Uint_1,
                                  UI_From_Int (
                                    Standard_Long_Long_Integer_Size)));

            function Can_Derive_From (E : Entity_Id) return Boolean is
            begin
               return UI_Le (
                        Size_Min,
                        UI_Exponentiate (Uint_2, Esize (E)));
            end Can_Derive_From;

         --  Start processing for Small attribute definition clause case

         begin
            if not Is_Ordinary_Fixed_Point_Type (Ent)
              or else not Is_First_Named_Subtype (Ent)
            then
               Error_Msg_N (
                 "small requires an ordinary fixed point type", Nam);

            elsif Has_Small_Clause (Ent) then
               Error_Msg_N ("small already given for &", Nam);

            elsif UR_Lt (Small, Small_Min) then
               Error_Msg_N (
                 "small value must not be less than Fine_Delta", Nam);

            elsif UR_Gt (Small, Delta_Value (Ent)) then
               Error_Msg_N (
                 "small value must not be greater then delta value", Nam);

            else
               Set_Small_Value (Ent, Small);
               Set_Small_Value (Implicit_Type, Small);
               Set_Has_Small_Clause (Ent, True);

               Size_Min :=
                 UI_Sum (
                   Uint_1,
                   UI_Product (
                     Uint_2,
                     Intval (High_Bound (Scalar_Range (Implicit_Type)))));

               if Can_Derive_From (Standard_Short_Short_Integer) then
                  Int_Type := Standard_Short_Short_Integer;

               elsif Can_Derive_From (Standard_Short_Integer) then
                  Int_Type := Standard_Short_Integer;

               elsif Can_Derive_From (Standard_Integer) then
                  Int_Type := Standard_Integer;

               elsif Can_Derive_From (Standard_Long_Integer) then
                  Int_Type := Standard_Long_Integer;

               elsif Can_Derive_From (Standard_Long_Long_Integer) then
                  Int_Type := Standard_Long_Long_Integer;

               else
                  Int_Type := Standard_Long_Long_Integer;
                  Error_Msg_N (
                   "small value with range of type is too small", Nam);
               end if;

               if Int_Type /= Corresponding_Integer_Type (Ent) then

                  Set_Esize (Implicit_Type, Esize (Int_Type));
                  Set_Corresponding_Integer_Type (Implicit_Type, Int_Type);

                  Set_Esize (Ent, Esize (Int_Type));
                  Set_Corresponding_Integer_Type (Ent, Int_Type);

               end if;

            end if;
         end Small;

         --  Storage_Size attribute definition clause

         when Attribute_Storage_Size => Storage_Size : declare
            Btype : constant Entity_Id := Base_Type (Ent);

         begin
            if Has_Storage_Size_Clause (Btype) then
               Error_Msg_N ("storage size already given for &", Nam);

            elsif not Is_Access_Type (Ent)
              and then Ekind (Ent) /= E_Task_Type
            then
               Error_Msg_N ("size cannot be given for &", Nam);

            elsif not Is_First_Named_Subtype (Ent) then
               Error_Msg_N ("cannot give storage size for subtypes", N);

            else
               Analyze (Expr);
               Resolve (Expr, Any_Integer);

               if Is_Access_Type (Ent)
                 and then Present (Associated_Storage_Pool (Ent))
               then
                  Error_Msg_N ("storage pool already given for &", Nam);
                  return;
               else
                  Set_Has_Storage_Size_Clause (Btype);
               end if;
            end if;
         end Storage_Size;

         --  Storage_Pool attribute definition clause

         when Attribute_Storage_Pool => Storage_Pool : declare
            Pool : Entity_Id;

         begin
            Note_Feature (New_Representation_Clauses, Sloc (N));
            Note_Feature (Storage_Pools, Sloc (N));

            if Ekind (Ent) /= E_Access_Type
              and then Ekind (Ent) /= E_General_Access_Type
            then
               Error_Msg_N (
                 "storage pool can only be given for access types", Nam);
               return;

            elsif Has_Storage_Size_Clause (Ent) then
               Error_Msg_N ("storage size already given for &", Nam);
               return;

            elsif Present (Associated_Storage_Pool (Ent)) then
               Error_Msg_N ("storage pool already given for &", Nam);
               return;
            end if;

            Analyze (Expr);
            Resolve (Expr, Class_Wide_Type (RTE (RE_Root_Storage_Pool)));

            if Is_Entity_Name (Expr) then
               Set_Associated_Storage_Pool (Ent, Entity (Expr));

            elsif Nkind (Expr) = N_Attribute_Reference
              and then Attribute_Name (Expr) = Name_Storage_Pool
            then
               Pool := Associated_Storage_Pool (Entity (Prefix (Expr)));

               if Present (Etype (Pool))
                 and then Etype (Pool) /= RTE (RE_Stack_Bounded_Pool)
                 and then Etype (Pool) /= RTE (RE_Unbounded_Reclaim_Pool)
               then
                  Set_Associated_Storage_Pool (Ent, Pool);
               else
                  Error_Msg_N ("Non sharable GNAT Pool", Expr);
               end if;
            else

               Error_Msg_N ("incorrect reference to a Storage Pool", Expr);
               return;
            end if;
         end Storage_Pool;

         --  All other attributes cannot be set

         when others =>
            Error_Msg_N
              ("attribute& cannot be set with definition clause", N);

      end case;
   end Analyze_Attribute_Definition_Clause;

   ----------------------------
   -- Analyze_Code_Statement --
   ----------------------------

   procedure Analyze_Code_Statement (N : Node_Id) is
   begin
      Unimplemented (N, "code statement");
   end Analyze_Code_Statement;

   -----------------------------------------------
   -- Analyze_Enumeration_Representation_Clause --
   -----------------------------------------------

   procedure Analyze_Enumeration_Representation_Clause (N : Node_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      Ident    : constant Node_Id    := Identifier (N);
      Aggr     : constant Node_Id    := Array_Aggregate (N);
      Enumtype : Entity_Id;
      Elit     : Entity_Id;
      Expr     : Node_Id;
      Assoc    : Node_Id;
      Choice   : Node_Id;
      Val      : Uint;
      Err      : Boolean := False;

      Lo  : constant Uint := Expr_Value (Type_Low_Bound (Universal_Integer));
      Hi  : constant Uint := Expr_Value (Type_High_Bound (Universal_Integer));
      Min : Uint;
      Max : Uint;

   begin
      --  First some basic error checks

      Analyze (Ident);
      Enumtype := Entity (Ident);

      if not Is_Enumeration_Type (Enumtype) then
         Error_Msg_N ("& is not an enumeration type", Ident);
         return;
      end if;

      if not Is_First_Named_Subtype (Enumtype) then
         Error_Msg_N ("cannot give enumeration clause for subtype", Ident);
         return;

      elsif Has_Enumeration_Rep_Clause (Enumtype) then
         Error_Msg_N
           ("duplicate enumeration rep clause ignored", N);
         return;

      elsif Already_Frozen (Enumtype, Ident) then
         return;

      else
         Set_Has_Enumeration_Rep_Clause (Enumtype);
      end if;

      --  Now we process the aggregate. Note that we don't use the normal
      --  aggregate code for this purpose, because we don't want any of the
      --  normal expansion activities, and a number of special semantic
      --  rules apply (including the component type being any integer type)

      --  Badent signals that we found some incorrect entries processing
      --  the list. The final checks for completeness and ordering are
      --  skipped in this case.

      Elit := First_Literal (Enumtype);

      --  First the positional entries if any

      if Present (Expressions (Aggr)) then
         Expr := First (Expressions (Aggr));
         while Present (Expr) loop

            if No (Elit) then
               Error_Msg_N ("too many entries in aggregate", Expr);
               return;
            end if;

            Val := Static_Integer (Expr);

            if Val = No_Uint then
               Err := True;

            elsif UI_Lt (Val, Lo) or else UI_Lt (Hi, Val) then
               Error_Msg_N ("value outside permitted range", Expr);
               Err := True;
            end if;

            Set_Enumeration_Rep (Elit, Val);
            Set_Enumeration_Rep_Expr (Elit, Expr);
            Expr := Next (Expr);
            Elit := Next (Elit);
         end loop;
      end if;

      --  Now process the named entries if present

      if Present (Component_Associations (Aggr)) then
         Assoc := First (Component_Associations (Aggr));
         while Present (Assoc) loop
            Choice := First (Choices (Assoc));

            if Present (Next (Choice)) then
               Error_Msg_N
                 ("multiple choice not allowed here", Next (Choice));
               Err := True;
            end if;

            if Nkind (Choice) = N_Others_Choice then
               Error_Msg_N ("others choice not allowed here", Choice);
               Err := True;

            elsif Nkind (Choice) = N_Range then
               --  ??? should allow zero/one element range here
               Error_Msg_N ("range not allowed here", Choice);
               Err := True;

            else
               Analyze (Choice);
               Resolve (Choice, Enumtype);

               if Is_Entity_Name (Choice)
                 and then Is_Type (Entity (Choice))
               then
                  Error_Msg_N ("subtype name not allowed here", Choice);
                  Err := True;
                  --  ??? should allow static subtype with zero/one entry

               elsif Etype (Choice) = Enumtype then
                  if not Is_Static_Expression (Choice) then
                     Check_Static_Expression (Choice);
                     Err := True;
                  else
                     Elit := Expr_Value (Choice);

                     if Present (Enumeration_Rep_Expr (Elit)) then
                        Error_Msg_Sloc := Sloc (Enumeration_Rep_Expr (Elit));
                        Error_Msg_NE
                          ("representation for& previously given#",
                           Choice, Elit);
                        Err := True;
                     end if;

                     Set_Enumeration_Rep_Expr (Elit, Choice);

                     Val := Static_Integer (Expression (Assoc));

                     if Val = No_Uint then
                        Err := True;
                     elsif UI_Lt (Val, Lo) or else UI_Lt (Hi, Val) then
                        Error_Msg_N ("value outside permitted range", Expr);
                        Err := True;
                     end if;

                     Set_Enumeration_Rep (Elit, Val);
                  end if;
               end if;
            end if;

            Assoc := Next (Assoc);
         end loop;
      end if;

      --  Aggregate is fully processed. Now we check that a full set of
      --  representations was given, and that they are in range and in order.
      --  These checks are only done if no other errors occurred.

      if not Err then
         Min  := No_Uint;
         Max  := No_Uint;

         Elit := First_Literal (Enumtype);
         while Present (Elit) loop
            if No (Enumeration_Rep_Expr (Elit)) then
               Error_Msg_NE ("missing representation for&!", N, Elit);

            else
               Val := Enumeration_Rep (Elit);

               if Min = No_Uint then
                  Min := Val;
               end if;

               if Val /= No_Uint then
                  if Max /= No_Uint and then UI_Le (Val, Max) then
                     Error_Msg_NE
                       ("enumeration value for& not ordered!",
                                       Enumeration_Rep_Expr (Elit), Elit);
                  end if;

                  Max := Val;
               end if;

            end if;

            Elit := Next (Elit);
         end loop;
      end if;

   end Analyze_Enumeration_Representation_Clause;

   ----------------------------
   -- Analyze_Free_Statement --
   ----------------------------

   procedure Analyze_Free_Statement (N : Node_Id) is
   begin
      Analyze (Expression (N));
   end Analyze_Free_Statement;

   ------------------------------------------
   -- Analyze_Record_Representation_Clause --
   ------------------------------------------

   procedure Analyze_Record_Representation_Clause (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Ident   : constant Node_Id    := Identifier (N);
      Rectype : Entity_Id;
      Mod_Val : Uint;
      CC      : Node_Id;
      Posit   : Uint;
      Fbit    : Uint;
      Lbit    : Uint;
      Adjust  : Uint;
      Comp    : Entity_Id;

   begin
      --  First some basic error checks

      Analyze (Ident);
      Rectype := Entity (Ident);

      if not Is_Record_Type (Rectype) then
         Error_Msg_N ("& is not a record type", Ident);
         return;
      end if;

      if not Is_First_Named_Subtype (Rectype) then
         Error_Msg_N
           ("cannot give record rep clause for subtype", Ident);
         return;

      elsif Has_Record_Rep_Clause (Rectype) then
         Error_Msg_N ("duplicate record rep clause ignored", N);
         return;

      elsif Already_Frozen (Rectype, Ident) then
         return;

      else
         Set_Has_Record_Rep_Clause (Rectype);
      end if;

      --  Processing for mod clause if present

      if Present (Mod_Clause (N)) then
         Mod_Val := Static_Integer (Expression (Mod_Clause (N)));

         --  If expression was valid and we are generating code (i.e. the
         --  expander is active), then the mod clause is removed and replaced
         --  by inserting an Alignment attribute definition clause.

         if Mod_Val /= No_Uint and then Expander_Active then
            Insert_After (N,
              Make_Attribute_Definition_Clause (Loc,
                Name       => New_Copy (N),
                Chars      => Name_Alignment,
                Expression => Make_Integer_Literal (Loc, Mod_Val)));
            Set_Mod_Clause (N, Empty);
         end if;
      end if;

      --  Process the component clauses

      CC := First (Component_Clauses (N));

      while Present (CC) loop

         Posit := Static_Integer (Position  (CC));
         Fbit  := Static_Integer (First_Bit (CC));
         Lbit  := Static_Integer (Last_Bit  (CC));

         if Posit /= No_Uint
          and then Fbit /= No_Uint
          and then Lbit /= No_Uint
         then

            if UI_Is_Negative (Posit) then
               Error_Msg_N ("position cannot be negative", Position (CC));

            elsif UI_Is_Negative (Fbit) then
               Error_Msg_N ("first bit cannot be negative", First_Bit (CC));

            --  Values look OK, so find the corresponding record component

            else
               Comp := First_Entity (Rectype);
               while Present (Comp) loop
                  exit when Chars (Comp) = Chars (Component_Name (CC));
                  Comp := Next_Entity (Comp);
               end loop;

               if No (Comp) then
                  Error_Msg_N
                    ("component clause is for non-existent field", N);

               elsif Present (Component_Clause (Comp)) then
                  Error_Msg_Sloc := Sloc (Component_Clause (Comp));
                  Error_Msg_N ("component clause previously given#", CC);

               else
                  Set_Component_Clause (Comp, CC);
                  Set_Component_First_Bit (Comp,
                    UI_Sum (Fbit,
                      UI_Product (UI_From_Int (System_Storage_Unit), Posit)));
                  Set_Esize (Comp,
                    UI_Sum (Uint_1, UI_Difference (Lbit, Fbit)));

                  if UI_Is_Negative (Esize (Comp)) then
                     Error_Msg_N ("component size is negative", CC);
                  end if;
               end if;
            end if;
         end if;

         CC := Next (CC);
      end loop;

   end Analyze_Record_Representation_Clause;

   -------------------
   -- Freeze_Entity --
   -------------------

   function Freeze_Entity (E : Entity_Id; Loc : Source_Ptr) return List_Id is
      Comp    : Entity_Id;
      Elmt    : Elmt_Id;
      F_Node  : Node_Id;
      Op_List : Elist_Id;
      Result  : List_Id;
      Subp    : Entity_Id;

      procedure Freeze_It is
      begin
         F_Node := New_Node (N_Freeze_Entity, Loc);
         Set_Entity (F_Node, E);
         Append (F_Node, Result);
      end Freeze_It;

   --  Start of processing for Freeze_Entity

   begin
      if not Is_Frozen (E)
        and then not Has_Private_Component (E)
      then
         Result := New_List;
         Set_Is_Frozen (E);

         if E /= Base_Type (E) then
            Append_List (Freeze_Entity (Base_Type (E), Loc), Result);

            --  Don't create Freeze Nodes for subtypes unless they are delayed

            if not Is_Delayed (E) then
               return Result;
            end if;
         end if;

         if Ekind (E) = E_Access_Type
           or else Ekind (E) = E_General_Access_Type
         then
            Freeze_It;

         elsif Is_Array_Type (E) then
            Append_List (Freeze_Entity (Component_Type (E), Loc), Result);
            Freeze_It;

         elsif Is_Record_Type (E) then
            Comp := First_Entity (E);

            while Present (Comp) loop
               Append_List
                 (Freeze_Entity (Base_Type (Etype (Comp)), Loc), Result);
               Comp := Next_Entity (Comp);
            end loop;

            if Is_Tagged_Type (E)
              and then Ekind (E) = E_Record_Type
            then
               Op_List := Primitive_Operations (E);

               --  ??? Our current model requires that the primitives must be
               --  frozen before the dispatch table is created. So for now,
               --  force the freezing here. Expand_Dispatch_Table recognize the
               --  obvious case where this approach is not correct and post an
               --  unimplemented message. The ultimate solution to this problem
               --  is to create an untyped dispatch table and recreate the
               --  appropriate type when expanding the dispatching call.  Also
               --  verify that no primitive operation of a non-abstract type is
               --  abstract

               Elmt := First_Elmt (Op_List);

               while Present (Elmt) loop
                  Subp := Node (Elmt);

                  if Is_Delayed (Subp) then
                     Append_List (Freeze_Entity (Subp, Loc), Result);
                  end if;

                  if Is_Abstract (Subp) and then not Is_Abstract (E) then
                     if Present (Alias (Subp)) then
                        Error_Msg_NE
                          ("type must be declared abstract or else "
                                              & "& overriden",
                           E, Subp);
                     else
                        Error_Msg_NE
                          ("non-abstract type has abstract subprogram&",
                           E, Subp);
                     end if;
                  end if;

                  Elmt := Next_Elmt (Elmt);
               end loop;
            end if;

            Freeze_It;

         elsif Is_Task_Type (E)
           and then Present (Corresponding_Record_Type (E))
         then
            Append_List
              (Freeze_Entity (Corresponding_Record_Type (E), Loc), Result);

         elsif Is_Delayed (E)
           and then not Is_Incomplete_Or_Private_Type (E)
         then
            --  Subprogram whose profile includes a private type,
            --  or subtype of private type,  or full declaration of
            --  private type. Note that the entity in the full declaration
            --  is currently a copy of the original private entity (it has
            --  not been swapped back yet) and does not receive a freeze node.

            Freeze_It;
         else
            null;
         end if;

         return Result;

      else
         return Empty_List;
      end if;

   end Freeze_Entity;

   -----------------
   --  Freeze_All --
   -----------------

   function Freeze_All (Scope : Entity_Id) return List_Id is
      E      : Entity_Id;
      Result : List_Id := New_List;
      Loc    : constant Source_Ptr := Sloc (Last_Entity (Current_Scope));

   begin
      E := First_Entity (Scope);

      while Present (E) loop

         if not Is_Frozen (E) then
            Append_List (Freeze_Entity (E, Loc), Result);
         end if;

         --  If the entity is a package which is not the renaming of another,
         --  make sure everything within is frozen. This is necessary for
         --  types derived from private types in inner packages.

         if Ekind (E) = E_Package
           and then No (Renamed_Object (E))
         then
            Append_List (Freeze_All (E), Result);
         end if;

         E := Next_Entity (E);
      end loop;

      return Result;
   end Freeze_All;

   -------------------
   -- Freeze_Before --
   -------------------

   procedure Freeze_Before (N : Node_Id; T : Entity_Id) is
      Freeze_Nodes : constant List_Id := Freeze_Entity (T, Sloc (N));
      F            : Node_Id;

   begin
      F := First (Freeze_Nodes);

      if Present (F) then

         if Nkind (N) = N_Object_Declaration then

            --  Implicit types are transfered into the Freeze Node because
            --  they may be frozen here!

            Transfer_Itypes (From => N, To => F);
         end if;

         Insert_List_Before (N, Freeze_Nodes);

         while F /= N loop
            Analyze (F);
            F := Next (F);
         end loop;
      end if;
   end Freeze_Before;

   -----------------------
   -- Freeze_Expression --
   -----------------------

   procedure Freeze_Expression (N : Node_Id) is
      Static    : constant Boolean   := Is_Static_Expression (N);
      Typ       : Entity_Id := Empty;
      Nam       : Entity_Id := Empty;
      Desig_Typ : Entity_Id := Empty;
      P         : Node_Id;
      Parent_P  : Node_Id;
      Null_Stmt : Node_Id;

   begin

      if Nkind (N) in N_Has_Etype and then not Is_Frozen (Etype (N)) then
         Typ := Etype (N);
      end if;

      if Is_Entity_Name (N) and then not Is_Frozen (Entity (N)) then
         Nam := Entity (N);
      end if;

      if Nkind (N) = N_Allocator
        and then not Is_Frozen (Designated_Type (Etype (N))) then
         Desig_Typ := Designated_Type (Etype (N));
      end if;

      --  Check if something needs freezing

      if No (Typ) and then No (Nam) and then No (Desig_Typ) then
         return;
      end if;

      --  Loop for looking at the right place to insert the freeze nodes
      --  check also that if the expr is not static, it is not part of a
      --  default expression (RM 13-14 (8))

      P := N;

      loop
         Parent_P := Parent (P);

         case Nkind (Parent_P) is
            when N_Parameter_Specification    |
                 N_Discriminant_Specification |
                 N_Component_Declaration      |
                 N_Formal_Object_Declaration  =>
            begin
               if not Static then
                  return;
               end if;
            end;

            when N_Subprogram_Body       |
                 N_Package_Specification |
                 N_Package_Body          |
                 N_Task_Body             |
                 N_Entry_Body            |
                 N_Block_Statement       |

               --  The expander is allowed to define types in any statements
               --  list, so any of the following parent nodes also mark a
               --  freezing point if the actual node is in a list of statements
               --  or declarations.

                 N_Handled_Sequence_Of_Statements |
                 N_Exception_Handler              |
                 N_If_Statement                   |
                 N_Elsif_Part                     |
                 N_Case_Statement_Alternative     |
                 N_Loop_Statement                 |
                 N_Selective_Accept               |
                 N_Accept_Alternative             |
                 N_Delay_Alternative              |
                 N_Conditional_Entry_Call         |
                 N_Entry_Call_Alternative         |
                 N_Triggering_Alternative         |
                 N_Abortable_Part                 =>
            begin
               if Is_List_Member (P) then
                  exit;
               end if;
            end;

            --  If the type is defined inside an expression-action and the
            --  expression uses this type. Freeze it at the end of the action
            --  part. To simplify processing, just create a Null_Statement at
            --  the end and freeze before this dummy node.

            when N_Expression_Actions =>
               if Present (Typ)
                 and then Present (Parent (Typ))
                 and then Parent (Parent (Typ)) = Parent_P
               then
                  Null_Stmt := Make_Null_Statement (Sloc (Parent_P));
                  Append_To (Actions (Parent_P),  Null_Stmt);
                  P := Null_Stmt;
                  exit;
               end if;

            when others => null;
         end case;

         P := Parent_P;

         if No (Parent_P) then

            --  Should never happen in a well-formed tree but there are some
            --  legitimate cases where a subtree is analyzed and resolved
            --  without beeing attached to the syntax tree (e.g. scalar range
            --  of a modular type). Such cases shouldn't require freezing.

            return;
         end if;
      end loop;

      --  RM 13.14(12)

      if Present (Desig_Typ) then
         Freeze_Before (P, Desig_Typ);
      end if;

      --  Simplified version of RM 13.13(9)
      --  ??? rep clause for enumaration type cannot work before 13.14 (9) is
      --  fully implemented

      if Present (Typ) then
         Freeze_Before (P, Typ);
      end if;

      --  RM 13.14(10)

      if Present (Nam) then
         Freeze_Before (P, Nam);
      end if;

   end Freeze_Expression;

   --------------------------------------
   -- Validate_Unchecked_Conversion --
   --------------------------------------

   procedure Validate_Unchecked_Conversion (N : Node_Id; Act_Unit : Entity_Id)
   is
      Source : Entity_Id;
      Target : Entity_Id;

      procedure No_Unconstrained_Type (T : Node_Id);
      --  Issue error if type T is an unconstrained type

      procedure No_Unconstrained_Type (T : Node_Id) is
      begin
         if Is_Indefinite_Subtype (T) then
            Error_Msg_NE
              ("unconstrained type& not allowed in unchecked conversion",
               N, T);
         end if;
      end No_Unconstrained_Type;

   --  Start of processing for Validate_Unchecked_Conversion

   begin
      --  If we are dealing with private types, then do the check on their
      --  fully declared counterparts if the full declarations have been
      --  encountered (they don't have to be visible, but they must exist!)

      Source := Etype (First_Formal (Act_Unit));

      if Is_Private_Type (Source)
        and then Present (Underlying_Type (Source))
      then
         Source := Underlying_Type (Source);
      end if;

      Target := Etype (Act_Unit);

      if Is_Private_Type (Target)
        and then Present (Underlying_Type (Target))
      then
         Target := Underlying_Type (Target);
      end if;

      No_Unconstrained_Type (Source);
      No_Unconstrained_Type (Target);

      if Esize (Source) /= Uint_0
        and then Esize (Target) /= Uint_0
        and then UI_Ne (Esize (Source), Esize (Target))
      then
         Error_Msg_N
           ("types for unchecked conversion have different sizes", N);
      end if;
   end Validate_Unchecked_Conversion;

end Sem_Ch13;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.92
--  date: Sat Jul 30 23:33:40 1994;  author: comar
--  (Freeze_Entity): Freeze primitive operations before its associated tagged
--   type (a bit beyong Ada rules but our current dispatch table format
--   requires it)
--  ----------------------------
--  revision 1.93
--  date: Thu Aug 18 16:26:46 1994;  author: comar
--  Add new procedure Analyze_Free_Statement.
--  ----------------------------
--  revision 1.94
--  date: Sun Aug 28 08:49:13 1994;  author: comar
--  (Analyze_Attribute_Definition_Clause): for Storage_Size and Storage_Pool,
--   General_Access_Types are acceptable.
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
