------------------------------------------------------------------------------
--                                                                          --
--                         GNAT COMPILER COMPONENTS                         --
--                                                                          --
--                              E X P _ C H 4                               --
--                                                                          --
--                                 B o d y                                  --
--                                                                          --
--                            $Revision: 1.114 $                            --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------

with Atree;    use Atree;
with Einfo;    use Einfo;
with Elists;   use Elists;
with Errout;   use Errout;
with Exp_Ch3;  use Exp_Ch3;
with Exp_Ch7;  use Exp_Ch7;
with Exp_Ch9;  use Exp_Ch9;
with Exp_Util; use Exp_Util;
with Fix_Util; use Fix_Util;
with Itypes;   use Itypes;
with Nlists;   use Nlists;
with Nmake;    use Nmake;
with Opt;      use Opt;
with Output;   use Output;
with Rtsfind;  use Rtsfind;
with Sem;      use Sem;
with Sem_Res;  use Sem_Res;
with Sem_Util; use Sem_Util;
with Sem_Ch5;  use Sem_Ch5;
with Sinfo;    use Sinfo;
with Sinfo.CN; use Sinfo.CN;
with Snames;   use Snames;
with Stand;    use Stand;
with Tbuild;   use Tbuild;
with Ttypes;   use Ttypes;
with Uintp;    use Uintp;
with Urealp;   use Urealp;

package body Exp_Ch4 is

   ------------------------
   --  Local Subprograms --
   ------------------------

   function Build_Qual_Expr
     (Val  : Uint;
      Typ  : Entity_Id;
      Loc  : Source_Ptr)
      return Node_Id;
   --  Make a qualified expression node of the specified integer type with the
   --  value given.

   procedure Expand_Arithmetic_Overflow_Check (N : Node_Id);
   --  Given a binary arithmetic operator (+ - * /) expand a software integer
   --  overflow check using range checks on a larger checking type.

   function Expand_Array_Equality
     (Loc : Source_Ptr; Typ : Entity_Id; Lhs, Rhs : Node_Id) return Node_Id;
   --  Expand an array equality into an expression-action containing a local
   --  function implementing this equality, and a call to it. Loc is the
   --  location for the generated nodes. Typ is the type of the array, and
   --  Lhs, Rhs are the array expressions to be compared.

   procedure Expand_Boolean_Operator (N : Node_Id);
   --  Common expansion processing for Boolean operators (And, Or, Xor)

   procedure Expand_Comparison_Operator (N : Node_Id);
   --  This routine handles expansion of the comparison operators (N_Op_Lt,
   --  N_Op_Le, N_Op_Gt, N_Op_Ge). Since the code is basicallly similar with
   --  the addition of some outer

   function Expand_Composite_Equality
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Lhs  : Node_Id;
      Rhs  : Node_Id)
      return Node_Id;
   --  Local recursive function used to expand equality for nested
   --  composite types. Used by Expand_Record_Equality, Expand_Array_Equality.

   procedure Expand_Concatenation (Node : Node_Id; Ops : List_Id);
   --  This routine handles expansion of concatenation operations, where
   --  N is the N_Op_Concat or N_Concat_Multiple node being expanded, and
   --  Ops is the list of operands (at least two are present).

   procedure Expand_Fixed_To_Fixed_Division (N : Node_Id);
   --  This routine expands the division between fixed point types, with a
   --  fixed point type result. It follows Hilfinger's algorithms.

   procedure Expand_Fixed_Integer_Division (N : Node_Id);
   --  This routine expands the division between a fixed point type and
   --  an standard integer type. The result type is the same fixed point type
   --  as the operand's one.

   procedure Expand_Fixed_Integer_Multiplication (N, Fix, Int : Node_Id);
   --  This routine expands the multiplication between a fixed point type and
   --  a standard integer type. The result type is the same fixed point type
   --  as the operand's one.

   procedure Expand_Fixed_To_Fixed_Multiplication (N : Node_Id);
   --  This routine expands the multiplication between fixed point types, with
   --  a fixed point type result. It follows Hilfinger's algorithms.

   procedure Expand_Fixed_To_Float_Conversion (N : Node_Id);
   --  This routine expands the conversion from one fixed point type to a
   --  floating point type.

   procedure Expand_Fixed_To_Float_Division (N : Node_Id);
   --  This routine expands the division between two fixed point
   --  type values with a floating point type result.

   procedure Expand_Fixed_To_Float_Multiplication (N : Node_Id);
   --  This routine expands the multiplication between two fixed point
   --  type values with a floating point type result.

   procedure Expand_Fixed_To_Integer_Division (N : Node_Id);
   --  This routine expands the division between two fixed point
   --  type values with an integer type result.

   procedure Expand_Fixed_To_Integer_Multiplication (N : Node_Id);
   --  This routine expands the multiplication between two fixed point
   --  type values with an integer type result.

   procedure Expand_Float_To_Fixed_Conversion (N : Node_Id);
   --  This routine expands the conversion from a floating point type to a
   --  fixed point type.

   procedure Expand_Zero_Divide_Check (N : Node_Id);
   --  The node kind is N_Op_Divide, N_Op_Mod, or N_Op_Rem. The right operand
   --  is replaced by an expression actions node that checks that the divisor
   --  (right operand) is non-zero. Note that in the divide case, but not in
   --  the other two cases, overflow can still occur with a non-zero divisor
   --  as a result of dividing the largest negative number by minus one.

   function Get_Double_Sized_Type (Size_Type : Uint) return Entity_Id;
   --  Find the standard integer type whose size is double that of Size_Type.

   function Make_Array_Comparison_Op
     (Typ   : Entity_Id;
      Loc   : Source_Ptr;
      Equal : Boolean)
      return  Node_Id;
   --  Comparisons between arrays are expanded in line. This function
   --  produces the body of the implementation of (a > b), or (a >= b), when
   --  a and b are one-dimensional arrays of some discrete type. The original
   --  node is then expanded into the appropriate call to this function.

   function Make_Boolean_Array_Op (N : Node_Id) return Node_Id;
   --  Boolean operations on boolean arrays are expanded in line. This
   --  function produce the body for (a and b), (a or b), or (a xor b).

   function Tagged_Membership (N : Node_Id) return Node_Id;
   --  Construct the expression corresponding to the tagged membership test.
   --  Deals with a second operand being (or not) a class-wide type.

   ---------------------
   -- Build_Qual_Expr --
   ---------------------

   function Build_Qual_Expr
     (Val  : Uint;
      Typ  : Entity_Id;
      Loc  : Source_Ptr)
      return Node_Id
   is
   begin
      pragma Assert (Is_Integer_Type (Etype (Typ)));
      return Make_Qualified_Expression (Loc,
         Subtype_Mark => New_Reference_To (Typ, Loc),
        Expression => Make_Integer_Literal (Loc, Intval => Val));
   end Build_Qual_Expr;

   --------------------------------------
   -- Expand_Arithmetic_Overflow_Check --
   --------------------------------------

   --  This routine is called only if the type is an integer type, and
   --  a software arithmetic overflow check must be performed.

   --    x op y

   --  is expanded into

   --    Typ (Checktyp (x) op Checktyp (y));

   --  where Typ is the type of the original expression, and Checktyp is an
   --  integer type of sufficient length to hold the largest possible result.

   --  Similarly,

   --    op x

   --  is expanded into

   --    Typ (op Checktyp (x));

   procedure Expand_Arithmetic_Overflow_Check (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Typ   : constant Entity_Id  := Etype (N);
      Rtyp  : constant Entity_Id  := Root_Type (Typ);
      Siz   : constant Int        := UI_To_Int (Esize (Rtyp));
      Dsiz  : constant Int        := Siz * 2;
      Opnod : Node_Id;
      Ctyp  : Entity_Id;
      Opnd  : Node_Id;

   begin
      --  Find check type

      if Dsiz <= Standard_Integer_Size then
         Ctyp := Standard_Integer;

      elsif Dsiz <= Standard_Long_Long_Integer_Size then
         Ctyp := Standard_Long_Long_Integer;

      else
         Ctyp := Huge_Integer;
      end if;

      --  Prepare adjusted version of operator node, with operands converted
      --  to the appropriate check type.

      Opnod := Relocate_Node (N);

      Opnd :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Ctyp, Loc),
          Expression => Right_Opnd (Opnod));

      Analyze (Opnd);
      Set_Etype (Opnd, Ctyp);
      Set_Analyzed (Opnd, True);
      Set_Right_Opnd (Opnod, Opnd);

      if Nkind (N) in N_Binary_Op then
         Opnd :=
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Ctyp, Loc),
             Expression => Left_Opnd (Opnod));

         Analyze (Opnd);
         Set_Etype (Opnd, Ctyp);
         Set_Analyzed (Opnd, True);

         Set_Left_Opnd (Opnod, Opnd);
      end if;

      --  The type of the operation changes to the base type of the check
      --  type, and we reset the overflow check indication, since clearly
      --  no overflow is possible now that we are using a double length
      --  type. We also set the Analyzed flag to avoid a recursive attempt
      --  to expand the node.

      Set_Etype             (Opnod, Base_Type (Ctyp));
      Set_Do_Overflow_Check (Opnod, False);
      Set_Analyzed          (Opnod, True);

      --  Now build the outer conversion

      Opnd :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Typ, Loc),
          Expression => Opnod);

      Analyze (Opnd);
      Set_Etype (Opnd, Typ);
      Set_Analyzed (Opnd, True);
      Set_Do_Range_Check (Opnd, True);
      Set_Do_Overflow_Check (Opnd, True);

      Replace_Substitute_Tree (N, Opnd);
   end Expand_Arithmetic_Overflow_Check;

   ---------------------------
   -- Expand_Array_Equality --
   ---------------------------

   --  Expand an equality function for multi-dimentionnal arrays. Here is
   --  an example of such a function for Nb_Dimension = 2

   --  function Enn (A : arr; B : arr) return boolean is
   --     J1 : integer := B'first (1);
   --     J2 : integer := B'first (2);

   --  begin
   --     if A'length (1) /= B'length (1) then
   --        return false;
   --     else
   --        for I1 in A'first (1) .. A'last (1) loop
   --           if A'length (2) /= B'length (2) then
   --              return false;
   --           else
   --              for I2 in A'first (2) .. A'last (2) loop
   --                 if A (I1, I2) /=  B (J1, J2) then
   --                    return false;
   --                 end if;
   --                 J2 := Integer'succ (J2);
   --              end loop;
   --           end if;
   --           J1 := Integer'succ (J1);
   --        end loop;
   --     end if;
   --     return true;
   --  end Enn;

   function Expand_Array_Equality
     (Loc      : Source_Ptr;
      Typ      : Entity_Id;
      Lhs, Rhs : Node_Id)
      return     Node_Id
   is
      Decls       : List_Id := New_List;
      Index_List1 : List_Id := New_List;
      Index_List2 : List_Id := New_List;
      Index       : Entity_Id := First_Index (Typ);
      Index_Type  : Entity_Id;
      Formals     : List_Id;
      Result      : Node_Id;
      Stats       : Node_Id;
      Func_Name   : Entity_Id;
      Func_Body   : Node_Id;

      A : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uA);
      B : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uB);

      function Component_Equality (Typ : Entity_Id) return Node_Id;
      --  Create one statement to compare corresponding components, designated
      --  by a full set of indices.

      function Loop_One_Dimension (N : Int) return Node_Id;
      --  Loop over the n'th dimension of the arrays. The single statement
      --  in the body of the loop is a loop over the next dimension, or
      --  the comparison of corresponding components.

      ------------------------
      -- Component_Equality --
      ------------------------

      function Component_Equality (Typ : Entity_Id) return Node_Id is
         Test : Node_Id;
         L, R : Node_Id;

      begin
         --  if a(i1...) /= b(j1...) then return false; end if;

         L :=
           Make_Indexed_Component (Loc,
             Prefix => Make_Identifier (Loc, Chars (A)),
             Expressions => Index_List1);

         R :=
           Make_Indexed_Component (Loc,
             Prefix => Make_Identifier (Loc, Chars (B)),
             Expressions => Index_List2);

         Test := Expand_Composite_Equality (Loc, Component_Type (Typ), L, R);

         return
           Make_If_Statement (Loc,
             Condition => Make_Op_Not (Loc, Right_Opnd => Test),
             Then_Statements => New_List (
               Make_Return_Statement (Loc,
                 Expression => New_Occurrence_Of (Standard_False, Loc))));

      end Component_Equality;

      ------------------------
      -- Loop_One_Dimension --
      ------------------------

      function Loop_One_Dimension (N : Int) return Node_Id is
         I : constant Entity_Id := Make_Defining_Identifier (Loc,
                                                  New_Internal_Name ('I'));
         J : constant Entity_Id := Make_Defining_Identifier (Loc,
                                                  New_Internal_Name ('J'));
         Stats : Node_Id;

      begin
         if N > Number_Dimensions (Typ) then
            return Component_Equality (Typ);

         else
            --  Generate the following:

            --  j: index_type := b'first (n);
            --  ...
            --  if a'length (n) /= b'length (n) then
            --    return false;
            --  else
            --     for i in a'range (n) loop
            --        --  loop over remaining dimensions.
            --        j := index_type'succ (j);
            --     end loop;
            --  end if;

            --  retrieve index type for current dimension.

            Index_Type := Base_Type (Etype (Index));
            Append (New_Reference_To (I, Loc), Index_List1);
            Append (New_Reference_To (J, Loc), Index_List2);

            --  Declare index for j as a local variable to the function.
            --  Index i is a loop variable.

            Append_To (Decls,
              Make_Object_Declaration (Loc,
                Defining_Identifier => J,
                Object_Definition   => New_Reference_To (Index_Type, Loc),
                Expression =>
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Reference_To (B, Loc),
                    Attribute_Name => Name_First,
                    Expressions => New_List (
                        Make_Integer_Literal (Loc, UI_From_Int (N))))));

            Stats :=
              Make_If_Statement (Loc,
                Condition =>
                  Make_Op_Ne (Loc,
                    Left_Opnd =>
                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (A, Loc),
                        Attribute_Name => Name_Length,
                        Expressions => New_List (
                          Make_Integer_Literal (Loc, UI_From_Int (N)))),
                    Right_Opnd =>
                      Make_Attribute_Reference (Loc,
                        Prefix => New_Reference_To (B, Loc),
                        Attribute_Name => Name_Length,
                        Expressions => New_List (
                          Make_Integer_Literal (Loc, UI_From_Int (N))))),

                Then_Statements => New_List (
                  Make_Return_Statement (Loc,
                    Expression => New_Occurrence_Of (Standard_False, Loc))),

                Else_Statements => New_List (
                  Make_Loop_Statement (Loc,
                    Identifier => Empty,
                    Iteration_Scheme =>
                      Make_Iteration_Scheme (Loc,
                        Loop_Parameter_Specification =>
                          Make_Loop_Parameter_Specification (Loc,
                            Defining_Identifier => I,
                            Discrete_Subtype_Definition =>
                              Make_Attribute_Reference (Loc,
                                Prefix => New_Reference_To (A, Loc),
                                Attribute_Name => Name_Range,
                                Expressions => New_List (
                                  Make_Integer_Literal (Loc,
                                    Intval => UI_From_Int (N)))))),

                    Statements => New_List (
                      Loop_One_Dimension (N + 1),
                      Make_Assignment_Statement (Loc,
                        Name => New_Reference_To (J, Loc),
                        Expression =>
                          Make_Attribute_Reference (Loc,
                            Prefix => New_Reference_To (Index_Type, Loc),
                            Attribute_Name => Name_Succ,
                            Expressions => New_List (
                              New_Reference_To (J, Loc))))))));

            Index := Next_Index (Index);
            return Stats;
         end if;
      end Loop_One_Dimension;

   ------------------------------------------
   -- Processing for Expand_Array_Equality --
   ------------------------------------------

   begin
      Formals := New_List (
        Make_Parameter_Specification (Loc,
          Defining_Identifier => A,
          Parameter_Type      => New_Reference_To (Typ, Loc)),

        Make_Parameter_Specification (Loc,
          Defining_Identifier => B,
          Parameter_Type      => New_Reference_To (Typ, Loc)));

      Func_Name := Make_Defining_Identifier (Loc,  New_Internal_Name ('E'));

      Stats := Loop_One_Dimension (1);

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name       => Func_Name,
              Parameter_Specifications => Formals,
              Subtype_Mark => New_Reference_To (Standard_Boolean, Loc)),
          Declarations               =>  Decls,
          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (
                Stats,
                Make_Return_Statement (Loc,
                  Expression => New_Occurrence_Of (Standard_True, Loc)))));

         Set_Has_Completion (Func_Name, True);

         Result :=
           Make_Expression_Actions (Loc,
             Actions    => New_List (Func_Body),
             Expression => Make_Function_Call (Loc,
               Name => New_Reference_To (Func_Name, Loc),
               Parameter_Associations => New_List (Lhs, Rhs)));

         return Result;
   end Expand_Array_Equality;

   -----------------------------
   -- Expand_Boolean_Operator --
   -----------------------------

   --  Expansion happens only for the array type cases. The expansion is
   --  to an expression actions node that declares a function to perform
   --  the desired operation, followed by a call to it. The construction
   --  of the function is performed by Make_Boolean_Array_Op.

   procedure Expand_Boolean_Operator (N : Node_Id) is
      Loc       : constant Source_Ptr := Sloc (N);
      Typ       : constant Entity_Id  := Etype (N);
      Result    : Node_Id;
      Func_Body : Node_Id;
      Func_Name : Entity_Id;

   begin
      if Is_Array_Type (Typ) then
         Func_Body := Make_Boolean_Array_Op (N);
         Func_Name := Defining_Unit_Name (Specification (Func_Body));

         Result :=
           Make_Expression_Actions (Loc,
             Actions => New_List (Func_Body),
             Expression =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (Func_Name, Loc),
                 Parameter_Associations =>
                   New_List (Left_Opnd (N), Right_Opnd (N))));

         Replace_Substitute_Tree (N, Result);
         Analyze (N);
         Resolve (N, Typ);
      end if;
   end Expand_Boolean_Operator;

   --------------------------------
   -- Expand_Comparison_Operator --
   --------------------------------

   --  Expansion is only required in the case of array types. The form of
   --  the expansion is:

   --     [body for greater_nn; boolean_expression]

   --  The body is built by Make_Array_Comparison_Op, and the form of the
   --  Boolean expression depends on the operator involved.

   procedure Expand_Comparison_Operator (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (N);
      Op1  : Node_Id             := Left_Opnd (N);
      Op2  : Node_Id             := Right_Opnd (N);
      Typ1 : constant Node_Id    := Base_Type (Etype (Op1));
      Typ2 : constant Node_Id    := Base_Type (Etype (Op2));
      Result    : Node_Id;
      Expr      : Node_Id;
      Func_Body : Node_Id;
      Func_Name : Entity_Id;

   --   ??? can't Op1, Op2 be constants, aren't assignments to Op1, Op2
   --   below redundant, if not why not? RBKD

   begin
      if Is_Array_Type (Typ1) then

         --  For <= the Boolean expression is
         --    greater__nn (op2, op1, true)

         if Chars (N) = Name_Op_Le then
            Func_Body := Make_Array_Comparison_Op (Typ1, Loc, True);
            Op1  := Right_Opnd (N);
            Op2  := Left_Opnd  (N);

         --  For < the Boolean expression is
         --    greater__nn (op2, op1)

         elsif Chars (N) = Name_Op_Lt then
            Func_Body := Make_Array_Comparison_Op (Typ1, Loc, False);
            Op1  := Right_Opnd (N);
            Op2  := Left_Opnd  (N);

         --  For >= the Boolean expression is
         --    op1 = op2 or else greater__nn (op1, op2)

         elsif Chars (N) = Name_Op_Ge then
            Func_Body := Make_Array_Comparison_Op (Typ1, Loc, True);

         --  For > the Boolean expression is
         --    greater__nn (op1, op2)

         elsif Chars (N) = Name_Op_Gt then
            Func_Body := Make_Array_Comparison_Op (Typ1, Loc, False);
         else
            return;
         end if;

         Func_Name := Defining_Unit_Name (Specification (Func_Body));
         Expr :=
           Make_Function_Call (Loc,
             Name => New_Reference_To (Func_Name, Loc),
             Parameter_Associations => New_List (Op1, Op2));

         Result :=
           Make_Expression_Actions (Loc,
             Actions => New_List (Func_Body),
             Expression => Expr);

         Rewrite_Substitute_Tree (N, Result);
         Analyze (N);
         Resolve (N, Standard_Boolean);

      elsif Is_Fixed_Point_Type (Typ1) then
         Expand_Literal_To_Fixed (Op2, Typ1);

      elsif Is_Fixed_Point_Type (Typ2) then
         Expand_Literal_To_Fixed (Op1, Typ2);
      end if;

   end Expand_Comparison_Operator;

   -------------------------------
   -- Expand_Composite_Equality --
   -------------------------------

   --  This function is only called for comparing internal fields of composite
   --  types when these fields are themselves composites. This is a special
   --  case because it is not possible to respect normal Ada visibility rules.

   function Expand_Composite_Equality
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Lhs  : Node_Id;
      Rhs  : Node_Id)
      return Node_Id
   is
      Full_Type : Entity_Id;
      Prim      : Elmt_Id;
   begin
      if Is_Private_Type (Typ) then
         Full_Type := Underlying_Type (Typ);
      else
         Full_Type := Typ;
      end if;

      Full_Type := Base_Type (Full_Type);

      if Is_Array_Type (Full_Type) then

         if Is_Scalar_Type (Component_Type (Full_Type)) then
            return Make_Op_Eq (Loc, Left_Opnd  => Lhs, Right_Opnd => Rhs);
         else
            return Expand_Array_Equality (Loc, Full_Type, Lhs, Rhs);
         end if;

      elsif Is_Tagged_Type (Full_Type) then

         --  Call the primitive operation "=" of this type

         if Is_Class_Wide_Type (Full_Type) then
            Full_Type := Etype (Full_Type);
         end if;

         Prim := First_Elmt (Primitive_Operations (Full_Type));

         while Chars (Node (Prim)) /= Name_Op_Eq loop
            Prim := Next_Elmt (Prim);
            pragma Assert (Present (Prim));
         end loop;

         return
           Make_Function_Call (Loc,
             Name => New_Reference_To (Node (Prim), Loc),
             Parameter_Associations => New_List (Lhs, Rhs));

      elsif Is_Record_Type (Full_Type) then
         return Expand_Record_Equality (Loc, Full_Type, Lhs, Rhs);
      else
         --  It can be a simple record or the full view of a scalar private

         return Make_Op_Eq (Loc, Left_Opnd => Lhs, Right_Opnd => Rhs);
      end if;
   end Expand_Composite_Equality;

   --------------------------
   -- Expand_Concatenation --
   --------------------------

   --  We construct the following expression actions node, where Atyp is
   --  the base type of the array involved and Ityp is the index type
   --  of this array:

   --    [function Cnn (S1 : Atyp; S2 : Atyp; .. Sn : Atyp) return Atyp is
   --        L : constant Ityp := S1'Length + S2'Length + ... Sn'Length;
   --        R : Atyp (S1'First .. S1'First + L - 1)
   --        P : Ityp := S1'First;
   --
   --     begin
   --        R (P .. P + S1'Length - 1) := S1;
   --        P := P + S1'Length;
   --        R (P .. P + S2'Length - 1) := S2;
   --        P := P + S2'Length;
   --        ...
   --        R (P .. P + Sn'Length - 1) := Sn;
   --        P := P + Sn'Length;
   --        return R;
   --     end Cnn;
   --
   --     Cnn (operand1, operand2, ... operandn)]

   --  Note: the low bound is not quite right, to be fixed later ???

   procedure Expand_Concatenation (Node : Node_Id; Ops : List_Id) is
      Loc   : constant Source_Ptr := Sloc (Node);
      Atyp  : constant Entity_Id  := Base_Type (Etype (Node));
      Ityp  : constant Entity_Id  := Etype (First_Index (Atyp));
      N     : constant Nat        := List_Length (Ops);

      Op    : Node_Id;
      Pspec : List_Id;
      Lexpr : Node_Id;
      Slist : List_Id;
      Alist : List_Id;
      Decls : List_Id;
      Func  : Node_Id;

      function L return Node_Id is
      begin
         return Make_Identifier (Loc, Name_uL);
      end L;

      function Nam (J : Nat) return Name_Id is
      begin
         return New_External_Name ('S', J);
      end Nam;

      function One return Node_Id is
      begin
         return Make_Integer_Literal (Loc, Uint_1);
      end One;

      function P return Node_Id is
      begin
         return Make_Identifier (Loc, Name_uP);
      end P;

      function R return Node_Id is
      begin
         return Make_Identifier (Loc, Name_uR);
      end R;

      function S1first return Node_Id is
      begin
         return
           Make_Attribute_Reference (Loc,
             Prefix => Make_Identifier (Loc, Nam (1)),
             Attribute_Name => Name_First);
      end S1first;

      function Slength (J : Nat) return Node_Id is
      begin
         return
           Make_Attribute_Reference (Loc,
             Prefix => Make_Identifier (Loc, Nam (J)),
             Attribute_Name => Name_Length);
      end Slength;

   --  Start of processing for Expand_Concatenation

   begin
      --  Construct parameter specification list

      Pspec := New_List;

      for J in 1 .. N loop
         Append_To (Pspec,
           Make_Parameter_Specification (Loc,
             Defining_Identifier => Make_Defining_Identifier (Loc, Nam (J)),
             Parameter_Type => New_Reference_To (Atyp, Loc)));
      end loop;

      --  Construct expression for total length of result

      Lexpr := Slength (1);

      for J in 2 .. N loop
         Lexpr := Make_Op_Add (Loc, Lexpr, Slength (J));
      end loop;

      --  Construct list of statements

      Slist := New_List;

      for J in 1 .. N loop
         Append_To (Slist,
           Make_Assignment_Statement (Loc,
             Name =>
               Make_Slice (Loc,
                 Prefix => R,
                 Discrete_Range =>
                   Make_Range (Loc,
                     Low_Bound => P,
                     High_Bound =>
                       Make_Op_Subtract (Loc,
                         Left_Opnd  => Make_Op_Add (Loc, P, Slength (J)),
                         Right_Opnd => One))),
             Expression => Make_Identifier (Loc, Nam (J))));

         Append_To (Slist,
           Make_Assignment_Statement (Loc,
             Name       => P,
             Expression => Make_Op_Add (Loc, P, Slength (J))));
      end loop;

      Append_To (Slist, Make_Return_Statement (Loc, Expression => R));

      --  Construct list of arguments for the call

      Alist := New_List;
      Op := First (Ops);

      for J in 1 .. N loop
         Append_To (Alist, New_Copy (Op));
         Op := Next (Op);
      end loop;

      --  Construct the declarations for the function

      Decls := New_List (
        Make_Object_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_uL),
          Object_Definition   => New_Reference_To (Ityp, Loc),
          Constant_Present    => True,
          Expression          => Lexpr),

        Make_Object_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_uR),

          Object_Definition =>
            Make_Subtype_Indication (Loc,
              Subtype_Mark => New_Reference_To (Atyp, Loc),
              Constraint =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints => New_List (
                    Make_Range (Loc,
                      Low_Bound  => S1first,
                      High_Bound =>
                        Make_Op_Subtract (Loc,
                          Left_Opnd => Make_Op_Add (Loc, S1first, L),
                          Right_Opnd => One)))))),

        Make_Object_Declaration (Loc,
          Defining_Identifier => Make_Defining_Identifier (Loc, Name_uP),
          Object_Definition   => New_Reference_To (Ityp, Loc),
          Expression          => S1first));

      --  Now construct the expression actions node and do the replace

      Func := Make_Defining_Identifier (Loc, New_Internal_Name ('C'));

      Rewrite_Substitute_Tree (Node,
        Make_Expression_Actions (Loc,
          Actions => New_List (
            Make_Subprogram_Body (Loc,
              Specification =>
                Make_Function_Specification (Loc,
                  Defining_Unit_Name       => Func,
                  Parameter_Specifications => Pspec,
                  Subtype_Mark => New_Reference_To (Atyp, Loc)),
              Declarations => Decls,
              Handled_Statement_Sequence =>
                Make_Handled_Sequence_Of_Statements (Loc, Slist))),
          Expression =>
            Make_Function_Call (Loc, New_Reference_To (Func, Loc), Alist)));

      Analyze (Node);
      Resolve (Node, Atyp);
      Set_Is_Inlined (Func);
   end Expand_Concatenation;

   ------------------------------------
   -- Expand_Fixed_To_Fixed_Division --
   ------------------------------------

   --  This procedure expands the division of fixed point types using
   --  Hilfinger's algorithm. The code expanded depends on some values that
   --  can be computed at compile time.

   --  The purpose of the algorithm is to find an integer value S that
   --  satisfies the following formula:

   --    |S - Beta * x / y| < 1    where  x, y  : the operand values
   --                                     Beta  : the scaling factor

   --  At compile time we compute the following values:

   --    Beta : the scaling factor (depends only on the smalls' values)
   --    Nmax : the maximum size (in bits) of the result and operands

   --  The types used are:

   --    Res_Type  : fixed point type of the result.
   --    Res_Int   : corresponding integer type of Res_Type.
   --    Dmax_Type : integer type with a size twice bigger Nmax.

   procedure Expand_Fixed_To_Fixed_Division (N : Node_Id) is

      Loc        : constant Source_Ptr := Sloc (N);
      Res_Type   : constant Entity_Id  := Etype (N);
      Res_Int    : constant Entity_Id  :=
                     Corresponding_Integer_Type (Res_Type);
      Expr       : constant Node_Id    := Expression (N);
      Left_Type  : constant Entity_Id  := Etype (Left_Opnd (Expr));
      Right_Type : constant Entity_Id  := Etype (Right_Opnd (Expr));
      Left_Int   : Node_Id;
      Right_Int  : Node_Id;
      Nmax       : constant Uint       :=
                     UI_Difference (
                       UI_Max (
                         Esize (Res_Type),
                         UI_Max (Esize (Left_Type), Esize (Right_Type))),
                       Uint_1);
      Dmax_Type  : constant Entity_Id  :=
                     Get_Double_Sized_Type (UI_Sum (Nmax, Uint_1));
      Beta       : constant Ureal :=
                    UR_Quotient (
                      Small_Value (Left_Type),
                        UR_Product (
                          Small_Value (Right_Type),
                          Small_Value (Etype (N))));
      Epsilon    : Ureal;
      B          : Ureal;
      Ksi        : Ureal;
      Tamp       : Ureal;
      M          : Uint;
      K          : Uint;
      X_Expr     : Node_Id;
      Y_Expr     : Node_Id;
      B_Expr     : Node_Id;
      M_Expr     : Node_Id;
      EN_Expr    : Node_Id;
      Rent       : RE_Id;
      Params     : List_Id;

   --  Start of processing for Expand_Fixed_Division;

   begin
      pragma Assert (Nkind (N) = N_Type_Conversion);

      UR_Normal_Form (UR_Round (Beta, Nmax), Ksi, K);
      B := UR_Product (UR_Exponentiate (Ureal_2, Nmax), Ksi);
      M := UI_Difference (K, Nmax);
      Tamp := UR_Exponentiate (Ureal_2, UI_Negate (M));
      Epsilon :=
        UR_Difference (UR_Quotient (UR_Product (Tamp, B), Beta), Ureal_1);

      Left_Int :=
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark =>
            New_Reference_To (Corresponding_Integer_Type (Left_Type), Loc),
            Expression => Left_Opnd (Expr));
      Set_Evaluate_Once (Left_Int);

      Right_Int :=
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark =>
            New_Reference_To (Corresponding_Integer_Type (Right_Type), Loc),
            Expression => Right_Opnd (Expr));
      Set_Evaluate_Once (Right_Int);

      if UI_Le (M, UI_Product (Uint_Minus_2, Nmax)) then

         --  Expr =  Res_Type! (Res_Int (0));

         Replace_Substitute_Tree (N,
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Res_Type, Loc),
             Expression => Build_Qual_Expr (Uint_0, Res_Int, Loc)));
         Analyze (N);
         Resolve (N, Res_Type);
         return;
      end if;

      if Dmax_Type = Standard_Long_Long_Integer then
         Rent := RE_Ffd_LLI;

      elsif Dmax_Type = Standard_Long_Integer then
         Rent := RE_Ffd_LI;

      elsif Dmax_Type = Standard_Integer then
         Rent := RE_Ffd_I;

      elsif Dmax_Type = Standard_Short_Integer then
         Rent := RE_Ffd_SI;
      end if;

      X_Expr  :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Left_Int);

      Y_Expr  :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Right_Int);

      B_Expr  := Build_Qual_Expr (UR_To_Uint (B), Dmax_Type, Loc);
      M_Expr  := Build_Qual_Expr (M, Standard_Integer, Loc);

      if UR_Lt (Epsilon, Ureal_0) then
         EN_Expr := New_Occurrence_Of (Standard_True, Loc);
      else
         EN_Expr := New_Occurrence_Of (Standard_False, Loc);
      end if;

      Params := New_List (X_Expr, Y_Expr, B_Expr, M_Expr);
      Append_To (Params, EN_Expr);

      Replace_Substitute_Tree (N,
        Make_Conditional_Expression (Loc,
          New_List (

      --  if x = 0

            Make_Op_Eq (Loc,
              Left_Opnd => New_Copy (Left_Int),
              Right_Opnd => Make_Integer_Literal (Loc, Uint_0)),

      --  then Expr =  Res_Type! (Res_Int (0))

            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Res_Type, Loc),
              Expression => Build_Qual_Expr (Uint_0, Res_Int, Loc)),

      --  else

            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Res_Type, Loc),
              Expression =>
                Make_Type_Conversion (Loc,
                  Subtype_Mark => New_Reference_To (Res_Int, Loc),
                  Expression =>
                    Make_Function_Call (Loc,
                      Name => New_Reference_To (RTE (Rent), Loc),
                      Parameter_Associations => Params))))));
      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_To_Fixed_Division;

   ------------------------------------
   -- Expand_Fixed_Integer_Division  --
   ------------------------------------

   procedure Expand_Fixed_Integer_Division (N : Node_Id) is
      Loc          : constant Source_Ptr := Sloc (N);
      Res_Type     : constant Entity_Id  := Etype (N);
      Res_Int_Type : constant Entity_Id
                       := Corresponding_Integer_Type (Res_Type);
      Left_Int     : Node_Id;
      Right_Int    : Node_Id;
      Right        : constant Node_Id := Relocate_Node (Right_Opnd (N));
      S            : Uint;

   begin
      Left_Int :=
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Res_Int_Type, Loc),
          Expression => Left_Opnd (N));
      Set_Evaluate_Once (Left_Int);

      if Res_Int_Type = Standard_Integer then
         Right_Int := Right;
      else
         Right_Int :=
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Res_Int_Type, Loc),
             Expression => Right);
         Set_Evaluate_Once (Right_Int);
      end if;

      if Standard_Integer_Size > UI_To_Int (Esize (Res_Int_Type)) then
         S := UI_Exponentiate (Uint_2, Esize (Res_Int_Type));

         Replace_Substitute_Tree (N,
           Make_Conditional_Expression (Loc,
             New_List (

         --  if Int < -2**S or Int > 2**S - 1

               Make_Op_Or_Else (Loc,

                 Left_Opnd =>
                   Make_Op_Lt (Loc,
                     Left_Opnd => New_Copy (Right),
                     Right_Opnd =>
                       Make_Qualified_Expression (Loc,
                         Subtype_Mark =>
                           New_Reference_To (Standard_Integer, Loc),
                         Expression =>
                           Make_Integer_Literal (Loc, UI_Negate (S)))),

                 Right_Opnd =>
                   Make_Op_Gt (Loc,
                     Left_Opnd => New_Copy (Right),
                     Right_Opnd =>
                       Make_Qualified_Expression (Loc,
                         Subtype_Mark =>
                           New_Reference_To (Standard_Integer, Loc),
                         Expression =>
                           Make_Integer_Literal (Loc, UI_Sum (S, Uint_1))))),

         --  then Expr = Res_Type! (Res_Int_Type (0))

               Make_Unchecked_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (Res_Type, Loc),
                 Expression =>
                   Make_Qualified_Expression (Loc,
                     Subtype_Mark => New_Reference_To (Res_Int_Type, Loc),
                     Expression => Make_Integer_Literal (Loc, Uint_0))),

         --  else Expr = Res_Type! (Left_Int / Right_Int);

               Make_Unchecked_Type_Conversion (Loc,
                 Subtype_Mark => New_Reference_To (Res_Type, Loc),
                 Expression =>
                   Make_Op_Divide (Loc,
                     Left_Opnd => Left_Int,
                     Right_Opnd => Right_Int)))));

      else
         Replace_Substitute_Tree (N,
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Res_Type, Loc),
             Expression =>
               Make_Op_Divide (Loc,
                 Left_Opnd => Left_Int,
                 Right_Opnd => Right_Int)));
      end if;

      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_Integer_Division;

   -----------------------------------------
   -- Expand_Fixed_Integer_Multiplication --
   -----------------------------------------

   procedure Expand_Fixed_Integer_Multiplication (N, Fix, Int : Node_Id) is

      Loc          : constant Source_Ptr := Sloc (N);
      Res_Type     : constant Entity_Id  := Etype (N);
      Res_Int_Type : constant Entity_Id  :=
                                  Corresponding_Integer_Type (Res_Type);
      Left_Int     : Node_Id;
      Right_Int    : Node_Id;

   begin

      Left_Int :=
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Res_Int_Type, Loc),
          Expression   => Fix);
      Set_Evaluate_Once (Left_Int);

      if Res_Int_Type = Standard_Integer then
         Right_Int := Relocate_Node (Int);
      else
         Right_Int :=
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Res_Int_Type, Loc),
             Expression   => Int);
         Set_Evaluate_Once (Right_Int);
      end if;

      Replace_Substitute_Tree (N,
        Make_Conditional_Expression (Loc,
          New_List (

      --  if Fix = 0;

            Make_Op_Eq (Loc,
              Left_Opnd => Left_Int,
              Right_Opnd => Make_Integer_Literal (Loc, Intval => Uint_0)),

      --  then Expr =  Res_Type! (Res_Int (0))

            Make_Unchecked_Type_Conversion (Loc,
               Subtype_Mark => New_Reference_To (Res_Type, Loc),
               Expression =>
                 Make_Qualified_Expression (Loc,
                   Subtype_Mark => New_Reference_To (Res_Int_Type, Loc),
                   Expression =>
                     Make_Integer_Literal (Loc, Intval => Uint_0))),

      --  else Expr = Res_Type! (Left_Int * Right_Int);

            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Res_Type, Loc),
              Expression =>
                Make_Op_Multiply (Loc,
                  Left_Opnd => Left_Int,
                  Right_Opnd => Right_Int)))));

      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_Integer_Multiplication;

   ------------------------------------------
   -- Expand_Fixed_To_Fixed_Multiplication --
   ------------------------------------------

   --  This procedure expands the multiplication of fixed point types using
   --  Hilfinger's algorithm. The code expanded depends on some values that
   --  can be computed at compile time.

   --  The purpose of the algorithm is to find an integer value R that
   --  satisfies the following formula:

   --    |R - Alpha * x * y| < 1    where  x, y  : the operand values
   --                                      Alpha : the scaling factor

   --  At compile time we compute the following values:

   --    Alpha     the scaling factor (depends only on the smalls' values)
   --    Nmax      the maximum size (in bits) of the result and operands - 1
   --    K, Ksi    with  (2 ** K) * Ksi normal form of round (1 / Alpha, Nres)
   --    A, M      with  A = (2 ** N) * Ksi  and  M = Nres - K
   --    Epsilon   with  Alpha * (1 + Epsilon) = (2 ** M) / A

   --  The types used are:

   --    Res_Type  : fixed point type of the result.
   --    Res_Int   : corresponding integer type of Res_Type.
   --    Dmax_Type : integer type with a size twice bigger Nmax + 1.

   procedure Expand_Fixed_To_Fixed_Multiplication (N : Node_Id) is
      Loc        : constant Source_Ptr := Sloc (N);
      Res_Type   : constant Entity_Id  := Etype (N);
      Res_Int    : constant Entity_Id  :=
                     Corresponding_Integer_Type (Res_Type);
      Expr       : constant Entity_Id  := Expression (N);
      Left_Type  : Entity_Id;
      Right_Type : Entity_Id;
      Left_Int   : Node_Id;
      Right_Int  : Node_Id;
      Alpha_1    : Ureal;
      Epsilon    : Ureal;
      A          : Ureal;
      Tamp       : Ureal;
      Nmax       : Uint;
      Dmax_Type  : Entity_Id;
      M          : Uint;
      X_Expr     : Node_Id;
      Y_Expr     : Node_Id;
      A_Expr     : Node_Id;
      M_Expr     : Node_Id;
      B1_Expr    : Node_Id;
      B2_Expr    : Node_Id;
      EN_Expr    : Node_Id;
      Rent       : RE_Id;
      Params     : List_Id;
      Mult       : Boolean := True;
      Int_Conv   : Boolean := False;

      procedure Initialize_Values;
      --  Initialize the some variables depending on wether we are doing just
      --  a simple conversion, or the conversion of a multiplication.

      procedure Initialize_Values is
         Ksi : Ureal;
         K   : Uint;

      begin
         if Mult then
            Left_Type  := Etype (Left_Opnd (Expr));
            Right_Type := Etype (Right_Opnd (Expr));
            Alpha_1    := UR_Quotient (
                            Small_Value (Res_Type),
                              UR_Product (
                                Small_Value (Left_Type),
                                Small_Value (Right_Type)));
            Nmax       := UI_Difference (
                            UI_Max (
                              Esize (Res_Type),
                              UI_Max (Esize (Left_Type), Esize (Right_Type))),
                            Uint_1);

         elsif not Int_Conv then
            Left_Type := Etype (Expr);
            Alpha_1   := UR_Quotient (
                           Small_Value (Res_Type), Small_Value (Left_Type));
            Nmax      := UI_Difference (
                           UI_Max (Esize (Res_Type), Esize (Left_Type)),
                           Uint_1);

         else
            Alpha_1   := Small_Value (Res_Type);
            Nmax      := UI_Difference (
                           UI_Max (Esize (Res_Type), Esize (Etype (Expr))),
                           Uint_1);
         end if;

         Dmax_Type   := Get_Double_Sized_Type (UI_Sum (Nmax, Uint_1));
         UR_Normal_Form (UR_Round (Alpha_1, Nmax), Ksi, K);
         A := UR_Product (UR_Exponentiate (Ureal_2, Nmax), Ksi);
         M := UI_Difference (Nmax, K);
         Tamp := UR_Exponentiate (Ureal_2, M);
         Epsilon := UR_Difference
                      (UR_Quotient (UR_Product (Tamp, Alpha_1), A),
                       Ureal_1);

      end Initialize_Values;


   --  Begin of processing for Expand_Fixed_To_Fixed_Multiplication

   begin
      pragma Assert (Nkind (N) = N_Type_Conversion);

      if Nkind (Expr) /= N_Op_Multiply then
         Mult := False;
      end if;

      if Is_Integer_Type (Etype (Expr)) then
         Int_Conv := True;
      end if;

      Initialize_Values;

      if Mult then
         Left_Int :=
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (
                 Corresponding_Integer_Type (Left_Type), Loc),
             Expression => Left_Opnd (Expr));
         Set_Evaluate_Once (Left_Int);

         Right_Int :=
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (
                 Corresponding_Integer_Type (Right_Type), Loc),
             Expression => Right_Opnd (Expr));
         Set_Evaluate_Once (Right_Int);

      elsif not Int_Conv then
         Left_Int :=
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (
                 Corresponding_Integer_Type (Left_Type), Loc),
             Expression => Expr);
         Set_Evaluate_Once (Left_Int);

         Right_Int := Build_Qual_Expr (Uint_1, Standard_Integer, Loc);

      else
         Left_Int  :=
            Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Base_Type (Etype (Expr)), Loc),
             Expression => Expr);
         Set_Evaluate_Once (Left_Int);

         Right_Int := Build_Qual_Expr (Uint_1, Standard_Integer, Loc);
      end if;


      if UI_Lt (M, UI_Difference (UI_Negate (Nmax), Uint_1)) then

         Replace_Substitute_Tree (N,
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Res_Type, Loc),
             Expression => Build_Qual_Expr (Uint_0, Res_Int, Loc)));
         Analyze (N);
         Resolve (N, Res_Type);
         return;
      end if;

      if Dmax_Type = Standard_Long_Long_Integer then
         Rent := RE_Ffm_LLI;

      elsif Dmax_Type = Standard_Long_Integer then
         Rent := RE_Ffm_LI;

      elsif Dmax_Type = Standard_Integer then
         Rent := RE_Ffm_I;

      elsif Dmax_Type = Standard_Short_Integer then
         Rent := RE_Ffm_SI;
      end if;

      X_Expr  :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Left_Int);

      Y_Expr  :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Right_Int);

      A_Expr  := Build_Qual_Expr (UR_To_Uint (A), Dmax_Type, Loc);
      M_Expr  := Build_Qual_Expr (M, Standard_Integer, Loc);

      if UR_Lt (Epsilon, Ureal_0) then
         B1_Expr := New_Occurrence_Of (Standard_True, Loc);
         EN_Expr := New_Occurrence_Of (Standard_True, Loc);
      else
         B1_Expr := New_Occurrence_Of (Standard_False, Loc);
         EN_Expr := New_Occurrence_Of (Standard_False, Loc);
      end if;

      if UI_Lt (M, Uint_0)
        and then UR_Ge (
                   UR_Abs (Epsilon),
                   UR_Product (
                     UR_Exponentiate (Ureal_2, UI_Negate (Nmax)),
                     UR_Quotient (
                       UR_Sum (UR_Difference (A, Ureal_1), Tamp),
                       A)))
      then
         B2_Expr := New_Occurrence_Of (Standard_True, Loc);
      else
         B2_Expr := New_Occurrence_Of (Standard_False, Loc);
      end if;

      Params := New_List (X_Expr, Y_Expr, A_Expr, M_Expr);
      Append_To (Params, B1_Expr);
      Append_To (Params, B2_Expr);
      Append_To (Params, EN_Expr);

      Replace_Substitute_Tree (N,
        Make_Conditional_Expression (Loc,
          New_List (

      --  if x = 0 or y = 0

            Make_Op_Or_Else (Loc,
              Left_Opnd =>
                Make_Op_Eq (Loc,
                  Left_Opnd => New_Copy (Left_Int),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_0)),

              Right_Opnd =>
                Make_Op_Eq (Loc,
                  Left_Opnd => New_Copy (Right_Int),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_0))),


      --  then Expr =  Res_Type! (Res_Int (0))

            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Res_Type, Loc),
              Expression => Build_Qual_Expr (Uint_0, Res_Int, Loc)),

      --  else

            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Res_Type, Loc),
              Expression =>
                Make_Type_Conversion (Loc,
                  Subtype_Mark => New_Reference_To (Res_Int, Loc),
                  Expression =>
                    Make_Function_Call (Loc,
                      Name => New_Reference_To (RTE (Rent), Loc),
                      Parameter_Associations => Params))))));
      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_To_Fixed_Multiplication;

   --------------------------------------
   -- Expand_Fixed_To_Float_Conversion --
   --------------------------------------

   --  This is a simple version of this procedure. The accuracy requirements of
   --  the RM are not implemented yet.

   procedure Expand_Fixed_To_Float_Conversion (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Expr        : constant Node_Id    := Expression (N);
      Fix_Type    : constant Entity_Id  := Etype (Expr);
      Res_Type    : constant Entity_Id  := Etype (N);
      Alpha       : constant Ureal      := Small_Value (Fix_Type);
      Left_Float  : Node_Id;
      Alpha_Float : Node_Id;

   begin
      Left_Float :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (Corresponding_Integer_Type (Fix_Type), Loc),
              Expression => Expr));

      Set_Evaluate_Once (Left_Float);

      Alpha_Float :=
        Make_Qualified_Expression (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression => Make_Real_Literal (Loc, Realval => Alpha));

      Replace_Substitute_Tree (N,
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Res_Type, Loc),
          Expression =>
            Make_Op_Multiply (Loc,
              Left_Opnd  => Left_Float,
              Right_Opnd => Alpha_Float)));

      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_To_Float_Conversion;

   ------------------------------------
   -- Expand_Fixed_To_Float_Division --
   ------------------------------------

   --  This is a simple version of this procedure. The accuracy requirements of
   --  the RM are not implemented yet.

   procedure Expand_Fixed_To_Float_Division (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Res_Type    : constant Entity_Id  := Etype (N);
      Expr        : constant Node_Id    := Expression (N);
      Left_Type   : constant Entity_Id  := Etype (Left_Opnd (Expr));
      Right_Type  : constant Entity_Id  := Etype (Right_Opnd (Expr));
      Alpha       : constant Ureal      :=
                       UR_Quotient (
                         Small_Value (Left_Type),
                         Small_Value (Right_Type));
      Left_Float  : Node_Id;
      Right_Float : Node_Id;
      Alpha_Float : Node_Id;

   begin
      Left_Float :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (Corresponding_Integer_Type (Left_Type), Loc),
              Expression => Left_Opnd (Expr)));

      Set_Evaluate_Once (Left_Float);

      Right_Float :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (
                  Corresponding_Integer_Type (Right_Type), Loc),
              Expression => Right_Opnd (Expr)));

      Set_Evaluate_Once (Right_Float);

      Alpha_Float :=
        Make_Qualified_Expression (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression => Make_Real_Literal (Loc, Realval => Alpha));

      Replace_Substitute_Tree (N,
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Res_Type, Loc),
          Expression =>
            Make_Op_Divide (Loc,
              Left_Opnd  =>
                Make_Op_Multiply (Loc,
                  Left_Opnd  => Alpha_Float,
                  Right_Opnd => Left_Float),
              Right_Opnd => Right_Float)));

      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_To_Float_Division;

   ------------------------------------------
   -- Expand_Fixed_To_Float_Multiplication --
   ------------------------------------------

   --  This is a simple version of this procedure. The accuracy requirements of
   --  the RM are not implemented yet.

   procedure Expand_Fixed_To_Float_Multiplication (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Res_Type    : constant Entity_Id  := Etype (N);
      Expr        : constant Node_Id    := Expression (N);
      Left_Type   : constant Entity_Id  := Etype (Left_Opnd (Expr));
      Right_Type  : constant Entity_Id  := Etype (Right_Opnd (Expr));
      Alpha       : constant Ureal      :=
                       UR_Product (
                         Small_Value (Left_Type),
                         Small_Value (Right_Type));
      Left_Float  : Node_Id;
      Right_Float : Node_Id;
      Alpha_Float : Node_Id;

   begin
      Left_Float :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (Corresponding_Integer_Type (Left_Type), Loc),
              Expression => Left_Opnd (Expr)));

      Set_Evaluate_Once (Left_Float);

      Right_Float :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression =>
            Make_Unchecked_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (
                  Corresponding_Integer_Type (Right_Type), Loc),
              Expression => Right_Opnd (Expr)));

      Set_Evaluate_Once (Right_Float);

      Alpha_Float :=
        Make_Qualified_Expression (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression => Make_Real_Literal (Loc, Realval => Alpha));

      Replace_Substitute_Tree (N,
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Res_Type, Loc),
          Expression =>
            Make_Op_Multiply (Loc,
              Left_Opnd  => Alpha_Float,
              Right_Opnd =>
                Make_Op_Multiply (Loc,
                  Left_Opnd  => Left_Float,
                  Right_Opnd => Right_Float))));

      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_To_Float_Multiplication;

   --------------------------------------
   -- Expand_Fixed_To_Integer_Division --
   --------------------------------------

   procedure Expand_Fixed_To_Integer_Division (N : Node_Id) is

      function Floor (Real : Ureal) return Uint;
      --  Returns the truncation towards minus infinity of the real value.

      procedure Initialize_Values;
      --  Computes the values of the major variables of the expansion process.

      Loc        : constant Source_Ptr := Sloc (N);
      Res_Type   : constant Entity_Id  := Etype (N);
      Expr       : constant Node_Id    := Expression (N);
      Left_Type  : constant Entity_Id  := Etype (Left_Opnd (Expr));
      Right_Type : constant Entity_Id  := Etype (Right_Opnd (Expr));
      Left_Int   : Node_Id;
      Right_Int  : Node_Id;
      Nmax       : constant Uint       :=
                     UI_Difference (
                       UI_Max (
                         Esize (Res_Type),
                         UI_Max (Esize (Left_Type), Esize (Right_Type))),
                       Uint_1);
      Dmax_Type  : constant Entity_Id  :=
                     Get_Double_Sized_Type (UI_Sum (Nmax, Uint_1));
      Beta       : constant Ureal      :=
                    UR_Quotient (
                      Small_Value (Left_Type), Small_Value (Right_Type));
      Epsilon    : Ureal;
      Tamp       : constant Uint       := UI_Exponentiate (Uint_2, Nmax);
      M          : Uint                := Uint_0;
      Beta1      : constant Uint       :=
                     Floor (UR_Quotient (Beta, UR_From_Uint (Tamp)));
      Beta2      : constant Uint       := UI_Mod (Floor (Beta), Tamp);
      A          : Uint;
      B          : Uint;
      X_Expr     : Node_Id;
      Y_Expr     : Node_Id;
      A_Expr     : Node_Id;
      B_Expr     : Node_Id;
      D_Expr     : Node_Id;
      Beta1_Expr : Node_Id;
      Beta2_Expr : Node_Id;
      M_Expr     : Node_Id;
      N_Expr     : Node_Id;
      Bool_Expr  : Node_Id;
      Rent       : RE_Id;
      Params     : List_Id;

      function Floor (Real : Ureal) return Uint is
      begin
         if UR_Is_Negative (Real) then
            return UI_Difference (UR_Trunc (Real), Uint_1);
         else
            return UR_Trunc (Real);
         end if;
      end Floor;

      procedure Initialize_Values is
      begin
         --  Computation of A, B and M. The goal of this computation is to have

         --    Beta := 2**M * A / B

         --  where 0 < A, B < 2**Nmax (= Tamp)
         --        M >= 0 or (M < 0 and A < 2 * B)

         A := Norm_Num (Beta);
         B := Norm_Den (Beta);

         while UI_Ge (A, Tamp) loop
            if UI_Eq (UI_Rem (A, Uint_2), Uint_0) then
               A := UI_Quotient (A, Uint_2);
               M := UI_Sum (M, Uint_1);
            else
               Error_Msg_N (
                 "small values incompatible for integer precision", N);
            end if;
         end loop;

         while UI_Ge (B, Tamp) loop
            if UI_Eq (UI_Rem (B, Uint_2), Uint_0) then
               B := UI_Quotient (B, Uint_2);
               M := UI_Difference (M, Uint_1);
            else
               Error_Msg_N (
                 "small values incompatible for integer precision", N);
            end if;
         end loop;

         if UI_Lt (M, Uint_0) then
            while UI_Ge (A, UI_Product (B, Uint_2)) loop
               B := UI_Product (B, Uint_2);
               M := UI_Sum (M, Uint_1);
               exit when UI_Eq (M, Uint_0);
            end loop;
         end if;

      end Initialize_Values;


   --  Start of processing for Expand_Fixed_To_Integer_Division.

   begin
      if UR_Le (Beta, UR_Quotient (Ureal_Half, UR_From_Uint (Tamp))) then
         Replace_Substitute_Tree (N, Build_Qual_Expr (Uint_0, Res_Type, Loc));
         Analyze (N);
         Resolve (N, Res_Type);
         return;
      end if;

      Left_Int :=
      Make_Unchecked_Type_Conversion (Loc,
        Subtype_Mark =>
          New_Reference_To (
            Corresponding_Integer_Type (Left_Type), Loc),
        Expression => Left_Opnd (Expr));
      Set_Evaluate_Once (Left_Int);

      Right_Int :=
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark =>
            New_Reference_To (
              Corresponding_Integer_Type (Right_Type), Loc),
          Expression => Right_Opnd (Expr));
      Set_Evaluate_Once (Right_Int);

      Initialize_Values;

      X_Expr :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Left_Int);

      Y_Expr :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Right_Int);

      A_Expr     := Build_Qual_Expr (A, Dmax_Type, Loc);
      B_Expr     := Build_Qual_Expr (B, Dmax_Type, Loc);
      M_Expr     := Build_Qual_Expr (M, Standard_Integer, Loc);
      N_Expr     := Build_Qual_Expr (Nmax, Standard_Integer, Loc);

      Beta1_Expr :=
        Build_Qual_Expr (
          Floor (UR_Quotient (Beta, UR_From_Uint (Tamp))), Dmax_Type, Loc);

      Beta2_Expr :=
        Build_Qual_Expr (UI_Mod (Floor (Beta), Tamp), Dmax_Type, Loc);

      D_Expr     :=
        Build_Qual_Expr (UI_Mod (UI_Product (A, Tamp), B), Dmax_Type, Loc);

      if UR_Ge (Beta, UR_From_Uint (UI_Product (Tamp, Tamp))) then
         Bool_Expr := New_Occurrence_Of (Standard_True, Loc);
      else
         Bool_Expr := New_Occurrence_Of (Standard_False, Loc);
      end if;

      Params := New_List (X_Expr, Y_Expr, A_Expr, B_Expr);
      Append_To (Params, D_Expr);
      Append_To (Params, Beta1_Expr);
      Append_To (Params, Beta2_Expr);
      Append_To (Params, M_Expr);
      Append_To (Params, N_Expr);
      Append_To (Params, Bool_Expr);

      if Dmax_Type = Standard_Long_Long_Integer then
         Rent := RE_Fid_LLI;

      elsif Dmax_Type = Standard_Long_Integer then
         Rent := RE_Fid_LI;

      elsif Dmax_Type = Standard_Integer then
         Rent := RE_Fid_I;

      elsif Dmax_Type = Standard_Short_Integer then
         Rent := RE_Fid_SI;
      end if;

      Replace_Substitute_Tree (N,
        Make_Conditional_Expression (Loc,
          New_List (

      --  if x = 0

            Make_Op_Eq (Loc,
              Left_Opnd => New_Copy (Left_Int),
              Right_Opnd => Make_Integer_Literal (Loc, Uint_0)),

      --  then Expr =  Res_Type! (Res_Int (0))

            Build_Qual_Expr (Uint_0, Res_Type, Loc),

      --  else

            Make_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Res_Type, Loc),
              Expression =>
                Make_Function_Call (Loc,
                  Name => New_Reference_To (RTE (Rent), Loc),
                  Parameter_Associations => Params)))));

      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_To_Integer_Division;

   --------------------------------------------
   -- Expand_Fixed_To_Integer_Multiplication --
   --------------------------------------------

   procedure Expand_Fixed_To_Integer_Multiplication (N : Node_Id) is
      type Intermediary is (SI, I, LI, LLI);
      Inter_Type  : Intermediary;
      Loc         : constant Source_Ptr := Sloc (N);
      Res_Type    : constant Entity_Id  := Etype (N);
      Expr        : constant Node_Id    := Expression (N);
      Left_Type   : Entity_Id;
      Right_Type  : Entity_Id;
      Alpha       : Ureal;
      A           : Uint;
      B           : Uint;
      M           : Uint                := Uint_0;
      Nmax        : Uint;
      Dmax_Type   : Entity_Id;
      Tamp        : Uint;
      Left_Int    : Node_Id;
      Right_Int   : Node_Id;
      X_Expr      : Node_Id;
      Y_Expr      : Node_Id;
      A_Expr      : Node_Id;
      B_Expr      : Node_Id;
      M_Expr      : Node_Id;
      Rent        : RE_Id;
      Params      : List_Id;
      Mult        : Boolean := True;

      procedure Initialize_Values;
      --  Initialize the some variables depending on wether we are doing just
      --  a simple conversion, or the conversion of a multiplication.

      procedure Initialize_Values is
      begin

         if Mult then
            Left_Type  := Etype (Left_Opnd (Expr));
            Right_Type := Etype (Right_Opnd (Expr));
            Alpha      := UR_Product (
                            Small_Value (Left_Type), Small_Value (Right_Type));
            Nmax       := UI_Difference (
                            UI_Max (
                              Esize (Res_Type),
                              UI_Max (Esize (Left_Type), Esize (Right_Type))),
                            Uint_1);

         else
            Left_Type := Etype (Expr);
            Alpha     := Small_Value (Left_Type);
            Nmax      := UI_Difference (
                           UI_Max (Esize (Res_Type), Esize (Left_Type)),
                           Uint_1);
         end if;

         A           := Norm_Num (Alpha);
         B           := Norm_Den (Alpha);
         Dmax_Type   := Get_Double_Sized_Type (UI_Sum (Nmax, Uint_1));
         Tamp        := UI_Exponentiate (Uint_2, Nmax);

      end Initialize_Values;

   --  Start of processing for Expand_Fixed_To_Integer_Multiplication

   begin
      if Nkind (Expr) /= N_Op_Multiply then
         Mult := False;
      end if;

      Initialize_Values;

      while UI_Ge (A, Tamp) loop
         if UI_Eq (UI_Rem (A, Uint_2), Uint_0) then
            A := UI_Quotient (A, Uint_2);
            M := UI_Sum (M, Uint_1);
         else
            Error_Msg_N ("small values incompatible for integer precision", N);
         end if;
      end loop;

      while UI_Ge (B, Tamp) loop
         if UI_Eq (UI_Rem (B, Uint_2), Uint_0) then
            B := UI_Quotient (B, Uint_2);
            M := UI_Difference (M, Uint_1);
         else
            Error_Msg_N ("small values incompatible for integer precision", N);
         end if;
      end loop;

      if Mult then
         Left_Int :=
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (
                 Corresponding_Integer_Type (Left_Type), Loc),
             Expression => Left_Opnd (Expr));
         Set_Evaluate_Once (Left_Int);

         Right_Int :=
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (
                 Corresponding_Integer_Type (Right_Type), Loc),
             Expression => Right_Opnd (Expr));
         Set_Evaluate_Once (Right_Int);

      else
         Left_Int :=
           Make_Unchecked_Type_Conversion (Loc,
             Subtype_Mark =>
               New_Reference_To (
                 Corresponding_Integer_Type (Left_Type), Loc),
             Expression => Expr);
         Set_Evaluate_Once (Left_Int);

         Right_Int := Build_Qual_Expr (Uint_1, Standard_Integer, Loc);
      end if;

      if UR_Le (
           Alpha,
           UR_Exponentiate (
             Ureal_2,
             UI_Difference (UI_Product (Nmax, Uint_Minus_2), Uint_1)))
      then
         Replace_Substitute_Tree (N, Build_Qual_Expr (Uint_0, Res_Type, Loc));
         Analyze (N);
         Resolve (N, Res_Type);
         return;
      end if;

      if Dmax_Type = Standard_Short_Integer then
         Inter_Type := SI;

      elsif Dmax_Type = Standard_Integer then
         Inter_Type := I;

      elsif Dmax_Type = Standard_Long_Integer then
         Inter_Type := LI;

      elsif Dmax_Type = Standard_Long_Long_Integer then
         Inter_Type := LLI;
      end if;

      if UI_Ge (M, Uint_0) then

         if UI_Ge (UI_Product (UI_Exponentiate (Uint_2, M), A), B) then
            case Inter_Type is
               when  SI => Rent := RE_Fim1_SI;
               when   I => Rent := RE_Fim1_I;
               when  LI => Rent := RE_Fim1_LI;
               when LLI => Rent := RE_Fim1_LLI;
            end case;

         else
            A := UI_Product (UI_Exponentiate (Uint_2, M), A);
            M := Uint_0;
            case Inter_Type is
               when  SI => Rent := RE_Fim2_SI;
               when   I => Rent := RE_Fim2_I;
               when  LI => Rent := RE_Fim2_LI;
               when LLI => Rent := RE_Fim2_LLI;
            end case;
         end if;

      else
         while UI_Ge (A, UI_Product (B, Uint_2)) loop
            B := UI_Product (B, Uint_2);
            M := UI_Sum (M, Uint_1);
            exit when UI_Eq (M, Uint_0);
         end loop;

         M := UI_Negate (M);

         if UI_Eq (M, Uint_0) then
            case Inter_Type is
               when  SI => Rent := RE_Fim1_SI;
               when   I => Rent := RE_Fim1_I;
               when  LI => Rent := RE_Fim1_LI;
               when LLI => Rent := RE_Fim1_LLI;
            end case;

         elsif UI_Lt (A, B) then
            case Inter_Type is
               when  SI => Rent := RE_Fim3_SI;
               when   I => Rent := RE_Fim3_I;
               when  LI => Rent := RE_Fim3_LI;
               when LLI => Rent := RE_Fim3_LLI;
            end case;

         else
            case Inter_Type is
               when  SI => Rent := RE_Fim4_SI;
               when   I => Rent := RE_Fim4_I;
               when  LI => Rent := RE_Fim4_LI;
               when LLI => Rent := RE_Fim4_LLI;
            end case;
         end if;
      end if;

      X_Expr :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Left_Int);

      Y_Expr :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Dmax_Type, Loc),
          Expression => Right_Int);

      A_Expr := Build_Qual_Expr (A, Dmax_Type, Loc);
      B_Expr := Build_Qual_Expr (B, Dmax_Type, Loc);
      M_Expr := Build_Qual_Expr (M, Standard_Natural, Loc);

      Params := New_List (A_Expr, B_Expr, X_Expr, Y_Expr);
      Append_To (Params, M_Expr);

      Replace_Substitute_Tree (N,
        Make_Conditional_Expression (Loc,
          New_List (

      --  if x = 0 or y = 0

            Make_Op_Or_Else (Loc,
              Left_Opnd =>
                Make_Op_Eq (Loc,
                  Left_Opnd => New_Copy (Left_Int),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_0)),

              Right_Opnd =>
                Make_Op_Eq (Loc,
                  Left_Opnd => New_Copy (Right_Int),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_0))),


      --  then Expr =  Res_Type! (Res_Int (0))

            Build_Qual_Expr (Uint_0, Res_Type, Loc),

      --  else

            Make_Type_Conversion (Loc,
              Subtype_Mark => New_Reference_To (Res_Type, Loc),
              Expression =>
                Make_Function_Call (Loc,
                  Name => New_Reference_To (RTE (Rent), Loc),
                  Parameter_Associations => Params)))));


      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Fixed_To_Integer_Multiplication;

   --------------------------------------
   -- Expand_Float_To_Fixed_Conversion --
   --------------------------------------

   --  This is a simple version of this procedure. The accuracy requirements of
   --  the RM are not implemented yet.

   procedure Expand_Float_To_Fixed_Conversion (N : Node_Id) is
      Loc         : constant Source_Ptr := Sloc (N);
      Expr        : constant Node_Id    := Expression (N);
      Float_Type  : constant Entity_Id  := Etype (Expr);
      Res_Type    : constant Entity_Id  := Etype (N);
      Alpha       : constant Ureal      := Small_Value (Res_Type);
      Left_Float  : Node_Id;
      Alpha_Float : Node_Id;

   begin
      Left_Float :=
        Make_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression => Relocate_Node (Expr));

      Alpha_Float :=
        Make_Qualified_Expression (Loc,
          Subtype_Mark => New_Reference_To (Standard_Long_Long_Float, Loc),
          Expression => Make_Real_Literal (Loc, Realval => Alpha));

      Replace_Substitute_Tree (N,
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Res_Type, Loc),
          Expression =>
            Make_Type_Conversion (Loc,
              Subtype_Mark =>
                New_Reference_To (Corresponding_Integer_Type (Res_Type), Loc),
            Expression =>
              Make_Op_Divide (Loc,
                Left_Opnd  => Left_Float,
                Right_Opnd => Alpha_Float))));

      Analyze (N);
      Resolve (N, Res_Type);

   end Expand_Float_To_Fixed_Conversion;

   ---------------------------
   -- Get_Double_Sized_Type --
   ---------------------------

   function Get_Double_Sized_Type (Size_Type : Uint) return Entity_Id is
      Dsize : constant Uint := UI_Product (Size_Type, Uint_2);

   begin
      if UI_Eq (Dsize, Esize (Standard_Short_Integer)) then
         return Standard_Short_Integer;

      elsif UI_Eq (Dsize, Esize (Standard_Integer)) then
         return Standard_Integer;

      elsif UI_Eq (Dsize, Esize (Standard_Long_Integer)) then
         return Standard_Long_Integer;

      elsif UI_Eq (Dsize, Esize (Standard_Long_Long_Integer)) then
         return Standard_Long_Long_Integer;

      else
         return Huge_Integer;

      end if;
   end Get_Double_Sized_Type;

   -----------------------------
   -- Expand_Literal_To_Fixed --
   -----------------------------

   procedure Expand_Literal_To_Fixed (Lit : Node_Id; Typ : Entity_Id) is
      Loc       : constant Source_Ptr := Sloc (Lit);
      Int_Val   : Uint                := Uint_1;
      Small_Val : constant Ureal      := Small_Value (Typ);

   begin

      if Nkind (Lit) = N_Op_Minus
        and then Nkind (Right_Opnd (Lit)) = N_Real_Literal
      then
         Int_Val :=
           UI_Negate (
             UR_To_Uint (UR_Quotient (Realval (Right_Opnd (Lit)), Small_Val)));

      elsif Nkind (Lit) = N_Op_Plus
        and then Nkind (Right_Opnd (Lit)) = N_Real_Literal
      then
         Int_Val :=
           UR_To_Uint (UR_Quotient (Realval (Right_Opnd (Lit)), Small_Val));

      elsif Nkind (Lit) = N_Real_Literal then
         Int_Val := UR_To_Uint (UR_Quotient (Realval (Lit), Small_Val));

      else
         return;
      end if;

      Replace_Substitute_Tree (Lit,
        Make_Unchecked_Type_Conversion (Loc,
          Subtype_Mark => New_Reference_To (Typ, Loc),
          Expression =>
            Build_Qual_Expr (Int_Val, Corresponding_Integer_Type (Typ), Loc)));
      Analyze (Lit);
      Resolve (Lit, Typ);
   end Expand_Literal_To_Fixed;

   ------------------------
   -- Expand_N_Allocator --
   ------------------------

   --  If the allocator is for a type which requires initialization, and
   --  there is no initial value (i.e. the operand is a subtype indication
   --  rather than a qualifed expression), then we must generate a call to
   --  the initialization routine. This is done using an expression actions
   --  node:
   --
   --     [Pnnn : constant ptr_T := new (T); Init (Pnnn.all,...); Pnnn]
   --
   --  Here ptr_T is the pointer type for the allocator, and T is the
   --  subtype of the allocator. A special case arises if the designated
   --  type of the access type is a task or contains tasks. In this case
   --  the call to Init (Temp.all ...) is replaced by code that ensures
   --  that the tasks get activated (see Exp_Ch9.Build_Task_Allocate_Block
   --  for details). In addition, if the type T is a task T, then the first
   --  argument to Init must be converted to the task record type.

   procedure Expand_N_Allocator (N : Node_Id) is
      PtrT  : constant Entity_Id  := Etype (N);
      Loc   : constant Source_Ptr := Sloc (N);
      Temp  : Entity_Id;
      Node  : Node_Id;

   begin
      if Nkind (Expression (N)) = N_Qualified_Expression then
         declare
            T   : constant Entity_Id := Entity (Subtype_Mark (Expression (N)));
            Exp : constant Node_Id   := Expression (Expression (N));
            Act : constant List_Id   := New_List;

            Obj_Decl : Node_Id;

         begin
            if Is_Tagged_Type (T) then

               --    output:  [
               --              Temp : constant ptr_T := new T'(Expression);
               --   <no CW>    Temp._tag := T'tag;
               --   <CTRL>     Adjust (Finalizable (Temp.all));
               --   <CTRL>     Attach_To_Final_List (Finalizable (Temp.all));
               --              Temp]

               --  we analyze by hand the new internal allocator to avoid
               --  any recursion and inappropriate call to Initialize

               Node := Relocate_Node (N);
               Set_Analyzed (Node, True);

               Temp :=
                 Make_Defining_Identifier (Loc, New_Internal_Name ('P'));

               Obj_Decl :=
                 Make_Object_Declaration (Loc,
                   Defining_Identifier => Temp,
                   Constant_Present    => True,
                   Object_Definition   => New_Reference_To (PtrT, Loc),
                   Expression          => Node);

               Append_To (Act, Obj_Decl);

               --  for a class wide allocation generate the following code
               --    type Equiv_Record is record ... end record;
               --    subtype CW is <Class_Wide_Subytpe>;
               --    temp : PtrT := new T'(CW!(expr));

               if Is_Class_Wide_Type (T) then
                  declare
                     Indic : constant Node_Id := New_Reference_To (T, Loc);

                     CW_Subtype   : constant Entity_Id :=
                          Make_Defining_Identifier (Loc,
                            New_Internal_Name ('R'));

                     Subtype_Decl : constant Node_Id :=
                          Make_Subtype_Declaration (Loc,
                            Defining_Identifier => CW_Subtype,
                            Subtype_Indication => Indic);

                  begin
                     Insert_Before (Obj_Decl, Subtype_Decl);
                     Expand_Subtype_From_Expr (Subtype_Decl, T, Indic, Exp);

                     Set_Expression (Expression (Node),
                       Make_Unchecked_Type_Conversion (Loc,
                         Subtype_Mark => New_Reference_To (CW_Subtype, Loc),
                         Expression => Exp));
                     Set_Etype (Expression (Expression (Node)), CW_Subtype);
                  end;

               else
                  Append_To (Act,
                    Make_Assignment_Statement (Loc,
                      Name =>
                        Make_Selected_Component (Loc,
                          Prefix => New_Reference_To (Temp, Loc),
                          Selector_Name =>
                            New_Reference_To (Tag_Component (T), Loc)),
                      Expression =>
                        Make_Unchecked_Type_Conversion (Loc,
                          Subtype_Mark =>
                            New_Reference_To (RTE (RE_Tag), Loc),
                          Expression =>
                            New_Reference_To (Access_Disp_Table (T), Loc))));

                  --  The previous assignment has to be done in any case

                  Set_Assignment_OK (Name (Next (First (Act))));
               end if;

               if Is_Controlled (T) then
                  Append_To (Act,
                      Make_Adjust_Call (
                        Ref => Make_Explicit_Dereference (Loc,
                                 Prefix => New_Reference_To (Temp, Loc)),
                        Typ => T));

                  Append_To (Act,
                      Make_Attach_Ctrl_Object (
                        Ref => Make_Explicit_Dereference (Loc,
                                 Prefix => New_Reference_To (Temp, Loc)),
                        Scop => Scope (PtrT)));
               end if;

               Rewrite_Substitute_Tree (N,
                 Make_Expression_Actions (Loc,
                   Actions    => Act,
                   Expression => New_Reference_To (Temp, Loc)));
               Analyze (N);
            else
               null;
            end if;
         end;

      --  in this case, an initialization routine is needed

      else
         declare
            T     : constant Entity_Id  := Entity (Expression (N));
            Init  : constant Entity_Id  := Base_Init_Proc (T);
            Arg1  : Node_Id;
            Args  : List_Id;
            Discr : Elmt_Id;
            Eact  : Node_Id;

         begin
            --  Nothing to do if no initialization routine required

            if No (Init) then
               null;

            --  Else we have the case that definitely needs a call

            else
               Node := N;
               Temp :=
                 Make_Defining_Identifier (Loc, New_Internal_Name ('P'));

               --  Construct argument list for the initialization routine call

               Arg1 :=
                 Make_Explicit_Dereference (Loc,
                   Prefix => New_Reference_To (Temp, Loc));

               --  The initialization procedure expects a specific type.
               --  if the context is access to class wide, indicate that
               --  the object being allocated has the right specific type.

               if Is_Class_Wide_Type (Designated_Type (PtrT)) then
                  Arg1 :=
                    Make_Type_Conversion (Loc,
                      Subtype_Mark => New_Reference_To (T,  Loc),
                      Expression => Arg1);
               end if;

               --  If designated type is a task type, then the first argument
               --  in the Init routine has to be unchecked converted to the
               --  corresponding record type, since that's what Init expects.

               if Is_Task_Type (T) then
                  Arg1 :=
                    Make_Unchecked_Type_Conversion (Loc,
                      Subtype_Mark =>
                        New_Reference_To (Corresponding_Record_Type (T), Loc),
                      Expression => Arg1);
               end if;

               Args := New_List (Arg1);

               --  For the task case, pass the Master_Id of the access type
               --  as the value of the _Master parameter, and _Chain as the
               --  value of the _Chain parameter (_Chain will be defined as
               --  part of the generated code for the allocator).

               if Has_Tasks (T) then

                  if No (Master_Id (PtrT)) then

                     --  The designated type was an incomplete type, and
                     --  the access type did not get expanded. Salvage
                     --  it now. This may be a more general problem.

                     Expand_N_Full_Type_Declaration (Parent (PtrT));
                  end if;

                  Append_To (Args, New_Reference_To (Master_Id (PtrT), Loc));
                  Append_To (Args, Make_Identifier (Loc, Name_uChain));
               end if;

               --  Add discriminants if discriminated type

               if Has_Discriminants (T) then
                  Discr := First_Elmt (Discriminant_Constraint (T));

                  while Present (Discr) loop
                     Append (New_Copy (Elists.Node (Discr)), Args);
                     Discr := Next_Elmt (Discr);
                  end loop;
               end if;

               --  We set the allocator as analyzed so that when we analyze the
               --  expression actions node, we do not get an unwanted recursive
               --  expansion of the allocator expression.

               Set_Analyzed (N, True);

               --  Now we can rewrite the allocator. First see if it is
               --  already in an expression actions node, which will often
               --  be the case, because this is how we handle the case of
               --  discriminants being present. If so, we can just modify
               --  that expression actions node that is there, otherwise
               --  we must create an expression actions node.

               Eact := Parent (N);

               if Nkind (Eact) = N_Expression_Actions
                 and then Expression (Eact) = N
               then
                  Node := N;

               else
                  Rewrite_Substitute_Tree (N,
                    Make_Expression_Actions (Loc,
                      Actions    => New_List,
                      Expression => New_Copy (N)));

                  Eact := N;
                  Node := Expression (N);
               end if;

               --  Now we modify the expression actions node as follows

               --    input:   [... ; new T]

               --    output:  [... ;
               --              Temp : constant ptr_T := new (T);
               --              Init (Temp.all, ...);
               --      <CTRL>  Attach_To_Final_List (Finalizable (Temp.all));
               --      <CTRL>  Initialize (Finalizable (Temp.all));
               --              Temp]

               --  Here ptr_T is the pointer type for the allocator, and T
               --  is the subtype of the allocator.

               Append_To (Actions (Eact),
                 Make_Object_Declaration (Loc,
                   Defining_Identifier => Temp,
                   Constant_Present    => True,
                   Object_Definition   => New_Reference_To (PtrT, Loc),
                   Expression          => Node));

               --  Case of designated type is task or contains task

               if Has_Tasks (T) then
                  Build_Task_Allocate_Block (Actions (Eact), Node, Args);

               else
                  Append_To (Actions (Eact),
                    Make_Procedure_Call_Statement (Loc,
                      Name => New_Reference_To (Init, Loc),
                      Parameter_Associations => Args));
               end if;

               if Is_Controlled (T) then
                  Append_List_To (Actions (Eact),
                    Make_Init_Attach_Calls (
                      Ref => Make_Explicit_Dereference (Loc,
                               Prefix => New_Reference_To (Temp, Loc)),
                      Scop => Scope (PtrT),
                      Typ => T));
               end if;

               Set_Expression (Eact, New_Reference_To (Temp, Loc));
               Analyze (Eact);

            end if;
         end;
      end if;
   end Expand_N_Allocator;

   ------------------------------
   -- Expand_N_Concat_Multiple --
   ------------------------------

   procedure Expand_N_Concat_Multiple (N : Node_Id) is
   begin
      Expand_Concatenation (N, Expressions (N));
   end Expand_N_Concat_Multiple;

   ---------------------
   -- Expand_N_Op_Abs --
   ---------------------

   procedure Expand_N_Op_Abs (N : Node_Id) is
   begin
      if Software_Overflow_Checking
         and then Is_Signed_Integer_Type (Etype (N))
         and then Do_Overflow_Check (N)
      then
         Expand_Arithmetic_Overflow_Check (N);
      end if;
   end Expand_N_Op_Abs;

   ---------------------
   -- Expand_N_Op_Add --
   ---------------------

   procedure Expand_N_Op_Add (N : Node_Id) is
   begin
      if Software_Overflow_Checking
         and then Is_Signed_Integer_Type (Etype (N))
         and then Do_Overflow_Check (N)
      then
         Expand_Arithmetic_Overflow_Check (N);
      end if;
   end Expand_N_Op_Add;

   ---------------------
   -- Expand_N_Op_And --
   ---------------------

   --  This is really just a renaming of Expand_Boolean_Operator ???

   procedure Expand_N_Op_And (N : Node_Id) is
   begin
      Expand_Boolean_Operator (N);
   end Expand_N_Op_And;

   ------------------------
   -- Expand_N_Op_Concat --
   ------------------------

   procedure Expand_N_Op_Concat (N : Node_Id) is
      Loc      : constant Source_Ptr := Sloc (N);
      Lhs      : Node_Id   := Left_Opnd (N);
      Rhs      : Node_Id   := Right_Opnd (N);
      Ltyp     : Entity_Id := Base_Type (Etype (Lhs));
      Rtyp     : Entity_Id := Base_Type (Etype (Rhs));
      Comp_Typ : Entity_Id := Base_Type (Component_Type (Etype (N)));

   begin
      --  If left operand is a single component, replace by an aggregate
      --  of the form (1 => operand), as required by concatenation semantics.

      if Ltyp = Comp_Typ then
         Lhs :=
           Make_Aggregate (Loc,
             Component_Associations => New_List (
               Make_Component_Association (Loc,
                 Choices    => New_List (Make_Integer_Literal (Loc, Uint_1)),
                 Expression => New_Copy (Lhs))));
         Ltyp := Base_Type (Etype (N));
      end if;

      --  Similar handling for right operand

      if Rtyp = Comp_Typ then
         Rhs :=
           Make_Aggregate (Loc,
             Component_Associations => New_List (
               Make_Component_Association (Loc,
                 Choices    => New_List (Make_Integer_Literal (Loc, Uint_1)),
                 Expression => New_Copy (Rhs))));
         Rtyp := Base_Type (Etype (N));
      end if;

      --  Handle case of concatenating Standard.String with runtime call

      if Ltyp = Standard_String and then Rtyp = Standard_String then
         Rewrite_Substitute_Tree (N,
           Make_Function_Call (Loc,
             Name => New_Reference_To (RTE (RE_Str_Concat), Loc),
             Parameter_Associations =>
                     New_List (New_Copy (Lhs), New_Copy (Rhs))));

         Analyze (N);
         Resolve (N, Standard_String);

      --  For other than Standard.String, use general routine

      else
         Expand_Concatenation (N, New_List (Lhs, Rhs));
      end if;

   end Expand_N_Op_Concat;

   ------------------------
   -- Expand_N_Op_Divide --
   ------------------------

   procedure Expand_N_Op_Divide (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Typ : constant Entity_Id  := Etype (N);
      Div : constant Node_Id    := New_Copy (N);

   begin
      if Is_Fixed_Point_Type (Typ) then

         if Base_Type (Etype (Right_Opnd (N))) = Standard_Integer then
            Expand_Fixed_Integer_Division (N);

         else
            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression => Div));
            Set_Etype (N, Typ);
            Expand_Fixed_To_Fixed_Division (N);
         end if;

      elsif Software_Overflow_Checking
         and then Is_Integer_Type (Typ)
         and then Do_Overflow_Check (N)
      then
         Expand_Zero_Divide_Check (N);

         if Is_Signed_Integer_Type (Etype (N)) then
            Expand_Arithmetic_Overflow_Check (N);
         end if;
      end if;
   end Expand_N_Op_Divide;

   --------------------
   -- Expand_N_Op_Eq --
   --------------------

   procedure Expand_N_Op_Eq (N : Node_Id) is
      Loc     : constant Source_Ptr := Sloc (N);
      Lhs     : constant Node_Id    := Left_Opnd (N);
      Rhs     : constant Node_Id    := Right_Opnd (N);
      Typl    : Entity_Id  := Etype (Lhs);
      Typr    : Entity_Id  := Etype (Rhs);
      Eq_Prim : Entity_Id;

   begin
      if Ekind (Typl) = E_Private_Type then
         Typl := Underlying_Type (Typl);
      end if;

      Typl := Base_Type (Typl);

      if  Is_Array_Type (Typl) then

         if Is_Scalar_Type (Component_Type (Typl)) then

            --  The case of two constrained arrays can be left to Gigi

            if Nkind (Lhs) /= N_Expression_Actions
              and then Nkind (Rhs) /= N_Expression_Actions
            then
               null;

               --  Kludge to avoid a bug in Gigi (work only for Strings) ???

            elsif Typl = Standard_String then
               Rewrite_Substitute_Tree (N,
                 Make_Function_Call (Loc,
                   Name => New_Reference_To (RTE (RE_Str_Equal), Loc),
                   Parameter_Associations =>
                     New_List (New_Copy (Lhs), New_Copy (Rhs))));

               Analyze (N);
               Resolve (N, Standard_Boolean);

            --  Other cases, we hope Gigi will not blow up ???

            else
               null;
            end if;
         else
            Replace_Substitute_Tree (N,
              Expand_Array_Equality
                (Loc, Typl, New_Copy (Lhs), New_Copy (Rhs)));

            Analyze (N);
            Resolve (N, Standard_Boolean);
         end if;

      elsif Is_Record_Type (Typl) then

         if Has_Discriminants (Typl)
           and then Present (Variant_Part (Component_List (
                               Type_Definition (Parent (Typl)))))
         then

            --  ???
            --  in this case a function has to be expanded and called using
            --  the same model as for initialization procedures  (use of
            --  the case statement in the record definition).
            --  It has to be dealt with as a special case because in the
            --  simple case (record without variant part), we prefer to
            --  generate a big expression which will be optimized by the
            --  back-end.

            Unimplemented (N, "?complex equality of discriminated records");

         else
            declare
               L       : Node_Id;
               R       : Node_Id;
               Local_L : Entity_Id;
               Local_R : Entity_Id;
               Actions : List_Id := New_List;

            begin
               if Is_Entity_Name (Lhs)  then
                  L := Relocate_Node (Lhs);
               else
                  Local_L :=
                    Make_Defining_Identifier (Loc, New_Internal_Name ('L'));

                  Append_To (Actions,
                    Make_Object_Declaration (Loc,
                      Defining_Identifier => Local_L,
                      Object_Definition   => New_Reference_To (Typl, Loc),
                      Expression => Relocate_Node (Lhs)));

                  L := New_Reference_To (Local_L, Loc);
               end if;

               if Is_Entity_Name (Rhs)  then
                  R := Relocate_Node (Rhs);
               else
                  Local_R :=
                    Make_Defining_Identifier (Loc, New_Internal_Name ('R'));

                  Append_To (Actions,
                    Make_Object_Declaration (Loc,
                      Defining_Identifier => Local_R,
                      Object_Definition   => New_Reference_To (Typl, Loc),
                      Expression => Relocate_Node (Rhs)));

                  R := New_Reference_To (Local_R, Loc);
               end if;

               if Is_Empty_List (Actions) then
                  Replace_Substitute_Tree (N,
                    Expand_Record_Equality (Loc, Typl, L, R));
               else
                  Replace_Substitute_Tree (N,
                    Make_Expression_Actions (Loc,
                       Actions    => Actions,
                       Expression =>
                         Expand_Record_Equality (Loc, Typl, L, R)));
               end if;

               Analyze (N);
               Resolve (N, Standard_Boolean);
            end;
         end if;

      elsif Is_Fixed_Point_Type (Typl) then
         Expand_Literal_To_Fixed (Rhs, Typl);

      elsif Is_Fixed_Point_Type (Typr) then
         Expand_Literal_To_Fixed (Lhs, Typr);
      end if;

   end Expand_N_Op_Eq;

   -----------------------
   -- Expand_N_Op_Expon --
   -----------------------

   procedure Expand_N_Op_Expon (N : Node_Id) is
      Loc   : constant Source_Ptr := Sloc (N);
      Typ   : constant Entity_Id  := Etype (N);
      Btyp  : constant Entity_Id  := Root_Type (Typ);
      Max   : constant Uint       := Uint_4;
      Min   : constant Uint       := Uint_Minus_4;
      Base  : constant Node_Id    := New_Copy (Left_Opnd (N));
      Exp   : constant Node_Id    := New_Copy (Right_Opnd (N));
      Ovflo : constant Boolean    := Do_Overflow_Check (N);
      Expv  : Uint;
      Xnode : Node_Id;
      Temp  : Node_Id;
      Rent  : RE_Id;
      Ent   : Entity_Id;

   begin
      --  At this point the exponentiation must be dynamic since the static
      --  case has already been folded after Resolve by Eval_Op_Expon.

      --  Test for case of literal right argument

      if Nkind (Exp) = N_Integer_Literal then
         Expv := Intval (Exp);

         if (Ekind (Typ) in Float_Kind
               and then UI_Ge (Expv, Min)
               and then UI_Le (Expv, Max))
           or else
            (Ekind (Typ) in Integer_Kind
               and then UI_Ge (Expv, Uint_0)
               and then UI_Le (Expv, Max))
         then
            Expv := UI_Abs (Expv);

            --  X ** 0 = 1 (or 1.0)

            if Expv = Uint_0 then
               if Ekind (Typ) in Integer_Kind then
                  Xnode := Make_Integer_Literal (Loc, Intval => Uint_1);
               else
                  Xnode := Make_Real_Literal (Loc, Ureal_1);
               end if;

            --  X ** 1 = X

            elsif Expv = Uint_1 then
               Xnode := Base;

            --  X ** 2 = X * X

            elsif Expv = Uint_2 then
               Set_Evaluate_Once (Base, True);
               Xnode :=
                 Make_Op_Multiply (Loc,
                   Left_Opnd  => Base,
                   Right_Opnd => New_Copy (Base));

            --  X ** 3 = X * X * X

            elsif Expv = Uint_3 then
               Set_Evaluate_Once (Base, True);
               Xnode :=
                 Make_Op_Multiply (Loc,
                   Left_Opnd =>
                     Make_Op_Multiply (Loc,
                       Left_Opnd  => New_Copy (Base),
                       Right_Opnd => New_Copy (Base)),

                   Right_Opnd  => Base);

            --  X ** 4 = {Xnnn : constant base'type := base * base} Xnnn * Xnnn

            elsif Expv = Uint_4 then
               Set_Evaluate_Once (Base, True);
               Temp :=
                 Make_Defining_Identifier (Loc, New_Internal_Name ('X'));

               Xnode :=
                 Make_Expression_Actions (Loc,
                   Actions => New_List (
                     Make_Object_Declaration (Loc,
                       Defining_Identifier => Temp,
                       Constant_Present    => True,
                       Object_Definition   => New_Reference_To (Typ, Loc),
                       Expression =>
                         Make_Op_Multiply (Loc,
                           Left_Opnd  => New_Copy (Base),
                           Right_Opnd => Base))),
                   Expression =>
                     Make_Op_Multiply (Loc,
                       Left_Opnd  => New_Reference_To (Temp, Loc),
                       Right_Opnd => New_Reference_To (Temp, Loc)));
            end if;

            --  For non-negative case, we are all set

            if not UI_Is_Negative (Intval (Exp)) then
               Rewrite_Substitute_Tree (N, Xnode);

            --  For negative cases, take reciprocal (base must be real)

            else
               Set_Paren_Count (Xnode, 1);
               Replace_Substitute_Tree (N,
                 Make_Op_Divide (Loc,
                   Left_Opnd   => Make_Real_Literal (Loc, Ureal_1),
                   Right_Opnd  => Xnode));
            end if;

            Analyze (N);
            Resolve (N, Typ);
            return;

         --  Don't fold cases of large literal exponents, and also don't fold
         --  cases of integer bases with negative literal exponents.

         end if;

      --  Don't fold cases where exponent is not integer literal

      end if;

      --  Fall through if exponentiation must be done using a runtime routine
      --  First deal with modular case.

      if Is_Modular_Integer_Type (Btyp) then

         --  Non-binary case, we call the special exponentiation routine for
         --  the non-binary case, converting the argument to Long_Long_Integer
         --  and passing the modulus value. Then the result is converted back
         --  to the base type.

         if Non_Binary_Modulus (Btyp) then

            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression   =>
                  Make_Function_Call (Loc,
                    Name => New_Reference_To (RTE (RE_Exp_Modular), Loc),
                    Parameter_Associations => New_List (
                      Make_Type_Conversion (Loc,
                        Subtype_Mark =>
                          New_Reference_To (Standard_Integer, Loc),
                        Expression => Base),
                      Make_Integer_Literal (Loc, Modulus (Btyp)),
                      Exp))));

         --  Binary case, in this case, we call one of two routines, either
         --  the unsigned integer case, or the unsigned long long integer
         --  case, with the final conversion doing the required truncation.

         else
            if UI_To_Int (Esize (Btyp)) <= Standard_Integer_Size then
               Ent := RTE (RE_Exp_Unsigned);
            else
               Ent := RTE (RE_Exp_Long_Long_Unsigned);
            end if;

            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression   =>
                  Make_Function_Call (Loc,
                    Name => New_Reference_To (Ent, Loc),
                    Parameter_Associations => New_List (
                      Make_Type_Conversion (Loc,
                        Subtype_Mark =>
                          New_Reference_To (Etype (First_Formal (Ent)), Loc),
                        Expression   => Base),
                      Exp))));
         end if;

         --  Common exit point for modular type case

         Analyze (N);
         Resolve (N, Typ);
         return;

      --  Signed integer cases

      elsif Btyp = Standard_Integer then
         if Ovflo then
            Rent := RE_Exp_Integer;
         else
            Rent := RE_Exn_Integer;
         end if;

      elsif Btyp = Standard_Short_Integer then
         if Ovflo then
            Rent := RE_Exp_Short_Integer;
         else
            Rent := RE_Exn_Short_Integer;
         end if;

      elsif Btyp = Standard_Short_Short_Integer then
         if Ovflo then
            Rent := RE_Exp_Short_Short_Integer;
         else
            Rent := RE_Exn_Short_Short_Integer;
         end if;

      elsif Btyp = Standard_Long_Integer then
         if Ovflo then
            Rent := RE_Exp_Long_Integer;
         else
            Rent := RE_Exn_Long_Integer;
         end if;

      elsif (Btyp = Standard_Long_Long_Integer
        or else Btyp = Universal_Integer)
      then
         if Ovflo then
            Rent := RE_Exp_Long_Long_Integer;
         else
            Rent := RE_Exn_Long_Long_Integer;
         end if;

      --  Floating-point cases

      elsif Btyp = Standard_Float then
         if Ovflo then
            Rent := RE_Exp_Float;
         else
            Rent := RE_Exn_Float;
         end if;

      elsif Btyp = Standard_Short_Float then
         if Ovflo then
            Rent := RE_Exp_Short_Float;
         else
            Rent := RE_Exn_Short_Float;
         end if;

      elsif Btyp = Standard_Long_Float then
         if Ovflo then
            Rent := RE_Exp_Long_Float;
         else
            Rent := RE_Exn_Long_Float;
         end if;

      elsif Btyp = Standard_Long_Long_Float
        or else Btyp = Universal_Real
      then
         if Ovflo then
            Rent := RE_Exp_Long_Long_Float;
         else
            Rent := RE_Exn_Long_Long_Float;
         end if;

      else
         pragma Assert (False); null;
      end if;

      --  Common processing for integer cases and floating-point cases.
      --  If we are in the base type, we can call runtime routine directly

      if Typ = Btyp
        and then Btyp /= Universal_Integer
        and then Btyp /= Universal_Real
      then
         Replace_Substitute_Tree (N,
           Make_Function_Call (Loc,
             Name => New_Reference_To (RTE (Rent), Loc),
             Parameter_Associations => New_List (Base, Exp)));

      --  Otherwise we have to introduce conversions (conversions are also
      --  required in the universal cases, since the runtime routine was
      --  typed using the largest integer or real case.

      else
         Replace_Substitute_Tree (N,
           Make_Type_Conversion (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Expression   =>
               Make_Function_Call (Loc,
                 Name => New_Reference_To (RTE (Rent), Loc),
                 Parameter_Associations => New_List (
                   Make_Type_Conversion (Loc,
                     Subtype_Mark => New_Reference_To (Btyp, Loc),
                     Expression   => Base),
                   Exp))));
      end if;

      Analyze (N);
      Resolve (N, Typ);
      return;

   end Expand_N_Op_Expon;

   --------------------
   -- Expand_N_Op_Ge --
   --------------------

   procedure Expand_N_Op_Ge (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Ge;

   --------------------
   -- Expand_N_Op_Gt --
   --------------------

   procedure Expand_N_Op_Gt (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Gt;

   --------------------
   -- Expand_N_Op_In --
   --------------------

   --  Expansion is only required for the tagged case. See specification of
   --  Tagged_Membership function for details of required transformation.

   procedure Expand_N_Op_In (N : Node_Id) is
      Typ : constant Entity_Id := Etype (N);

   begin
      if Is_Tagged_Type (Etype (Etype (Right_Opnd (N)))) then
         Replace_Substitute_Tree (N, Tagged_Membership (N));
         Analyze (N);
         Resolve (N, Typ);
      end if;
   end Expand_N_Op_In;

   --------------------
   -- Expand_N_Op_Le --
   --------------------

   procedure Expand_N_Op_Le (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Le;

   --------------------
   -- Expand_N_Op_Lt --
   --------------------

   procedure Expand_N_Op_Lt (N : Node_Id) is
   begin
      Expand_Comparison_Operator (N);
   end Expand_N_Op_Lt;

   -----------------------
   -- Expand_N_Op_Minus --
   -----------------------

   procedure Expand_N_Op_Minus (N : Node_Id) is
   begin
      if Software_Overflow_Checking
         and then Is_Signed_Integer_Type (Etype (N))
         and then Do_Overflow_Check (N)
      then
         Expand_Arithmetic_Overflow_Check (N);
      end if;
   end Expand_N_Op_Minus;

   ---------------------
   -- Expand_N_Op_Mod --
   ---------------------

   procedure Expand_N_Op_Mod (N : Node_Id) is
   begin
      if Software_Overflow_Checking
         and then Is_Integer_Type (Etype (N))
         and then Do_Overflow_Check (N)
      then
         Expand_Zero_Divide_Check (N);
      end if;
   end Expand_N_Op_Mod;

   --------------------------
   -- Expand_N_Op_Multiply --
   --------------------------

   procedure Expand_N_Op_Multiply (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Typ : constant Entity_Id  := Etype (N);
      Mul : constant Node_Id    := New_Copy (N);

   begin
      if Is_Fixed_Point_Type (Typ) then

         if Base_Type (Etype (Left_Opnd (N))) = Standard_Integer then
            Expand_Fixed_Integer_Multiplication
              (N, Right_Opnd (N), Left_Opnd (N));

         elsif Base_Type (Etype (Right_Opnd (N))) = Standard_Integer then
            Expand_Fixed_Integer_Multiplication
              (N, Left_Opnd (N), Right_Opnd (N));

         else
            Replace_Substitute_Tree (N,
              Make_Type_Conversion (Loc,
                Subtype_Mark => New_Reference_To (Typ, Loc),
                Expression => Mul));
            Set_Etype (N, Typ);
            Expand_Fixed_To_Fixed_Multiplication (N);
         end if;

      elsif Software_Overflow_Checking
         and then Is_Signed_Integer_Type (Etype (N))
         and then Do_Overflow_Check (N)
      then
         Expand_Arithmetic_Overflow_Check (N);
      end if;
   end Expand_N_Op_Multiply;

   ---------------------
   -- Expand_N_Op_Not --
   ---------------------

   --  If the argument of negation is a Boolean array type, generate the
   --  following in line function definition:

   --     function Nnnn (A : arr) is
   --       B : arr; (or arr (A'range) if arr is unconstrained)
   --     begin
   --       for J in a'range loop
   --          B (J) := not A (J);
   --       end loop;
   --       return B;
   --     end Nnnn;

   procedure Expand_N_Op_Not (N : Node_Id) is
      Loc : constant Source_Ptr := Sloc (N);
      Typ : constant Entity_Id  := Etype (N);
      A   : constant Entity_Id  := Make_Defining_Identifier (Loc, Name_uA);
      B   : constant Entity_Id  := Make_Defining_Identifier (Loc, Name_uB);
      I   : constant Entity_Id  := Make_Defining_Identifier (Loc, Name_uI);
      A_I : Node_Id;
      B_I : Node_Id;

      Func_Name      : Entity_Id;
      Func_Body      : Node_Id;
      Loop_Statement : Node_Id;
      Result         : Node_Id;
      Type_Of_B      : Node_Id;

   begin
      if not Is_Array_Type (Typ) then
         return;
      end if;

      A_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (A, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      B_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (B, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      Loop_Statement :=
        Make_Loop_Statement (Loc,
          Identifier => Empty,

          Iteration_Scheme =>
            Make_Iteration_Scheme (Loc,
              Loop_Parameter_Specification =>
                Make_Loop_Parameter_Specification (Loc,
                  Defining_Identifier => I,
                  Discrete_Subtype_Definition =>
                    Make_Attribute_Reference (Loc,
                      Prefix => Make_Identifier (Loc, Chars (A)),
                      Attribute_Name => Name_Range))),

          Statements => New_List (
            Make_Assignment_Statement (Loc,
              Name       => B_I,
              Expression => Make_Op_Not (Loc, A_I))));


      Func_Name := Make_Defining_Identifier (Loc, New_Internal_Name ('N'));

      if Is_Constrained (Typ) then
         Type_Of_B := New_Reference_To (Typ, Loc);
      else
         Type_Of_B :=
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Constraint   =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (Make_Attribute_Reference (Loc,
                      Attribute_Name => Name_Range,
                      Prefix => New_Reference_To (A, Loc)))));
      end if;

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name => Func_Name,
              Parameter_Specifications => New_List (
                Make_Parameter_Specification (Loc,
                  Defining_Identifier => A,
                  Parameter_Type      => New_Reference_To (Typ, Loc))),
              Subtype_Mark => New_Reference_To (Typ,  Loc)),

          Declarations => New_List (
            Make_Object_Declaration (Loc,
              Defining_Identifier => B,
              Object_Definition   => Type_Of_B)),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (
                Loop_Statement,
                Make_Return_Statement (Loc,
                  Expression =>
                    Make_Identifier (Loc, Chars (B))))));

      Result :=
        Make_Expression_Actions (Loc,
          Actions => New_List (Func_Body),
          Expression =>
            Make_Function_Call (Loc,
              Name => New_Reference_To (Func_Name, Loc),
              Parameter_Associations =>
                New_List (Right_Opnd (N))));

      Replace_Substitute_Tree (N, Result);
      Analyze (N);
      Resolve (N, Typ);
   end Expand_N_Op_Not;

   ------------------------
   -- Expand_N_Op_Not_In --
   ------------------------

   --  Expansion is only required for the tagged case. See specification of
   --  Tagged_Membership function for details of required transformation.

   procedure Expand_N_Op_Not_In (N : Node_Id) is
      Typ : constant Entity_Id := Etype (N);

   begin
      if Is_Tagged_Type (Etype (Etype (Right_Opnd (N)))) then
         Replace_Substitute_Tree (N,
           Make_Op_Not (Sloc (N), Tagged_Membership (N)));
         Analyze (N);
         Resolve (N, Typ);
      end if;
   end Expand_N_Op_Not_In;

   --------------------
   -- Expand_N_Op_Or --
   --------------------

   procedure Expand_N_Op_Or (N : Node_Id) is
   begin
      Expand_Boolean_Operator (N);
   end Expand_N_Op_Or;

   ---------------------
   -- Expand_N_Op_Rem --
   ---------------------

   procedure Expand_N_Op_Rem (N : Node_Id) is
   begin
      if Software_Overflow_Checking
         and then Is_Integer_Type (Etype (N))
         and then Do_Overflow_Check (N)
      then
         Expand_Zero_Divide_Check (N);
      end if;
   end Expand_N_Op_Rem;

   --------------------------
   -- Expand_N_Op_Subtract --
   --------------------------

   procedure Expand_N_Op_Subtract (N : Node_Id) is
   begin
      if Software_Overflow_Checking
         and then Is_Signed_Integer_Type (Etype (N))
         and then Do_Overflow_Check (N)
      then
         Expand_Arithmetic_Overflow_Check (N);
      end if;
   end Expand_N_Op_Subtract;

   ---------------------
   -- Expand_N_Op_Xor --
   ---------------------

   procedure Expand_N_Op_Xor (N : Node_Id) is
   begin
      Expand_Boolean_Operator (N);
   end Expand_N_Op_Xor;

   --------------------
   -- Expand_N_Slice --
   --------------------

   --  Build an implicit subtype declaration to represent the type delivered
   --  by the slice. This subtype has an Ekind of E_Slice_Subtype, which is
   --  a special kind of type used just for this purpose. Logically, what is
   --  needed is a full array subtype declaration, but that would take a lot
   --  of nodes. On the other hand if we don't provide any kind of subtype
   --  for the slice, Gigi gets really confused. The compromise of building
   --  a special kind of very economical subtype declaration node, and then
   --  putting a bit of specialized code in Gigi to deal with this special
   --  declaration meets the need with minimum overhead.

   --  The procesing consists of building this subtype and then resetting the
   --  Etype of the slice node to have this type.

   procedure Expand_N_Slice (N : Node_Id) is
      Impl_Subtype : Entity_Id;

   begin
      --  First we build a defining occurrence for the "slice subtype"

      Impl_Subtype := New_Itype (E_Slice_Subtype, N);
      Set_Component_Type (Impl_Subtype, Component_Type (Etype (N)));
      Set_Slice_Range (Impl_Subtype, Discrete_Range (N));
      Set_Etype (Impl_Subtype, Etype (N));

      --  The Etype of the existing Slice node is reset to this anymous
      --  subtype. This node will be marked as Analyzed when we return and
      --  nothing else needs to be done to it.

      Set_Etype (N, Impl_Subtype);
   end Expand_N_Slice;

   ------------------------------
   -- Expand_N_Type_Conversion --
   ------------------------------

   procedure Expand_N_Type_Conversion (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (N);
      Expr : constant Node_Id    := Expression (N);
      T    : constant Entity_Id  := Etype (N);

   begin
      --  If it is a tagged type, that's a view conversion and the node
      --  gets converted to an unchecked type conversion since we don't
      --  want Gigi doing anything to such a node

      if Is_Variable (Expr)
        and then Is_Tagged_Type (T)
        and then Is_Tagged_Type (Etype (Expr))
      then
         Change_Conversion_To_Unchecked (N);
      end if;

      if Etype (Expr) = Universal_Fixed then

         if Nkind (Expr) = N_Op_Multiply then

            if Is_Fixed_Point_Type (T) then
               Expand_Fixed_To_Fixed_Multiplication (N);

            elsif Is_Integer_Type (T) then
               Expand_Fixed_To_Integer_Multiplication (N);

            elsif Is_Floating_Point_Type (T) then
               Expand_Fixed_To_Float_Multiplication (N);

            else
               Unimplemented (N,
                 "fixed by fixed multiplication with wanted result type");
            end if;

         elsif Nkind (Expr) = N_Op_Divide then

            if Is_Fixed_Point_Type (T) then
               Expand_Fixed_To_Fixed_Division (N);

            elsif Is_Integer_Type (T) then
               Expand_Fixed_To_Integer_Division (N);

            elsif Is_Floating_Point_Type (T) then
               Expand_Fixed_To_Float_Division (N);

            else
               Unimplemented (N,
                 "fixed by fixed division with wanted result type");
            end if;
         end if;

      --  Expansion of conversions whose source is a fixed-point type

      elsif Is_Fixed_Point_Type (Etype (Expr)) then

         if Is_Fixed_Point_Type (T) then
            Expand_Fixed_To_Fixed_Multiplication (N);

         elsif Is_Integer_Type (T) then
            Expand_Fixed_To_Integer_Multiplication (N);

         elsif Is_Floating_Point_Type (T) then
            Expand_Fixed_To_Float_Conversion (N);
         end if;

      --  Expansions of conversions whose result type is fixed-point

      elsif Is_Fixed_Point_Type (T) then

         if Is_Integer_Type (Etype (Expr)) then
            Expand_Fixed_To_Fixed_Multiplication (N);

         elsif Is_Floating_Point_Type (Etype (Expr)) then
            Expand_Float_To_Fixed_Conversion (N);
         end if;

      --  Expansion of float-to-integer conversions

      elsif Is_Integer_Type (T)
        and then Is_Floating_Point_Type (Etype (Expr))
      then
         --  Nothing to do if Float_Truncate already set (means that this
         --  conversion is the result of a previous extension step)

         if Float_Truncate (N) then
            null;

         --  Special case, if the expression is a typ'Truncation attribute,
         --  then this attribute can be eliminated, and Float_Truncate set
         --  on the conversion node.

         elsif Nkind (Expr) = N_Attribute_Reference
           and then Attribute_Name (Expr) = Name_Truncation
         then
            Replace_Substitute_Tree (Expr,
              Relocate_Node (First (Expressions (Expr))));
            Set_Float_Truncate (N, True);

         --  Otherwise, we expand T (S) into

         --    [Tnn : constant rtyp := S;
         --       [if Tnn >= 0.0 then ityp^(Tnn + 0.5) else ityp^(Tnn - 0.5)]]

         --  where rtyp is the base type of the floating-point source type,
         --  and itype is the base type of the integer target type.

         else
            declare
               Tnn : constant Entity_Id :=
                       Make_Defining_Identifier
                         (Loc, New_Internal_Name ('T'));

               Ityp : constant Entity_Id := T;
               Rtyp : constant Entity_Id := Etype (Expr);

               function Make_Float_Truncate_Type_Conversion
                 (Expr : Node_Id)
                  return Node_Id;
               --  Builds a type conversion with the Float_Truncate flag set,
               --  the given argument Expr as the source, and the base type'
               --  as the destination subtype

               function Make_Float_Truncate_Type_Conversion
                 (Expr : Node_Id)
                  return Node_Id
               is
                  Cnode : constant Node_Id :=
                    Make_Type_Conversion (Loc,
                      Subtype_Mark => New_Reference_To (Ityp, Loc),
                      Expression => Expr);
               begin
                  Set_Float_Truncate (Cnode, True);
                  return Cnode;
               end Make_Float_Truncate_Type_Conversion;

            begin
               Replace_Substitute_Tree (N,
                 Make_Expression_Actions (Loc,
                    Actions => New_List (
                      Make_Object_Declaration (Loc,
                        Defining_Identifier => Tnn,
                        Constant_Present    => True,
                        Object_Definition   => New_Reference_To (Rtyp, Loc),
                        Expression          => Expression (N))),

                    Expression =>
                      Make_Conditional_Expression (Loc, New_List (
                        Make_Op_Ge (Loc,
                          Left_Opnd  => New_Reference_To (Tnn, Loc),
                          Right_Opnd => Make_Real_Literal (Loc, Ureal_0)),

                        Make_Float_Truncate_Type_Conversion (
                          Make_Op_Add (Loc,
                            Left_Opnd  => New_Reference_To (Tnn, Loc),
                            Right_Opnd =>
                              Make_Real_Literal (Loc, Ureal_Half))),

                        Make_Float_Truncate_Type_Conversion (
                          Make_Op_Subtract (Loc,
                            Left_Opnd  => New_Reference_To (Tnn, Loc),
                            Right_Opnd =>
                              Make_Real_Literal (Loc, Ureal_Half)))))));

               Analyze (N);
               Resolve (N, T);
            end;
         end if;
      end if;

   end Expand_N_Type_Conversion;

   ----------------------------
   -- Expand_Record_Equality --
   ----------------------------

   --  For non-variant records, Equality is expanded when needed into:

   --      and then Lhs.Discr1 = Rhs.Discr1
   --      and then ...
   --      and then Lhs.Discrn = Rhs.Discrn
   --      and then Lhs.Cmp1 = Rhs.Cmp1
   --      and then ...
   --      and then Lhs.Cmpn = Rhs.Cmpn

   --  The expression is folded by the back-end for adjacent fields. This
   --  function is called for tagged record in only one occasion: for imple-
   --  menting predefined primitive equality (see Predefined_Primitives_Bodies)
   --  otherwise the primitive "=" is used directly.

   function Expand_Record_Equality
     (Loc  : Source_Ptr;
      Typ  : Entity_Id;
      Lhs  : Node_Id;
      Rhs  : Node_Id)
      return Node_Id
   is
      function Suitable_Element (C : Entity_Id) return Entity_Id;
      --  return the first field to compare beginning with C, skipping the
      --  inherited components

      function Suitable_Element (C : Entity_Id) return Entity_Id is
      begin

         if No (C) then
            return Empty;

         elsif (Ekind (C) /= E_Discriminant and then Ekind (C) /= E_Component)
           or else (Is_Tagged_Type (Typ)
             and then C /= Original_Record_Component (C))
         then
            return Suitable_Element (Next_Entity (C));
         else
            return C;
         end if;
      end Suitable_Element;

      Result : Node_Id;
      C      : Entity_Id;

   --  Start of processing for Expand_Record_Equality

   begin
      --  Generates the following code: (assuming that Typ has one Discr and
      --  component C2 is also a record)

      --   True
      --     and then Lhs.Discr1 = Rhs.Discr1
      --     and then Lhs.C1 = Rhs.C1
      --     and then Lhs.C2.C1=Rhs.C2.C1 and then ... Lhs.C2.Cn=Rhs.C2.Cn
      --     and then ...
      --     and then Lhs.Cmpn = Rhs.Cmpn

      Result := New_Reference_To (Standard_True, Loc);
      C := Suitable_Element (First_Entity (Typ));

      while Present (C) loop

         Result :=
           Make_Op_And_Then (Loc,
             Left_Opnd  => Result,
             Right_Opnd =>
               Expand_Composite_Equality (Loc, Etype (C),
                Lhs => Make_Selected_Component (Loc,
                         Prefix => Lhs,
                         Selector_Name => New_Reference_To (C, Loc)),
                Rhs => Make_Selected_Component (Loc,
                         Prefix => Rhs,
                         Selector_Name => New_Reference_To (C, Loc))));

         C := Suitable_Element (Next_Entity (C));
      end loop;

      return Result;
   end Expand_Record_Equality;

   ------------------------------
   -- Expand_Zero_Divide_Check --
   ------------------------------

   --  This routine is called only if a software zero divide check is needed,
   --  i.e. if the operation is a signed integer divide (or mod/rem) operation
   --  and software overflow checking is enabled, and Do_Overflow_Check is
   --  True. The expression a op b is expanded to:

   --     a op [temp : constant Typ := b;
   --           if temp = 0 then
   --             raise Constraint_Error;
   --           end if;
   --           temp]

   --  The check is required if software overflow checking is enabled, the
   --  operation is for a signed integer type, and Do_Overflow_Check is True

   procedure Expand_Zero_Divide_Check (N : Node_Id) is
      Loc  : constant Source_Ptr := Sloc (Right_Opnd (N));
      Typ  : constant Entity_Id  := Etype (N);
      Opnd : Node_Id;
      Temp : Entity_Id;

   begin
      Temp := Make_Defining_Identifier (Loc, New_Internal_Name ('T'));

      Opnd :=
        Make_Expression_Actions (Loc,
          Actions => New_List (

            Make_Object_Declaration (Loc,
              Defining_Identifier => Temp,
              Constant_Present    => True,
              Object_Definition   => New_Reference_To (Typ, Loc),
              Expression          => Relocate_Node (Right_Opnd (N))),

            Make_If_Statement (Loc,
              Condition =>
                Make_Op_Eq (Loc,
                  Left_Opnd => New_Reference_To (Temp, Loc),
                  Right_Opnd => Make_Integer_Literal (Loc, Uint_0)),
              Then_Statements => New_List (
                Make_Raise_Statement (Loc,
                  Name =>
                    New_Reference_To (
                      Standard_Constraint_Error, Loc))))),

          Expression => New_Reference_To (Temp, Loc));

      Analyze (Opnd);
      Resolve (Opnd, Typ);

      Replace_Substitute_Tree (Right_Opnd (N), Opnd);
      Set_Do_Overflow_Check (N, False);

   end Expand_Zero_Divide_Check;

   ------------------------------
   -- Make_Array_Comparison_Op --
   ------------------------------

   --  This is a hand-coded expansion of the following generic function:

   --  generic
   --    type elem is  (<>);
   --    type index is (<>);
   --    type a is array (index range <>) of elem;
   --
   --  function Gnnn (X : a; Y: a) return boolean is
   --    J : index := Y'first;
   --
   --  begin
   --    if X'length = 0 then
   --       return false;
   --
   --    elsif Y'length = 0 then
   --       return true;
   --
   --    else
   --      for I in X'range loop
   --        if X (I) = Y (J) then
   --          if J = Y'last then
   --            exit;
   --          else
   --            J := index'succ (J);
   --          end if;
   --
   --        else
   --           return X (I) > Y (J);
   --        end if;
   --      end loop;
   --
   --      return X'length > Y'length;
   --    end if;
   --  end Gnnn;

   --  If the flag Equal is true, the procedure generates the body for
   --  >= instead. This only affects the last return statement.

   --  Note that since we are essentially doing this expansion by hand, we
   --  do not need to generate an actual or formal generic part, just the
   --  instantiated function itself.

   function Make_Array_Comparison_Op
     (Typ   : Entity_Id;
      Loc   : Source_Ptr;
      Equal : Boolean)
      return  Node_Id
   is
      X : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uX);
      Y : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uY);
      I : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uI);
      J : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uJ);

      Index : constant Entity_Id := Base_Type (Etype (First_Index (Typ)));

      Loop_Statement : Node_Id;
      Loop_Body      : Node_Id;
      If_Stat        : Node_Id;
      Inner_If       : Node_Id;
      Final_Expr     : Node_Id;
      Func_Body      : Node_Id;
      Func_Name      : Entity_Id;
      Formals        : List_Id;
      Length1        : Node_Id;
      Length2        : Node_Id;

   begin
      --  if J = Y'last then
      --     exit;
      --  else
      --     J := index'succ (J);
      --  end if;

      Inner_If :=
        Make_If_Statement (Loc,
          Condition =>
            Make_Op_Eq (Loc,
              Left_Opnd => New_Reference_To (J, Loc),
              Right_Opnd =>
                Make_Attribute_Reference (Loc,
                  Prefix => New_Reference_To (Y, Loc),
                  Attribute_Name => Name_Last)),

          Then_Statements => New_List (
                Make_Exit_Statement (Loc)),

          Else_Statements =>
            New_List (
              Make_Assignment_Statement (Loc,
                Name => New_Reference_To (J, Loc),
                Expression =>
                  Make_Attribute_Reference (Loc,
                    Prefix => New_Reference_To (Index, Loc),
                    Attribute_Name => Name_Succ,
                    Expressions => New_List (New_Reference_To (J, Loc))))));

      --  if X (I) = Y (J) then
      --     if ... end if;
      --  else
      --     return X (I) > Y (J);
      --  end if;

      Loop_Body :=
        Make_If_Statement (Loc,
          Condition =>
            Make_Op_Eq (Loc,
              Left_Opnd =>
                Make_Indexed_Component (Loc,
                  Prefix      => New_Reference_To (X, Loc),
                  Expressions => New_List (New_Reference_To (I, Loc))),

              Right_Opnd =>
                Make_Indexed_Component (Loc,
                  Prefix      => New_Reference_To (Y, Loc),
                  Expressions => New_List (New_Reference_To (J, Loc)))),

          Then_Statements => New_List (Inner_If),

          Else_Statements => New_List (
            Make_Return_Statement (Loc,
              Expression =>
                Make_Op_Gt (Loc,
                  Left_Opnd =>
                    Make_Indexed_Component (Loc,
                      Prefix      => New_Reference_To (X, Loc),
                      Expressions => New_List (New_Reference_To (I, Loc))),

                  Right_Opnd =>
                    Make_Indexed_Component (Loc,
                      Prefix      => New_Reference_To (Y, Loc),
                      Expressions => New_List (
                        New_Reference_To (J, Loc)))))));

      --  for I in X'range loop
      --     if ... end if;
      --  end loop;

      Loop_Statement :=
        Make_Loop_Statement (Loc,
          Identifier => Empty,

          Iteration_Scheme =>
            Make_Iteration_Scheme (Loc,
              Loop_Parameter_Specification =>
                Make_Loop_Parameter_Specification (Loc,
                  Defining_Identifier => I,
                  Discrete_Subtype_Definition =>
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Reference_To (X, Loc),
                      Attribute_Name => Name_Range))),

          Statements => New_List (Loop_Body));

      --    if X'length = 0 then
      --       return false;
      --    elsif Y'length = 0 then
      --       return true;
      --    else
      --      for ... loop ... end loop;
      --      return X'length > Y'length;
      --    --  return X'length >= Y'length to implement >=.
      --    end if;

      Length1 :=
        Make_Attribute_Reference (Loc,
          Prefix => New_Reference_To (X, Loc),
          Attribute_Name => Name_Length);

      Length2 :=
        Make_Attribute_Reference (Loc,
          Prefix => New_Reference_To (Y, Loc),
          Attribute_Name => Name_Length);

      if Equal then
         Final_Expr :=
           Make_Op_Ge (Loc,
             Left_Opnd  => Length1,
             Right_Opnd => Length2);
      else
         Final_Expr :=
           Make_Op_Gt (Loc,
             Left_Opnd  => Length1,
             Right_Opnd => Length2);
      end if;

      If_Stat :=
        Make_If_Statement (Loc,
          Condition =>
            Make_Op_Eq (Loc,
              Left_Opnd =>
                Make_Attribute_Reference (Loc,
                  Prefix => New_Reference_To (X, Loc),
                  Attribute_Name => Name_Length),
              Right_Opnd =>
                Make_Integer_Literal (Loc, Uint_0)),

          Then_Statements =>
            New_List (
              Make_Return_Statement (Loc,
                Expression => New_Reference_To (Standard_False, Loc))),

          Elsif_Parts => New_List (
            Make_Elsif_Part (Loc,
              Condition =>
                Make_Op_Eq (Loc,
                  Left_Opnd =>
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Reference_To (Y, Loc),
                      Attribute_Name => Name_Length),
                  Right_Opnd =>
                    Make_Integer_Literal (Loc, Uint_0)),

              Then_Statements =>
                New_List (
                  Make_Return_Statement (Loc,
                     Expression => New_Reference_To (Standard_True, Loc))))),

          Else_Statements => New_List (
            Loop_Statement,
            Make_Return_Statement (Loc,
              Expression => Final_Expr)));


      --  (X : a; Y: a)

      Formals := New_List (
        Make_Parameter_Specification (Loc,
          Defining_Identifier => X,
          Parameter_Type      => New_Reference_To (Typ, Loc)),

        Make_Parameter_Specification (Loc,
          Defining_Identifier => Y,
          Parameter_Type      => New_Reference_To (Typ, Loc)));

      --  function Gnnn (...) return boolean is
      --    J : index := Y'first;
      --  begin
      --    if ... end if;
      --  end Gnnn;

      Func_Name := Make_Defining_Identifier (Loc, New_Internal_Name ('G'));

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name       => Func_Name,
              Parameter_Specifications => Formals,
              Subtype_Mark => New_Reference_To (Standard_Boolean, Loc)),

          Declarations => New_List (
            Make_Object_Declaration (Loc,
              Defining_Identifier => J,
              Object_Definition   => New_Reference_To (Index, Loc),
              Expression =>
                Make_Attribute_Reference (Loc,
                  Prefix => New_Reference_To (Y, Loc),
                  Attribute_Name => Name_First))),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (If_Stat)));

      return Func_Body;

   end Make_Array_Comparison_Op;

   ---------------------------
   -- Make_Boolean_Array_Op --
   ---------------------------

   --  For logical operations on boolean arrays, expand in line the
   --  following, replacing 'and' with 'or' or 'xor' where needed:

   --    function Annn (A : arr; B: arr) is
   --       C : arr;   (or arr (A'range) if arr is unconstrained)
   --    begin
   --       for I in A'range loop
   --          C (i) := A (i) and B (i);
   --       end loop;
   --       return C;
   --    end Annn;

   function Make_Boolean_Array_Op (N : Node_Id) return Node_Id is

      Loc : Source_Ptr := Sloc (N);
      Typ : Entity_Id := Etype (Left_Opnd (N));

      A   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uA);
      B   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uB);
      C   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uC);
      I   : constant Entity_Id := Make_Defining_Identifier (Loc, Name_uI);

      A_I : Node_Id;
      B_I : Node_Id;
      C_I : Node_Id;
      Op  : Node_Id;

      Formals        : List_Id;
      Func_Name      : Entity_Id;
      Func_Body      : Node_Id;
      Loop_Statement : Node_Id;
      Type_Of_C      : Node_Id;

   begin
      A_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (A, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      B_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (B, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      C_I :=
        Make_Indexed_Component (Loc,
          Prefix      => New_Reference_To (C, Loc),
          Expressions => New_List (New_Reference_To (I, Loc)));

      if Nkind (N) = N_Op_And then
         Op :=
           Make_Op_And (Loc,
             Left_Opnd  => A_I,
             Right_Opnd => B_I);

      elsif Nkind (N) = N_Op_Or then
         Op :=
           Make_Op_Or (Loc,
             Left_Opnd  => A_I,
             Right_Opnd => B_I);

      else
         Op :=
           Make_Op_Xor (Loc,
             Left_Opnd  => A_I,
             Right_Opnd => B_I);
      end if;

      Loop_Statement :=
        Make_Loop_Statement (Loc,
          Identifier => Empty,

          Iteration_Scheme =>
            Make_Iteration_Scheme (Loc,
              Loop_Parameter_Specification =>
                Make_Loop_Parameter_Specification (Loc,
                  Defining_Identifier => I,
                  Discrete_Subtype_Definition =>
                    Make_Attribute_Reference (Loc,
                      Prefix => New_Reference_To (A, Loc),
                      Attribute_Name => Name_Range))),

          Statements => New_List (
            Make_Assignment_Statement (Loc,
              Name       => C_I,
              Expression => Op)));

      Formals := New_List (
        Make_Parameter_Specification (Loc,
          Defining_Identifier => A,
          Parameter_Type      => New_Reference_To (Typ, Loc)),

        Make_Parameter_Specification (Loc,
          Defining_Identifier => B,
          Parameter_Type      => New_Reference_To (Typ, Loc)));

      Func_Name := Make_Defining_Identifier (Loc, New_Internal_Name ('A'));

      if Is_Constrained (Typ) then
         Type_Of_C := New_Reference_To (Typ, Loc);
      else
         Type_Of_C :=
           Make_Subtype_Indication (Loc,
             Subtype_Mark => New_Reference_To (Typ, Loc),
             Constraint   =>
                Make_Index_Or_Discriminant_Constraint (Loc,
                  Constraints =>
                    New_List (Make_Attribute_Reference (Loc,
                      Attribute_Name => Name_Range,
                      Prefix => New_Reference_To (A, Loc)))));
      end if;

      Func_Body :=
        Make_Subprogram_Body (Loc,
          Specification =>
            Make_Function_Specification (Loc,
              Defining_Unit_Name       => Func_Name,
              Parameter_Specifications => Formals,
              Subtype_Mark             => New_Reference_To (Typ, Loc)),

          Declarations => New_List (
            Make_Object_Declaration (Loc,
              Defining_Identifier => C,
              Object_Definition   => Type_Of_C)),

          Handled_Statement_Sequence =>
            Make_Handled_Sequence_Of_Statements (Loc,
              Statements => New_List (
                Loop_Statement,
                Make_Return_Statement (Loc,
                  Expression => New_Reference_To (C, Loc)))));

      return Func_Body;

   end Make_Boolean_Array_Op;

   ------------------------
   --  Tagged_Membership --
   ------------------------

   --  There are two different cases to consider depending on whether
   --  the right operand is a class-wide type or not. If not we just
   --  compare the actual tag of the left expr to the target type tag:
   --
   --     Left_Expr.Tag = Tag_Of (Right_Type);
   --
   --  If it is a class-wide type, it is more complex. We use the table of
   --  ancestors accessed by the "Tags" field of the Dispatch table. We have
   --  to ensure that the inheritance depth of the operand if greater or equal
   --  than the target types's and that they are on the inheritance path :
   --
   --  <action>
   --    N : Integer := Left_Expr.Tag.all.Inheritance_Depth -
   --                     Tag_Of (Right_Type).all.Inheritance_Depth;
   --  <expression>
   --    (N >= 0)
   --      and then Left_Expr.Tag.all.Tags.all (N) = Tag_Of (Right_Type)
   --
   --  the real expressions are a bit more complicated due to type conversions

   function Tagged_Membership (N : Node_Id) return Node_Id is
      Left       : constant Node_Id    := Left_Opnd  (N);
      Right      : constant Node_Id    := Right_Opnd (N);
      Sloc_N     : constant Source_Ptr := Sloc (N);

      Left_Type  : Entity_Id;
      Right_Type : Entity_Id;
      Var_N      : Node_Id;

   begin
      Left_Type  := Etype (Left);
      Right_Type := Etype (Right);

      if Is_Class_Wide_Type (Left_Type) then
         Left_Type := Etype (Left_Type);
      end if;

      if not Is_Class_Wide_Type (Right_Type) then

         --  Left_Type (Left)._Tag =
         --    System.Tag (Access_Disp_Table (Right_Type));

         return
           Make_Op_Eq (Sloc_N,
             Left_Opnd =>
               Make_Selected_Component (Sloc_N,
                 Prefix =>
                   Make_Unchecked_Type_Conversion (Sloc_N,
                     Subtype_Mark => New_Reference_To (Left_Type, Sloc_N),
                     Expression => New_Copy (Left)),
                 Selector_Name =>
                   New_Reference_To (Tag_Component (Left_Type), Sloc_N)),
             Right_Opnd =>
               Make_Unchecked_Type_Conversion (Sloc_N,
                 Subtype_Mark => New_Reference_To (RTE (RE_Tag), Sloc_N),
                 Expression =>
                   New_Reference_To (
                     Access_Disp_Table (Right_Type), Sloc_N)));

      else
         --  Replace N by expression-actions
         --
         --    <actions>
         --    N : Integer :=
         --      Acc_Dt (Left_Type(Left)._Tag).all.Inheritance_Depth
         --        - Access_Disp_Table (Right_Type).all.Inheritance_Depth;

         Var_N := Make_Defining_Identifier (Sloc_N,  New_Internal_Name ('N'));

         --  Use the root type of the class

         Right_Type := Etype (Right_Type);

         return
           Make_Expression_Actions (Sloc_N,
             Actions => New_List (
               Make_Object_Declaration (Sloc_N,
                 Defining_Identifier => Var_N,

                 Object_Definition   =>
                   New_Reference_To (Standard_Integer, Sloc_N),

                 Expression =>
                   Make_Op_Subtract (Sloc_N,
                     Left_Opnd =>
                       Make_Selected_Component (Sloc_N,
                         Prefix =>  Make_DT_Access (Sloc_N, Left, Left_Type),
                         Selector_Name =>
                           Make_DT_Component (Sloc_N, Left_Type, 1)),

                     Right_Opnd =>
                       Make_Selected_Component (Sloc_N,
                         Prefix =>
                           Make_Explicit_Dereference (Sloc_N,
                             Prefix =>
                               Make_Identifier (Sloc_N,
                                 Chars (Access_Disp_Table (Right_Type)))),

                       Selector_Name =>
                         Make_DT_Component (Sloc_N, Right_Type, 1))))),

         --  (N >= 0)
         --  and then
         --    (Acc_Dt (Left_Type (Left).__Tag).all.Tags.all (N)
         --     = System.Tag (Access_Disp_Table (Right_Type)))

             Expression =>
               Make_Op_And_Then (Sloc_N,
                 Left_Opnd =>
                   Make_Op_Ge (Sloc_N,
                     Left_Opnd => New_Reference_To (Var_N, Sloc_N),
                     Right_Opnd => Make_Integer_Literal (Sloc_N, Uint_0)),

                 Right_Opnd =>
                   Make_Op_Eq (Sloc_N,
                     Left_Opnd =>
                       Make_Indexed_Component (Sloc_N,
                         Prefix =>
                           Make_Explicit_Dereference (Sloc_N,
                             Prefix =>
                               Make_Selected_Component (Sloc_N,
                                 Prefix =>
                                   Make_DT_Access (Sloc_N, Left, Left_Type),
                                 Selector_Name =>
                                   Make_DT_Component (Sloc_N, Left_Type, 2))),
                         Expressions =>
                           New_List (New_Reference_To (Var_N, Sloc_N))),

                     Right_Opnd =>
                       Make_Unchecked_Type_Conversion (Sloc_N,
                         Subtype_Mark =>
                           New_Reference_To (RTE (RE_Tag), Sloc_N),
                         Expression =>
                           New_Reference_To (
                             Access_Disp_Table (Right_Type), Sloc_N)))));
      end if;
   end Tagged_Membership;

end Exp_Ch4;


----------------------
-- REVISION HISTORY --
----------------------

--  ----------------------------
--  revision 1.112
--  date: Sun Aug 14 07:12:49 1994;  author: dewar
--  (Expand_N_Type_Conversion): Implement rounding of floating-point
--   conversions, and also handling of Truncation attribute as argument
--   to conversions.
--  (Expand_N_Op_Expon): Avoid duplicate references to Base operand by
--   using New_Copy. This isn't quite right, but it's better than what
--   we have now, which bombs for sure.
--  ----------------------------
--  revision 1.113
--  date: Wed Aug 17 22:42:13 1994;  author: dewar
--  (Expand_Arithmetic_Overflow_Check): Simplify to use one conversion
--  ----------------------------
--  revision 1.114
--  date: Sun Aug 28 08:48:31 1994;  author: comar
--  (Expand_N_Slice): Use new protocol for New_Itype
--  ----------------------------
--  New changes after this line.  Each line starts with: "--  "
