/* tty.c -- Looking up terminals and access times. */

/* Copyright (C) 1988, 1990, 1992  Free Software Foundation, Inc.

   This file is part of GNU Finger.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <stdio.h>
#include <config.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>

#include <general.h>
#include <fingerpaths.h>
#include <util.h>

/* The directory containing the tty devices. */
#define TTYPATH "/dev"

/* **************************************************************** */
/*                                                                  */
/*                  TTY Location and Access Time                    */
/*                                                                  */
/* **************************************************************** */

/* List of devices attached to the console */
static char *console_devices[] =
{
#ifdef HAVE_DEV_CONSOLE
  "console",
#endif
#ifdef HAVE_DEV_KBD
  "kbd",
#endif
#ifdef HAVE_DEV_KEYBOARD
  "keyboard",
#endif
#ifdef HAVE_DEV_MOUSE
  "mouse",
#endif
#ifdef HAVE_DEV_SMF
  "smf",
#endif
  NULL
  };

/* Return the time that TTY was last accessed.  TTY is the name of the
   terminal device, as in "ttyp0", or "console". */
long
get_last_access (tty)
  char *tty;
{
  char *device;
  struct stat stats;
  time_t access_time;
  char **condev;


  access_time = 0;

  /* If the user is on the console, then return the access time
     of the most recently accessed device in console_devices[]. */

  if (is_console (tty))
    for (condev = console_devices; *condev; condev++)
      {
	device = xmalloc (strlen (TTYPATH) + 2 + strlen (*condev));
	sprintf (device, "%s/%s", TTYPATH, *condev);
	
	if (stat (device, &stats) >= 0)
	  access_time = MAX (access_time, stats.st_atime);
	
	free (device);
      }
  else
    {
      /* Not on console, stat device directly */
      device = xmalloc (strlen (TTYPATH) + 2 + strlen (tty));
      sprintf (device, "%s/%s", TTYPATH, tty);
      
      if (lstat (device, &stats) == -1
	  || (stats.st_mode & S_IFMT) == S_IFLNK)
	access_time = time (NULL);
      else
	access_time = stats.st_atime;

      free (device);
    }

  return access_time;
}


/* Lookup the associated location of TTY.  Return NULL if not found. */
char *
lookup_tty (tty)
  char *tty;
{
  char *fingerdir, *ttylocfile, *path_concat ();
  FILE *stream;
  static char *buffer = NULL;
  static int buffer_size = 0;


  stream = fopen (TTYLOCFILE, "r");

  if (!stream)
    return ((char *)NULL);

  while (getline (&buffer, &buffer_size, stream) >= 0)
    {
      register char c, *t = buffer + strlen (buffer);

      /* Get rid of trailing whitespace. */
      if (t != buffer)
	t--;

      while (*t && whitespace (*t) && t != buffer) t--;
      if (!whitespace (*t)) t++;
      *t = '\0';

      /* Isolate first word.  This is the tty.  The remainder of the
	 line is the location. */
      t = buffer;

      while (*t && whitespace (*t)) t++;
      while (*t && !whitespace (*t)) t++;
      c = *t;
      *t = '\0';

      if (strcmp (tty, buffer) == 0)
	{
	  fclose (stream);

	  /* Skip to the start of the next word.  That is the start of
	     the tty location string. */
	  if (c && t++)
	    while (*t && whitespace (*t)) t++;

	  return (t);
	}
    }
  fclose (stream);
  return ((char *)NULL);
}
