/*
 * man-config.c
 *
 * Read the man.config file
 *
 * Input line types:
 *	MANDATORY_MANPATH       /usr/X386/man
 *	MANPATH_MAP     /usr/bin     /usr/man
 *	NROFF           /usr/bin/groff -Tascii -mandoc
 *	.gz             /usr/bin/gunzip -c
 *	# Comment
 */

#ifdef STDC_HEADERS
#include <string.h>
#include <stdlib.h>
#else
char *malloc ();
char *rindex ();
#endif

#include <stdio.h>
#include "config.h"
#include "man.h"
#include "paths.h"

extern int debug;

#define whitespace(x) ((x) == ' ' || (x) == '\t')

void
addval (buf)
     char *buf;
{
  int i, len;
  char *bp;

  for (i = 0; i < sizeof(paths)/sizeof(paths[0]); i++) {
      len = strlen (paths[i].name);
      bp = buf + len;
      if(!strncmp (buf, paths[i].name, len) && (!*bp || whitespace(*bp))) {
	  while(whitespace(*bp))
	    bp++;
	  paths[i].path = strdup(bp);
	  return;
      }
  }
  fprintf (stderr, "Unrecognized line in config file (ignored)\n%s\n", buf);
}

char *
getval (cmd)
     char *cmd;
{
  int i;

  for (i = 0; i < sizeof(paths)/sizeof(paths[0]); i++)
      if (!strcmp (cmd, paths[i].name))
	return paths[i].path;
  fprintf (stderr, "man-config.c: internal error: string %s not found\n", cmd);
  return "";			/* impossible */
}

static struct xp {
    char *extension;		/* non-null, including initial . */
    char *expander;
    struct xp *nxt;
} uncompressors;		/* linked list, 1st entry unused */

void
addext (bp)
     char *bp;
{
  char *p, csv;
  struct xp *xpp;

  xpp = &uncompressors;
  while (xpp->nxt)
    xpp = xpp->nxt;
  xpp->nxt = (struct xp *) malloc (sizeof(struct xp));
  xpp = xpp->nxt;
  if (xpp == NULL)
    gripe_alloc (sizeof(struct xp), "xpp");
  xpp->nxt = 0;

  p = bp;
  while(*p && !whitespace(*p))
    p++;
  csv = *p;
  *p = 0;
  xpp->extension = strdup(bp);

  *p = csv;
  while(whitespace(*p))
    p++;
  xpp->expander = strdup(p);
}

char *
get_expander (file)
     char *file;
{
  struct xp *xp;
  char *extp;

  extp = rindex (file, '.');
  if (extp != NULL) {
      for (xp = uncompressors.nxt; xp; xp = xp->nxt)
	if (!strcmp (extp, xp->extension))
	  return (xp->expander);
  }
  return NULL;
}

struct dirs cfdirlist;		/* linked list, 1st entry unused */

void
adddir (bp, mandatory)
     char *bp;
     int mandatory;
{
  int i;
  struct dirs *dlp;

  while (whitespace(*bp))
    bp++;
  if (*bp == 0)
    gripe_reading_mp_config ();

  dlp = &cfdirlist;
  while (dlp->nxt)
    dlp = dlp->nxt;
  dlp->nxt = (struct dirs *) malloc (sizeof(struct dirs));
  dlp = dlp->nxt;
  if (dlp == NULL)
    gripe_alloc (sizeof(struct dirs), "dlp");
  dlp->mandatory = mandatory;
  dlp->nxt = 0;

  if (!mandatory) {
      i = 0;
      while (*bp && !whitespace(*bp)) {
	  if (i < MAXPATHLEN - 1)
	    dlp->bindir[i++] = *bp;
	  bp++;
      }
      dlp->bindir[i] = 0;

      while (whitespace(*bp))
	bp++;
  } else {
      dlp->bindir[0] = 0;
  }

  i = 0;
  while (*bp && !whitespace(*bp)) {
      if (i < MAXPATHLEN - 1)
	dlp->mandir[i++] = *bp;
      bp++;
  }
  dlp->mandir[i] = 0;
      
  if (debug) {
      if (dlp->mandatory)
	fprintf (stderr, "found mandatory man directory %s\n",
		 dlp->mandir);
      else
	fprintf (stderr, "found manpath map %s --> %s\n",
		 dlp->bindir, dlp->mandir);
  }
}

void
read_config_file (cf)
     char *cf;
{
  char *bp;
  char *p;
  char buf[BUFSIZ];
  FILE *config;

  if (cf == NULL)
    cf = CONFIG_FILE;

  if ((config = fopen (cf, "r")) == NULL) {
      perror (cf);
      gripe_getting_mp_config (cf);
      return;
  }

  while ((bp = fgets (buf, BUFSIZ, config)) != NULL) {
      while (whitespace(*bp))
	bp++;

      for (p = bp; *p && *p != '#' && *p != '\n'; p++) ;
      if (!*p) {
	  fprintf (stderr, "Line too long in config file\n");
	  gripe_reading_mp_config ();
      }
      while (p > bp && whitespace(p[-1]))
	p--;
      *p = 0;
      
      if (*bp == 0)
	continue;
      
      if (!strncmp ("MANDATORY_MANPATH", bp, 17))
	adddir (bp+17, 1);
      else if (!strncmp ("MANPATH_MAP", bp, 11))
	adddir (bp+11, 0);
      else if (*bp == '.')
	addext (bp);
      else
	addval (bp);
  }
}

