/****************************************************************************
**
** Copyright (C) 2010 Nokia Corporation and/or its subsidiary(-ies).
** All rights reserved.
** Contact: Nokia Corporation (qt-info@nokia.com)
**
** This file is part of the QtCore module of the Qt Toolkit.
**
** $QT_BEGIN_LICENSE:LGPL$
** Commercial Usage
** Licensees holding valid Qt Commercial licenses may use this file in
** accordance with the Qt Commercial License Agreement provided with the
** Software or, alternatively, in accordance with the terms contained in
** a written agreement between you and Nokia.
**
** GNU Lesser General Public License Usage
** Alternatively, this file may be used under the terms of the GNU Lesser
** General Public License version 2.1 as published by the Free Software
** Foundation and appearing in the file LICENSE.LGPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU Lesser General Public License version 2.1 requirements
** will be met: http://www.gnu.org/licenses/old-licenses/lgpl-2.1.html.
**
** In addition, as a special exception, Nokia gives you certain additional
** rights.  These rights are described in the Nokia Qt LGPL Exception
** version 1.1, included in the file LGPL_EXCEPTION.txt in this package.
**
** GNU General Public License Usage
** Alternatively, this file may be used under the terms of the GNU
** General Public License version 3.0 as published by the Free Software
** Foundation and appearing in the file LICENSE.GPL included in the
** packaging of this file.  Please review the following information to
** ensure the GNU General Public License version 3.0 requirements will be
** met: http://www.gnu.org/copyleft/gpl.html.
**
** If you have questions regarding the use of this file, please contact
** Nokia at qt-info@nokia.com.
** $QT_END_LICENSE$
**
****************************************************************************/

#include "quuid.h"

#include "qdatastream.h"

QT_BEGIN_NAMESPACE

/*!
    \class QUuid
    \brief The QUuid class stores a Universally Unique Identifier (UUID).

    \reentrant

    Using \e{U}niversally \e{U}nique \e{ID}entifiers (UUID) is a
    standard way to uniquely identify entities in a distributed
    computing environment. A UUID is a 16-byte (128-bit) number
    generated by some algorithm that is meant to guarantee that the
    UUID will be unique in the distributed computing environment where
    it is used. The acronym GUID is often used instead, \e{G}lobally
    \e{U}nique \e{ID}entifiers, but it refers to the same thing.

    \target Variant field
    Actually, the GUID is one \e{variant} of UUID. Multiple variants
    are in use. Each UUID contains a bit field that specifies which
    type (variant) of UUID it is. Call variant() to discover which
    type of UUID an instance of QUuid contains. It extracts the three
    most signifcant bits of byte 8 of the 16 bytes. In QUuid, byte 8
    is \c{QUuid::data4[0]}. If you create instances of QUuid using the
    constructor that accepts all the numeric values as parameters, use
    the following table to set the three most significant bits of
    parameter \c{b1}, which becomes \c{QUuid::data4[0]} and contains
    the variant field in its three most significant bits. In the
    table, 'x' means \e {don't care}.

    \table
    \header
    \o msb0
    \o msb1
    \o msb2
    \o Variant 

    \row
    \o 0
    \o x
    \o x
    \o NCS (Network Computing System)

    \row
    \o 1
    \o 0
    \o x
    \o DCE (Distributed Computing Environment)

    \row
    \o 1
    \o 1
    \o 0
    \o Microsoft (GUID)

    \row
    \o 1
    \o 1
    \o 1
    \o Reserved for future expansion

    \endtable

    \target Version field
    If variant() returns QUuid::DCE, the UUID also contains a
    \e{version} field in the four most significant bits of
    \c{QUuid::data3}, and you can call version() to discover which
    version your QUuid contains. If you create instances of QUuid
    using the constructor that accepts all the numeric values as
    parameters, use the following table to set the four most
    significant bits of parameter \c{w2}, which becomes
    \c{QUuid::data3} and contains the version field in its four most
    significant bits.

    \table
    \header
    \o msb0
    \o msb1
    \o msb2
    \o msb3
    \o Version

    \row
    \o 0
    \o 0
    \o 0
    \o 1
    \o Time

    \row
    \o 0
    \o 0
    \o 1
    \o 0
    \o Embedded POSIX

    \row
    \o 0
    \o 0
    \o 1
    \o 1
    \o Name

    \row
    \o 0
    \o 1
    \o 0
    \o 0
    \o Random

    \endtable

    The field layouts for the DCE versions listed in the table above
    are specified in the \l{http://www.ietf.org/rfc/rfc4122.txt}
    {Network Working Group UUID Specification}.
    
    Most platforms provide a tool for generating new UUIDs, e.g. \c
    uuidgen and \c guidgen. You can also use createUuid().  UUIDs
    generated by createUuid() are of the random type.  Their
    QUuid::Version bits are set to QUuid::Random, and their
    QUuid::Variant bits are set to QUuid::DCE. The rest of the UUID is
    composed of random numbers. Theoretically, this means there is a
    small chance that a UUID generated by createUuid() will not be
    unique. But it is
    \l{http://en.wikipedia.org/wiki/Universally_Unique_Identifier#Random_UUID_probability_of_duplicates}
    {a \e{very} small chance}.

    UUIDs can be constructed from numeric values or from strings, or
    using the static createUuid() function. They can be converted to a
    string with toString(). UUIDs have a variant() and a version(),
    and null UUIDs return true from isNull().
*/

/*!
    \fn QUuid::QUuid(const GUID &guid)

    Casts a Windows \a guid to a Qt QUuid.

    \warning This function is only for Windows platforms.
*/

/*!
    \fn QUuid &QUuid::operator=(const GUID &guid)

    Assigns a Windows \a guid to a Qt QUuid.

    \warning This function is only for Windows platforms.
*/

/*!
    \fn QUuid::operator GUID() const

    Returns a Windows GUID from a QUuid.

    \warning This function is only for Windows platforms.
*/

/*!
    \fn QUuid::QUuid()

    Creates the null UUID. toString() will output the null UUID
    as "{00000000-0000-0000-0000-000000000000}".
*/

/*!
    \fn QUuid::QUuid(uint l, ushort w1, ushort w2, uchar b1, uchar b2, uchar b3, uchar b4, uchar b5, uchar b6, uchar b7, uchar b8)

    Creates a UUID with the value specified by the parameters, \a l,
    \a w1, \a w2, \a b1, \a b2, \a b3, \a b4, \a b5, \a b6, \a b7, \a
    b8.

    Example:
    \snippet doc/src/snippets/code/src_corelib_plugin_quuid.cpp 0
*/

#ifndef QT_NO_QUUID_STRING
/*!
  Creates a QUuid object from the string \a text, which must be
  formatted as five hex fields separated by '-', e.g.,
  "{xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx}" where 'x' is a hex
  digit. The curly braces shown here are optional, but it is normal to
  include them. If the conversion fails, a null UUID is created.  See
  toString() for an explanation of how the five hex fields map to the
  public data members in QUuid.

    \sa toString(), QUuid()
*/
QUuid::QUuid(const QString &text)
{
    bool ok;
    if (text.isEmpty()) {
        *this = QUuid();
        return;
    }
    QString temp = text.toUpper();
    if (temp[0] != QLatin1Char('{'))
        temp = QLatin1Char('{') + text;
    if (text[(int)text.length()-1] != QLatin1Char('}'))
        temp += QLatin1Char('}');

    data1 = temp.mid(1, 8).toULongLong(&ok, 16);
    if (!ok) {
        *this = QUuid();
        return;
    }

    data2 = temp.mid(10, 4).toUInt(&ok, 16);
    if (!ok) {
        *this = QUuid();
        return;
    }
    data3 = temp.mid(15, 4).toUInt(&ok, 16);
    if (!ok) {
        *this = QUuid();
        return;
    }
    data4[0] = temp.mid(20, 2).toUInt(&ok, 16);
    if (!ok) {
        *this = QUuid();
        return;
    }
    data4[1] = temp.mid(22, 2).toUInt(&ok, 16);
    if (!ok) {
        *this = QUuid();
        return;
    }
    for (int i = 2; i<8; i++) {
        data4[i] = temp.mid(25 + (i-2)*2, 2).toUShort(&ok, 16);
        if (!ok) {
            *this = QUuid();
            return;
        }
    }
}

/*!
    \internal
*/
QUuid::QUuid(const char *text)
{
    *this = QUuid(QString::fromLatin1(text));
}
#endif

/*!
    \fn bool QUuid::operator==(const QUuid &other) const

    Returns true if this QUuid and the \a other QUuid are identical;
    otherwise returns false.
*/

/*!
    \fn bool QUuid::operator!=(const QUuid &other) const

    Returns true if this QUuid and the \a other QUuid are different;
    otherwise returns false.
*/
#ifndef QT_NO_QUUID_STRING
/*!
    \fn QUuid::operator QString() const

    Returns the string representation of the uuid.

    \sa toString()
*/

static QString uuidhex(uint data, int digits)
{
    return QString::number(data, 16).rightJustified(digits, QLatin1Char('0'));
}

/*!
    Returns the string representation of this QUuid. The string is
    formatted as five hex fields separated by '-' and enclosed in
    curly braces, i.e., "{xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx}" where
    'x' is a hex digit.  From left to right, the five hex fields are
    obtained from the four public data members in QUuid as follows:

    \table
    \header
    \o Field #
    \o Source
    
    \row
    \o 1
    \o data1
    
    \row
    \o 2
    \o data2
    
    \row
    \o 3
    \o data3
    
    \row
    \o 4
    \o data4[0] .. data4[1]
    
    \row
    \o 5
    \o data4[2] .. data4[7]

    \endtable
*/
QString QUuid::toString() const
{
    QString result;

    QChar dash = QLatin1Char('-');
    result = QLatin1Char('{') + uuidhex(data1,8);
    result += dash;
    result += uuidhex(data2,4);
    result += dash;
    result += uuidhex(data3,4);
    result += dash;
    result += uuidhex(data4[0],2);
    result += uuidhex(data4[1],2);
    result += dash;
    for (int i = 2; i < 8; i++)
        result += uuidhex(data4[i],2);

    return result + QLatin1Char('}');
}
#endif

#ifndef QT_NO_DATASTREAM
/*!
    \relates QUuid
    Writes the UUID \a id to the data stream \a s.
*/
QDataStream &operator<<(QDataStream &s, const QUuid &id)
{
    s << (quint32)id.data1;
    s << (quint16)id.data2;
    s << (quint16)id.data3;
    for (int i = 0; i < 8; i++)
        s << (quint8)id.data4[i];
    return s;
}

/*!
    \relates QUuid
    Reads a UUID from the stream \a s into \a id.
*/
QDataStream &operator>>(QDataStream &s, QUuid &id)
{
    quint32 u32;
    quint16 u16;
    quint8 u8;
    s >> u32;
    id.data1 = u32;
    s >> u16;
    id.data2 = u16;
    s >> u16;
    id.data3 = u16;
    for (int i = 0; i < 8; i++) {
        s >> u8;
        id.data4[i] = u8;
    }
    return s;
}
#endif // QT_NO_DATASTREAM

/*!
    Returns true if this is the null UUID
    {00000000-0000-0000-0000-000000000000}; otherwise returns false.
*/
bool QUuid::isNull() const
{
    return data4[0] == 0 && data4[1] == 0 && data4[2] == 0 && data4[3] == 0 &&
           data4[4] == 0 && data4[5] == 0 && data4[6] == 0 && data4[7] == 0 &&
           data1 == 0 && data2 == 0 && data3 == 0;
}

/*!
    \enum QUuid::Variant

    This enum defines the values used in the \l{Variant field}
    {variant field} of the UUID. The value in the variant field
    determines the layout of the 128-bit value.

    \value VarUnknown Variant is unknown
    \value NCS Reserved for NCS (Network Computing System) backward compatibility
    \value DCE Distributed Computing Environment, the scheme used by QUuid
    \value Microsoft Reserved for Microsoft backward compatibility (GUID)
    \value Reserved Reserved for future definition
*/

/*!
    \enum QUuid::Version

    This enum defines the values used in the \l{Version field}
    {version field} of the UUID. The version field is meaningful
    only if the value in the \l{Variant field} {variant field}
    is QUuid::DCE.

    \value VerUnknown Version is unknown
    \value Time Time-based, by using timestamp, clock sequence, and
    MAC network card address (if available) for the node sections
    \value EmbeddedPOSIX DCE Security version, with embedded POSIX UUIDs
    \value Name Name-based, by using values from a name for all sections
    \value Random Random-based, by using random numbers for all sections
*/

/*!
    \fn QUuid::Variant QUuid::variant() const

    Returns the value in the \l{Variant field} {variant field} of the
    UUID. If the return value is QUuid::DCE, call version() to see
    which layout it uses. The null UUID is considered to be of an
    unknown variant.

    \sa version()
*/
QUuid::Variant QUuid::variant() const
{
    if (isNull())
        return VarUnknown;
    // Check the 3 MSB of data4[0]
    if ((data4[0] & 0x80) == 0x00) return NCS;
    else if ((data4[0] & 0xC0) == 0x80) return DCE;
    else if ((data4[0] & 0xE0) == 0xC0) return Microsoft;
    else if ((data4[0] & 0xE0) == 0xE0) return Reserved;
    return VarUnknown;
}

/*!
    \fn QUuid::Version QUuid::version() const

    Returns the \l{Version field} {version field} of the UUID, if the
    UUID's \l{Variant field} {variant field} is QUuid::DCE. Otherwise
    it returns QUuid::VerUnknown.

    \sa variant()
*/
QUuid::Version QUuid::version() const
{
    // Check the 4 MSB of data3
    Version ver = (Version)(data3>>12);
    if (isNull()
         || (variant() != DCE)
         || ver < Time
         || ver > Random)
        return VerUnknown;
    return ver;
}

/*!
    \fn bool QUuid::operator<(const QUuid &other) const

    Returns true if this QUuid has the same \l{Variant field}
    {variant field} as the \a other QUuid and is lexicographically
    \e{before} the \a other QUuid. If the \a other QUuid has a
    different variant field, the return value is determined by
    comparing the two \l{QUuid::Variant} {variants}.

    \sa variant()
*/
#define ISLESS(f1, f2) if (f1!=f2) return (f1<f2);
bool QUuid::operator<(const QUuid &other) const
{
    if (variant() != other.variant())
        return variant() < other.variant();

    ISLESS(data1, other.data1);
    ISLESS(data2, other.data2);
    ISLESS(data3, other.data3);
    for (int n = 0; n < 8; n++) {
        ISLESS(data4[n], other.data4[n]);
    }
    return false;
}

/*!
    \fn bool QUuid::operator>(const QUuid &other) const

    Returns true if this QUuid has the same \l{Variant field}
    {variant field} as the \a other QUuid and is lexicographically
    \e{after} the \a other QUuid. If the \a other QUuid has a
    different variant field, the return value is determined by
    comparing the two \l{QUuid::Variant} {variants}.

    \sa variant()
*/
#define ISMORE(f1, f2) if (f1!=f2) return (f1>f2);
bool QUuid::operator>(const QUuid &other) const
{
    if (variant() != other.variant())
        return variant() > other.variant();

    ISMORE(data1, other.data1);
    ISMORE(data2, other.data2);
    ISMORE(data3, other.data3);
    for (int n = 0; n < 8; n++) {
        ISMORE(data4[n], other.data4[n]);
    }
    return false;
}

/*!
    \fn QUuid QUuid::createUuid()

    On any platform other than Windows, this function returns a new
    UUID with variant QUuid::DCE and version QUuid::Random. The random
    numbers used to construct the UUID are obtained from the local
    pseudo-random generator, qrand(), which is usually not a cryptographic
    quality random number generator. Therefore, a UUID generated by
    this function can't be guaranteed to be unique. If the pseudo-random
    number generator for the calling thread has not yet been seeded, this
    function will seed the pseudo-random number generator by calling qsrand().

    On a Windows platform, a GUID is generated, which almost certainly
    \e{will} be unique, on this or any other system, networked or not.

    \sa variant(), version()
*/
#if defined(Q_OS_WIN32) && ! defined(Q_CC_MWERKS)

QT_BEGIN_INCLUDE_NAMESPACE
#include <objbase.h> // For CoCreateGuid
QT_END_INCLUDE_NAMESPACE

QUuid QUuid::createUuid()
{
    GUID guid;
    CoCreateGuid(&guid);
    QUuid result = guid;
    return result;
}

#else // !Q_OS_WIN32

QT_BEGIN_INCLUDE_NAMESPACE
#include "qdatetime.h"
#include "stdlib.h" // For srand/rand
QT_END_INCLUDE_NAMESPACE

extern void qsrand(); // in qglobal.cpp

QUuid QUuid::createUuid()
{
    static const int intbits = sizeof(int)*8;
    static int randbits = 0;
    if (!randbits) {
        int max = RAND_MAX;
        do { ++randbits; } while ((max=max>>1));
    }

    // reseed, but only if not already seeded
    qsrand();

    QUuid result;
    uint *data = &(result.data1);
    int chunks = 16 / sizeof(uint);
    while (chunks--) {
        uint randNumber = 0;
        for (int filled = 0; filled < intbits; filled += randbits)
            randNumber |= qrand()<<filled;
         *(data+chunks) = randNumber;
    }

    result.data4[0] = (result.data4[0] & 0x3F) | 0x80;        // UV_DCE
    result.data3 = (result.data3 & 0x0FFF) | 0x4000;        // UV_Random

    return result;
}
#endif // !Q_OS_WIN32

/*!
    \fn bool QUuid::operator==(const GUID &guid) const

    Returns true if this UUID is equal to the Windows GUID \a guid;
    otherwise returns false.
*/

/*!
    \fn bool QUuid::operator!=(const GUID &guid) const

    Returns true if this UUID is not equal to the Windows GUID \a
    guid; otherwise returns false.
*/

QT_END_NAMESPACE
